package LL;

#
# Anders Waananen <waananen@nbi.dk>
#

use File::Basename;
use lib dirname($0);
@ISA = ('Exporter');

# Module implements these subroutines for the LRMS interface

@EXPORT_OK = ('cluster_info',
              'queue_info',
              'jobs_info',
              'users_info');

use LogUtils ( 'start_logging', 'error', 'warning', 'debug' ); 
use strict;

##########################################
# Saved private variables
##########################################

our(%lrms_queue);

##########################################
# Private subs
##########################################

sub get_cpu_distribution($) { 

	my ( $path ) = shift;

	# FIX ME !!
	my $single_job_per_box = 1;

        if ($single_job_per_box == 1) {

        # Without hyperthreading
	    unless (open LLSTATUSOUT,  "$path/llstatus -f %sta|") {
		error("Error in executing llstatus");
	    }

	} else {

        # Use all available cpus/cores including hyperthreading:
	    unless (open LLSTATUSOUT,  "$path/llstatus -r %cpu %sta|") {
		error("Error in executing llstatus");
	    }
	}

	my %cpudist;
	while (<LLSTATUSOUT>) {
		chomp;
		# We only want CPU lines (and there must be at least one)
		next if !/^[1-9]/;
		# An empty line denotes end of CPUs
		last if /^$/;

		my $cpus;
		my $startd;

		if ($single_job_per_box == 1) {
		    ($startd) = split/\!/;
		    $cpus = 1;
		} else {
		    ($cpus, $startd) = split/\!/;
		}

		# Only count those machines which have startd running
		if ($startd != "0") {
		    $cpudist{$cpus}++;
		}
	}

	close LLSTATUSOUT;

	return %cpudist;
}

sub get_used_cpus($) { 

	my ( $path ) = shift;

	unless (open LLSTATUSOUT,  "$path/llstatus -f %r |") {
		error("Error in executing llstatus");
    	}

	my $cpus_used;
	my $cpus_queued;

##       Sum over all startd's
#	$usedcpus = 0;
#	while (<LLSTATUSOUT>) {
#		chomp;
#		# We only want CPU lines (and there must be at least one)
#		next if !/^[1-9]/;
#		# An empty line denotes end of CPUs
#		last if /^$/;
#		my $cpus = $_;
#		$usedcpus = $usedcpus + $cpus;
#	}
#	close LLSTATUSOUT;

#	Or simply use the summary:
	while (<LLSTATUSOUT>) {
		chomp;
		# We only want CPU lines (and there must be at least one)
		next if !/^Total Machines/;
		tr / //s;
		my @fields = split;
		$cpus_used   = $fields[6];
		$cpus_queued = $fields[4] - $cpus_used;
		last;
	}

	close LLSTATUSOUT;

	return ($cpus_used, $cpus_queued);
}

sub get_long_status($) { 

	my ( $path ) = shift;

	unless (open LLSTATUSOUT,  "$path/llstatus -l |") {
		error("Error in executing llstatus");
	}

	my %cpudist;
	my $machine_name;
	my %machines;

	while (<LLSTATUSOUT>) {
		# Discard trailing information separated by a newline
		if ( /^$/ ) {
		    next;
		}
		chomp;
		my ($par, $val) = split/\s*=\s*/,$_,2;
		if ($par eq 'Name') {
		    $machine_name=$val;
		    next;
		}
		$machines{$machine_name}{$par} = $val;
	}
	close LLSTATUSOUT;

	return %machines;
}


sub get_long_queue_info($) { 

	my ( $path ) = shift;

	unless (open LLCLASSOUT,  "$path/llclass -l |") {
		error("Error in executing llclass");
	}

	my %queue_info;
	my $queue_name;

	while (<LLCLASSOUT>) {
		# Discard trailing information separated by a newline and header
		if ( /^$/ || /^==========/ ) {
		    next;
		}

		# Info ends with a line of dashes
		last if /^----------/;

		s/^\s*//;
		chomp;
		my ($par, $val) = split/\s*:\s*/,$_,2;

		if ($par eq 'Name') {
		    $queue_name=$val;
		    next;
		}

		$queue_info{$queue_name}{$par} = $val;
	}

	close LLCLASSOUT;

	return %queue_info;
}

sub get_queues($) {

	my ( $path ) = shift;

	unless (open LLCLASSOUT,  "$path/llclass |") {
		error("Error in executing llclass");
	}

	# llclass outputs queues (classes) delimited by ---- markers

	my @queues;
	my $queue_sect;
	while (<LLCLASSOUT>) {

		# Now reading queues
		if ( /^----------/ && $queue_sect == 0) {
		    if ($#queues == -1) {
				$queue_sect = 1;
				next;
			}
		}

		# Normal ending after reading final queue
		if ( /^----------/ && $queue_sect == 1) {
			$queue_sect = 0;
			return @queues;
		}

		if ( $queue_sect == 1 ) {
			chomp;
			s/ .*//;
			push @queues, $_;
		}

	}

	# We only end here if there were no queues
	return @queues;
}

sub get_short_job_info($$) { 

    # Path to LRMS commands
    my ($path) = shift;
    # Name of the queue to query
    my ($queue) = shift;

    if ($queue != "") {
	unless (open LLQOUT, "$path/llq -c $queue |") {
	    error("Error in executing llq");
	}
    } else {
	unless (open LLQOUT, "$path/llq |") {
	    error("Error in executing llq");
	}
    }

    my %jobstatus;

    while (<LLQOUT>) {

        my ($total, $waiting, $pending, $running, $held, $preempted);

        if (/(\d*) .* (\d*) waiting, (\d*) pending, (\d*) running, (\d*) held, (\d*) preempted/) {
          $total     = $1;
          $waiting   = $2;
          $pending   = $3;
          $running   = $4;
          $held      = $5;
          $preempted = $6;
        }
	$jobstatus{total}     = $total;
	$jobstatus{waiting}   = $waiting;
        $jobstatus{pending}   = $pending;
	$jobstatus{running}   = $running;
	$jobstatus{held}      = $held;
	$jobstatus{preempted} = $preempted;
    }
    close LLQOUT;
    return %jobstatus;
}


sub get_long_job_info($$$) { 

    # Path to LRMS commands
    my ($path) = shift;
    # Name of the queue to query
    my ($queue) = shift;
    # LRMS job IDs from Grid Manager (jobs with "INLRMS" GM status)
    my ($lrms_ids) = @_;

    my $lrmsidstr = join(" ", @{$lrms_ids});

    if ($queue == "") {
	unless (open LLQOUT, "$path/llq -l $lrmsidstr |") {
	    error("Error in executing llq");
	}
    } else {
	unless (open LLQOUT, "$path/llq -c $queue |") {
	    error("Error in executing llq");
	}
    }

    my $jobid;
    my $skip=0;
    my %jobinfo;

    while (<LLQOUT>) {

        # Discard trailing information separated by a newline
        if ( /^$/ ) {
            last;
        }

	# Discard header lines
	if (/^===/) {
	    $skip=0;
	    next;
	}
	if ($skip == 1) {
	    next;
	}
	# Drop extra info
        if (/^------/) {
	    $skip=1;
	}
	chomp;
	# Create variables using text before colon
	my ($par, $val) = split/: */,$_,2;
	$par =~ s/^ *//g;
	$par =~ s/ /_/g;
	# Assign variables
	if ($par eq 'Job_Step_Id') {
	    $jobid = $val;
	    next;
	}
		$jobinfo{$jobid}{$par} = $val;
    }
    close LLQOUT;
    return %jobinfo;
}

############################################
# Public subs
#############################################

sub cluster_info ($) {

    # Path to LRMS commands
    my ($path) = shift;

    # Return data structure %lrms_cluster{$keyword}
    # should contain the keyvords listed in LRMS.pm namely:
    #
    # lrms_type          LRMS type (eg. LoadLeveler)
    # lrms_version       LRMS version
    # totalcpus          Total number of cpus in the system
    # queuedcpus         Number of cpus requested in queueing jobs in LRMS ???????????
    # usedcpus           Used cpus in the system
    # cpudistribution    CPU distribution string
    # queue              Names of the LRMS queues
    #
    # All values should be defined, empty values "" are ok if field
    # does not apply to particular LRMS.

    my (%lrms_cluster);

    # lrms_type
    $lrms_cluster{lrms_type} = "LoadLeveler";

    # lrms_version
    my $status_string=`$path/llstatus -v`;
    if ( $? != 0 ) {    
	warning("Can't run llstatus");
    }
    $status_string =~ /^\S+\s+(\S+)/;
    $lrms_cluster{lrms_version} = $1;

	# cpudistribution
    # totalcpus
    $lrms_cluster{cpudistribution} = "";
    $lrms_cluster{totalcpus} = 0;
    my %cpudist = get_cpu_distribution($path);
    my $sep = "";
    foreach my $key (keys %cpudist) {
		$lrms_cluster{cpudistribution} .= $sep.$key."cpu:".$cpudist{$key};
	    if ($sep == "") {
			$sep = " ";
		}
		$lrms_cluster{totalcpus} += $key * $cpudist{$key};
    }

    # usedcpus
#    ($lrms_cluster{usedcpus},$lrms_cluster{queuedcpus}) = get_used_cpus($path);
    my %jobstatus       = get_short_job_info($path,"");
    $lrms_cluster{usedcpus}   = $jobstatus{running} + $jobstatus{held} + $jobstatus{preempted};
    $lrms_cluster{queuedcpus} = $jobstatus{waiting} + $jobstatus{pending};;

    # queues
    @{$lrms_cluster{queue}} = get_queues($path);

    return %lrms_cluster;
}

sub queue_info ($$) {

    # Path to LRMS commands
    my ($path) = shift;

    # Name of the queue to query
    my ($queue) = shift;

    # The return data structure is %lrms_queue.
    # In this template it is defined as persistent module data structure,
    # because it is later used by jobs_info(), and we wish to avoid
    # re-construction of it. If it were not needed later, it would be defined
    # only in the scope of this subroutine, as %lrms_cluster previously.
    # This is just an example of a possible implementation detail, not
    # required by the LRMS interface.

    # Return data structure %lrms_queue{$keyword}
    # should contain the keyvords listed in LRMS.pm:
    #
    # status        available slots in the queue, negative number signals
    #               some kind of LRMS error state for the queue
    # maxrunning    queue limit for number of running jobs
    # maxqueuable   queue limit for number of queueing jobs
    # maxuserrun    queue limit for number of running jobs per user
    # maxcputime    queue limit for max cpu time for a job
    # mincputime    queue limit for min cpu time for a job
    # defaultcput   queue default for cputime
    # running       number of procs used by running jobs in the queue
    # queued        number of procs requested by queueing jobs in the queue
    # totalcpus     number of procs in the queue

    #
    # All values should be defined, empty values "" are ok if field
    # does not apply to particular LRMS.

    my %long_queue_info = get_long_queue_info($path);
    my %jobstatus       = get_short_job_info($path,$queue);

    # Translate between LoadLeveler and ARC

    $lrms_queue{status} = $long_queue_info{$queue}{'Free_slots'};
    $lrms_queue{maxrunning} = $long_queue_info{$queue}{'Maximum_slots'};

    $lrms_queue{maxqueuable} = "";
    $lrms_queue{maxuserrun} = $lrms_queue{maxrunning};

    # Note we use  Wall Clock!
    $_ = $long_queue_info{$queue}{'Wall_clock_limit'};
    if (/\((.*) seconds,/) {
	$lrms_queue{maxcputime} = $1 / 60;
    }

    # There is no lower limit enforced
    $lrms_queue{mincputime} = 0;

    $_ = $long_queue_info{$queue}{'Def_wall_clock_limit'};
    if (/\((.*) seconds,/) {
	$lrms_queue{defaultcput} = $1 / 60;
    }

    $lrms_queue{running} = $jobstatus{running} + $jobstatus{held} + $jobstatus{preempted};
    $lrms_queue{queued}  = $jobstatus{waiting} + $jobstatus{pending};
    $lrms_queue{totalcpus} =  $long_queue_info{$queue}{'Max_processors'};
    
    return %lrms_queue;
}

sub jobs_info ($$$) {

    # Path to LRMS commands
    my ($path) = shift;
    # Name of the queue to query
    my ($queue) = shift;
    # LRMS job IDs from Grid Manager (jobs with "INLRMS" GM status)
    my ($lrms_ids) = @_;

    # status        Status of the job: Running 'R', Queued'Q',
    #                                  Suspended 'S', Exiting 'E', Other 'O'
    # rank          Position in the queue
    # mem           Used (virtual) memory
    # walltime      Used walltime
    # cputime       Used cpu-time
    # reqwalltime   Walltime requested from LRMS
    # reqcputime    Cpu-time requested from LRMS
    # node          Execution host, master node in case of parallel jobs
    # comment       Comment about the job in LRMS, if any

    my (%lrms_jobs);

    my %jobinfo = get_long_job_info($path,"",$lrms_ids);

    foreach my $id (keys %jobinfo) {
	if (
	    $jobinfo{$id}{Status} == "Completed" ||
	    $jobinfo{$id}{Status} == "Canceled"  ||
	    $jobinfo{$id}{Status} == "Removed"
	    ) {
	}
	$lrms_jobs{$id}{mem} = -1;
	my $dispt = `date +%s -d "$jobinfo{$id}{Dispatch_Time}\n"`;
	chomp $dispt;
	$lrms_jobs{$id}{walltime} = POSIX::ceil((time() - $dispt) /60),"\n";
	$lrms_jobs{$id}{cputime} = -1;
	$lrms_jobs{$id}{reqwalltime} = $jobinfo{$id}{Wall_Clk_Hard_Limit};
	$lrms_jobs{$id}{reqcputime} = -1;
	$lrms_jobs{$id}{comment} = $jobinfo{$id}{Status};
    }

    return %lrms_jobs;
}

sub users_info($$@) {

    my ($path) = shift;
    my ($qname) = shift;
    my ($accts) = shift;

    my (%lrms_users);

    # freecpus
    # queue length
    #
    # Using simple estimate independent of users

    if ( ! exists $lrms_queue{status} ) {
        %lrms_queue = queue_info( $path, $qname );
    }

    foreach my $u ( @{$accts} ) {
        $lrms_users{$u}{freecpus} = $lrms_queue{status};
        $lrms_users{$u}{queuelength} = "$lrms_queue{queued}";
    }
    return %lrms_users;

}

1;
