package config_parser;

use strict;
use warnings;

BEGIN {
    use base 'Exporter';

    # Set the version for version checking.
    our $VERSION = '1.000';
    our @EXPORT = qw(
        config_parse
        config_set_defaults
        config_get_option
        config_get_blocknames
        config_get_blocks
        config_has_block
    );
}

my %data=(_defaults_ => []);
my @blocknames=("_defaults_");

#
# Sets default settings.
# Expects a hash array of settings.
# It creates a _defaults_ block, similar
# to the common block in the config file.
#
sub config_set_defaults(%) {
    my %defaults = @_;
    delete $data{_defaults_};
    $data{_defaults_}{$_} = $defaults{$_} for keys %defaults;
}

#
# Expects the filename of the arc.conf config file. 
# Returns undef if it cannot open the file.
#
sub config_parse($) {
    my $blockname;
    my $arcconf=shift;

    # Clear old data
    delete $data{$_} for grep { $_ ne "_defaults_" } keys %data; 
    @blocknames=("_defaults_");

    unless (open (CONFIGFILE, $arcconf)) {
        return undef;
    }
    while (my $line =<CONFIGFILE>) {
       next if $line =~/^\s*#/;
       next if $line =~/^\s*$/;
    
       if ($line =~/^\s*\[(.+)\]/ ) {
          $blockname = $1;
          unshift @blocknames, $blockname;
          next;
       }
       # Single or double quotes can be used. Quotes are removed from the values
       next unless $line =~ /^(\w+)\s*=\s*(["'])(.*)(\2)\s*$/;
       my $key=$1;
       my $value=$3;
       if (not defined $data{$blockname}{$key}) {
          $data{$blockname}{$key} = $value;
       }
       else {
          $data{$blockname}{$key} .= "[separator]".$value;
       }
    }
    close CONFIGFILE;
}

#
# Returns the list of all blocks found in the config file in reverse order.  A
# single hash containing all the options in the config file can be generated
# like this:
# %config = config_get_blocks(config_get_blocknames());
#
sub config_get_blocknames() {
    return grep {$_ ne "_defaults_"} @blocknames;
}

#
# Checks whether a block is present in the config file.
#
sub config_has_block() {
    my $blockname = shift;
    return grep {$_ eq $blockname} @blocknames;
}

#
# Expects the name of an option from the config file, followed by a list of
# config block names.  The specified blocks are searched consecutively, until
# the option is found.  The first found value of that option is returned.  If
# option was not found in the listed blocks, or if no blocks were listed, the
# common block is also searched.
#
sub config_get_option {
    my $option = shift;
    foreach my $blockname (@_, "common", "_defaults_") {
#       next unless defined $data{$blockname};
        my $val = $data{$blockname}{$option};
        return $val if defined $val;
    }
    return undef;
}

#
# Expects a list of block names from the configuration file and returns a
# single hash array with all the options defined in those blocks. Options
# defined in the common block are also included. The order of blocks is
# important: Options in blocks listed first mask same-name options in latter
# blocks.  
#
sub config_get_blocks(@) {
    my %config = ();
    foreach my $bn (reverse @_, "common", "_defaults_") {
#     return undef unless defined $data{$bn};
      $config{$_}=$data{$bn}{$_} for (keys %{$data{$bn}});
    }
    return %config;
} 

# test
#config_set_defaults(gm_port => 2811, homogeneity => 'true');
#config_parse('/etc/arc.conf') or die 'Cannot parse config file';
#config_get_option('lrms');
#config_get_option('nodecpu','queue/large');
#config_get_option('nodecpu','queue/large','infosys');
#%config = config_get_blocks('queue/large','infosys');
#%config = config_get_blocks(config_get_blocknames());

1;
