#!/usr/bin/perl 
use Getopt::Long;
use lib "$ENV{NORDUGRID_LOCATION}/libexec";
use lib "/opt/nordugrid/libexec";
use infosys_shared;

#
# generates everything under the nordugrid-queue-name=xz. creates the queue entry, job & user entries
#  

my %config;
GetOptions("dn:s" => \$config{dn},	     	  
	   "queue:s" => \$config{queue}, 
	   "config:s" => \$conf_file, 
	   "valid-to:i" => \$config{ttl},   
	   "loglevel:i" => \$loglevel,
	   "help|h" => \$print_help   
	  ); 
 
if ($print_help) { 
    print "\n  
     		script usage: 
		mandatory arguments: --dn
				     --queue
				     --config
				      
		optional arguments:  --valid-to 
		
		this help:	     --help				  		
		\n";
		exit;
}

if (! ($config{dn} and $config{queue} and $conf_file)) {
    $loglevel and  &infosys_shared::write_log("a command line argument is missing, see --help ");
    die "a command line argument is missing, see --help "
};


#
#  default values
#
$config{ttl}           ||= 600;
$config{gm_mount_point}  = "/jobs";
$config{gm_port}         = 2811;
$config{gridmap} 	 = "/etc/grid-security/grid-mapfile";
chomp ($config{hostname} = `/bin/hostname -f`);

#
# initialization from the info.conf 
#

my $queue_block;
unless (open (CONFIGFILE, "<$conf_file")) {
   $loglevel and  &infosys_shared::write_log("can't open $conf_file configuration file");
   die "Can't open $conf_file configuration file\n";
}

while (my $line =<CONFIGFILE>) {
   next if $line =~/^#/;   
   $line=~m/^(\w+)\s*=\s*"(.+)"\s*$/;

   ($1 eq "hostname")	  	    and   $config{$1} = $2;
   ($1 eq "controldir") 	    and   $config{$1} = $2;
   ($1 eq "gm_mount_point")	    and   $config{$1} = $2;
   ($1 eq "gm_port")		    and   $config{$1} = $2;
   ($1 eq "gridmap")		    and   $config{$1} = $2;	  	          
   ($1 eq "sessiondir") 	    and   $config{$1} = $2; 
   ($1 eq "dedicated_node_string")  and   $config{$1} = $2; 
   ($1 eq "maui_bin_path")	    and   $showbf_command = "$2/showbf" ; 
   ($1 eq "pbs_bin_path")	    and   ( $pbsnodes_command="$2/pbsnodes" and
					    $qstat_command="$2/qstat" and 
					    $qmgr_command="$2/qmgr" 
        				  );	           
   if ($1 eq "queue_name" and $2 eq $config{queue}) {	       
       $queue_block = "true";
   }

   if ($queue_block) {
       ($1 eq "totalcpus")          and    $config{$1}=$2;
       ($1 eq "nodecpu")  	    and    $config{$1}=$2;
       ($1 eq "nodememory")  	    and    $config{$1}=$2;
       ($1 eq "architecture")  	    and    $config{$1}=$2;
       ($1 eq "opsys") 		    and    push @opsys, $2;	       
       ($1 eq "queue_node_string")  and    $config{$1} = $2; 
       ($1 eq "scheduling_policy")  and    $config{$1} = $2;
       ($1 eq "comment") 	    and    $config{$1} = $2;  
       ($line =~ /^$/) and $queue_block = "";
   }	

}	
close CONFIGFILE;

#
# read the mappings from the grid-mapfile to a %users hash, the SN is the key
#

unless (open MAPFILE, "<$config{gridmap}") {
   $loglevel and  &infosys_shared::write_log("can't open gridmapfile");
   die "can't open gridmapfile";
}   
while(my $line = <MAPFILE>) {
    chomp($line);
    if ($line =~ m/\"([^"]+)\"\s+(\S+)/) {
        $users{$1}=$2; 	
    }
}
close MAPFILE;

#
# if sessiondir ="*" read $HOME of the local unix users from "passwd" into the %homedirs{$uid} 
#
my %homedirs;
if ($config{"sessiondir"} =~ /^\s*\*\s*$/) {   
   unless (open PASSWD, "</etc/passwd") {
      $loglevel and  &infosys_shared::write_log("can't open passwd file");
      die "can't open passwd file";
   }	     
   while  (my $line= <PASSWD>) {
       my @passwd_fields = split (":", $line);  
       $homedirs{$passwd_fields[0]}=$passwd_fields[5];
   }
   close PASSWD;
}


# determine the flavour and version of PBS
my $qmgr_string=`$qmgr_command -c "list server"`;
if ($? != 0 and $loglevel) {    
    &infosys_shared::write_log("Can't run qmgr");
}
$qmgr_string =~ /pbs_version = \b(\D+)_(\d\S+)\b/;
my $pbs_flavour=$1;
my $pbs_version=$2;

   
#
# read the queue information for the queue entry from the qstat
#

unless (open QSTATOUTPUT,   "$qstat_command -f -Q $config{queue} |") {
    $loglevel and  &infosys_shared::write_log("Error in executing qstat");
    die "Error in executing qstat\n";
}

while (my $line= <QSTATOUTPUT>) {       
      if ($line =~ m/ = /) {
      	 chomp($line);	 
      	 my ($qstat_var,$qstat_value) = split("=", $line);	
	 $qstat_var =~ s/\s+//g;      	 
      	 $qstat{$qstat_var}=$qstat_value;
      }
}	    
close QSTATOUTPUT;

#
# read the job info from the PBS qstat -f to a hash of hashes %hoh_pbsjobs


unless (open QSTATOUTPUT,  "$qstat_command -f $config{queue} |") { 
    $loglevel and  &infosys_shared::write_log("Error in executing qstat");
    die "Error in executing qstat\n";
}

my ($jobid, $job_var, $job_value, @d);
while (my $line= <QSTATOUTPUT>) { 
      chomp($line); 
      next if ($line =~ /^$/); 
      if ($line =~ /^.*Job Id:/) {            
         $line =~ s/.*Job Id:\s*//;
	 $line =~ s/\s*$//;
	 @d = split(/\./, $line);
	 $jobid = "$d[0].$d[1]";
	 next;
      }
      if ($line =~ / = /)  {
          ($job_var,$job_value) = split (/ = /, $line);
	  $job_var =~ s/^\s+//g;  
      }
      else {                     
          $line =~ s/^\s*//;   
          $job_value = $job_value.$line;	  
      }   
      $hoh_pbsjobs{$jobid}{$job_var}=$job_value;     
} 
close QSTATOUTPUT;


#processing the pbsnodes output by using a hash of hashes %hoh_pbsnodes
my %hoh_pbsnodes;
&infosys_shared::read_pbsnodes($pbsnodes_command, \%hoh_pbsnodes);

if ($loglevel == 2) {  
   my $runtime = time - $^T; 
   &infosys_shared::write_log("PBS+GRIDMAP I/O time: $runtime");  
}


# loop over the PBS jobs (both grid and nongrid) and 
# count the number of running (plus exiting) jobs with their multiciplity  
# count the jobs per unix users and the number of queueing jobs per users in the queue


my $queue_rank = 0;
my $totalrunning_in_the_queue = 0;
my $runningjobs_in_the_queue = 0; 
my %user_jobs_queued;
my %user_jobs_running;
foreach my $jobid (sort {$a <=> $b} keys %hoh_pbsjobs) {
    if ( $hoh_pbsjobs{$jobid}{"Resource_List.ncpus"} ) {	         
    	$job_ncpus = $hoh_pbsjobs{$jobid}{"Resource_List.ncpus"};		         
    }
    elsif ( $hoh_pbsjobs{$jobid}{"Resource_List.nodes"} ) {          
    	$job_ncpus = $hoh_pbsjobs{$jobid}{"Resource_List.nodes"};			         
    }
    elsif ( defined $hoh_pbsjobs{$jobid}{"Resource_List.neednodes"}) {      
    	$job_ncpus = $hoh_pbsjobs{$jobid}{"Resource_List.neednodes"}			         
    }						         
    else {
    	$job_ncpus = 1;
    }
    if ( $hoh_pbsjobs{$jobid}{"Job_Owner"} ) {
        $username = $hoh_pbsjobs{$jobid}{"Job_Owner"};
	$username =~ s/@.*$//;
    }
    if ( ($hoh_pbsjobs{$jobid}{"job_state"} eq "R") or ($hoh_pbsjobs{$jobid}{"job_state"} eq "E") ) {
       $totalrunning_in_the_queue += $job_ncpus;
       $runningjobs_in_the_queue++;
       $user_jobs_running{$username}++;
    }
    if ( $hoh_pbsjobs{$jobid}{"job_state"} eq "Q" ) {
       $user_jobs_queued{$username}++;
       $queue_rank++;
       $hoh_pbsjobs{$jobid}{"rank"} = $queue_rank;
    }          
}   


# determine @acl_users, queue limits from the $qstat{$qstat_var} hash
if (defined ($qstat{max_running})) {
    $max_running = $qstat{max_running};
}      
if (defined ($qstat{max_user_run})) {
    $max_userrun = $qstat{max_user_run};
}
if (defined ($qstat{resources_max.cput})) {
    my @time=split(":",$qstat{resources_max.cput});
    $max_cputime = ":".($time[0]*60+$time[1]);
}
if (defined ($qstat{acl_user_enable})) {
    $acl_enable = $qstat{acl_user_enable};
}
if ($acl_enable eq "True" && defined ($qstat{acl_users})) {
    @acl_users = split (/\,/, $qstat{acl_users});
}


# calculate the totalcpus & number of used cpus
# taking into account the minor output difference of OpenPBS and PBSPRO
# skipping the nondedicated nodes in case the $dedicated_node_string or the
# queue_node_string is defined
$totalcpus=0;
$usedcpus=0;
my $npstring;
if (lc($pbs_flavour) eq "openpbs"){
   $npstring="np" 
}
elsif (lc($pbs_flavour) eq "spbs"){
   $npstring="np"
}
elsif (lc($pbs_flavour) eq "torque"){
   $npstring="np"
}
elsif (lc($pbs_flavour) eq "pbspro"){
   $npstring="pcpus"
}
else {
   $loglevel and  &infosys_shared::write_log("the given flavour of PBS is not supported");
   die "The given flavour of the PBS is not supported"
};  

foreach my $node (keys %hoh_pbsnodes){
   if ($config{dedicated_node_string}){ 						      
      next unless ($hoh_pbsnodes{$node}{"properties"} =~ m/$config{dedicated_node_string}/)     			 
   }											      
   if ($config{queue_node_string}){							      
       next unless ( $hoh_pbsnodes{$node}{"properties"} =~ m/$config{queue_node_string}/ or
       		     $hoh_pbsnodes{$node}{"queue"} =~ m/$config{queue_node_string}/)        	      
   }	  										      
   my $nodestate=$hoh_pbsnodes{$node}{"state"};      					      
   if ($nodestate=~/down/ or $nodestate=~/offline/ or $nodestate=~/[^-]busy/) {next};	      
   $totalcpus += $hoh_pbsnodes{$node}{$npstring};       				      
   if ($hoh_pbsnodes{$node}{"jobs"}){							      
     $usedcpus++;									      
     my @comma = ($hoh_pbsnodes{$node}{"jobs"}=~ /,/g); 				      
     $usedcpus+=@comma;
   } 											      
}      


#
# read the list of jobs from the jobdir and create the @gridmanager_jobs 
# the @gridmanager_jobs contains the IDs from the job.ID.status
#

$timestamp = time;
unless (opendir JOBDIR,  $config{controldir}) {
   $loglevel and  &infosys_shared::write_log("can't access the directory of the jobstatus files");
   die "Can't access the directory of the jobstatus files\n";
}  

my @allfiles= readdir JOBDIR;
@allfiles= grep /\.status/, @allfiles;
closedir JOBDIR;
@gridmanager_jobs = map {$_=~m/job\.(.+)\.status/; $_=$1;} @allfiles;


#
# read the gridmanager jobinfo into a hash of hashes %hoh_gmjobs 
# filter out jobs not belonging to this $queue
#

my $controldir = $config{controldir};
foreach $ID (@gridmanager_jobs) { 
    my $gmjob_local=$controldir."/job.".$ID.".local";
    my $gmjob_status=$controldir."/job.".$ID.".status";
    my $gmjob_failed=$controldir."/job.".$ID.".failed";  
    my $gmjob_description=$controldir."/job.".$ID.".description"; 
    my $gmjob_diag=$controldir."/job.".$ID.".diag"; 
    
    open (GMJOB_LOCAL, "<$gmjob_local");
    my @local_allines=<GMJOB_LOCAL>;
    
    # check that this job belongs to this queue  
    my $the_right_queue= grep /^queue=$config{queue}$/, @local_allines;
    if (!$the_right_queue) {
       close GMJOB_LOCAL;
       next;
    }
    
    # parse the content of the job.ID.local into the %hoh_gmjobs hash 
    foreach my $line (@local_allines) {		  
  	$line=~m/^(\w+)=(.+)$/; 		   	
  	$hoh_gmjobs{$ID}{$1}=$2;    
    }	 
    close GMJOB_LOCAL;

    # read the job.ID.status into "status"
    open (GMJOB_STATUS, "<$gmjob_status");
    my @status_allines=<GMJOB_STATUS>;    
    chomp (my $job_status_firstline=@status_allines[0]);    
    $hoh_gmjobs{$ID}{"status"}= $job_status_firstline;
    close GMJOB_STATUS;    
    
    if ($hoh_gmjobs{$ID}{"status"} eq "FINISHED") {
      	my @file_stat = stat $gmjob_status;       	  
      	($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime (@file_stat[9]);
        my $file_time = sprintf "%4d%02d%02d%02d%02d%02d%1s", $year+1900,$mon+1,$mday,$hour,$min,$sec,"Z";    
      	$hoh_gmjobs{$ID}{"status"} .= " at: $file_time";	
    }  

    # check for job failure, (job.ID.failed )   "errors"
    if (-e $gmjob_failed) {
       open (GMJOB_FAILED, "<$gmjob_failed");
       my @failed_allines=<GMJOB_FAILED>;    
       chomp (my $job_failed_firstline=@failed_allines[0]);    
       $hoh_gmjobs{$ID}{"errors"}="JOB FAILURE: $job_failed_firstline";
       close GMJOB_FAILED;	        
    } 
    
    # do something with the job.ID.description file
    # read the stdin,stdout,stderr 
    open (GMJOB_DESC, "<$gmjob_description");
    my @desc_allines=<GMJOB_DESC>;    
    chomp (my $rsl_string=@desc_allines[0]);      
    if ($rsl_string=~m/"stdin"\s+=\s+"(\S+)"/) {
    	$hoh_gmjobs{$ID}{"stdin"}=$1;
    }
    if ($rsl_string=~m/"stdout"\s+=\s+"(\S+)"/) {
    	$hoh_gmjobs{$ID}{"stdout"}=$1;	
    }
    if ($rsl_string=~m/"stderr"\s+=\s+"(\S+)"/) {
    	$hoh_gmjobs{$ID}{"stderr"}=$1;		
    }
    if ($rsl_string=~m/"count"\s+=\s+"(\S+)"/) {
    	$hoh_gmjobs{$ID}{"count"}=$1;	
    }
    else {
        $hoh_gmjobs{$ID}{"count"}="1";
    }
    if ($rsl_string=~m/"cputime"\s+=\s+"(\S+)"/i) {
    	$hoh_gmjobs{$ID}{"reqcputime"}=$1;	
    }      
    # the runtimeenvironment is multivalued, needs further processing at printout
    if ($rsl_string=~m/\("runtimeenvironment"\s+=\s+([^\)]+)/i) {
    	$hoh_gmjobs{$ID}{"runtimeenvironment"}=$1;	
    }
    close GMJOB_DESC;
     
    #read the job.ID.diag file        
    if (-s $gmjob_diag) {
       open (GMJOB_DIAG, "<$gmjob_diag");
          while ($line = <GMJOB_DIAG>) {
	    $line=~m/nodename=(\S+)/     and   $hoh_gmjobs{$ID}{"exec_host"}.=$1."+";
	    $line=~m/WallTime=(\S+)\./   and   $hoh_gmjobs{$ID}{"WallTime"}=int($1/60);
	    $line=~m/KernelTime=(\S+)\./ and   $kerneltime=$1;
	    $line=~m/UserTime=(\S+)\./	 and   $usertime=$1;  
	  }	
 	$hoh_gmjobs{$ID}{"CpuTime"}= int(($kerneltime+$usertime)/60); 
	close GMJOB_DIAG;	  		
   } 	 
}

if ($loglevel == 2) {   
   my $runtime = time - $timestamp; 
   &infosys_shared::write_log("GM I/O time: $runtime");
}

#
#Count the running and queued GRID jobs (a subset of the PBS jobs)
#running jobs are counted with their multiciplity  
#

$gridrunning = 0;
$gridqueued=  0;
foreach my $ID (keys %hoh_gmjobs){
	 next unless  $hoh_gmjobs{$ID}{"status"} eq "INLRMS";
	 $pbsjobid=$hoh_gmjobs{$ID}{"localid"};	 
	 if ($hoh_pbsjobs{$pbsjobid}{"job_state"} eq "R"){	      
	     $gridrunning+=$hoh_gmjobs{$ID}{"count"};
	 }
	 if ($hoh_pbsjobs{$pbsjobid}{"job_state"} eq "E"){	      
	     $gridrunning+=$hoh_gmjobs{$ID}{"count"};
	 }	 
	 if ($hoh_pbsjobs{$pbsjobid}{"job_state"} eq "Q"){
	     $gridqueued++;
	 }                 
}


#
# The queue entry 
#
print "dn: $config{dn}\n";

# The objectclasses
print "objectclass: Mds\n";
print "objectclass: nordugrid-queue\n";

# The attributes
print "nordugrid-queue-name: $config{queue}\n";

# nordugrid-queue-status
my $gm_running = `pgrep grid-manager`;
if ($? != 0) {
    $gm_running = `ps -C grid-manager  --no-heading`;     
    if ($? != 0) {
        $loglevel and   &infosys_shared::write_log("Failed checking the grid-manager process");        	
    }   
}

my $gridftp_running = `pgrep gridftpd`;
if ($? != 0) {
    $gridftp_running = `ps -C gridftpd --no-heading`;     
    if ($? != 0) {
        $loglevel and   &infosys_shared::write_log("Failed checking the gridftpd process");        	
    }   
}

if (not $gm_running) {
    print "nordugrid-queue-status: inactive, grid-manager is down\n";   
}
elsif  (  !($gridftp_running) )  {
   print "nordugrid-queue-status: inactive, gridftp is down\n";  
}   
elsif ( !( ($qstat{"enabled"} eq " True") and ($qstat{"started"} eq " True"))) {
	print "nordugrid-queue-status: inactive\n";
}
else {
	print "nordugrid-queue-status: active\n";
}

# nordugrid-queue-running
print "nordugrid-queue-running: $totalrunning_in_the_queue\n";

#nordugrid-queue-gridrunning
print "nordugrid-queue-gridrunning: $gridrunning\n";

# nordugrid-queue-queued
$qstat{"state_count"}=~ m/Queued:(\d+)/;
print "nordugrid-queue-queued: $1\n";

#nordugrid-queue-gridqueued
print "nordugrid-queue-gridqueued: $gridqueued\n";

# nordugrid-queue-maxrunning
$qstat{"max_running"} and print "nordugrid-queue-maxrunning:", $qstat{"max_running"}, "\n";

# nordugrid-queue-maxinqueue
$qstat{"max_queuable"} and print "nordugrid-queue-maxqueuable:", $qstat{"max_queuable"}, "\n";

# nordugrid-queue-maxuserrun
$qstat{"max_user_run"} and print "nordugrid-queue-maxuserrun:", $qstat{"max_user_run"}, "\n";

# nordugrid-queue-maxcputime
if ($qstat{"resources_max.cput"}) {
    @time=split(":",$qstat{"resources_max.cput"});
    $in_minutes=$time[0]*60+$time[1];
    print "nordugrid-queue-maxcputime: ", $in_minutes, "\n";
}
# nordugrid-queue-mincput
if ($qstat{"resources_min.cput"}) {
   @time=split(":",$qstat{"resources_min.cput"});
   $in_minutes=$time[0]*60+$time[1]+1;
   print "nordugrid-queue-mincputime: ",$in_minutes , "\n";
}

# nordugrid-queue-defaultcputime
if ($qstat{"resources_default.cput"}) {
    @time=split(":",$qstat{"resources_default.cput"});
    $in_minutes=$time[0]*60+$time[1];
    print "nordugrid-queue-defaultcputime: ", $in_minutes, "\n";
}

# nordugrid-queue-schedulingpolicy
$config{scheduling_policy} and print "nordugrid-queue-schedulingpolicy: $config{scheduling_policy}\n";

# nordugrid-queue-comment
$config{comment} and print "nordugrid-queue-comment: $config{comment}\n";

# nordugrid-queue-totalcpus
if ($config{queue_node_string}) {
   print "nordugrid-queue-totalcpus: $totalcpus\n";
}
elsif ( $config{totalcpus}) {
   print "nordugrid-queue-totalcpus: $config{totalcpus}\n";
}   


# nordugrid-queue-nodecpu
$config{nodecpu} and print "nordugrid-queue-nodecpu: $config{nodecpu}\n";

# nordugrid-queue-nodememory
$config{nodememory} and print "nordugrid-queue-nodememory: $config{nodememory}\n";

# nordugrid-queue-architecture
$config{architecture} and print "nordugrid-queue-architecture: $config{architecture}\n";

#nordugrid-queue-opsys
foreach my $listentry (@opsys) {
   print "nordugrid-queue-opsys: $listentry\n";
}

# Mds-validfrom/to
&infosys_shared::mds_valid($config{ttl});

# The nordugrid-info-group=jobs entry
print "dn: nordugrid-info-group-name=jobs, $config{dn}\n"; 
print "objectclass: Mds\n";
print "objectclass: nordugrid-info-group\n";
print "nordugrid-info-group-name: jobs\n";
# Mds-validfrom/to
&infosys_shared::mds_valid($config{ttl});


#
# The job entries
#

# jobs are sorted in a descending order with respect to their jobsubmission time
foreach my $ID (sort {$hoh_gmjobs{$b}{"starttime"} <=> $hoh_gmjobs{$a}{"starttime"}} keys %hoh_gmjobs){
     
     my $pbsjobid=$hoh_gmjobs{$ID}{"localid"};
     my $nordugrid_globalid="gsiftp://".$config{hostname}.":".$config{gm_port}.$config{gm_mount_point}."/".$ID; 	
     print "dn: nordugrid-job-globalid=$nordugrid_globalid, nordugrid-info-group-name=jobs, $config{dn}\n";
     print "objectclass: Mds\n";
     print "objectclass: nordugrid-job\n";   
     print "nordugrid-job-globalid: $nordugrid_globalid\n";
     print "nordugrid-job-globalowner: ", $hoh_gmjobs{$ID}{"subject"}, "\n";
     if ( $hoh_gmjobs{$ID}{"jobname"}) {
         print "nordugrid-job-jobname: ", $hoh_gmjobs{$ID}{"jobname"}, "\n";
     }	
     print "nordugrid-job-submissiontime: ",$hoh_gmjobs{$ID}{"starttime"}, "\n";              
     print "nordugrid-job-execcluster: ", $config{hostname}, "\n";
     print "nordugrid-job-execqueue: ",$config{queue}, "\n";
     #nordugrid-job-executionnodes
     if ( $hoh_gmjobs{$ID}{"exec_host"}) { 
	 my @exec_hosts = split('\+',$hoh_gmjobs{$ID}{"exec_host"});
	 foreach my $listentry (@exec_hosts) {
	    print "nordugrid-job-executionnodes: $listentry\n";
	 }
     }	     
     #nordugrid-job-status
     if ($hoh_gmjobs{$ID}{"status"} eq "INLRMS") {
         if ($hoh_pbsjobs{"$pbsjobid"}{"job_state"}) {   
              print "nordugrid-job-status: INLRMS: ",     
              $hoh_pbsjobs{"$pbsjobid"}{"job_state"}, "\n";
	 }
	 else {      
	      print "nordugrid-job-status: FINISHING\n";     
         }     	      
	 if ($hoh_pbsjobs{$pbsjobid}{"rank"}) {  
              print "nordugrid-job-queuerank: ", $hoh_pbsjobs{$pbsjobid}{"rank"}, "\n"; 
	 }     
         if ($hoh_pbsjobs{$pbsjobid}{"resources_used.cput"}) {
	     $usedcputime_tmp=$hoh_pbsjobs{$pbsjobid}{"resources_used.cput"};
	     $usedcputime_tmp=~/(^\d+):(\d+):/;	   
	     print "nordugrid-job-usedcputime: ",$1*60+$2, "\n"; 
	 }
         if ($hoh_pbsjobs{$pbsjobid}{"resources_used.walltime"}) {
	    $usedwall_tmp=$hoh_pbsjobs{$pbsjobid}{"resources_used.walltime"};	 
	    $usedwall_tmp=~/(^\d+):(\d+):/;	    
	    print "nordugrid-job-usedwalltime: ",$1*60+$2, "\n" ;	   
	} 
	if ( $hoh_pbsjobs{$pbsjobid}{"resources_used.mem"}) {
            print "nordugrid-job-usedmem: ", $hoh_pbsjobs{$pbsjobid}{"resources_used.mem"}-"kb", "\n";
	}
	if ($hoh_pbsjobs{$pbsjobid}{"Resource_List.cput"}) {
            @time=split(":",$hoh_pbsjobs{$pbsjobid}{"Resource_List.cput"});
            $in_minutes=$time[0]*60+$time[1];
            print "nordugrid-job-reqcput: $in_minutes\n";		
	}
	#nordugrid-job-executionnodes	
	if ($hoh_pbsjobs{$pbsjobid}{"exec_host"}) {
	    my @exec_hosts = split('\+',$hoh_pbsjobs{$pbsjobid}{"exec_host"});
	    foreach my $listentry (@exec_hosts) {
	       print "nordugrid-job-executionnodes: $listentry\n";
	    }
	}
	
	if ($hoh_pbsjobs{$pbsjobid}{"comment"}) {	 
	    print "nordugrid-job-lrmscomment:", $hoh_pbsjobs{$pbsjobid}{"comment"}, "\n";    
	}    
     }
     else {
         print "nordugrid-job-status: ", $hoh_gmjobs{$ID}{"status"}, "\n";
	 if ($hoh_gmjobs{$ID}{"WallTime"}) {
             print "nordugrid-job-usedwalltime: ", $hoh_gmjobs{$ID}{"WallTime"}, "\n";
	 }
	 if ($hoh_gmjobs{$ID}{"CpuTime"})  {   
	     print "nordugrid-job-usedcputime: ", $hoh_gmjobs{$ID}{"CpuTime"}, "\n";
	 }
	 if ($hoh_gmjobs{$ID}{"reqcputime"})  {   
	     print "nordugrid-job-reqcput: ", $hoh_gmjobs{$ID}{"reqcputime"}, "\n";
	 }    
     }
     if ($hoh_gmjobs{$ID}{"count"}) {
         print "nordugrid-job-cpucount: ", $hoh_gmjobs{$ID}{"count"}, "\n";
     }
     else {
         print "nordugrid-job-cpucount: 1\n";
     }  
     if ($hoh_gmjobs{$ID}{"errors"}) {
         print "nordugrid-job-errors: ", $hoh_gmjobs{$ID}{"errors"}, "\n";
     }
     if ($hoh_gmjobs{$ID}{"cleanuptime"}) {
         print "nordugrid-job-sessiondirerasetime: ",$hoh_gmjobs{$ID}{"cleanuptime"}, "\n";
     } 
     if ($hoh_gmjobs{$ID}{"stdin"}) {
         print "nordugrid-job-stdin: ",$hoh_gmjobs{$ID}{"stdin"}, "\n";
     }
     if ($hoh_gmjobs{$ID}{"stdout"}) {
         print "nordugrid-job-stdout: ",$hoh_gmjobs{$ID}{"stdout"}, "\n";
     } 
     if ($hoh_gmjobs{$ID}{"stderr"}) {
         print "nordugrid-job-stderr: ",$hoh_gmjobs{$ID}{"stderr"}, "\n";
     }
     if ($hoh_gmjobs{$ID}{"gmlog"}) {
         print "nordugrid-job-gmlog: ",$hoh_gmjobs{$ID}{"gmlog"}, "\n";
     }     
     if (my $tmpruntimestring = $hoh_gmjobs{$ID}{"runtimeenvironment"}) {          
         while( $tmpruntimestring =~ m/"(\S+)"/g ) {
	    print "nordugrid-job-runtimeenvironment: ",$1, "\n";    
	 }                      
     }
     if ($hoh_gmjobs{$ID}{"clientname"}) {
         print "nordugrid-job-submissionui: ",$hoh_gmjobs{$ID}{"clientname"}, "\n";
     }              	 
     if ($hoh_gmjobs{$ID}{"clientsoftware"}) {
         print "nordugrid-job-clientsoftware: ",$hoh_gmjobs{$ID}{"clientsoftware"}, "\n";
     } 
     if ($hoh_gmjobs{$ID}{"delegexpiretime"}) {
         print "nordugrid-job-proxyexpirationtime: ",$hoh_gmjobs{$ID}{"delegexpiretime"}, "\n";
     }         
     # Mds-validfrom/to
     &infosys_shared::mds_valid($config{ttl});
}

#
# generates the user entries under the  nordugrid-info-group-name=users
#


# calculate the diskspace on the sessiondir, %gridareas{$user} is filled if "sessiondir=*",
# otherwise %gridareas{"common"} set to sessiondir

my %gridareas;
my %diskspaces;
if ($config{"sessiondir"} =~ /^\s*\*\s*$/) {
     while ( ($key,$user) = each %users ) { 	 
         next unless ($gridareas{$user} eq "");
	 if ($homedirs{$user} eq "") {
	    &infosys_shared::write_log("$user is not listed in the passwd file");
	    $gridareas{$user} = "";
	    next;
	 }
         $gridareas{$user} = $homedirs{$user}."/.jobs";	 
     }
}
else {
    $gridareas{"common"} = $config{sessiondir};
}  


foreach $user (keys %gridareas) {
    my $gridarea = $gridareas{$user};
    if ( -d "$gridarea") {						 
      # check if on afs 						 
      if ($gridarea =~ /\/afs\//) {					 
    	    $diskspace =`fs listquota $gridarea 2>/dev/null`;	
	    if ($? != 0) {
                &infosys_shared::write_log("Failed checking diskspace for user=$user, gridarea=$gridarea");
		$diskspace = 0;
	    }	
    	    if ($diskspace) {						
    	      $diskspace =~ /\n\S+\s+(\d+)\s+(\d+)\s+\d+%\s+\d+%/;	
    	      $diskspace = int (($1 - $2)/1024);			
    	    } else {							
    	      $diskspace = 0;						
    	    }								
      # "ordinary" disk 						 
      } else {  							 
    	    $diskspace =`df -k $gridarea 2>/dev/null`;    	    	
	    if ($? != 0) {                
		&infosys_shared::write_log("Failed checking diskspace for user=$user, gridarea=$gridarea");
		$diskspace = 0;
	    }
    	    if ($diskspace) {						
    	      $diskspace =~ /\n\S+\s+\d+\s+\d+\s+(\d+)\s+\d+/;  	
    	      $diskspace=int $1/1024;					
    	    } else {
    	      $diskspace = 0;						
    	    }
      } 								 
    } else {								 
      $diskspace = 0;							 
      $loglevel and &infosys_shared::write_log("sessiondir $gridarea was not found");   
    }
    $diskspaces{$user} = $diskspace;
}


#
# in case of a Maui scheduler fill a hash (%unix_mappings) with the unix usernames as keys
# and freecpu info as value
#    

if (lc($config{scheduling_policy}) eq "maui") {
   for ( values(%users)) {
     $unix_mappings{$_}="";   
   } 
#commented out for the temporary solution below    
#    for ( keys %unix_mappings) {
#      open showbfoutput,   "$showbf_command -u  $_ |" or die "Error in executing showbf";
#      while ($line= <showbfoutput>) {  
#    
#      	if ($line =~ /no procs available/) {
#      	  $unix_mappings{$_}= " 0";
#      	# print "no procs\n";
#      	  last; 
#      	} 
#      	if ($line =~ /(\d+).+available for\s+([\w:]+)/) {
# 	  @tmp= reverse split /:/, $2;	  
# 	  $minutes=$tmp[1] + 60*$tmp[2] + 24*60*$tmp[3];
# 	#  print  @tmp,"\t", $minutes, "\n";	  
#      	  $unix_mappings{$_} .= " ".$1.":".$minutes;
#      	#  print "$1 available for $2\n";
#      	}
#      	if ($line =~ /(\d+).+available with no timelimit/) {
#      	  $unix_mappings{$_}.= " ".$1;
#      	#  print "$1 procs with no limit\n";
#      	}	
#      }
#      close  showbfoutput; 
#    }    

   #start temporary solution to speed up the showbf, checks only a single user   	  
   unless (open SHOWBFOUTPUT,	"$showbf_command  |") {
      $loglevel and  &infosys_shared::write_log("error in executing showbf");
      die "Error in executing showbf";
   }     
   while (my $line= <SHOWBFOUTPUT>) {						  
      if ($line =~ /^partition/) {						  
   	last;									  
      }   									  
      if ($line =~ /no procs available/) {					  
   	$maui_freecpus= " 0";      						  
   	last; 									  
      } 									  
      if ($line =~ /(\d+).+available for\s+([\w:]+)/) { 			  
   	@tmp= reverse split /:/, $2;	
   	$minutes=$tmp[1] + 60*$tmp[2] + 24*60*$tmp[3];  		
   	$maui_freecpus .= " ".$1.":".$minutes;	      			  
      } 									  
      if ($line =~ /(\d+).+available with no timelimit/) {			  
   	$maui_freecpus.= " ".$1;     					  
   	last; 
      }       									  
   }										  
   close  SHOWBFOUTPUT; 							  
   for ( keys %unix_mappings) {    						  
   	$unix_mappings{$_}=$maui_freecpus;					  
   }										  
   #end temporary solution							  
}

# The nordugrid-info-group=users entry 
print "dn: nordugrid-info-group-name=users, $config{dn}\n";
print "objectclass: Mds\n";
print "objectclass: nordugrid-info-group\n";
print "nordugrid-info-group-name: users\n";
# Mds-validfrom/to
&infosys_shared::mds_valid($config{ttl});

#
# create the user entries for those members of the hash
# which are PBS allowed
#

my $usernumber = 0;
foreach my $key (sort sort_by_cn keys %users) {

   if ( @acl_users and !(grep /$users{$key}/, @acl_users)) {
   #   print "we skip $users{$key} \n";
      next
   }
   

   #nordugrid-authuser-name= CN from the SN  + unique number 
   $key =~ m/\/CN=([^\/]+)(\/Email)?/;
   my $cn = $1;
   $usernumber++;
   print "dn: nordugrid-authuser-name=", $cn, "_", $usernumber, ", nordugrid-info-group-name=users, ", $config{dn}, "\n";
   print "objectclass: Mds\n";
   print "objectclass: nordugrid-authuser\n";
   print "nordugrid-authuser-name: ", $cn, "_", $usernumber, "\n";
   print "nordugrid-authuser-sn: $key\n";
   
 
   #nordugrid-authuser-freecpus
   if (lc($config{scheduling_policy}) eq "maui") {
      print "nordugrid-authuser-freecpus:", $unix_mappings{$users{$key}}, "\n";
   }
   else {   
      $systemlimit = $totalcpus - $usedcpus;
      if  ($max_running) {   
           $queuelimit =  $max_running - $runningjobs_in_the_queue;
      }    
      else  { $queuelimit =  $systemlimit; }
      if  ($max_userrun) {   
             $userlimit  =  $max_userrun - $user_jobs_running{$users{$key}};
      }
      else { $userlimit  =  $systemlimit; }    
   
      #print "SYSTEM:$systemlimit\n";
      #print "QUEUE:$queuelimit\n";;
      #print "USER:$userlimit\n";   
         
      @sorted_limits=sort {$a<=>$b}($systemlimit, $queuelimit, $userlimit);
  
      if ($sorted_limits[0] >0 ) {
         print "nordugrid-authuser-freecpus: $sorted_limits[0]$max_cputime\n";
      }
      else {
         print "nordugrid-authuser-freecpus: 0\n";
      }
   }
      
       
   #nordugrid-authuser-queuelength 
   if ($user_jobs_queued{$users{$key}}) {
       $queuelength=$user_jobs_queued{$users{$key}};
   }
   else { $queuelength=0};
   print "nordugrid-authuser-queuelength: ",$queuelength , "\n";   
   
   #nordugrid-authuser-diskspace
   if ($config{"sessiondir"} =~ /^\s*\*\s*$/) {
      print "nordugrid-authuser-diskspace: $diskspaces{$users{$key}}\n";
   }
   else {      
      print "nordugrid-authuser-diskspace: ", $diskspaces{"common"},"\n";
   }
   
   #Mds-validfrom/to
   &infosys_shared::mds_valid($config{ttl});
}



my $runtime =  time - $^T;
if ($loglevel == 2) {
   &infosys_shared::write_log("execution time: $runtime");
}
elsif ($loglevel == 1 and $runtime >= 4 ) {  
   &infosys_shared::write_log("SLOW script: $runtime");  
}


#sort by CN
sub sort_by_cn{
   $a =~ m/\/CN=([^\/]+)(\/Email)?/;
   $cn_a=$1;
   $b =~ m/\/CN=([^\/]+)(\/Email)?/;
   $cn_b=$1;   
   $cn_a cmp $cn_b;
}
