#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <arc/certificate.h>
#include <arc/datetime.h>
#include <arc/globusmodules.h>
#include <arc/stringconv.h>
#include <arc/globuserrorutils.h>

#include <globus_gsi_credential.h>
#include <globus_gsi_system_config.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


Certificate::Certificate(certtype type,
                         std::string filename) throw(CertificateError) {

	GlobusGSISysconfigModule module;
	GlobusGSIGSSAPIModule module2;

	GlobusResult res;

	if (filename.empty()) {
		char* cert_path = NULL;
		char* key_path = NULL;

		if (type == PROXY) {
			res = GLOBUS_GSI_SYSCONFIG_GET_PROXY_FILENAME(&cert_path,
			                                              GLOBUS_PROXY_FILE_INPUT);
			if (!cert_path) {
				throw CertificateError(
					_("Could not determine location of a proxy certificate") + (": " + res.str()));
			}
		}

		else if (type == USERCERT) {
			res = GLOBUS_GSI_SYSCONFIG_GET_USER_CERT_FILENAME(&cert_path, &key_path);
			if (!cert_path) {
				throw CertificateError(
					_("Could not determine location of a user certificate") + (": " + res.str()));
			}
		}

		else if (type == HOSTCERT) {
			res = GLOBUS_GSI_SYSCONFIG_GET_HOST_CERT_FILENAME(&cert_path, &key_path);
			if (!cert_path) {
				throw CertificateError(
					_("Could not determine location of a host certificate") + (": " + res.str()));
			}
		}

		else {
			throw CertificateError(
			      _("Could not determine location of a certificate"));
		}

		filename = cert_path;
		free(cert_path);
		if (key_path)
			free(key_path);
	}

	cert_filename = filename;
	cert_type = type;

	RetrieveCertInfo(cert_filename);
}


void Certificate::RetrieveCertInfo(std::string path) throw(CertificateError) {

	GlobusGSICredentialModule module;
	GlobusGSIGSSAPIModule module2;

	globus_gsi_cred_handle_t cred;
	char* subject;

	if (globus_gsi_cred_handle_init(&cred, NULL) != GLOBUS_SUCCESS)
		throw CertificateError(
			_("Could not initialize credential handle"));

	if (globus_gsi_cred_read_cert(cred, (char*)path.c_str()) != GLOBUS_SUCCESS) {
		ERR_clear_error();
		throw CertificateError(
			_("Could not read certificate") + (": " + path));
	}
	ERR_clear_error();

	if (globus_gsi_cred_get_subject_name(cred, &subject) != GLOBUS_SUCCESS)
		throw CertificateError(
			_("Could not get a valid subject name from the certificate"));
	sn = subject;
	OPENSSL_free(subject);

	if (globus_gsi_cred_get_issuer_name(cred, &subject) != GLOBUS_SUCCESS)
		throw CertificateError(
			_("Could not get a valid issuer name from the certificate"));
	issuer_sn = subject;
	OPENSSL_free(subject);

	if (globus_gsi_cred_get_identity_name(cred, &subject) != GLOBUS_SUCCESS)
		throw CertificateError(
		    _("Could not get a valid identity name from the certificate"));

	identity_sn = subject;
	OPENSSL_free(subject);

	time_t exp;
	if (globus_gsi_cred_get_goodtill(cred, &exp) != GLOBUS_SUCCESS)
		throw CertificateError(
			_("Could not get a valid lifetime for the certificate"));
	expires = exp;

	globus_gsi_cred_handle_destroy(cred);
}


std::string Certificate::ConvertSN(std::string sn, SNFormat format) {

	std::string::size_type pos = 0;

	switch (format) {
		case PLAIN:
			while ((pos = sn.find("\\x", pos)) != std::string::npos) {
				try {
					sn.replace(pos, 4, 1,
					           stringtoi("0x" + sn.substr(pos + 2, 2)));
					pos++;
				} catch (StringConvError e) {
					// Illegal \xHH sequence
					pos += 2;
				}
			}
			break;
		case X509:
			break;
		case LDAP1:
			while ((pos = sn.find_first_of("()*\\", pos))!=std::string::npos) {
				if (sn[pos] == '\\' && sn[pos + 1] == 'x') {
					sn.erase(pos + 1, 1);
					pos += 1;
				} else {
					sn.insert(pos, 1, '\\');
					pos += 2;
				}
			}
			break;
		case LDAP2:
			while ((pos = sn.find_first_of("()*\\", pos))!=std::string::npos) {
				sn.insert(pos, 1, '\\');
				pos += 2;
			}
			break;
	}
	return sn;
}


Certificate Certificate::GetIssuerCert() const throw(CertificateError) {

	if (cert_type == PROXY) {
		Certificate user;
		if (user.sn != issuer_sn)
			throw CertificateError(
				_("Could not locate issuer certificate of proxy"));
		return user;
	}

	if (cert_type == USERCERT || cert_type == HOSTCERT) {
		std::list<Certificate> ca_list = GetCAList();

		std::list<Certificate>::iterator it;
		for (it = ca_list.begin(); it != ca_list.end(); it++) {
			if (it->sn == issuer_sn) {
				break;
			}
		}

		if (it == ca_list.end())
			throw CertificateError(
				_("Could not find corresponding issuer CA certificate"));

		return (*it);
	}

	throw CertificateError(_("Unknown certificate type"));
}


std::string Certificate::GetIssuerSN(SNFormat format) const {
	return ConvertSN(issuer_sn, format);
}


std::string Certificate::GetIdentitySN(SNFormat format) const {
	return ConvertSN(identity_sn, format);
}


std::string Certificate::GetSN(SNFormat format) const {
	return ConvertSN(sn, format);
}


Time Certificate::Expires() const {
	return expires;
}


std::string Certificate::ExpiryTime() const {
	return expires.str();
}


std::string Certificate::ValidFor() const {
	if (IsExpired())
		return _("expired");
	else
		return Period(expires.GetTime() - time(NULL));
}


std::string Certificate::GetCertFilename() const {
	return cert_filename;
}


bool Certificate::IsExpired () const {
	return (expires.GetTime() - time(NULL) < 0);
}


bool Certificate::IsSelfSigned() const {
	return (issuer_sn == sn);
}


certtype Certificate::GetCertificateType() const {
	return cert_type;
}

std::list<Certificate> GetCAList() {

	GlobusGSISysconfigModule module;

	// caching list of CA-certificates
	static std::list<Certificate> certs;
	if (certs.size()>0) return certs;

	char* cert_dir;
	if (GLOBUS_GSI_SYSCONFIG_GET_CERT_DIR(&cert_dir) != GLOBUS_SUCCESS)
		return certs;

	globus_fifo_t ca_cert_files;
	globus_fifo_init(&ca_cert_files);
	if (GLOBUS_GSI_SYSCONFIG_GET_CA_CERT_FILES(cert_dir, &ca_cert_files)
	    != GLOBUS_SUCCESS) {
		free(cert_dir);
		return certs;
	}

	free(cert_dir);

	char* cert_filename;
	while ((cert_filename = (char*)globus_fifo_dequeue(&ca_cert_files))) {
		Certificate newcert(USERCERT, cert_filename);
		certs.push_back(newcert);
	}

	globus_fifo_destroy(&ca_cert_files);
	return certs;
}


bool CheckIssuer(std::string issuer) {

	std::list<Certificate> listofcerts = GetCAList();

	std::list<Certificate>::iterator it;
	for (it = listofcerts.begin(); it != listofcerts.end(); it++) {
		if (it->GetSN() == issuer)
			break;
	}

	if (it == listofcerts.end())
		return false;

	return true;
}


Certificate GetEffectiveCredential() throw(CertificateError) {

	try {
		Certificate proxy(PROXY);
		if (!proxy.IsExpired()) return proxy;
	} catch (CertificateError e) { }

	try {
		Certificate usercert(USERCERT);
		if (!usercert.IsExpired()) return usercert;
	} catch (CertificateError e) { }

	throw CertificateError(_("Neither a valid proxy- nor user-certificate "
	                         "was found."));
}


std::string GetEffectiveSN(Certificate::SNFormat format)
throw(CertificateError) {

	Certificate cert = GetEffectiveCredential();

	// if (cert.GetCertificateType()==PROXY)
	// 	return cert.GetIssuerSN(format);

	return cert.GetIdentitySN(format);
}
