#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <fstream>
#include <iostream>
#include <string>

#include <arc/common.h>
#include <arc/jobftpcontrol.h>
#include <arc/jobsubmission.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/standardbrokers.h>
#include <arc/stringconv.h>
#include <arc/target.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif

JobSubmission::JobSubmission(Xrsl axrsl,
                             std::list<Target> targetlist,
                             bool adryrun) : targets(targetlist),
                                             xrsl(axrsl),
                                             dryrun(adryrun) {}


std::string JobSubmission::Submit(int timeout)
throw(XrslError, JobSubmissionError) {

	if (targets.empty())
		throw JobSubmissionError(_("No targets available for job-submission"));

	std::list<Target>::iterator targit;
	std::string jobid;
	for (targit = targets.begin(); targit != targets.end(); targit++) {

		notify(INFO) << _("Queue selected") << ": " << targit->name << "@"
		             << targit->cluster.hostname << std::endl;

		Xrsl jobxrsl;
		try {
			jobxrsl = PrepareXrsl(*targit);
		}
		catch (JobSubmissionError e) {
			notify(INFO) << e.what() << std::endl;
			continue;
		}

		if (dryrun && !jobxrsl.IsRelation("dryrun")) {
			XrslRelation dry("dryrun", operator_eq, "yes");
			jobxrsl.AddRelation(dry);
		}

		std::string contactstring = targit->cluster.contact;
		JobFTPControl ftpc;
		notify(DEBUG) << _("Submitting xrsl") << ": " << jobxrsl.str()
		              << std::endl;

		try {
			jobid = ftpc.Submit(contactstring,
			                    jobxrsl.str(),
			                    localinputfiles,
			                    timeout);
		}
		catch (FTPControlError e) {
			notify(INFO) << _("Job submission failed to queue") << " "
			             << targit->name << "@" << targit->cluster.hostname
			             << ": " << e.what() << std::endl
			             << _("Trying remaining queues") << std::endl;
			continue;
		}

		// save variables needed for RegisterJobsubmission
		jobtarget = targit;
		jobcount = 1;
		if (jobxrsl.IsRelation("count"))
			jobcount = stringtoi(jobxrsl.GetRelation("count").GetSingleValue());

		jobcputime = UNDEFINED;
		if (jobxrsl.IsRelation("cputime"))
			jobcputime = stringtoi(jobxrsl.GetRelation("cputime").GetSingleValue());

		jobdisk = 0;
		if (jobxrsl.IsRelation("disk"))
			jobdisk = stringtoll(jobxrsl.GetRelation("disk").GetSingleValue());

		return jobid;
	}

	throw JobSubmissionError(_("All targets rejected job requests"));
}


void JobSubmission::RegisterJobsubmission(std::list<Queue>& queues) {

	// host- and queue-name
	std::string hostname = jobtarget->cluster.hostname;
	std::string queuename = jobtarget->name;

	// Find corresponding queue in queue-list
	std::list<Queue>::iterator it;
	for (it = queues.begin(); it != queues.end(); it++) {
		if (it->cluster.hostname != hostname) continue;
		if (it->name != queuename) continue;
		break;
	}

	if (it==queues.end()) return; // not found

	// Corresponding queue in queuelist has been found. Now modify it.
	if (it->running<it->max_running) {
		it->running++;
	} else {
		it->queued++;
	}

	if (jobcputime==UNDEFINED) jobcputime = it->default_cpu_time;
	if (jobcputime==UNDEFINED) jobcputime = LONG_MAX;

	std::map<long,int>::iterator cpuit, cpuit2;
	std::list<User>::iterator userit;
	for (userit = it->users.begin(); userit != it->users.end(); userit++) {
		// diskspace
		userit->free_diskspace -= jobdisk*1024*1024;
		if (userit->free_diskspace<0) userit->free_diskspace = 0;
		notify(VERBOSE) << "User free diskspace is now: " << userit->free_diskspace << std::endl;

		// cputime
		cpuit = userit->free_cpus.lower_bound(jobcputime);
		if (cpuit != userit->free_cpus.end()) {
			if (jobcount>=cpuit->second) {
				cpuit->second = 0;
			} else {
				for (cpuit2 = userit->free_cpus.begin();
				     cpuit2 != userit->free_cpus.end(); cpuit2++) {
					if (cpuit2->first <= cpuit->first) {
						cpuit2->second -= jobcount;
					} else if (cpuit2->second >= cpuit->second) {
						cpuit2->second = cpuit->second;
						int oldkey = cpuit->first;
						cpuit++;
						userit->free_cpus.erase(oldkey);
					}
				}
			}

			if (cpuit->second==0) userit->free_cpus.erase(cpuit->first);

			if (userit->free_cpus.empty()) {
				if (it->max_cpu_time==UNDEFINED) {
					userit->free_cpus[LONG_MAX] = 0;
				} else {
					userit->free_cpus[it->max_cpu_time] = 0;
				}
			}
		}
	}

	
}


Xrsl JobSubmission::PrepareXrsl(Target& chosentarget)
throw(XrslError, JobSubmissionError) {

	Xrsl jobxrsl(chosentarget.GetXrsls().front());

	std::list<RuntimeEnvironment> middlewares =
		chosentarget.cluster.middlewares;

	RuntimeEnvironment nordugridenv("nordugrid-unknown");
	std::list<RuntimeEnvironment>::iterator runit;
	for (runit = middlewares.begin(); runit!=middlewares.end(); runit++) {
		if (runit->Name()=="nordugrid") nordugridenv = *runit;
		break;
	}

	// rewrite xrsl so it's ready for jobsubmission
	XrslRelation rel1("queue", operator_eq, chosentarget.name);
	jobxrsl.AddRelation(rel1);
	XrslRelation rel2("action", operator_eq, "request");
	jobxrsl.AddRelation(rel2);
	XrslRelation rel3("savestate", operator_eq, "yes");
	jobxrsl.AddRelation(rel3);
	XrslRelation rel4("clientxrsl", operator_eq, xrsl.str());
	jobxrsl.AddRelation(rel4);

	char hostname[1024];
	globus_libc_gethostname(hostname, 1024);
	XrslRelation rel5("hostname", operator_eq, hostname);
	jobxrsl.AddRelation(rel5);

	std::string version("nordugrid-arc-");
	version += VERSION;
	XrslRelation rel6("clientsoftware", operator_eq, version);
	jobxrsl.AddRelation(rel6);

	PrepareInputOutput(jobxrsl, "stdin", "inputfiles");
	PrepareInputOutput(jobxrsl, "stdout", "outputfiles");
	PrepareInputOutput(jobxrsl, "stderr", "outputfiles");
	PrepareInputOutput(jobxrsl, "gmlog", "outputfiles");

	XrslRelation rel7 = jobxrsl.GetRelation("executable");
	std::string exeval = rel7.GetSingleValue();

	if (exeval[0] != '/' && exeval[0] != '$') {

		std::list<std::list<std::string> > llstr;
		if (jobxrsl.IsRelation("inputfiles"))
			llstr = jobxrsl.GetRelation("inputfiles").GetDoubleListValue();

		bool found = false;
		for (std::list<std::list<std::string> >::iterator it = llstr.begin();
		     it != llstr.end(); it++) {
			if (it->front() == exeval) {
				found = true;
				break;
			}
		}

		if (!found) {
			struct stat st;
			stat(exeval.c_str(), &st);
			if (S_ISREG(st.st_mode)) {
				std::list<std::string> newlist;
				newlist.push_front("");
				newlist.push_front(exeval);
				llstr.push_back(newlist);
				if (jobxrsl.IsRelation("inputfiles"))
					jobxrsl.RemoveRelation("inputfiles");
				XrslRelation rel("inputfiles", operator_eq, llstr);
				jobxrsl.AddRelation(rel);

				// Add executable to executables
				std::list<std::string> exevals;
				if (jobxrsl.IsRelation("executables"))
					exevals = jobxrsl.GetRelation("executables").GetListValue();

				bool found = false;
				for (std::list<std::string>::iterator it = exevals.begin();
				     it != exevals.end(); it++) {
					if (*it == exeval) {
						found = true;
						break;
					}
				}

				if (!found) {
					exevals.push_front(exeval);
					if (jobxrsl.IsRelation("executables"))
						jobxrsl.RemoveRelation("executables");
					XrslRelation rel("executables", operator_eq, exevals);
					jobxrsl.AddRelation(rel);
				}
			}
			else if (nordugridenv < RuntimeEnvironment("nordugrid-0.5.42")) {
				throw JobSubmissionError(_("Server does not support "
							   "executable from PATH"));
			}
		}
	}

	if (nordugridenv < RuntimeEnvironment("nordugrid-0.5.26")) {

		// Add executable to arguments

		std::list<std::string> argvals;
		if (jobxrsl.IsRelation("arguments")) {
			argvals = jobxrsl.GetRelation("arguments").GetListValue();
			jobxrsl.RemoveRelation("arguments");
		}
		argvals.push_front(exeval);

		XrslRelation rel9("arguments", operator_eq, argvals);
		jobxrsl.AddRelation(rel9);

		jobxrsl.RemoveRelation("executable");
		XrslRelation rel10("executable", operator_eq, "/bin/echo");
		jobxrsl.AddRelation(rel10);
	}

	if (jobxrsl.IsRelation("join")) {
		XrslRelation joinrel = jobxrsl.GetRelation("join");
		std::string joinval = joinrel.GetSingleValue();
		if (joinval=="true" || joinval=="yes") {

			if (!jobxrsl.IsRelation("stdout"))
				throw XrslError(_("Xrsl attribute join is set but "
				                  "attribute stdout is not set"));
			if (jobxrsl.IsRelation("stderr"))
				throw XrslError(_("Xrsl attribute join is set but "
				                  "attribute stderr is also set"));

			XrslRelation out = jobxrsl.GetRelation("stdout");
			std::string outvalue = out.GetSingleValue();
			XrslRelation rel11("stderr", operator_eq, outvalue);
			jobxrsl.AddRelation(rel11);
		}
	}

	/*if (jobxrsl.IsRelation("stdin")) {
		XrslRelation stdin = jobxrsl.GetRelation("stdin");
		xrsl_operator op = stdin.GetOperator();
		std::string value = stdin.GetSingleValue();

		XrslRelation sstdin("sstdin", op, value);
		jobxrsl.RemoveRelation("stdin");
		jobxrsl.AddRelation(sstdin);
	}*/

	if (jobxrsl.IsRelation("starttime")) {
		XrslRelation starttime = jobxrsl.GetRelation("starttime");
		xrsl_operator op = starttime.GetOperator();

		try {
			Time start(starttime.GetSingleValue());
			XrslRelation newstarttime("starttime",
			                          op,
			                          TimeStamp(start, MDSTime));
			jobxrsl.RemoveRelation("starttime");
			jobxrsl.AddRelation(newstarttime);
		} catch(TimeError e) {
			throw XrslError(_("Illegal timeformat given in starttime"));
		}
	}

	if (jobxrsl.IsRelation("lifetime")) {
		XrslRelation lifetime = jobxrsl.GetRelation("lifetime");
		xrsl_operator op = lifetime.GetOperator();
		unsigned long value = Seconds(lifetime.GetSingleValue());

		XrslRelation newlifetime("lifetime", op, tostring(value));
		jobxrsl.RemoveRelation("lifetime");
		jobxrsl.AddRelation(newlifetime);
	}

	long walltime = chosentarget.GetCputime(jobxrsl);

	long cputime = walltime;
	if (jobxrsl.IsRelation("cputime"))
		cputime = Seconds(jobxrsl.GetRelation("cputime").GetSingleValue());

	if (jobxrsl.IsRelation("cputime"))
		jobxrsl.RemoveRelation("cputime");
	if (jobxrsl.IsRelation("walltime"))
		jobxrsl.RemoveRelation("walltime");
	if (jobxrsl.IsRelation("gridtime"))
		jobxrsl.RemoveRelation("gridtime");
	if (jobxrsl.IsRelation("benchmarks"))
		jobxrsl.RemoveRelation("benchmarks");

	if (cputime!=UNDEFINED) {
		if (nordugridenv < RuntimeEnvironment("nordugrid-0.5.27")) {
			// use minutes for pre 0.5.27 servers
			cputime = (long)((cputime+59.0)/60.0);
		}
		XrslRelation cpurel("cputime", operator_eq, tostring(cputime));
		jobxrsl.AddRelation(cpurel);
	}

	if (walltime!=UNDEFINED) {
		// pre 0.5.27 servers do not support the walltime attribute
		if (nordugridenv >= RuntimeEnvironment("nordugrid-0.5.27")) {
			XrslRelation wallrel("walltime", operator_eq, tostring(walltime));
			jobxrsl.AddRelation(wallrel);
		}
	}

	PrepareRE(jobxrsl,
	          "runtimeenvironment",
		  chosentarget,
	          nordugridenv < RuntimeEnvironment("nordugrid-0.5.40"));
	PrepareRE(jobxrsl,
	          "middleware",
	          chosentarget,
	          nordugridenv < RuntimeEnvironment("nordugrid-0.5.40"));
	PrepareRE(jobxrsl,
	          "opsys",
	          chosentarget,
	          nordugridenv < RuntimeEnvironment("nordugrid-0.5.40"));

	PrepareUpload(jobxrsl);

	return jobxrsl;
}


void JobSubmission::PrepareInputOutput(Xrsl& jobxrsl,
                                       const std::string& attr,
                                       const std::string& inoutattr) {

	if (!jobxrsl.IsRelation(attr)) return;
	std::string val = jobxrsl.GetRelation(attr).GetSingleValue();

	std::list<std::list<std::string> > llstr;
	if (jobxrsl.IsRelation(inoutattr)) {
		llstr = jobxrsl.GetRelation(inoutattr).GetDoubleListValue();
		jobxrsl.RemoveRelation(inoutattr);
	}

	bool found = false;
	std::list<std::list<std::string> >::iterator it;
	for (it = llstr.begin(); it != llstr.end(); it++) {
		if (it->front()==val) {
			found = true;
			break;
		}
	}

	if (!found) {
		std::list<std::string> newlist;
		newlist.push_front("");
		newlist.push_front(val);
		llstr.push_back(newlist);
	}

	XrslRelation rel(inoutattr, operator_eq, llstr);
	jobxrsl.AddRelation(rel);
}


void JobSubmission::PrepareUpload(Xrsl& jobxrsl) throw(XrslError) {

	std::list<std::list<std::string> > inputfiles;
	if (jobxrsl.IsRelation("inputfiles")) {
		inputfiles = jobxrsl.GetRelation("inputfiles").GetDoubleListValue();
	}

	localinputfiles.clear();
	std::list<std::list<std::string> > newinputfiles;

	std::list<std::list<std::string> >::iterator it;
	for (it = inputfiles.begin(); it != inputfiles.end(); it++) {
		std::string rfile = *(it->begin());
		std::string lfile = *(++it->begin());
		if (lfile.empty()) lfile = rfile;

		if (lfile.substr(0, 7)=="file://" ||
		    lfile.find("://")==std::string::npos) {

			if (lfile.substr(0, 7)=="file://") {
				lfile = lfile.substr(7);
			} else if (lfile[0]!='/') {
				char path[PATH_MAX];
				getcwd(path, PATH_MAX);
				lfile = (std::string)path + '/' + lfile;
			}

			std::ifstream file(lfile.c_str());
			if (!file)
				throw XrslError(_("Could not open input file") + (" " +lfile));

			file.seekg(0, std::ios::end);
			size_t length = file.tellg();
			file.close();
			// missing: checksum evaluation

			std::list<std::string> newlist;
			newlist.push_back(rfile);
			newlist.push_back(tostring(length));

			newinputfiles.push_back(newlist);
			localinputfiles.insert(std::make_pair(lfile, rfile));
		} else {
			newinputfiles.push_back(*it);
		}
	}

	if (!newinputfiles.empty()) {
		XrslRelation newrel("inputfiles", operator_eq, newinputfiles);
		if (jobxrsl.IsRelation("inputfiles"))
			jobxrsl.RemoveRelation("inputfiles");
		jobxrsl.AddRelation(newrel);
	}

	return;
}


void JobSubmission::PrepareRE(Xrsl& jobxrsl,
                              const std::string& attr,
                              Target& chosentarget,
                              bool oldserver) throw(XrslError) {

	if (!jobxrsl.IsRelation(attr)) return;
	std::list<XrslRelation> allrel = jobxrsl.GetAllRelations(attr);

	std::list<RuntimeEnvironment> res;
	std::list<RuntimeEnvironment> clusterres;
	if (attr=="runtimeenvironment") {
		res = chosentarget.runtime_environments;
		clusterres = chosentarget.cluster.runtime_environments;
	} else if (attr=="middleware") {
		res = chosentarget.middlewares;
		clusterres = chosentarget.cluster.middlewares;
	} else if (attr=="opsys") {
		res = chosentarget.operating_systems;
		clusterres = chosentarget.cluster.operating_systems;
	} else {
		return;
	}

	if (!res.empty()) {
		res.insert(res.end(), clusterres.begin(), clusterres.end());
	} else {
		res = clusterres;
	}
	res.sort();
	res.unique();

	std::list<RuntimeEnvironment> chosenres;
	std::list<RuntimeEnvironment>::iterator it;
	for (it = res.begin(); it != res.end(); it++) {

		bool good = true;
		unsigned int sz = allrel.size();

		// this constructs the possible RE's under consideration.
		std::list<XrslRelation>::iterator relit;
		for (relit = allrel.begin(); relit != allrel.end(); relit++) {
			xrsl_operator op = relit->GetOperator();
			std::string value = relit->GetSingleValue();
			RuntimeEnvironment re(value);

			if (it->Name()!=re.Name()) {
				sz--;
				continue;
			}
			if (op==operator_eq && *it==re) continue;
			if (op==operator_lt && *it<re) continue;
			if (op==operator_gt && *it>re) continue;
			if (op==operator_lteq && *it<=re) continue;
			if (op==operator_gteq && *it>=re) continue;

			good = false; // if we reach this point, the RE is not valid.
		}

		if (good==true && sz!=0) chosenres.push_back(*it);
	}

	chosenres.sort();
	chosenres.unique();

	if (chosenres.size()>allrel.size()) {
		std::list<RuntimeEnvironment>::iterator end = chosenres.end();
		end--;
		it = chosenres.begin();
		while (it != end && chosenres.size()>allrel.size()) {
			if (it->Name()==(++it)->Name()) {
				--it;
				it = chosenres.erase(it);
			}
		}
	}

	while (jobxrsl.IsRelation(attr)) jobxrsl.RemoveRelation(attr);

	if (oldserver) {
		std::list<std::string> restrings;
		for (it = chosenres.begin(); it != chosenres.end(); it++)
			restrings.push_back(it->str());
		XrslRelation rel(attr, operator_eq, restrings);
		jobxrsl.AddRelation(rel);
	} else {
		for (it = chosenres.begin(); it != chosenres.end(); it++) {
			XrslRelation rel(attr, operator_eq, it->str());
			jobxrsl.AddRelation(rel);
		}
	}

	return;
}


std::list<Target> PrepareJobSubmission(Xrsl axrsl)
throw(MDSQueryError, XrslError, GIISError, TargetError) {
	PerformXrslValidation(axrsl);
	std::list<Queue> queues = GetQueueInfo();
	std::list<Target> targets = ConstructTargets(queues, axrsl);
	PerformStandardBrokering(targets);
	return targets;
}


std::string SubmitJob(Xrsl axrsl,
                      std::list<Target> targets,
                      int timeout,
                      bool dryrun) throw(JobSubmissionError, XrslError) {

	JobSubmission jobsub(axrsl, targets, dryrun);
	std::string jobid = jobsub.Submit(timeout);
	// Write jobid to file
	return jobid;
}
