#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <algorithm>
#include <iostream>
#include <vector>

#include <arc/giis.h>
#include <arc/ldapquery.h>
#include <arc/mdsdiscovery.h>
#include <arc/notify.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


/** Resource-discovery callback class */
class ResourceDiscovery {

	public:
		/** Constructs a new callback object. */
		ResourceDiscovery(std::list<URL>& giislist);

		std::list<URL> GetGIISes();
		std::list<URL> GetClusters();
		std::list<URL> GetSEs();
		std::list<URL> GetRCs();

		void QueryGIISes(bool anonymous = true,
		                 std::string usersn = "",
		                 int timeout = TIMEOUT);

		static void Callback(const std::string& attribute,
		                     const std::string& value,
		                     void* ref);
		void ProcessCallback(const std::string& attribute,
		                     const std::string& value);

	private:
		std::list<URL> giis_urls;
		std::list<URL> cluster_urls;
		std::list<URL> se_urls;
		std::list<URL> rc_urls;

		std::string host;
		std::string port;
		std::string suffix;

		bool new_giis;
		bool new_cluster;
		bool new_se;
		bool new_rc;
};


ResourceDiscovery::ResourceDiscovery(std::list<URL>& giisurls) {
	giis_urls = giisurls;
	new_giis = false;
	new_cluster = false;
}


std::list<URL> ResourceDiscovery::GetGIISes() {
	return giis_urls;
}


std::list<URL> ResourceDiscovery::GetClusters() {
	return cluster_urls;
}

std::list<URL> ResourceDiscovery::GetSEs() {
	return se_urls;
}

std::list<URL> ResourceDiscovery::GetRCs() {
	return rc_urls;
}


void ResourceDiscovery::Callback(const std::string& attribute,
                                 const std::string& value,
                                 void* ref) {
	ResourceDiscovery* object = (ResourceDiscovery*)ref;
	object->ProcessCallback(attribute, value);
}


void ResourceDiscovery::ProcessCallback(const std::string& attribute,
                                        const std::string& value) {

	notify(VERBOSE) << _("attribute") << ": " << attribute << "  "
	                << _("value") << ": " << value << std::endl;

	std::string attr(attribute);
	std::string val(value);
	transform(attr.begin(), attr.end(), attr.begin(), tolower);
	transform(val.begin(), val.end(), val.begin(), tolower);

	if (attr=="mds-service-hn") {
		host = value;
		new_giis = false;
		new_cluster = false;
		new_se = false;
		new_rc = false;
	}

	if (attr=="mds-service-port") {
		port = value;
	}

	if (attr=="mds-service-ldap-suffix") {
		if (val.substr(0, 17)=="mds-vo-name=local" ||
			val.substr(0, 22)=="nordugrid-cluster-name") {
			new_cluster = true;
		} else if (val.substr(0, 17)=="nordugrid-se-name") {
			new_se = true;
		} else if (val.substr(0, 17)=="nordugrid-rc-name") {
			new_rc = true;
		} else if (val.substr(0, 11)=="mds-vo-name") {
			new_giis = true;
			suffix = value;
		}
	}

	if (attr=="mds-reg-status") {
		if (val=="valid") {
			if (new_cluster || new_se || new_rc) {
				std::string newresource;
				if (port.empty()) {
					newresource = "ldap://" + host +
					              "/o=grid/mds-vo-name=local";
				} else {
					newresource = "ldap://" + host + ":" + port +
					              "/o=grid/mds-vo-name=local";
				}

				if (new_cluster) {
					cluster_urls.push_back(URL(newresource));
					notify(DEBUG) << _("Found new cluster") << ": "
					              << newresource << std::endl;
				}
				if (new_se) {
					se_urls.push_back(URL(newresource));
					notify(DEBUG) << _("Found new storage element") << ": "
					              << newresource << std::endl;
				}
				if (new_rc) {
					rc_urls.push_back(URL(newresource));
					notify(DEBUG) << _("Found new replica catalog") << ": "
					              << newresource << std::endl;
				}
			}

			if (new_giis) {

				std::string giis("ldap://" + host);
				if (!port.empty()) giis += ":" + port;
				giis += "/" + suffix;

				URL giisurl(giis);
				notify(VERBOSE) << _("Found new GIIS") << ": " << giisurl
				                << std::endl;

				if (find(giis_urls.begin(), giis_urls.end(), giisurl) ==
				    giis_urls.end()) {

					notify(VERBOSE) << _("Adding new GIIS") << ": " << giisurl
					                << std::endl;
					giis_urls.push_back(URL(giisurl));
				}
			}
		}
	}
}

void ResourceDiscovery::QueryGIISes(bool anonymous,
                                    std::string usersn,
                                    int timeout) {

	std::vector<std::string> attrs;
	attrs.push_back("giisregistrationstatus");
	std::string filter = "(objectclass=*)";

	std::list<URL>::iterator begin = giis_urls.begin();
	std::list<URL>::iterator end = giis_urls.end();
	std::list<URL>::iterator back = --giis_urls.end();

	while (begin != end) {

		std::list<URL> tmplist(begin, end);

		ParallelLdapQueries pldapq(tmplist,
		                           filter,
		                           attrs,
		                           &ResourceDiscovery::Callback,
		                           this,
		                           LdapQuery::base,
		                           usersn,
		                           anonymous,
		                           timeout);
		pldapq.Query();

		begin = ++back;
		back = --giis_urls.end();
	}
}


std::list<URL> GetClusterResources(const URL& url,
                                   bool anonymous,
                                   std::string usersn,
                                   int timeout) {
	return GetResources(url, cluster, anonymous, usersn, timeout);
}


std::list<URL> GetSEResources(const URL& url,
                              bool anonymous,
                              std::string usersn,
                              int timeout) {
	return GetResources(url, storageelement, anonymous, usersn, timeout);
}


std::list<URL> GetRCResources(const URL& url,
                             bool anonymous,
                             std::string usersn,
                             int timeout) {
	return GetResources(url, replicacatalog, anonymous, usersn, timeout);
}


std::list<URL> GetResources(const URL& url,
                            resource id,
                            bool anonymous,
                            std::string usersn,
                            int timeout) {

	std::list<URL> urls;
	urls.push_back(url);

	return GetResources(urls, id, anonymous, usersn, timeout);
}


std::list<URL> GetClusterResources(std::list<URL> urls,
                                   bool anonymous,
                                   std::string usersn,
                                   int timeout) {
	return GetResources(urls, cluster, anonymous, usersn, timeout);
}


std::list<URL> GetSEResources(std::list<URL> urls,
                              bool anonymous,
                              std::string usersn,
                              int timeout) {
	return GetResources(urls, storageelement, anonymous, usersn, timeout);
}


std::list<URL> GetRCResources(std::list<URL> urls,
                              bool anonymous,
                              std::string usersn,
                              int timeout) {
	return GetResources(urls, replicacatalog, anonymous, usersn, timeout);
}


std::list<URL> GetResources(std::list<URL> urls,
                            resource id,
                            bool anonymous,
                            std::string usersn,
                            int timeout) {

	if (urls.empty()) urls = GetGIISList();

	std::list<URL> resources;

	std::list<URL>::iterator it;
	for (it = urls.begin(); it != urls.end(); it++) {
		std::string path = it->Path();
		transform(path.begin(), path.end(), path.begin(), tolower);
		if (path=="/o=grid/mds-vo-name=local") { // GRIS
			notify(VERBOSE) << _("Found GRIS") << ": "
			                << it->str() << std::endl;
			resources.push_back(*it);
			it = --urls.erase(it);
		}
	}

	ResourceDiscovery resourcedisc(urls);
	resourcedisc.QueryGIISes(anonymous, usersn, timeout);

	std::list<URL> found;
	if (id==cluster) {
		found = resourcedisc.GetClusters();
	} else if (id==storageelement) {
		found = resourcedisc.GetSEs();
	} else if (id==replicacatalog) {
		found = resourcedisc.GetRCs();
	}

	if (!found.empty())
		resources.insert(resources.end(), found.begin(), found.end());

	resources.sort();
	resources.unique();
	return resources;
}
