#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstdlib>

#include <arc/certificate.h>
#include <arc/common.h>
#include <arc/notify.h>
#include <arc/stringconv.h>
#include <arc/target.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


Target::Target(Queue aqueue, Xrsl axrsl) : Queue(aqueue) {
	xrsls = axrsl.SplitOrRelation();
}


void Target::AddXrsl(Xrsl axrsl) {
	xrsls = axrsl.SplitOrRelation();
}


std::list<Xrsl>& Target::GetXrsls() {
	return xrsls;
}


long Target::GetCputime(Xrsl xrsl) throw(XrslError) {

	bool cputimegiven = xrsl.IsRelation("cputime");
	bool walltimegiven = xrsl.IsRelation("walltime");
	bool gridtimegiven = xrsl.IsRelation("gridtime");
	bool benchmarksgiven = xrsl.IsRelation("benchmarks");

	if (cputimegiven && gridtimegiven)
		throw XrslError(_("Both cputime and gridtime given in the xrsl"));

	if (cputimegiven && benchmarksgiven)
		throw XrslError(_("Both cputime and benchmarks given in the xrsl"));

	if (walltimegiven && gridtimegiven)
		throw XrslError(_("Both walltime and gridtime given in the xrsl"));

	if (walltimegiven && benchmarksgiven)
		throw XrslError(_("Both walltime and benchmarks given in the xrsl"));

	if (benchmarksgiven && gridtimegiven)
		throw XrslError(_("Both gridtime and benchmarks given in the xrsl"));

	if (gridtimegiven) {
		std::string gridstr = xrsl.GetRelation("gridtime").GetSingleValue();
		float cpufreq = cpu_freq;
		if (cpufreq == UNDEFINED) cpufreq = cluster.cpu_freq;
		if (cpufreq == UNDEFINED) {
			notify(DEBUG) << _("Could not determine CPU frequency for queue")
			              << " " << name << "@" << cluster.hostname << " - "
			              << _("1 GHz is assumed") << std::endl;
			cpufreq = 1000;
		}

		return (long)(Seconds(gridstr)*2800.0/cpufreq);
	}

	if (benchmarksgiven) {
		long benchtime = UNDEFINED;

		std::map<std::string, float> queben = benchmarks;
		std::map<std::string, float> cluben = cluster.benchmarks;

		std::list<std::list<std::string> > bench =
		    xrsl.GetRelation("benchmarks").GetDoubleListValue();
		std::list<std::list<std::string> >::iterator it;

		for (it = bench.begin(); it != bench.end(); it++) {
			long thisbenchtime = UNDEFINED;

			std::list<std::string>::iterator listit = it->begin();
			std::string name = *listit;

			listit++;
			double value = stringtod(*listit);
			listit++;
			double time = Seconds(*listit);

			if (queben.find(name)!=queben.end()) {
				thisbenchtime = (long)(value*time/queben.find(name)->second);
			}
			else if (cluben.find(name)!=cluben.end()) {
				thisbenchtime = (long)(value*time/cluben.find(name)->second);
			}                     

			if (benchtime < thisbenchtime) benchtime = thisbenchtime;
		}

		if (benchtime!=UNDEFINED) return benchtime;
	}

	if (walltimegiven)
		return Seconds(xrsl.GetRelation("walltime").GetSingleValue());

	if (cputimegiven)
		return Seconds(xrsl.GetRelation("cputime").GetSingleValue());

	if (default_wall_time!=UNDEFINED)
		return default_wall_time;

	if (default_cpu_time!=UNDEFINED)
		return default_cpu_time;

	return UNDEFINED;
}


std::list<Target> ConstructTargets(std::list<Queue> queues, Xrsl axrsl)
throw(TargetError) {

	if (queues.size()==0)
		throw TargetError(_("No cluster information available for brokering"));

	std::list<Target> targets;
	unsigned int authreject = 0;

	std::list<Queue>::iterator it;
	for (it = queues.begin(); it != queues.end(); it++) {

		/** The user should be authorized. */
		if (it->users.size()==0) {
			notify(DEBUG) << _("Queue rejected because the user is not "
			                   "authorized") << ": " << it->name << "@"
			              << it->cluster.hostname << std::endl;
			authreject++;
			continue;
		}

		/** Queue should be active. */
		if (it->status != "active") {
			notify(DEBUG) << _("Queue rejected because of status") << " "
			              << it->status << ": " << it->name << "@"
			              << it->cluster.hostname << std::endl;
			continue;
		}

		/** totalcpus should be > 0. */
		int totalcpus = 0;
		if (it->cluster.total_cpus != UNDEFINED)
			totalcpus = it->cluster.total_cpus;
		if (it->total_cpus != UNDEFINED)
			totalcpus = it->total_cpus;
		if (totalcpus == 0) {
			notify(DEBUG) << _("Queue rejected because it does not have any "
			                   "CPUs") << ": " << it->name << "@"
			              << it->cluster.hostname << std::endl;
			continue;
		}

		/** number of queued jobs should be less than maxqueuable. */
		if (it->max_queuable != UNDEFINED && it->queued > it->max_queuable) {
			notify(DEBUG) << _("Queue rejected because the number of queued "
			                   "jobs is larger than maximum allowed")
			              << ": " << it->name << "@"
			              << it->cluster.hostname << std::endl;
			continue;
		}

		/** the cluster's issuer-certificate must be installed locally. */
		std::list<Certificate> localcerts = GetCAList();
		std::list<Certificate>::iterator ceit;

		bool found = false;

		// temporary workaround for issuers that have email in their SN
		std::string issuerca = it->cluster.issuer_ca;
		
		std::string::size_type pos = issuerca.find("Email");
		if (pos != std::string::npos) issuerca = issuerca.substr(0, pos);

		pos = issuerca.find("emailAddress");
		if (pos != std::string::npos) issuerca = issuerca.substr(0, pos);

		for (ceit = localcerts.begin(); ceit != localcerts.end(); ceit++) {
			std::string sn = ceit->GetSN();
			sn = sn.substr(0, issuerca.size());
			if (sn==issuerca) {
				found = true;
				break;
			}
		}

		if (found==false) {
			notify(DEBUG) << _("Queue rejected because the certificate of "
			                   "cluster's CA is not installed on the client")
			              << ": " << it->name << "@"
			              << it->cluster.hostname << std::endl;
			continue;
		}

		Target target(*it, axrsl);
		targets.push_back(target);
	}

	if (authreject==queues.size())
		throw TargetError(_("The user is not authorized at any of the "
		                    "possible targets"));

	return targets;
}
