#include "arccli.h"

#include <iostream>
#include <list>
#include <string>
#include <algorithm>

#include <arc/common.h>
#include <arc/datetime.h>
#include <arc/giis.h>
#include <arc/joblist.h>
#include <arc/mdsdiscovery.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/resource.h>
#include <arc/runtimeenvironment.h>
#include <arc/url.h>
#include <arc/jobftpcontrol.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#define __(A, B, C) dngettext("arccli", (A), (B), (C))
#else
#define _(A) (A)
#define __(A, B, C) ((C) == 1 ? (A) : (B))
#endif


void arcstat(const std::list<std::string>& jobs,
             const std::list<std::string>& clusterselect,
             const std::list<std::string>& clusterreject,
             const std::list<std::string>& status,
             const std::list<std::string>& giisurls,
             const bool clusters,
             const bool longlist,
             const int timeout,
             const bool anonymous) {

	if (clusters) {

		std::list<URL> clusterurllist;

		for (std::list<std::string>::const_iterator it = clusterselect.begin();
		     it != clusterselect.end(); it++) {
			bool found = false;
			for (std::list<URL>::iterator cli = clusterurllist.begin();
			     !found && cli != clusterurllist.end(); cli++)
				if (*it == cli->Host()) found = true;
			if (!found)
				clusterurllist.push_back("ldap://" + *it +
				                         ":2135/O=Grid/Mds-Vo-Name=local");
		}

		if (clusterurllist.empty()) {
			std::list<URL> giisurllist = ConvertToURLs (giisurls);
			clusterurllist =
				GetClusterResources(giisurllist, anonymous, "", timeout);
			if (clusterurllist.empty())
				throw ARCCLIError(_("Could not retrieve cluster "
				                    "list from GIIS"));
		}

		for (std::list<std::string>::const_iterator it = clusterreject.begin();
		     it != clusterreject.end(); it++)
			for (std::list<URL>::iterator cli = clusterurllist.begin();
			     cli != clusterurllist.end(); cli++)
				if (cli->Host() == *it) {
					notify(INFO) << _("Rejecting cluster")
					             << ": " << *it << std::endl;
					clusterurllist.erase(cli);
					break;
				}

		std::list<Cluster> clusterlist =
			GetClusterInfo(clusterurllist, MDS_FILTER_CLUSTERINFO,
			               anonymous, "", timeout);

		std::list<Cluster>::iterator cli;
		for (cli = clusterlist.begin(); cli != clusterlist.end(); cli++) {
			if (cli->issuer_ca.size()==0) {
				notify(ERROR) << _("Error: No information obtained from "
				                   "cluster") << ": " << cli->hostname << ". " 
				              << _("Cluster is not responding.") << std::endl;
				cli = clusterlist.erase(cli);
			}
		}

		for (cli = clusterlist.begin(); cli != clusterlist.end(); cli++) {

			std::cout << _("Cluster") << " " << cli->hostname
			          << std::endl;
			if (!cli->alias.empty())
				std::cout << "  " << _("Alias")
				          << ": " << cli->alias << std::endl;
			if (longlist) {
				if (!cli->comment.empty())
					std::cout << "  " << _("Comment")
					          << ": " << cli->comment << std::endl;
				if (!cli->contact.empty())
					std::cout << "  " << _("Contact")
					          << ": " << cli->contact << std::endl;
				if (!cli->interactive_contact.empty())
					std::cout << "  " << _("Interactive Contact")
					          << ": " << cli->interactive_contact << std::endl;
				if (!cli->location.empty())
					std::cout << "  " << _("Location")
					          << ": " << cli->location << std::endl;
				if (!cli->issuer_ca.empty())
					std::cout << "  " << _("Host Certificate Issuer")
					          << ": " << cli->issuer_ca << std::endl;
				if (!cli->owners.empty()) {
					std::cout << "  " << _("Owner")
					          << ":" << std::endl;
					for (std::list<std::string>::const_iterator
					     it = cli->owners.begin();
					     it != cli->owners.end(); it++)
						std::cout << "    " << *it << std::endl;
				}
				if (!cli->architecture.empty())
					std::cout << "  " << _("Architecture")
					          << ": " << cli->architecture << std::endl;
				if (!cli->operating_systems.empty()) {
					std::cout << "  " << _("Operating System")
					          << ":" << std::endl;
					for (std::list<RuntimeEnvironment>::const_iterator
					     it = cli->operating_systems.begin();
					     it != cli->operating_systems.end(); it++)
						std::cout << "    " << it->str() << std::endl;
				}
				if (!cli->node_cpu.empty())
					std::cout << "  " << (cli->homogeneity ? _("CPU") :
					                      _("CPU (worst)"))
					          << ": " << cli->node_cpu << std::endl;
				if (cli->node_memory != UNDEFINED)
					std::cout << "  " << (cli->homogeneity ?
					                      _("Memory on Each Node") :
					                      _("Memory on Each Node (smallest)"))
					          << ": " << cli->node_memory
					          << " " << _("MiB") << std::endl;
				if (cli->session_dir_total != UNDEFINED)
					std::cout << "  " << _("Size of Scratch Directory")
					          << ": " << cli->session_dir_total << " "
					          << __("byte", "bytes", cli->session_dir_total)
					          << std::endl;
				if (cli->session_dir_free != UNDEFINED)
					std::cout << "  "
					          << _("Free Space in Scratch Directory")
					          << ": " << cli->session_dir_free << " "
					          << __("byte", "bytes", cli->session_dir_free)
					          << std::endl;
				if (cli->cache_total != UNDEFINED)
					std::cout << "  " << _("Size of Cache Directory")
					          << ": " << cli->cache_total << " "
					          << __("byte", "bytes", cli->cache_total)
					          << std::endl;
				if (cli->cache_free != UNDEFINED)
					std::cout << "  " << _("Free Space in Cache Directory")
					          << ": " << cli->cache_free << " "
					          << __("byte", "bytes", cli->cache_free)
					          << std::endl;
				if (cli->total_cpus != UNDEFINED)
					std::cout << "  " << _("Number of CPUs")
					          << ": " << cli->total_cpus << std::endl;
				if (cli->used_cpus != UNDEFINED)
					std::cout << "  " << _("Number of Used CPUs")
					          << ": " << cli->used_cpus << std::endl;
				if (cli->total_jobs != UNDEFINED)
					std::cout << "  " << _("Number of Jobs")
					          << ": " << cli->total_jobs << std::endl;
				if (cli->queued_jobs != UNDEFINED)
					std::cout << "  " << _("Number of Queued Jobs")
					          << ": " << cli->queued_jobs << std::endl;
				if (!cli->cpu_distribution.empty()) {
					std::cout << "  " << _("Number of Computers")
					          << ":" << std::endl;
					for (std::map<int, int>::const_iterator
					     it = cli->cpu_distribution.begin();
					     it != cli->cpu_distribution.end(); it++)
						std::cout << "    " << it->first << " "
						          << __("processor", "processors", it->first)
						          << ": " << it->second << std::endl;
				}
				if (!cli->benchmarks.empty()) {
					std::cout << "  " << _("Evaluated Benchmarks")
					          << ":" << std::endl;
					for (std::map<std::string, float>::const_iterator
					     it = cli->benchmarks.begin();
					     it != cli->benchmarks.end(); it++)
						std::cout << "    " << it->first
						          << ": " << it->second << std::endl;
				}
				if (!cli->support.empty()) {
					std::cout << "  " << _("Cluster Support")
					          << ":" << std::endl;
					for (std::list<std::string>::const_iterator
					     it = cli->support.begin();
					     it != cli->support.end(); it++)
						std::cout << "    " << *it << std::endl;
				}
				if (!cli->local_se.empty()) {
					std::cout << "  " << _("Local Storage Elements")
					          << ":" << std::endl;
					for (std::list<std::string>::const_iterator
					     it = cli->local_se.begin();
					     it != cli->local_se.end(); it++)
						std::cout << "    " << *it << std::endl;
				}
				if (!cli->middlewares.empty()) {
					std::cout << "  " << _("Installed Middleware")
					          << ":" << std::endl;
					for (std::list<RuntimeEnvironment>::const_iterator
					     it = cli->middlewares.begin();
					     it != cli->middlewares.end(); it++)
						std::cout << "    " << it->str() << std::endl;
				}
				if (!cli->runtime_environments.empty()) {
					std::cout << "  " << _("Installed Runtime Environments")
					          << ":" << std::endl;
					for (std::list<RuntimeEnvironment>::const_iterator
					     it = cli->runtime_environments.begin();
					     it != cli->runtime_environments.end(); it++)
						std::cout << "    " << it->str() << std::endl;
				}
				if (!cli->node_access.empty()) {
					std::cout << "  " << _("Network Access on Cluster Nodes")
					          << ":" << std::endl;
					for (std::list<std::string>::const_iterator
					     it = cli->node_access.begin();
					     it != cli->node_access.end(); it++)
						std::cout << "    " << *it << std::endl;
				}
				if (cli->session_dir_lifetime != UNDEFINED)
					std::cout << "  " << _("Session directory lifetime")
					          << ": " << Period(cli->session_dir_lifetime)
					          << std::endl;
				if (!cli->trusted_ca.empty()) {
					std::cout << "  " << _("Trusted Certificate Authorities")
					          << ":" << std::endl;
					for (std::list<std::string>::const_iterator
					     it = cli->trusted_ca.begin();
					     it != cli->trusted_ca.end(); it++)
						std::cout << "    " << *it << std::endl;
				}
				if (cli->mds_validfrom != UNDEFINED)
					std::cout << "  " << _("Entry valid from")
					          << ": " << TimeStamp(cli->mds_validfrom)
					          << std::endl;
				if (cli->mds_validto != UNDEFINED)
					std::cout << "  " << _("Entry valid to")
					          << ": " << TimeStamp(cli->mds_validto)
					          << std::endl;
			}

			for (std::list<Queue>::iterator qli = cli->queues.begin();
			     qli != cli->queues.end(); qli++) {
				
				std::cout << _("Queue") << " " << qli->name
				          << std::endl;
				if (!qli->status.empty())
					std::cout << "  " << _("Status")
					          << ": " << qli->status << std::endl;
				if (longlist) {
					if (!qli->comment.empty())
						std::cout << "  " << _("Comment")
						          << ": " << qli->comment << std::endl;
					if (!qli->architecture.empty())
						std::cout << "  " << _("Architecture")
						          << ": " << qli->architecture << std::endl;
					if (!qli->operating_systems.empty()) {
						std::cout << "  " << _("Operating System")
						          << ":" << std::endl;
						for (std::list<RuntimeEnvironment>::const_iterator
						     it = qli->operating_systems.begin();
						     it != qli->operating_systems.end(); it++)
							std::cout << "    " << it->str() << std::endl;
					}
					if (!qli->node_cpu.empty())
						std::cout << "  " << _("CPU")
						          << ": " << qli->node_cpu << std::endl;
					if (qli->node_memory != UNDEFINED)
						std::cout << "  " << _("Memory on Each Node")
						          << ": " << qli->node_memory
						          << " " << _("MiB") << std::endl;
					if (!qli->middlewares.empty()) {
						std::cout << "  " << _("Installed Middleware")
						          << ":" << std::endl;
						for (std::list<RuntimeEnvironment>::const_iterator
						     it = qli->middlewares.begin();
						     it != qli->middlewares.end(); it++)
							std::cout << "    " << it->str() << std::endl;
					}
					if (!qli->runtime_environments.empty()) {
						std::cout << "  "
						          << _("Installed Runtime Environments")
						          << ":" << std::endl;
						for (std::list<RuntimeEnvironment>::const_iterator
						     it = qli->runtime_environments.begin();
						     it != qli->runtime_environments.end(); it++)
							std::cout << "    " << it->str() << std::endl;
					}
					if (!qli->benchmarks.empty()) {
						std::cout << "  " << _("Evaluated Benchmarks")
						          << ":" << std::endl;
						for (std::map<std::string, float>::const_iterator
						     it = qli->benchmarks.begin();
						     it != qli->benchmarks.end(); it++)
							std::cout << "    " << it->first
							          << ": " << it->second << std::endl;
					}
					if (qli->running != UNDEFINED)
						std::cout << "  " << _("Number of Running Jobs")
						          << ": " << qli->running << std::endl;
					if (qli->queued != UNDEFINED)
						std::cout << "  " << _("Number of Queued Jobs")
						          << ": " << qli->queued << std::endl;
					if (qli->max_running != UNDEFINED)
						std::cout << "  " << _("Max Number of Running Jobs")
						          << ": " << qli->max_running << std::endl;
					if (qli->max_queuable != UNDEFINED)
						std::cout << "  " << _("Max Number of Queued Jobs")
						          << ": " << qli->max_queuable << std::endl;
					if (qli->max_user_run != UNDEFINED)
						std::cout << "  " << _("Max Number of Running Jobs "
						                       "per Local User")
						          << ": " << qli->max_user_run << std::endl;
					if (qli->max_cpu_time != UNDEFINED)
						std::cout << "  " << _("Max CPU Time")
						          << ": " << Period(qli->max_cpu_time)
						          << std::endl;
					if (qli->min_cpu_time != UNDEFINED)
						std::cout << "  " << _("Min CPU Time")
						          << ": " << Period(qli->min_cpu_time)
						          << std::endl;
					if (qli->default_cpu_time != UNDEFINED)
						std::cout << "  " << _("Default CPU Time")
						          << ": " << Period(qli->default_cpu_time)
						          << std::endl;
					if (qli->max_wall_time != UNDEFINED)
						std::cout << "  " << _("Max Wall Time")
						          << ": " << Period(qli->max_wall_time)
						          << std::endl;
					if (qli->min_wall_time != UNDEFINED)
						std::cout << "  " << _("Min Wall Time")
						          << ": " << Period(qli->min_wall_time)
						          << std::endl;
					if (qli->default_wall_time != UNDEFINED)
						std::cout << "  " << _("Default Wall Time")
						          << ": " << Period(qli->default_wall_time)
						          << std::endl;
					if (!qli->scheduling_policy.empty())
						std::cout << "  " << _("Scheduling Policy")
						          << ": " << qli->scheduling_policy
						          << std::endl;
					if (qli->total_cpus != UNDEFINED)
						std::cout << "  " << _("Number of CPUs")
						          << ": " << qli->total_cpus << std::endl;
					if (qli->mds_validfrom != UNDEFINED)
						std::cout << "  " << _("Entry valid from")
						          << ": " << TimeStamp(qli->mds_validfrom)
						          << std::endl;
					if (qli->mds_validto != UNDEFINED)
						std::cout << "  " << _("Entry valid to")
						          << ": " << TimeStamp(qli->mds_validto)
						          << std::endl;

					for (std::list<User>::iterator uli = qli->users.begin();
					     uli != qli->users.end(); uli++) {

						std::cout << _("User") << " " << uli->name
						          << std::endl;
						if (!uli->free_cpus.empty()) {
							std::cout << "  " << _("Number of free CPUs")
							          << ":" << std::endl;
							for (std::map<long int, int>::const_iterator
							     it = uli->free_cpus.begin();
							     it != uli->free_cpus.end(); it++)
								if (it->second)
									std::cout << "    " << it->second << " " 
									          << __("processor", "processors",
									                it->second)
									          << ": "
									          << (it->first == LONG_MAX ?
									              _("indefinitely") :
									              Period(it->first))
									          << std::endl;
								else
									std::cout << "    " << _("None")
									          << std::endl;
						}
						if (uli->free_diskspace != UNDEFINED)
							std::cout << "  " << _("Free Disk Space")
							          << ": " << uli->free_diskspace
							          << " " << __("byte", "bytes",
							                       uli->free_diskspace)
							          << std::endl;
						if (uli->queue_length != UNDEFINED)
							std::cout << "  " << _("Queue Length")
							          << ": " << uli->queue_length
							          << std::endl;

						if (uli->mds_validfrom != UNDEFINED)
							std::cout << "  " << _("Entry valid from")
							          << ": " << TimeStamp(uli->mds_validfrom)
							          << std::endl;
						if (uli->mds_validto != UNDEFINED)
							std::cout << "  " << _("Entry valid to")
							          << ": " << TimeStamp(uli->mds_validto)
							          << std::endl;
					}

				}

			}

			std::cout << std::endl;
		}
	}

	else {

		std::list<std::string> jobids =
			GetJobIDsList(jobs, clusterselect, clusterreject);

		if (jobs.empty() && clusterselect.empty() && jobids.empty())
			throw ARCCLIError(_("No jobs"));

		if (jobids.empty())
			throw ARCCLIError(_("No valid jobnames/jobids given"));

		std::list<Job> joblist =
			GetJobInfo(jobids, MDS_FILTER_JOBINFO, anonymous, "", timeout);

		Time now;
		std::map<std::string, Time> jobhistory = GetJobHistory();

		for (std::list<Job>::iterator jli = joblist.begin();
		     jli != joblist.end(); jli++) {

			if (jli->status.empty()) {
				
				notify(WARNING) << _("Job information not found")
				                << ": " << jli->id << ". ";

				std::map<std::string, Time>::iterator it =
				    jobhistory.find(jli->id);
				if (it != jobhistory.end()) {

					if (now.GetTime() - it->second.GetTime()<90) {
						notify(WARNING) << _("This job was only very "
						                     "recently submitted and it might "
						                     "not yet have been reached "
						                     "the information-system.");
					}
				}

				std::cout << std::endl;
				continue;
			}

			if (!status.empty() &&
			    (std::find(status.begin(), status.end(), jli->status) == status.end()))
				continue;

			std::cout << _("Job") << " " << jli->id << std::endl;
			if (!jli->job_name.empty())
				std::cout << "  " << _("Job Name")
				          << ": " << jli->job_name << std::endl;
			if (!jli->status.empty())
				std::cout << "  " << _("Status")
				          << ": " << jli->status << std::endl;
			if (jli->exitcode != UNDEFINED)
				std::cout << "  " << _("Exit Code")
				          << ": " << jli->exitcode << std::endl;
			if (!jli->errors.empty())
				std::cout << "  " << _("Error")
				          << ": " << jli->errors << std::endl;
			if (longlist) {
				if (!jli->rerunable.empty())
					std::cout << "  " << _("Rerunable from state")
						  << ": " << jli->rerunable << std::endl;
				if (!jli->owner.empty())
					std::cout << "  " << _("Owner")
					          << ": " << jli->owner << std::endl;
				if (!jli->comment.empty())
					std::cout << "  " << _("Comment")
					          << ": " << jli->comment << std::endl;
				if (!jli->cluster.empty())
					std::cout << "  " << _("Cluster")
					          << ": " << jli->cluster << std::endl;
				if (!jli->queue.empty())
					std::cout << "  " << _("Queue")
					          << ": " << jli->queue << std::endl;
				if (jli->cpu_count != UNDEFINED)
					std::cout << "  " << _("Requested Number of CPUs")
					          << ": " << jli->cpu_count << std::endl;
				if (!jli->execution_nodes.empty()) {
					std::cout << "  " << _("Execution Nodes")
					          << ":" << std::endl;
					for (std::list<std::string>::const_iterator
					     it = jli->execution_nodes.begin();
					     it != jli->execution_nodes.end(); it++)
						std::cout << "    " << *it << std::endl;
				}
				if (jli->queue_rank != UNDEFINED)
					std::cout << "  " << _("Rank")
					          << ": " << jli->queue_rank << std::endl;
				if (!jli->sstdin.empty())
					std::cout << "  " << _("stdin")
					          << ": " << jli->sstdin << std::endl;
				if (!jli->sstdout.empty())
					std::cout << "  " << _("stdout")
					          << ": " << jli->sstdout << std::endl;
				if (!jli->sstderr.empty())
					std::cout << "  " << _("stderr")
					          << ": " << jli->sstderr << std::endl;
				if (!jli->gmlog.empty())
					std::cout << "  " << _("Grid Manager Log Directory")
					          << ": " << jli->gmlog << std::endl;
				if (jli->submission_time != UNDEFINED)
					std::cout << "  " << _("Submitted")
					          << ": " << TimeStamp(jli->submission_time)
					          << std::endl;
				if (jli->completion_time != UNDEFINED)
					std::cout << "  " << _("Completed")
					          << ": " << TimeStamp(jli->completion_time)
					          << std::endl;
				if (!jli->submission_ui.empty())
					std::cout << "  " << _("Submitted from")
					          << ": " << jli->submission_ui << std::endl;
				if (!jli->client_software.empty())
					std::cout << "  " << _("Submitting Client")
					          << ": " << jli->client_software << std::endl;
				if (!jli->runtime_environments.empty()) {
					std::cout << "  " << _("Requested Runtime Environments")
					          << ":" << std::endl;
					for (std::list<RuntimeEnvironment>::const_iterator
					     it = jli->runtime_environments.begin();
					     it != jli->runtime_environments.end(); it++)
						std::cout << "    " << it->str() << std::endl;
				}
				if (jli->requested_cpu_time != UNDEFINED)
					std::cout << "  " << _("Required CPU Time")
					          << ": " << Period(jli->requested_cpu_time)
					          << std::endl;
				if (jli->used_cpu_time != UNDEFINED)
					std::cout << "  " << _("Used CPU Time")
					          << ": " << Period(jli->used_cpu_time)
					          << std::endl;
				if (jli->used_wall_time != UNDEFINED)
					std::cout << "  " << _("Used Wall Time")
					          << ": " << Period(jli->used_wall_time)
					          << std::endl;
				if (jli->used_memory != UNDEFINED)
					std::cout << "  " << _("Used Memory")
					          << ": " << jli->used_memory
					          << " " << _("KiB") << std::endl;
				if (jli->erase_time != UNDEFINED)
					std::cout << "  "
					          << (jli->status == "DELETED" ?
					              _("Results were deleted") :
					              _("Results must be retrieved before"))
					          << ": " << TimeStamp(jli->erase_time)
					          << std::endl;
				if (jli->proxy_expire_time != UNDEFINED)
					std::cout << "  " << _("Proxy valid to")
					          << ": " << TimeStamp(jli->proxy_expire_time)
					          << std::endl;
				if (jli->mds_validfrom != UNDEFINED)
					std::cout << "  " << _("Entry valid from")
					          << ": " << TimeStamp(jli->mds_validfrom)
					          << std::endl;
				if (jli->mds_validto != UNDEFINED)
					std::cout << "  " << _("Entry valid to")
					          << ": " << TimeStamp(jli->mds_validto)
					          << std::endl;
			}
		}
	}
}
