#include "arccli.h"

#include <fstream>
#include <iostream>
#include <list>
#include <string>
#include <sstream>

#include <arc/common.h>
#include <arc/giis.h>
#include <arc/joblist.h>
#include <arc/jobsubmission.h>
#include <arc/mdsdiscovery.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/standardbrokers.h>
#include <arc/target.h>
#include <arc/url.h>
#include <arc/xrsl.h>
#include <arc/job_xrsl.h>
#include <arc/job_jsdl.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


std::list<std::string> arcsub(const std::list<std::string>& xrslfiles,
                              const std::list<std::string>& xrslstrings,
                              const std::list<std::string>& clusterselect,
                              const std::list<std::string>& clusterreject,
                              const std::list<std::string>& giisurls,
                              const std::string& joblistfile,
                              const bool dryrun,
                              const bool dumpxrsl,
                              const bool unknownattr,
                              const int timeout,
                              const bool anonymous) {

	if (xrslfiles.empty() && xrslstrings.empty())
		throw ARCCLIError (_("No xrsl input specified"));

	std::list<Xrsl> xrsllist;

	for (std::list<std::string>::const_iterator it = xrslfiles.begin();
	     it != xrslfiles.end(); it++) {

		std::ifstream xrslfile(it->c_str());
		if (!xrslfile)
			throw ARCCLIError (_("Can not open file") + (": " + *it));
		xrslfile.seekg(0, std::ios::end);
		std::streamsize length = xrslfile.tellg();
		xrslfile.seekg(0, std::ios::beg);

		char* buffer = new char[length+1];
		xrslfile.read(buffer, length);
		xrslfile.close();

		buffer[length] = '\0';
		std::string xrsl(buffer);
		delete[] buffer;

		std::string::size_type pos = xrsl.find_first_not_of(" \t\r\n");
		if (xrsl[pos] == '&' || xrsl[pos] == '+') {
			// it's a Xrsl-string
			std::list<Xrsl> xrsls = Xrsl(xrsl).SplitMulti(); 
			xrsllist.insert(xrsllist.end(), xrsls.begin(), xrsls.end());
		} else if (xrsl[pos] == '<') {
			// it's a JSDL-string. Convert it.
			JobRequestJSDL jobjsdl(xrsl);
			JobRequestXRSL jobxrsl(jobjsdl);
			std::ostringstream os;
			os << jobxrsl;
			std::list<Xrsl> xrsls = Xrsl(os.str()).SplitMulti(); 
			xrsllist.insert(xrsllist.end(), xrsls.begin(), xrsls.end());
		} else {
			throw ARCCLIError(_("Unknown job request type."));
		}
	}

	for (std::list<std::string>::const_iterator it = xrslstrings.begin();
	     it != xrslstrings.end(); it++) {

		std::string::size_type pos = it->find_first_not_of(" \t\r\n");
		if ((*it)[pos] == '&' || (*it)[pos] == '+') {
			// it's a Xrsl-string
			std::list<Xrsl> xrsls = Xrsl(*it).SplitMulti(); 
			xrsllist.insert(xrsllist.end(), xrsls.begin(), xrsls.end());
		} else if ((*it)[pos] == '<') {
			// it's a JSDL-string. Convert it.
			JobRequestJSDL jobjsdl(*it);
			JobRequestXRSL jobxrsl(jobjsdl);
			std::ostringstream os;
			os << jobxrsl;
			std::list<Xrsl> xrsls = Xrsl(os.str()).SplitMulti(); 
			xrsllist.insert(xrsllist.end(), xrsls.begin(), xrsls.end());
		} else {
			throw ARCCLIError(_("Unknown job request type."));
		}
	}

	// find available clusters

	std::list<URL> clusterurllist;

	for (std::list<std::string>::const_iterator it = clusterselect.begin();
	     it != clusterselect.end(); it++) {
		bool found = false;
		for (std::list<URL>::iterator cli = clusterurllist.begin();
		     !found && cli != clusterurllist.end(); cli++)
			if (*it == cli->Host()) found = true;
		if (!found)
			clusterurllist.push_back("ldap://" + *it +
			                         ":2135/O=Grid/Mds-Vo-Name=local");
	}

	if (clusterurllist.empty()) {
		std::list<URL> giisurllist = ConvertToURLs(giisurls);
		clusterurllist =
			GetClusterResources(giisurllist, anonymous, "", timeout);
		if (clusterurllist.empty())
			throw ARCCLIError(_("Could not retrieve cluster list from GIIS"));
	}

	for (std::list<std::string>::const_iterator it = clusterreject.begin();
	     it != clusterreject.end(); it++)
		for (std::list<URL>::iterator cli = clusterurllist.begin();
		     cli != clusterurllist.end(); cli++)
			if (cli->Host() == *it) {
				notify(INFO) << _("Rejecting cluster")
				             << ": " << *it << std::endl;
				clusterurllist.erase (cli);
				break;
			}

	std::list<Queue> queuelist =
		GetQueueInfo (clusterurllist, MDS_FILTER_CLUSTERINFO,
		              anonymous, "", timeout);
	Time infotime;

	std::map<int, std::string> notsubmitted;

	int jobnr = 1;
	std::list<std::string> jobids;

	if (clusterselect.size()==1 && queuelist.size()==0) {
		notify(ERROR) << _("Job submission failed due to")
		              << ": " << _("The specified cluster") << " (" << *clusterselect.begin()
		              << ") " << _("did not return any information") << ". " << std::endl;
		return jobids;
	}

	for (std::list<Xrsl>::iterator it = xrsllist.begin();
	     it != xrsllist.end(); it++, jobnr++) {

		// if more than 5 minutes has passed, renew queue-info
		Time now;
		if (now.GetTime() - infotime.GetTime() > 300) {
			queuelist = GetQueueInfo(clusterurllist, MDS_FILTER_CLUSTERINFO,
			                         anonymous, "", timeout);
			infotime = now;
		}

		std::string jobid;

		std::string jobname;
		try {
			jobname = it->GetRelation("jobname").GetSingleValue();
		}
		catch (XrslError e) {}

		try {
			it->Eval();

			PerformXrslValidation(*it, unknownattr);

			std::list<Target> targetlist = ConstructTargets(queuelist, *it);

			PerformStandardBrokering(targetlist);

			JobSubmission submit(*it, targetlist, dryrun);

			if (dumpxrsl) {
				if(targetlist.begin() != targetlist.end()) {
					notify(INFO) << _("Selected queue") << ": "
					             << targetlist.begin()->name << "@"
					             << targetlist.begin()->cluster.hostname << std::endl;

					Xrsl jobxrsl = submit.PrepareXrsl(*targetlist.begin());
					std::cout << jobxrsl.str() << std::endl;
				} else {
					notify(WARNING) << _("No suitable target found") << std::endl;
				};
				continue;
			}

			jobid = submit.Submit(timeout);

			submit.RegisterJobsubmission(queuelist);

		}
		catch (ARCLibError e) {
			notify(ERROR) << _("Job submission failed due to")
			              << ": " << e.what() << std::endl;
			notsubmitted[jobnr] = jobname;
			continue;
		}

		AddJobID(jobid, jobname);

		if (!joblistfile.empty()) {
			LockFile(joblistfile);
			std::ofstream jobs(joblistfile.c_str(), std::ios::app);
			jobs << jobid << std::endl;
			jobs.close();
			UnlockFile(joblistfile);
		}

		std::string histfilename = GetEnv("HOME");
		histfilename.append ("/.arc/history");
		LockFile(histfilename);
		std::ofstream nghist (histfilename.c_str(), std::ios::app);
		nghist << TimeStamp() << "  " << jobid << std::endl;
		nghist.close();
		UnlockFile(histfilename);

		std::cout << _("Job submitted with jobid")
				  << ": " << jobid << std::endl;

		jobids.push_back(jobid);
	}

	if (xrsllist.size()>1) {
	  std::cout << std::endl << _("Job submission summary:") << std::endl;
		std::cout << "-----------------------" << std::endl;
		std::cout << xrsllist.size() - notsubmitted.size()
		          << " " << _("of") << " " << xrsllist.size() << " " << _("jobs") << " " 
			  << _("were submitted") << "." << std::endl << std::endl;
		if (notsubmitted.size()) {
		  std::cout << _("The following") << " " << notsubmitted.size() << " " << _("jobs") << " "
			    << _("were not submitted") << ":" << std::endl;
			std::map<int, std::string>::iterator it;
			for (it = notsubmitted.begin(); it != notsubmitted.end(); it++) {
			  std::cout << _("Job nr.") << " " << it->first;
				if (it->second.size()>0) std::cout << ": " << it->second;
				std::cout << std::endl;
			}
		}
	}

	return jobids;
}
