#include "arccli.h"

#include <fstream>
#include <iostream>
#include <list>
#include <string>

#include <arc/common.h>
#include <arc/datetime.h>
#include <arc/giis.h>
#include <arc/mdsdiscovery.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/resource.h>
#include <arc/url.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


void arcsync (const std::list<std::string>& clusterselect,
              const std::list<std::string>& clusterreject,
              const std::list<std::string>& giisurls,
              const bool force,
              const int timeout,
              const bool anonymous) {

	if (!force) {
		std::cout << _(
"Synchronizing the local list of active jobs with the information in the MDS\n"
"can result in some inconsistencies. Very recently submitted jobs might not\n"
"yet be present in the MDS information, whereas jobs very recently scheduled\n"
"for deletion can still be present."
		                 ) << std::endl;
		std::cout << _(
"Are you sure you want to synchronize your local job list?"
		                 ) << " ["<< _("y") << "/" << _("n") << "] ";
		std::string response;
		std::cin >> response;
		if (response != _("y")) {
			notify(ERROR) << _("Cancelling synchronization request")
			              << std::endl;
			return;
		}
	}

	std::list<URL> clusterurllist;

	for (std::list<std::string>::const_iterator it = clusterselect.begin();
	     it != clusterselect.end(); it++) {
		bool found = false;
		for (std::list<URL>::iterator cli = clusterurllist.begin();
		     !found && cli != clusterurllist.end(); cli++)
			if (*it == cli->Host()) found = true;
		if (!found)
			clusterurllist.push_back("ldap://" + *it +
			                         ":2135/O=Grid/Mds-Vo-Name=local");
	}

	if (clusterurllist.empty()) {
		std::list<URL> giisurllist = ConvertToURLs(giisurls);
		clusterurllist =
			GetClusterResources(giisurllist, anonymous, "", timeout);
		if (clusterurllist.empty())
			throw ARCCLIError(_("Could not retrieve cluster list from GIIS"));
	}

	for (std::list<std::string>::const_iterator it = clusterreject.begin();
	     it != clusterreject.end(); it++)
		for (std::list<URL>::iterator cli = clusterurllist.begin();
		     cli != clusterurllist.end(); cli++)
			if (cli->Host() == *it) {
				notify(INFO) << _("Rejecting cluster")
				             << ": " << *it << std::endl;
				clusterurllist.erase (cli);
				break;
			}

	std::list<Job> joblist =
		GetAllJobs(clusterurllist, anonymous, "", timeout);

	std::string filename = GetEnv("HOME");
	filename.append("/.ngjobs");

	LockFile(filename);

	if (!joblist.empty()) {
		notify(WARNING) << _("Found the following jobs") << ":" << std::endl;
		std::ofstream jobfile(filename.c_str());
		while (!joblist.empty()) {
			std::list<Job>::iterator job = joblist.begin();
			for(std::list<Job>::iterator jit = job;
				jit != joblist.end(); jit++) {
				if ((jit->job_name < job->job_name) ||
				    ((jit->job_name == job->job_name) &&
				     (jit->submission_time < job->submission_time)))
					job = jit;
			}
			jobfile << job->id << '#' << job->job_name << std::endl;
			if (job->job_name.empty())
				notify(WARNING) << job->id << std::endl;
			else
				notify(WARNING) << job->job_name << std::endl;
			joblist.erase(job);
		}
		jobfile.close();
	}
	else {
		notify(ERROR) << _("No jobs found") << std::endl;
		unlink(filename.c_str());
	}

	UnlockFile(filename);
}
