#include "../std.h"

#include <arc/notify.h>
#include <arc/url.h>

#include "datamove.h"
#include "datahandle.h"

#include "databroker.h"


// Assume cached files are always available
#define CACHE_RATIO (0.0)

// Assume files at local SE may be accessed at least 10 time faster
#define LOCAL_RATIO (0.1)

class DataSize {
 public:
  long long int full;
  long long int cached;
  long long int local;
  long long int effective;
  DataSize(void):full(0),cached(0),local(0),effective(0) { };
};

class DataTarget {
 public:
  Target* target;
  DataSize* size;
  bool cached;
  bool local;
  DataTarget(Target& target_,DataSize& size_):target(&target_),size(&size_),cached(false),local(false) {};
};

class DataLocation {
 public:
  std::string url;
  std::list<DataTarget> targets;
  long long int size;
  bool resolved;
  DataLocation(const std::string& url_):url(url_),resolved(false),size(0) { }; 
};

class DataLocations {
 public:
  std::list<DataLocation> locations;
  DataLocations(void) { };
  void Add(const std::string& url_,Target& target_,DataSize& size_) {
    for(std::list<DataLocation>::iterator loc = locations.begin();
                                        loc!=locations.end();++loc) {
      if(url_ == loc->url) {
        loc->targets.push_back(DataTarget(target_,size_));
        return;
      };
    };
    std::list<DataLocation>::iterator loc = 
               locations.insert(locations.end(),DataLocation(url_));
    loc->targets.push_back(DataTarget(target_,size_));
  };
};


void DataBroker::DoBrokering(std::list<Target>& targets) {

  DataLocations locations;
  std::list<DataSize> sizes;

  // Collect all needed files
  std::list<Target>::iterator targit = targets.begin();
  for(;targit != targets.end();++targit) {
    std::list<Xrsl>& xrsls = targit->GetXrsls(); // target sub-xrsls
    std::list<Xrsl>::iterator xrslit = xrsls.begin();
    std::list<DataSize>::iterator s = sizes.insert(sizes.end(),DataSize());
    for(;xrslit != xrsls.end();++xrslit) {
      std::list<std::list<std::string> > files;
      if (xrslit->IsRelation("inputfiles")) {
        files = xrslit->GetRelation("inputfiles").GetDoubleListValue();
      };
      std::list<std::list<std::string> >::iterator fileit;
      for (fileit = files.begin(); fileit != files.end(); fileit++) {
        std::string location = *(++fileit->begin());
        if(location.find(':') != std::string::npos) {
          locations.Add(location,*targit,*s);
        };
      };
    };
  };
  if(locations.locations.size() == 0) return;

  // Collect information about files (size, cache location)
  for(std::list<DataLocation>::iterator locit = locations.locations.begin();
                                locit != locations.locations.end();++locit) {
    notify(INFO)<<"Retrieving information about "<<locit->url<<std::endl;
    DataPoint point(locit->url.c_str());
    if(!point) continue;
    if(!point.meta_resolve(true)) continue;
    if(!point.have_locations()) continue;
    // Check for cached and local files first
    point.tries(1);
    while(point.have_location()) {
      if(strncasecmp("cache://",point.current_location(),8) == 0) {
        try {
          URL u(point.current_location());
          for(std::list<DataTarget>::iterator dtit = locit->targets.begin();
                                     dtit != locit->targets.end();++dtit) {
            if(dtit->target->cluster.hostname == u.Host()) {
              dtit->cached=true;
            };
          };
        } catch(ARCLibError) {};
      } else {
        try {
          URL u(point.current_location());
          std::string point_url = u.CanonicalURL();
          // TODO: better comparison
          for(std::list<DataTarget>::iterator dtit = locit->targets.begin();
                                     dtit != locit->targets.end();++dtit) {
            for(std::list<std::string>::iterator lsit = 
                     dtit->target->cluster.local_se.begin();
                     lsit!=dtit->target->cluster.local_se.end();++lsit) {
           
              if(strncasecmp(u.CanonicalURL().c_str(),lsit->c_str(),lsit->length())==0){
                dtit->local=true; break;
              };
            };
          }; 
        } catch(ARCLibError) {};
      };
      point.next_location();
    };
    // Look for size
    if(point.meta_size_available()) {
      locit->size=point.meta_size();
      locit->resolved=true;
      notify(DEBUG)<<"File size is "<<locit->size<<std::endl;
      continue;
    };
    point.tries(1);
    while(point.have_location()) {
      notify(DEBUG)<<"Retrieving information directly about "<<point.current_location()<<std::endl;
      DataHandle handle(&point);
      if(handle.check()) {
        if(point.meta_size_available()) {
          locit->size=point.meta_size();
          locit->resolved=true;
          notify(DEBUG)<<"File size is "<<locit->size<<std::endl;
          break;
        };
      };
      point.next_location();
    };
  };

  // Calculate size per target
  for(std::list<DataLocation>::iterator locit = locations.locations.begin();
                             locit != locations.locations.end();++locit) {
    for(std::list<DataTarget>::iterator dtit = locit->targets.begin();
                               dtit != locit->targets.end();++dtit) {
      dtit->size->full+=locit->size;
      notify(DEBUG)<<"File "<<locit->url<<
                     " - size "<<locit->size<<
                     " - for target "<<dtit->target->cluster.hostname<<" is "<<
                     ((dtit->cached)?"":"not ")<<"cached, "<<
                     ((dtit->local)?"":"not ")<<"local"<<std::endl;
      if(dtit->cached) {
        dtit->size->effective+=(long long int)(locit->size * CACHE_RATIO);
        dtit->size->cached+=locit->size;
      } else if(dtit->local) {
        dtit->size->effective+=(long long int)(locit->size * LOCAL_RATIO);
        dtit->size->local+=locit->size;
      } else {
        dtit->size->effective+=locit->size;
      };
    };
  };

  // Sort by (effective) size
  // TODO: options to control which size to take into account
  std::list<Target>::iterator tgit1 = targets.begin();
  std::list<DataSize>::iterator dsit1 = sizes.begin();
  for(;(tgit1!=targets.end()) && (dsit1!=sizes.end());++tgit1,++dsit1) {
    notify(DEBUG)<<"Target "<<tgit1->name<<"@"<<
                   tgit1->cluster.hostname<<" has effective data size "<<
                   dsit1->effective<<" of "<<dsit1->full<<std::endl;
  };
  tgit1 = targets.begin();
  dsit1 = sizes.begin();
  for(;(tgit1!=targets.end()) && (dsit1!=sizes.end());++tgit1,++dsit1) {
    std::list<Target>::iterator tgit2 = tgit1; ++tgit2;
    std::list<DataSize>::iterator dsit2 = dsit1; ++dsit2;
    for(;(tgit2!=targets.end()) && (dsit2!=sizes.end());++tgit2,++dsit2) {
      if(dsit2->effective < dsit1->effective) {
        Target t_tmp = *tgit1; *tgit1=*tgit2; *tgit2=t_tmp;
        DataSize d_tmp = *dsit1; *dsit1=*dsit2; *dsit2=d_tmp;
      };
    };
  };
}


extern "C" {

  void DataBrokerWrapper (std::list<Target> &targets) {

    DataBroker b;
    b.DoBrokering(targets);
  }

}
