#include "../std.h"
#include <iostream>

#include "../misc/url_options.h"
#include "../misc/stringtoint.h"
#include "../misc/mkdir_recursive.h"
#include "../misc/checkfile.h"
#include "../misc/log_time.h"
#include "../misc/inttostring.h"
#include "datapoint.h"
#include "databufferpar.h"
#include "datahandle_file.h"
#include "datacallback.h"

DataHandle* DataHandleFile::CreateInstance(DataPoint* url_) {
  if((!url_) || (!*url_)) return NULL;
  const char* cur_url = url_->current_location();
  if(strncasecmp("file://",cur_url,7) && strcmp("-",cur_url)) return NULL;
  return new DataHandleFile(url_);
}

DataHandleFile::DataHandleFile(DataPoint* url_):DataHandleCommon(url_) {
}

DataHandleFile::~DataHandleFile(void) {
  stop_reading();
  stop_writing();
  deinit_handle();
}

bool DataHandleFile::init_handle(void) {
  if(!DataHandleCommon::init_handle()) return false;
  const char* cur_url = url->current_location();
  if(!strncasecmp("file:/",cur_url,6)) {
    cacheable=false; is_channel=false;
  } else if(!strcmp("-",cur_url)) {
    cacheable=false; linkable=false; is_channel=true;
  } else {
    return false;
  };
  return true;
}

bool DataHandleFile::deinit_handle(void) {
  if(!DataHandleCommon::deinit_handle()) return false;
  return true;
}

void* DataHandleFile::read_file(void* arg) {
  DataHandleFile* it = (DataHandleFile*)arg;
  bool limit_length = false;
  unsigned long long int range_length;
  unsigned long long int offset = 0;
  if(it->range_end > it->range_start) {
    range_length = it->range_end - it->range_start;
    limit_length = true;
    lseek(it->fd,it->range_start,SEEK_SET);
    offset=it->range_start;
  } else {
    lseek(it->fd,0,SEEK_SET);
  };
  for(;;) {
    if(limit_length) if(range_length==0) break;
  /* read from fd here and push to buffer */
  /* 1. claim buffer */
    int h;
    unsigned int l;
    if(!it->buffer->for_read(h,l,true)) { 
      /* failed to get buffer - must be error or request to exit */
      it->buffer->error_read(true);
      break;
    };
    if(it->buffer->error()) {
      it->buffer->is_read(h,0,0);
      break;
    };
  /* 2. read */
    if(limit_length) if(l>range_length) l=range_length;
    unsigned long long int p = lseek(it->fd,0,SEEK_CUR);
    if(p == (unsigned long long int)(-1)) p=offset;
    int ll = read(it->fd,(*(it->buffer))[h],l);
    if(ll == -1) { /* error */
      it->buffer->is_read(h,0,0);
      it->buffer->error_read(true);
      break;
    };
    if(ll == 0) { /* eof */
      it->buffer->is_read(h,0,0);
      break;
    };
  /* 3. announce */
    it->buffer->is_read(h,ll,p);
    if(limit_length) {
      if(ll>range_length) { range_length=0; } else { range_length-=ll; };
    };
    offset+=ll; // for non-seakable files
  };
  close(it->fd);
  it->buffer->eof_read(true);
  /* inform parent thread about exit */
  it->file_thread_exited.signal();
  return NULL;
}

void* DataHandleFile::write_file(void* arg) {
  DataHandleFile* it = (DataHandleFile*)arg;
  for(;;) {
  /* take from buffer and write to fd */
  /* 1. claim buffer */
    int h;
    unsigned int l;
    unsigned long long int p;
    if(!it->buffer->for_write(h,l,p,true)) {
      /* failed to get buffer - must be error or request to exit */
      if(!it->buffer->eof_read()) it->buffer->error_write(true);
      it->buffer->eof_write(true);
      break;
    };
    if(it->buffer->error()) {
      it->buffer->is_written(h);
      it->buffer->eof_write(true);
      break;
    };
  /* 2. write */
    lseek(it->fd,p,SEEK_SET);
    int l_=0;
    int ll = 0;
    for(;l_<l;) {
      ll = write(it->fd,(*(it->buffer))[h]+l_,l-l_);
      if(ll == -1) { /* error */
        it->buffer->is_written(h);
        it->buffer->error_write(true);
        it->buffer->eof_write(true);
        break;
      };
      l_+=ll;
    };
    if(ll == -1) break;
  /* 3. announce */
    it->buffer->is_written(h);
  };
  close(it->fd);
  /* inform parent thread about exit */
  it->file_thread_exited.signal();
  return NULL;
}

bool DataHandleFile::check(void) {
  if(!DataHandleCommon::check()) return false;
  const char* url_path = get_url_path(c_url.c_str());
  int res=check_file_access(url_path,O_RDONLY,get_user_id(),(gid_t)(-1));
  if(res != 0) {
    odlog(INFO)<<"File is not accessible: "<<url_path<<std::endl;
    return false;
  };
  struct stat st;
  if(stat(url_path,&st) != 0) {
    odlog(INFO)<<"Can't stat file: "<<url_path<<std::endl;
    return false;
  };
  url->meta_size(st.st_size);
  url->meta_created(st.st_mtime);
  return true;
}

bool DataHandleFile::remove(void) {
  if(!DataHandleCommon::remove()) return false;
  const char* path = get_url_path(c_url.c_str());
  if(unlink(path) == -1) {
    if(errno != ENOENT) return false;
  };
  return true;
}

bool DataHandleFile::start_reading(DataBufferPar &buf) {
  if(!DataHandleCommon::start_reading(buf)) return false;
  /* try to open */
  file_thread_exited.reset();
  if(!strcmp("-",c_url.c_str())) {
    fd=dup(STDIN_FILENO);
  } else {
    if(check_file_access(get_url_path(c_url.c_str()),O_RDONLY,
                       get_user_id(),(gid_t)(-1)) != 0) {
      DataHandleCommon::stop_reading();
      return false;
    };
    fd=open(get_url_path(c_url.c_str()),O_RDONLY);
  };
  if(fd == -1) {
    DataHandleCommon::stop_reading();
    return false;
  };
  /* provide some metadata */
  struct stat st;
  if(fstat(fd,&st) == 0) {
    url->meta_size(st.st_size);
    url->meta_created(st.st_mtime);
  };
  buffer=&buf;
  /* create thread to maintain reading */
  pthread_attr_init(&file_thread_attr);
  pthread_attr_setdetachstate(&file_thread_attr,PTHREAD_CREATE_DETACHED);
  if(pthread_create(&file_thread,&file_thread_attr,&read_file,this) != 0) {
    pthread_attr_destroy(&file_thread_attr);
    close(fd); fd=-1; 
    DataHandleCommon::stop_reading();
    return false;
  };
  return true;
}

bool DataHandleFile::stop_reading(void) {
  if(!DataHandleCommon::stop_reading()) return false;
  if(!buffer->eof_read()) {
    buffer->error_read(true);        /* trigger transfer error */
    close(fd); fd=-1;
  };
  file_thread_exited.wait();  /* wait till reading thread exited */
  pthread_attr_destroy(&file_thread_attr);
  return true;
}

bool DataHandleFile::start_writing(DataBufferPar &buf,DataCallback *space_cb) {
  if(!DataHandleCommon::start_writing(buf,space_cb)) return false;
  /* try to open */
  file_thread_exited.reset();
  buffer=&buf;
  if(!strcmp("-",c_url.c_str())) {
    fd=dup(STDOUT_FILENO);
    if(fd == -1) {
      odlog(ERROR)<<"Failed to use channel stdout"<<std::endl;
      buffer->error_write(true); buffer->eof_write(true);
      DataHandleCommon::stop_writing();
      return false;
    };
  } else {
    uid_t uid=get_user_id();
    gid_t gid=get_user_group(uid);
    /* do not check permissions to create anything here - 
       suppose it path was checked at higher level */
    /* make directories */
    const char* path_ = get_url_path(c_url.c_str());
    if(path_ == NULL) {
      odlog(ERROR)<<"Invalid url: "<<c_url<<std::endl;
      buffer->error_write(true); buffer->eof_write(true);
      DataHandleCommon::stop_writing();
      return false;
    };
    std::string path = path_;
    {
      if(path[0] != '/') path="/"+path;
      std::string dirpath = path;
      int n = dirpath.rfind('/');
      if(n == 0) {
        dirpath="/";
      } else {
        dirpath.erase(n,dirpath.length()-n+1);
      };
      if(mkdir_recursive(NULL,dirpath.c_str(),S_IRWXU,uid,gid) != 0) {
        if(errno != EEXIST) {
          odlog(ERROR)<<"Failed to create/find directory "<<dirpath<<std::endl;
          buffer->error_write(true); buffer->eof_write(true);
          DataHandleCommon::stop_writing();
          return false;
        };
      }; 
    };
    /* try to create file, if failed - try to open it */
    fd=open(path.c_str(),O_WRONLY | O_CREAT | O_EXCL,S_IRUSR | S_IWUSR);
    if(fd == -1) {
      fd=open(path.c_str(),O_WRONLY | O_TRUNC,S_IRUSR | S_IWUSR);
    }
    else { /* this file was created by us. Hence we can set it's owner */
      fchown(fd,uid,gid);
    };
    if(fd == -1) {
      odlog(ERROR)<<"Failed to create/open file "<<path<<std::endl;
      buffer->error_write(true); buffer->eof_write(true);
      DataHandleCommon::stop_writing();
      return false;
    };

    /* preallocate space */
    buffer->speed.hold(true);
    unsigned long long int fsize = url->meta_size();
    if(fsize > 0) {
      odlog(INFO)<<"setting file "<<get_url_path(c_url.c_str())<<
                " to size "<<fsize<<std::endl;
      /* because filesytem can skip empty blocks do real write */
      unsigned long long int old_size=lseek(fd,0,SEEK_END);
      if(old_size < fsize) {
        char buf[65536];
        memset(buf,0xFF,sizeof(buf));
        unsigned int l=1;
        for(;l>0;) {
          old_size=lseek(fd,0,SEEK_END);
          l=sizeof(buf);
          if(l > (fsize-old_size)) l=fsize-old_size;
          if(write(fd,buf,l) == -1) {
            /* out of space */
            perror("write"); /* not good, but let it be */
            if(space_cb != NULL) {
              if(space_cb->cb((unsigned long long int)l)) continue;
            };
            lseek(fd,0,SEEK_SET); 
            ftruncate(fd,0);
            close(fd); fd=-1; 
            odlog(INFO)<<"Failed to preallocate space"<<std::endl;
            buffer->speed.reset();
            buffer->speed.hold(false);
            buffer->error_write(true); buffer->eof_write(true);
            DataHandleCommon::stop_writing();
            return false;
          };
        };
      };
    };
  };
  buffer->speed.reset();
  buffer->speed.hold(false);
  /* create thread to maintain writing */
  pthread_attr_init(&file_thread_attr);
  pthread_attr_setdetachstate(&file_thread_attr,PTHREAD_CREATE_DETACHED);
  if(pthread_create(&file_thread,&file_thread_attr,&write_file,this) != 0) {
    pthread_attr_destroy(&file_thread_attr);
    close(fd); fd=-1;
    buffer->error_write(true); buffer->eof_write(true);
    DataHandleCommon::stop_writing();
    return false;
  };
  return true;
}

bool DataHandleFile::stop_writing(void) {
  if(!DataHandleCommon::stop_writing()) return false;
  if(!buffer->eof_write()) {
    buffer->error_write(true);        /* trigger transfer error */
    close(fd); fd=-1;
  };
  file_thread_exited.wait();  /* wait till reading thread exited */
  pthread_attr_destroy(&file_thread_attr);
  return true;
}

bool DataHandleFile::list_files(std::list<DataPoint::FileInfo> &files,bool long_list,bool resolve,bool metadata) {
  if(!DataHandleCommon::list_files(files,long_list,resolve,metadata)) return false;
  std::string dirname = get_url_path(c_url.c_str());
  if(dirname[dirname.length()-1] == '/') dirname.resize(dirname.length()-1);
  DIR *dir=opendir(dirname.c_str());
  if(dir == NULL || metadata) {
    // maybe a file
    struct stat st;
    std::list<DataPoint::FileInfo>::iterator f = 
               files.insert(files.end(),DataPoint::FileInfo(dirname.c_str()));
    if(stat(dirname.c_str(),&st) == 0) {
      f->metadata["path"] = dirname;
      f->size=st.st_size; f->size_available=true;
      f->metadata["size"] = inttostring(st.st_size);
      f->created=st.st_mtime; f->created_available=true;
      const time_t creationtime = st.st_mtime;
      std::string ctimestr = ctime(&creationtime);
      f->metadata["mtime"] = ctimestr.erase(ctimestr.length()-1);
      if(S_ISDIR(st.st_mode)) {
        f->type=DataPoint::FileInfo::file_type_dir;
        f->metadata["type"] = "file";
      } else if(S_ISREG(st.st_mode)) {
        f->type=DataPoint::FileInfo::file_type_file;
        f->metadata["type"] = "dir";
      };
      // fill some more metadata
      const time_t accesstime = st.st_atime;
      std::string atimestr = ctime(&accesstime);
      f->metadata["atime"] = atimestr.erase(atimestr.length()-1);
      const time_t changetime = st.st_ctime;
      std::string chtimestr = ctime(&changetime);
      f->metadata["ctime"] = chtimestr.erase(chtimestr.length()-1);
      f->metadata["group"] = inttostring(st.st_gid);
      f->metadata["owner"] = inttostring(st.st_uid);
      unsigned int mode = st.st_mode;
      std::string perms;
      if (mode & S_IRUSR) perms += 'r'; else perms += '-';
      if (mode & S_IWUSR) perms += 'w'; else perms += '-';
      if (mode & S_IXUSR) perms += 'x'; else perms += '-';
      if (mode & S_IRGRP) perms += 'r'; else perms += '-';
      if (mode & S_IWGRP) perms += 'w'; else perms += '-';
      if (mode & S_IXGRP) perms += 'x'; else perms += '-';
      if (mode & S_IROTH) perms += 'r'; else perms += '-';
      if (mode & S_IWOTH) perms += 'w'; else perms += '-';
      if (mode & S_IXOTH) perms += 'x'; else perms += '-';
      f->metadata["accessperm"] = perms;
      return true;
    };
    files.erase(f);
    odlog(INFO)<<"Failed to read object: "<<dirname<<std::endl;
    return false;
  };
  for(;;) {
    struct dirent file_;
    struct dirent *file;
    readdir_r(dir,&file_,&file);
    if(file == NULL) break;
    if(file->d_name[0] == '.') {
      if(file->d_name[1] == 0) continue;
      if(file->d_name[1] == '.') if(file->d_name[2] == 0) continue;
    };
    std::list<DataPoint::FileInfo>::iterator f = 
               files.insert(files.end(),DataPoint::FileInfo(file->d_name));
    if(long_list) {
      std::string fname =  dirname+"/"+file->d_name;
      struct stat st;
      if(stat(fname.c_str(),&st) == 0) {
        f->size=st.st_size; f->size_available=true;
        f->created=st.st_mtime; f->created_available=true;
        if(S_ISDIR(st.st_mode)) {
          f->type=DataPoint::FileInfo::file_type_dir;
        } else if(S_ISREG(st.st_mode)) {
          f->type=DataPoint::FileInfo::file_type_file;
        };
      };
    };
  }; 
  return true;
}

bool DataHandleFile::analyze(analyze_t &arg) {
  if(!DataHandleCommon::analyze(arg)) return false;
  const char* cur_url = url->current_location();
  if(!strcmp("-",cur_url)) { arg.cache=false; arg.readonly=false; };
  if(!strncasecmp("file:/",cur_url,6)) {
    arg.local=true; arg.cache=false;
  };
  return true;
}

bool DataHandleFile::out_of_order(void) {
  if(!url) return false;
  const char* cur_url = url->current_location();
  if(!strncasecmp("file:/",cur_url,6)) return true;
  return false;
}

