#include "../std.h"
#include <string>
#include <list>
#include <iostream>

//#include "../misc/condition.h"
//#include "../misc/checksum.h"

//#include "../https/SRM/srm1_soapH.h"
//extern SOAP_NMAC struct Namespace srm1_soap_namespaces[];
#include "../https/SRM/srm_client.h"
#include "../misc/random.h"
#include "../misc/url_options.h"
#include "../misc/log_time.h"
#include "../misc/inttostring.h"
#include "databufferpar.h"
#include "datapoint.h"
#include "datahandle_srm.h"

DataHandle* DataHandleSRM::CreateInstance(DataPoint* url_) {
  if((!url_) || (!*url_)) return NULL;
  const char* cur_url = url_->current_location();
  if(strncasecmp("srm://",cur_url,6)) return NULL;
  return new DataHandleSRM(url_);
}

DataHandleSRM::DataHandleSRM(DataPoint* url_):DataHandleCommon(url_) {
  r_handle=NULL; r_url=NULL;
  srm_request=NULL;
}

DataHandleSRM::~DataHandleSRM(void) {
  stop_reading();
  stop_writing();
  deinit_handle();
}

bool DataHandleSRM::init_handle(void) {
  if(!DataHandleCommon::init_handle()) return false;
  const char* cur_url = url->current_location();
  if(!strncasecmp("srm://",cur_url,6)) return true;
  return false;
}

bool DataHandleSRM::deinit_handle(void) {
  if(!DataHandleCommon::deinit_handle()) return false;
  if(r_handle) {
    //r_handle->stop_reading(); // just in case
    //r_handle->stop_writing(); // just in case
    //r_handle->deinit_handle();
    delete r_handle; r_handle=NULL;
  };
  if(r_url) { delete r_url; r_url=NULL; };
  return true;
}

bool DataHandleSRM::analyze(analyze_t &arg) {
  return DataHandleCommon::analyze(arg);
}

bool DataHandleSRM::start_reading(DataBufferPar &buf) {
#ifndef SRM_MISSING
  if(r_handle) return false; // must be operation in progress
  if(!DataHandleCommon::start_reading(buf)) return false;
  buffer=&buf;
  SRMClient * client = SRMClient::getInstance(std::string(url->current_location()), buffer->speed.get_max_inactivity_time());
  if(!client) {
    DataHandleCommon::stop_reading();
    return false;
  };
  r_url=NULL; r_handle=NULL; srm_request=NULL;
  std::list<std::string> turls;

  // take out options in srm url
  std::string plain_url = url->current_location();
  if (canonic_url(plain_url)) goto error_exit;
  srm_request = new SRMClientRequest(plain_url);

  if(!srm_request) goto error_exit;
  if(!no_checks) {
    odlog(DEBUG)<<"start_reading_srm: looking for metadata: "<<c_url.c_str()<<std::endl;
    std::list<struct SRMFileMetaData> metadata;
    if(!client->info(*srm_request,metadata)) {
      DataHandleCommon::stop_reading();
      return false;
    };
    /* provide some metadata */
    if(!metadata.empty()){
      odlog(INFO)<<"start_reading_srm: obtained size: "<<metadata.front().size<<std::endl;
      if(metadata.front().size > 0) url->meta_size(metadata.front().size);
      odlog(INFO)<<"start_reading_srm: obtained checksum: "<<metadata.front().checkSumType<<":"<<metadata.front().checkSumValue<<std::endl;
      if(metadata.front().checkSumValue.length() > 0 &&
         metadata.front().checkSumType.length() > 0) {
        std::string csum(metadata.front().checkSumType+":"+metadata.front().checkSumValue);
        url->meta_checksum(csum.c_str());
      };
    };
  };
  if(!client->getTURLs(*srm_request,turls)) {
    goto error_exit;
  };
  client->disconnect();
  // Choose handled URL randomly
  for(;;) {
    if(turls.size() <= 0) break;
    int n = Random::get(turls.size()-1);
    std::list<std::string>::iterator i = turls.begin(); for(;n;++i,++n) { };
    if(i == turls.end()) continue;
    // Avoid redirection to SRM
    odlog(DEBUG)<<"Checking URL returned by SRM: "<<*i<<std::endl;
    if(strncasecmp(i->c_str(),"srm://",6) == 0) { turls.erase(i); continue; };
    // Try to use this TURL + old options
    {
      std::string options;
      get_url_options(url->current_location(),options);
      if(options.length()) add_url_options(*i,options.c_str(),0);
    };
    r_url = DataPoint::CreateInstance(i->c_str());
    // check if url can be handled
    if(!r_url) { turls.erase(i); continue; };
    if(r_url->meta()) { delete r_url; r_url=NULL; turls.erase(i); continue; };
    break;
  };
  if(r_url == NULL) {
    odlog(INFO)<<"SRM returned no useful Transfer URLs: "<<c_url<<std::endl;
    goto error_exit;
  };
  r_handle = new DataHandle(r_url);
  r_handle->additional_checks(false); // checks at higher levels are always done on SRM metadata
  r_handle->secure(force_secure);
  r_handle->passive(force_passive);
  //r_handle->transfer_streams=transfer_streams;
  //r_handle->cacheable=cacheable;
  //r_handle->linkable=linkable;
  odlog(INFO)<<"Redirecting to new URL: "<<*r_url<<std::endl;
  if(!(r_handle->start_reading(buf))) goto error_exit;
  return true;
error_exit:
  if(r_handle) delete r_handle; r_handle=NULL;
  if(r_url) delete r_url; r_url=NULL;
  if(srm_request) delete srm_request; srm_request=NULL;
  if(client) { delete client; client=NULL; }
  DataHandleCommon::stop_reading();
#endif
  return false;
}

bool DataHandleSRM::stop_reading(void) {
#ifndef SRM_MISSING
  if(!r_handle) return true;
  if(!DataHandleCommon::stop_reading()) return false;
  bool r = r_handle->stop_reading();
  delete r_handle;
  if(r_url) delete r_url;
  if(srm_request) {
    SRMClient * client = SRMClient::getInstance(std::string(url->current_location()), buffer->speed.get_max_inactivity_time());
    if(client) {
      client->releaseGet(*srm_request);
    };
    delete srm_request;
    if(client) { delete client; client=NULL; }
  };
  r_handle=NULL; r_url=NULL; srm_request=NULL;
  return r;
#else
  return false;
#endif
}

bool DataHandleSRM::start_writing(DataBufferPar &buf,DataCallback *space_cb) {
#ifndef SRM_MISSING
  if(r_handle) return false; // must be operation in progress
  if(!DataHandleCommon::start_writing(buf)) return false;
  buffer = &buf;
  SRMClient * client = SRMClient::getInstance(std::string(url->current_location()), buffer->speed.get_max_inactivity_time());
  if(!client) {
    DataHandleCommon::stop_writing();
    return false;
  };
  r_url=NULL; r_handle=NULL; srm_request=NULL;
  std::list<std::string> turls;
  std::string space_token;

  // take out options in srm url
  std::string plain_url = url->current_location();
  if (canonic_url(plain_url)) goto error_exit;
  srm_request = new SRMClientRequest(plain_url);

  if(!srm_request) goto error_exit;
  // set space token
  if (get_url_option(url->current_location(), "spacetoken", space_token)) {
    if(client->getVersion().compare("v2.2") == 0) {
      // only print message if using v2.2
      odlog(DEBUG)<<"No space token specified"<<std::endl;
    };
  }
  else {
    if(client->getVersion().compare("v2.2") != 0) {
      // print warning if not using srm2.2
      odlog(WARNING)<<"Warning: Using SRM protocol v1 which does not support space tokens"<<std::endl;
    }
    else {
      odlog(DEBUG)<<"Using space token description "<<space_token<<std::endl;
      // get token from SRM that matches description
      std::list<std::string> tokens;
      if(client->getSpaceTokens(tokens, space_token) != SRM_OK) {
        // not critical so log a warning
        odlog(WARNING)<<"Warning: Error looking up space tokens matching description "<<space_token<<". Will copy without using token"<<std::endl;
      }
      else if (tokens.empty()) {
        // not critical so log a warning
        odlog(WARNING)<<"Warning: No space tokens found matching description! Will copy without using token"<<std::endl;
      }
      else{
        // take the first one in the list
        odlog(DEBUG)<<"Using space token "<<tokens.front()<<std::endl;
        srm_request->space_token(tokens.front());
      };
    };
  };
  if(!client->putTURLs(*srm_request,turls,url->meta_size())) goto error_exit;
  client->disconnect();
  // Choose handled URL randomly
  for(;;) {
    if(turls.size() <= 0) break;
    int n = Random::get(turls.size()-1);
    std::list<std::string>::iterator i = turls.begin(); for(;n;++i,++n) { };
    if(i == turls.end()) continue;
    odlog(DEBUG)<<"Checking URL returned by SRM: "<<*i<<std::endl;
    // Avoid redirection to SRM
    if(strncasecmp(i->c_str(),"srm://",6) == 0) { turls.erase(i); continue; };
    // Try to use this TURL + old options
    {
      std::string options;
      get_url_options(url->current_location(),options);
      if(options.length()) add_url_options(*i,options.c_str(),0);
    };
    r_url = DataPoint::CreateInstance(i->c_str());
    // check if url can be handled
    if(!r_url) { turls.erase(i); continue; };
    if(r_url->meta()) { delete r_url; r_url=NULL; turls.erase(i); continue; };
    break;
  };
  if(r_url == NULL) {
    odlog(INFO)<<"SRM returned no useful Transfer URLs: "<<c_url<<std::endl;
    goto error_exit;
  };
  r_handle = new DataHandle(r_url);
  //r_handle->ftp_threads=ftp_threads;
  //r_handle->cacheable=cacheable;
  //r_handle->linkable=linkable;
  //r_handle->force_secure=force_secure;
  //r_handle->force_passive=force_passive;
  odlog(INFO)<<"Redirecting to new URL: "<<*r_url<<std::endl;
  if(!(r_handle->start_writing(buf))) goto error_exit;
  return true;
error_exit:
  if(r_handle) delete r_handle; r_handle=NULL;
  if(r_url) delete r_url; r_url=NULL;
  if(srm_request) delete srm_request; srm_request=NULL;
  if(client) { delete client; client=NULL; }
  DataHandleCommon::stop_writing();
#endif
  return false;
}

bool DataHandleSRM::stop_writing(void) {
#ifndef SRM_MISSING
  if(!r_handle) return true;
  if(!DataHandleCommon::stop_writing()) return false;
  bool r = r_handle->stop_reading();
  delete r_handle;
  if(r_url) delete r_url;
  if(srm_request) {
    SRMClient * client = SRMClient::getInstance(std::string(url->current_location()), buffer->speed.get_max_inactivity_time());
    if(client) {
      // call abort if failure, or releasePut on success
      if(buffer->error()) client->abort(*srm_request);
      else client->releasePut(*srm_request);
    };
    delete srm_request;
    if (client) { delete client; client=NULL; }
  };
  r_handle=NULL; r_url=NULL; srm_request=NULL;
  return r;
#else
  return false;
#endif
}

bool DataHandleSRM::remove(void) {
#ifndef SRM_MISSING
  if(!DataHandleCommon::remove()) return false;
  SRMClient * client = SRMClient::getInstance(std::string(url->current_location()));
  if(!client) return false;
  // take out options in srm url
  std::string plain_url = url->current_location();
  if (canonic_url(plain_url)) {
    odlog(ERROR)<<"Error converting URL "<<plain_url<<" to canonic URL"<<std::endl;
    delete client; client=NULL;
    return false;
  }
  srm_request = new SRMClientRequest(plain_url);
  if(!srm_request) { delete client; client=NULL; return false; }
  odlog(DEBUG)<<"remove_srm: deleting: "<<url->current_location()<<std::endl;
  if(!client->remove(*srm_request)) { delete client; client=NULL; return false; }
  if(client) { delete client; client=NULL; }
  return true;
#else
  return false;
#endif
}

bool DataHandleSRM::list_files(std::list<DataPoint::FileInfo> &files,bool long_list,bool resolve,bool metadata) {
#ifndef SRM_MISSING
  if(!DataHandleCommon::list_files(files,long_list,resolve,metadata)) return false;
  SRMClient * client = SRMClient::getInstance(std::string(url->current_location()));
  if(!client) return false;
  // take out options in srm url
  std::string plain_url = url->current_location();
  if (canonic_url(plain_url)) {
    odlog(ERROR)<<"Error converting URL "<<plain_url<<" to canonic URL"<<std::endl;
    delete client; client=NULL;
    return false;
  }
  srm_request = new SRMClientRequest(plain_url);
  if(long_list || metadata) srm_request->long_list(true);
  if(!srm_request) { delete client; client=NULL; return false; }
  odlog(DEBUG)<<"list_files_srm: looking for metadata: "<<plain_url<<std::endl;
  std::list<struct SRMFileMetaData> srm_metadata;

  // get info from SRM
  int recursion = 0;
  if (metadata) recursion = -1;
  if(!client->info(*srm_request,srm_metadata,recursion)) { delete client; client=NULL; return false; }
  
  if(srm_metadata.empty()) { delete client; client=NULL; return true; }
  // set URL attributes for surl requested (file or dir)
  if(srm_metadata.front().size > 0) url->meta_size(srm_metadata.front().size);
  if(srm_metadata.front().checkSumType.length() > 0 &&
     srm_metadata.front().checkSumValue.length() > 0) {
    std::string csum(srm_metadata.front().checkSumType+":"+srm_metadata.front().checkSumValue);
    url->meta_checksum(csum.c_str());
  };
  if(srm_metadata.front().createdAtTime > 0) url->meta_created(srm_metadata.front().createdAtTime);

  // set FileInfo attributes for surl requested and any files within a dir
  for (std::list<struct SRMFileMetaData>::iterator i = srm_metadata.begin();
       i != srm_metadata.end();
       ++i) {

    std::list<DataPoint::FileInfo>::iterator f =
      files.insert(files.end(),DataPoint::FileInfo((*i).path.c_str()));
    f->metadata["path"] = i->path;

    if ((*i).fileType==SRM_FILE) {
      f->type=DataPoint::FileInfo::file_type_file;
      f->metadata["type"] = "file";
    }
    else if ((*i).fileType==SRM_DIRECTORY) {
      f->type=DataPoint::FileInfo::file_type_dir;
      f->metadata["type"] = "dir";
    }

    if((*i).size >= 0) {
      f->size=(*i).size;
      f->size_available=true;
      f->metadata["size"] = inttostring(i->size);
    };
    if((*i).createdAtTime > 0) {
      f->created=(*i).createdAtTime;
      f->created_available=true;
      const time_t creationtime = i->createdAtTime;
      std::string ctimestr = ctime(&creationtime);
      f->metadata["ctime"] = ctimestr.erase(ctimestr.length()-1);
    };
    if((*i).checkSumType.length() > 0 &&
       (*i).checkSumValue.length() > 0) {
      std::string csum((*i).checkSumType+":"+(*i).checkSumValue);
      f->checksum=csum.c_str();
      f->checksum_available=true;
      f->metadata["checksum"] = csum;
    };
    if((*i).fileLocality == SRM_ONLINE) {
      f->latency="ONLINE";
      f->latency_available=true;
      f->metadata["latency"] = "ONLINE";
    }
    else if ((*i).fileLocality == SRM_NEARLINE) {
      f->latency="NEARLINE";
      f->latency_available=true;
      f->metadata["latency"] = "NEARLINE";
    };
    if(!(*i).arrayOfSpaceTokens.empty()) f->metadata["spacetokens"] = (*i).arrayOfSpaceTokens;
    if(!(*i).owner.empty()) f->metadata["owner"] = (*i).owner;
    if(!(*i).group.empty()) f->metadata["group"] = (*i).group;
    if(!(*i).permission.empty()) f->metadata["accessperm"] = (*i).permission;
    if((*i).lastModificationTime > 0) {
      const time_t modtime = i->lastModificationTime;
      std::string mtimestr = ctime(&modtime);
      f->metadata["mtime"] = mtimestr.erase(mtimestr.length()-1);
    };
    if((*i).lifetimeLeft != 0) f->metadata["lifetimeleft"] = inttostring((*i).lifetimeLeft);
    if((*i).lifetimeAssigned != 0) f->metadata["lifetimeassigned"] = inttostring((*i).lifetimeAssigned);

    if ((*i).retentionPolicy == SRM_REPLICA) f->metadata["retentionpolicy"] = "REPLICA";
    else if ((*i).retentionPolicy == SRM_OUTPUT) f->metadata["retentionpolicy"] = "OUTPUT";
    else if ((*i).retentionPolicy == SRM_CUSTODIAL)  f->metadata["retentionpolicy"] = "CUSTODIAL";

    if ((*i).fileStorageType == SRM_VOLATILE) f->metadata["filestoragetype"] = "VOLATILE";
    else if ((*i).fileStorageType == SRM_DURABLE) f->metadata["filestoragetype"] = "DURABLE";
    else if ((*i).fileStorageType == SRM_PERMANENT) f->metadata["filestoragetype"] = "PERMANENT"; 
  };
  delete client; client=NULL;
  return true;
#else
  return false;
#endif
}

bool DataHandleSRM::check(void) {
#ifndef SRM_MISSING
  if(!DataHandleCommon::check()) return false;
  SRMClient * client = SRMClient::getInstance(std::string(url->current_location()));
  if(!client) return false;
  // take out options in srm url
  std::string plain_url = url->current_location();
  if (canonic_url(plain_url)) {
    odlog(ERROR)<<"Error converting URL "<<plain_url<<" to canonic URL"<<std::endl;
    delete client; client=NULL;
    return false;
  }
  srm_request = new SRMClientRequest(plain_url);
  if(!srm_request) return false;
  odlog(DEBUG)<<"check_srm: looking for metadata: "<<url->current_location()<<std::endl;
  std::list<struct SRMFileMetaData> metadata;
  if(!client->info(*srm_request,metadata)) return false;
  if(metadata.empty()) return false;
  odlog(INFO)<<"check_srm: obtained size: "<<metadata.front().size<<std::endl;
  if(metadata.front().size > 0) url->meta_size(metadata.front().size);
  odlog(INFO)<<"check_srm: obtained checksum: "<<metadata.front().checkSumValue<<std::endl;
  if(metadata.front().checkSumValue.length() > 0 &&
     metadata.front().checkSumType.length() > 0) {
    std::string csum(metadata.front().checkSumType+":"+metadata.front().checkSumValue);
    url->meta_checksum(csum.c_str());
  };
  if(metadata.front().createdAtTime > 0) {
    odlog(INFO)<<"check_srm: obtained creation date: "<<ctime(&(metadata.front().createdAtTime));
    url->meta_created(metadata.front().createdAtTime);
  }
  return true;
#else
  return false;
#endif
}
