#include "../std.h"
#include "../misc/url_options.h"
#include "../misc/checkfile.h"
#include "../misc/checksum.h"
#include "../misc/mkdir_recursive.h"
#include "../misc/globus_modules.h"
#include "../misc/log_time.h"

#include <arc/certificate.h>

#include "datapoint.h"
#include "datahandle.h"
#include "databufferpar.h"

#include "datamove.h"

#ifndef BUFLEN
#define BUFLEN 1024
#endif

#ifndef HAVE_STRERROR_R
static char * strerror_r (int errnum, char * buf, size_t buflen) {
  char * estring = strerror (errnum);
  strncpy (buf, estring, buflen);
  return buf;
}
#endif

DataMove::DataMove() {
  be_verbose=false;
  force_secure=false;
  force_passive=false;
  force_registration=false;
  do_retries=true;
  do_checks=true;
  default_min_speed=0;
  default_min_speed_time=0;
  default_min_average_speed=0;
  default_max_inactivity_time=300;
  show_progress=NULL;
}

DataMove::~DataMove() {
}

bool DataMove::verbose(void) { return be_verbose; };

void DataMove::verbose(bool val) { be_verbose=val; };

void DataMove::verbose(const std::string &prefix) { 
  be_verbose=true; verbose_prefix=prefix;
};

bool DataMove::retry(void) { return do_retries; };

void DataMove::retry(bool val) { do_retries=val; };

bool DataMove::checks(void) { return do_checks; };

void DataMove::checks(bool val) { do_checks=val; };

typedef struct {
  DataPoint *source;
  DataPoint *destination;
  FileCache *cache;
  const UrlMap *map;
  unsigned long long int min_speed;
  time_t min_speed_time;
  unsigned long long int min_average_speed;
  time_t max_inactivity_time;
  std::string* failure_description;
  DataMove::callback cb;
  DataMove* it;
  void* arg;
  const char* prefix;
} transfer_struct;


DataMove::result DataMove::Delete(DataPoint &url,bool errcont) {
  bool remove_lfn = !url.have_locations(); // pfn or plain url
  if(!url.meta_resolve(true)) {
    // TODO: Check if error is real or "not exist".
    if(remove_lfn) {
      odlog(INFO)<<"No locations found - probably no more physical instances"
              <<std::endl;
    };
  };
  std::list<std::string> removed_urls;
  if(url.have_locations()) for(;url.have_location();) {
    odlog(INFO)<<"Removing "<<url.current_location()<<std::endl;
    // It can happen that after resolving list contains duplicated
    // physical locations obtained from different meta-data-services.
    // Because not all locations can reliably say if files does not exist
    // or access is not allowed, avoid duplicated delete attempts.
    bool url_was_deleted = false;
    for(std::list<std::string>::iterator u = removed_urls.begin();
                                           u!=removed_urls.end();++u) {
      if(url.current_location() == (*u)) { url_was_deleted=true; break; };
    };
    if(url_was_deleted) {
      odlog(VERBOSE)<<"This instance was already deleted"<<std::endl;
    } else {
      DataHandle handle(&url);
      handle.secure(false);
      if(!handle.remove()) {
        odlog(INFO)<<"Failed to delete physical file"<<std::endl;
        if(!errcont) {
          url.next_location(); continue;
        };
      } else {
        removed_urls.push_back(url.current_location());
      };
    };
    if(!url.meta()) {
      url.remove_location();
    } else {
      odlog(INFO)<<"Removing metadata in "
                          <<url.current_meta_location()<<std::endl;
      if(!url.meta_unregister(false)) {
        odlog(ERROR)<<"Failed to delete meta-information"<<std::endl;
        url.next_location();
      }
      else { url.remove_location(); };
    };
  };
  if(url.have_locations()) {
    odlog(ERROR)<<"Failed to remove all physical instances"<<std::endl;
    return DataMove::delete_error;
  };
  if(url.meta()) {
    if(remove_lfn) {
      odlog(INFO)<<"Removing logical file from metadata "
                          <<url.canonic_url()<<std::endl;
      if(!url.meta_unregister(true)) {
        odlog(ERROR)<<"Failed to delete logical file"<<std::endl;
        return DataMove::unregister_error;
      };
    };
  };
  return DataMove::success;
}

void* transfer_func(void* arg) {
  transfer_struct* param = (transfer_struct*)arg;
  std::string failure_description;
  DataMove::result res = param->it->Transfer(
         *(param->source),*(param->destination),*(param->cache),*(param->map),
         param->min_speed,param->min_speed_time,
         param->min_average_speed,param->max_inactivity_time,
         &failure_description,
         NULL,NULL,param->prefix);
  if(param->failure_description) 
    *(param->failure_description)=failure_description;
  (*(param->cb))(param->it,res,failure_description.c_str(),param->arg);
  if(param->prefix) free((void*)(param->prefix));
  delete param->cache;
  free(param);
  return NULL;
}

/* transfer data from source to destination */
DataMove::result DataMove::Transfer(
      DataPoint &source,DataPoint &destination,
      FileCache &cache,const UrlMap &map,
      std::string* failure_description,
      DataMove::callback cb,void* arg,const char* prefix) {
  return Transfer(source,destination,cache,map,
           default_min_speed,default_min_speed_time,
           default_min_average_speed,
           default_max_inactivity_time,
           failure_description,cb,arg,prefix);
}

DataMove::result DataMove::Transfer(
      DataPoint &source,DataPoint &destination,
      FileCache &cache,const UrlMap &map,
      unsigned long long int min_speed,time_t min_speed_time,
      unsigned long long int min_average_speed,
      time_t max_inactivity_time,
      std::string* failure_description,
      DataMove::callback cb,void* arg,const char* prefix) {
  if(cb != NULL) {
    odlog(DEBUG)<<"DataMove::Transfer : starting new thread"<<std::endl;
    pthread_t thread;
    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    pthread_attr_setdetachstate(&thread_attr,PTHREAD_CREATE_DETACHED);
    transfer_struct* param = (transfer_struct*)malloc(sizeof(transfer_struct));
    if(param==NULL) {
      pthread_attr_destroy(&thread_attr); return system_error;
    };
    param->source=&source;
    param->destination=&destination;
    param->cache=new FileCache(cache);
    param->map=&map;
    param->min_speed=min_speed;
    param->min_speed_time=min_speed_time;
    param->min_average_speed=min_average_speed;
    param->max_inactivity_time=max_inactivity_time;
    param->failure_description=failure_description;
    param->cb=cb;
    param->it=this;
    param->arg=arg;
    param->prefix=NULL;
    if(prefix) param->prefix=strdup(prefix);
    if(param->prefix == NULL) param->prefix=strdup(verbose_prefix.c_str());
    if(pthread_create(&thread,&thread_attr,&transfer_func,param) != 0) {
      free(param); pthread_attr_destroy(&thread_attr); return system_error;
    };
    pthread_attr_destroy(&thread_attr);
    return success;
  };
  if(failure_description) *failure_description="";
  odlog(INFO)<<"Transfer from "<<source.canonic_url()<<" to "<<destination.canonic_url()<<std::endl;
  if(!source) {
    odlog(ERROR)<<"Not valid source"<<std::endl; return read_acquire_error;
  };
  if(!destination) {
    odlog(ERROR)<<"Not valid destination"<<std::endl; return write_acquire_error;
  };
  for(;;) {
    if(source.meta_resolve(true,map)) {
      if(source.have_locations()) break;
      odlog(ERROR)<<"No locations for source found: "<<source<<std::endl;
    } else odlog(ERROR)<<"Failed to resolve source: "<<source<<std::endl;
    source.next_location(); /* try again */
    if(!do_retries) return read_resolve_error;
    if(!source.have_location()) return read_resolve_error;
  };
  for(;;) {
    if(destination.meta_resolve(false,UrlMap())) {
      if(destination.have_locations()) break;
      odlog(ERROR)<<"No locations for destination found: "<<destination<<std::endl;
    } else odlog(ERROR)<<"Failed to resolve destination: "<<destination<<std::endl;
    destination.next_location(); /* try again */
    if(!do_retries) return write_resolve_error;
    if(!destination.have_location()) return write_resolve_error;
  };
  bool replication=false;
  if(source.meta() && destination.meta()) {
    // check for possible replication
    if(source.canonic_url() == destination.canonic_url()) {
      replication=true;
      // we do not want to replicate to same site
      destination.remove_locations(source);
      if(!destination.have_locations()) {
        odlog(ERROR)<<"No locations for destination different from source found: "<<destination<<std::endl;
        return write_resolve_error;
      };
    };
  };
  //  Try to avoid any additional checks meant to provide 
  // meta-information whenever possible
  bool checks_required = destination.accepts_meta() && (!replication);
  bool destination_meta_initially_stored = destination.meta_stored();
  bool destination_overwrite = false;
  if(!replication) {  // overwriting has no sense in case of replication
    std::string value;
    if(get_url_option(destination.base_url(),"overwrite",0,value) == 0) {
      if(strcasecmp(value.c_str(),"no") != 0) destination_overwrite=true;
    };
  };
  if(destination_overwrite) {
    if( (destination.meta() && destination_meta_initially_stored) ||
        (!destination.meta()) ) {
      std::string del_url = destination.canonic_url();
      odlog(DEBUG)<<"DataMove::Transfer: trying to destroy/overwrite destination: "
              <<del_url<<std::endl;
      int try_num = destination.tries();
      for(;;) {
        DataPoint del(del_url.c_str());
        del.tries(1);
        result res = Delete(del);
        if(res == success) break;
        if(!destination.meta()) break; // pfn has chance to be overwritten directly
        odlog(INFO)<<"Failed to delete "<<del_url<<std::endl;

        destination.next_location(); /* try again */
        if(!do_retries) return res;
        if((--try_num) <= 0) return res;
      };
      if(destination.meta()) {
        for(;;) {
          if(destination.meta_resolve(false)) {
            if(destination.have_locations()) break;
            odlog(ERROR)<<"No locations for destination found: "<<destination<<std::endl;
          } else odlog(ERROR)<<"Failed to resolve destination: "<<destination<<std::endl;
          destination.next_location(); /* try again */
          if(!do_retries) return write_resolve_error;
          if(!destination.have_location()) return write_resolve_error;
        };
        destination_meta_initially_stored=destination.meta_stored();
        if(destination_meta_initially_stored) {
          odlog(INFO)<<"Deleted but still have locations at "<<destination<<std::endl;
          return write_resolve_error;
        };
      };
    };
  };
  result res = transfer_error;
  int try_num;
  for(try_num=0;;try_num++) {  /* cycle for retries */
    odlog(DEBUG)<<"DataMove: cycle"<<std::endl;
    if((try_num != 0) && (!do_retries)) {
      odlog(DEBUG)<<"DataMove: no retries requested - exit"<<std::endl;
      return res;
    };
    if( (!source.have_location()) || (!destination.have_location()) ) {
      odlog(DEBUG)<<"DataMove: ";
      if(!source.have_location()) odlog_(DEBUG)<<" source ";
      if(!destination.have_location()) odlog_(DEBUG)<<" destination ";
      odlog_(DEBUG)<<" out of tries - exit"<<std::endl;
      /* out of tries */
      return res;
    };
    // By putting DataBufferPar here, one makes sure it will be always
    // destroyed AFTER all DataHandle. This allows for not bothering
    // to call stop_reading/stop_writing because they are called in
    // destructor of DataHandle.
    DataBufferPar buffer;
    odlog(INFO)<<"Real transfer from "<<source.current_location()<<" to "<<destination.current_location()<<std::endl;
    /* creating handler for transfer */
    DataHandle source_h(&source);
    DataHandle destination_h(&destination);
    source_h.secure(force_secure);
    source_h.passive(force_passive);
    destination_h.secure(force_secure);
    destination_h.passive(force_passive);
    destination_h.additional_checks(do_checks);
    /* take suggestion from DataHandle about buffer, etc. */
    DataHandle::analyze_t hint_read;
    DataHandle::analyze_t hint_write;
    bool cacheable=false;
    /* is file executable */
    bool executable = false;
    std::string exec_option;
    if (get_url_option(source.current_location(), "exec", exec_option) == 0 && exec_option == "yes") executable = true;
    long int bufsize;
    int bufnum;
    source_h.analyze(hint_read);
    destination_h.analyze(hint_write);
    if(hint_read.cache && hint_write.local) {
      if(cache) cacheable=true;
    };
    /* tune buffers */
    bufsize=65536; /* have reasonable buffer size */
    bool seekable = destination_h.out_of_order();
    source_h.out_of_order(seekable);
    bufnum=1;
    if(hint_read.bufsize  > bufsize) bufsize=hint_read.bufsize;
    if(hint_write.bufsize > bufsize) bufsize=hint_write.bufsize;
    if(seekable) {
      if(hint_read.bufnum  > bufnum)  bufnum=hint_read.bufnum;
      if(hint_write.bufnum > bufnum)  bufnum=hint_read.bufnum;
    };
    bufnum=bufnum*2;
    odlog(DEBUG)<<"Creating buffer: "<<bufsize<<" x "<<bufnum<<std::endl;
    /* prepare crc */
    CheckSumAny crc;
    // Shold we trust indexing service or always compute checksum ?
    // Let's trust.
    if(destination.accepts_meta()) { // may need to compute crc
      // Let it be CRC32 by default.
      std::string crc_type;
      {
        std::string d_url = destination.base_url();
        if(get_url_option(d_url,"checksum",0,crc_type) != 0) {
          crc_type="cksum";
        };
        odlog(DEBUG)<<"DataMove::Transfer: checksum type is "<<crc_type<<std::endl;
      };
      if(!source.meta_checksum_available()) {
        crc=crc_type.c_str();
        odlog(DEBUG)<<"DataMove::Transfer: will try to compute crc"<<std::endl;
      } else if(CheckSumAny::Type(crc_type.c_str()) != CheckSumAny::Type(source.meta_checksum())) {
        crc=crc_type.c_str();
        odlog(DEBUG)<<"DataMove::Transfer: will try to compute crc"<<std::endl;
      };
    };
    /* create buffer and tune speed control */
    buffer.set(&crc,bufsize,bufnum);
    if(!buffer) {
      odlog(INFO)<<"Buffer creation failed !"<<std::endl;
    };
    buffer.speed.set_min_speed(min_speed,min_speed_time);
    buffer.speed.set_min_average_speed(min_average_speed);
    buffer.speed.set_max_inactivity_time(max_inactivity_time);
    buffer.speed.verbose(be_verbose);
    if(be_verbose) {
      if(prefix) { buffer.speed.verbose(std::string(prefix)); }
      else { buffer.speed.verbose(verbose_prefix); };
      buffer.speed.set_progress_indicator(show_progress);
    };
    /* checking if current source should be mapped to different location */
    /*
       TODO: make mapped url to be handled by source handle directly
    */
    bool mapped = false;
    std::string mapped_url = "";
    if(hint_write.local) {
      mapped_url = source.current_location();
      ::canonic_url(mapped_url);
      mapped=map.map(mapped_url);
      /* TODO: copy options to mapped_url */
      if(!mapped) {
        mapped_url="";
      } else {
        odlog(DEBUG)<<"Url is mapped to: "<<mapped_url<<std::endl;
        if((strncasecmp(mapped_url.c_str(),"link:/",6) == 0) ||
           (strncasecmp(mapped_url.c_str(),"file:/",6) == 0)) {
          /* can't cache links */
          cacheable=false;
        };
      };
    };
    // Do not link if user asks. Replace link:// with file://
    if(hint_read.readonly && mapped) {
      if(strncasecmp(mapped_url.c_str(),"link:/",6) == 0) {
        mapped_url.replace(0,4,"file");
      };
    };
    DataPoint mapped_p(mapped_url.c_str());
    DataHandle mapped_h(&mapped_p);
    mapped_h.secure(force_secure);
    mapped_h.passive(force_passive);
    /* Try to initiate cache (if needed) */
    std::string canonic_url = source.canonic_url();
    if(cacheable) {
      res=success;
      std::string cache_option;
      bool use_remote_caches = true;
      for(;;) { /* cycle for outdated cache files */
        bool is_in_cache = false;
        bool is_locked = false;
        if(!cache.start(canonic_url,is_in_cache,is_locked,use_remote_caches)) {
          if(is_locked) {
            odlog(DEBUG)<<"Cached file is locked - should retry"<<std::endl;
            return cache_error_retryable;
          }
          cacheable=false;
          odlog(INFO)<<"Failed to initiate cache"<<std::endl;
          break;
        };
        if(is_in_cache) { 
          // check for forced re-download option
          std::string cache_option;
          if(get_url_option(source.base_url(),"cache",0,cache_option) == 0 && cache_option == "renew") {
            odlog(DEBUG)<<"Forcing re-download of file "<<canonic_url<<std::endl;
            cache.stopAndDelete(canonic_url);
            use_remote_caches = false; // to avoid endless loop with remote caches
            continue;
          }
          /* just need to check permissions */
          odlog(INFO)<<"File "<<canonic_url<<" is cached ("<<cache.file(canonic_url)<<") - checking permissions"<<std::endl;
          // check the list of cached DNs
          bool have_permission = false;
          std::string dn;
          time_t exp_time(0);
          try {
            Certificate ci(PROXY);
            dn = ci.GetIdentitySN();
            if(cache.checkDN(canonic_url, dn)) have_permission = true;
            exp_time = ci.Expires().GetTime();
          } catch (CertificateError e) {
            odlog(WARNING)<<"Couldn't handle certificate: "<<e.what()<<std::endl;
          }
          if(!have_permission) {
            if(!source_h.check()) {
              odlog(ERROR)<<"Permission checking failed: "<<source<<std::endl;
              cache.stop(canonic_url);
              source.next_location(); /* try another source */
              odlog(DEBUG)<<"source.next_location"<<std::endl;
              res=read_start_error;
              break;
            }
            cache.addDN(canonic_url, dn, exp_time);
          };
          odlog(DEBUG)<<"Permission checking passed"<<std::endl;
          /* check if file is fresh enough */
          bool outdated = true;
          if(have_permission) outdated = false; // cached DN means don't check creation date
          if(source.meta_created_available() && cache.created_available(canonic_url)) {
            time_t source_ctime = source.meta_created();
            time_t cache_ctime = cache.created(canonic_url);
            odlog(DEBUG)<<"Source creation date: "<<ctime(&source_ctime);
            odlog(DEBUG)<<"Cache creation date: "<<ctime(&cache_ctime);
            if(source_ctime <= cache_ctime) outdated=false;
          }
          if(cache.validtill_available(canonic_url)) {
            time_t cache_vtime = cache.validtill(canonic_url);
            odlog(DEBUG)<<"Cache file valid until: "<<ctime(&cache_vtime);
            if(cache_vtime > time(NULL)) outdated=false;
            else outdated=true;
          };
          if(outdated) {
            cache.stopAndDelete(canonic_url);
            odlog(INFO)<<"Cached file is outdated, will re-download"<<std::endl;
            use_remote_caches = false; // to avoid endless loop with remote caches
            continue;
          }
          odlog(DEBUG)<<"Cached copy is still valid"<<std::endl;
          if(hint_read.readonly && !executable) {
            odlog(DEBUG)<<"Linking/copying cached file"<<std::endl;
            if(!cache.link_file(std::string(get_url_path(destination.current_location())), canonic_url)){
              /* failed cache link is unhandable */
              cache.stop(canonic_url);
              return cache_error;
            };
          } else {
            odlog(DEBUG)<<"Copying cached file"<<std::endl;
            if(!cache.copy_file(std::string(get_url_path(destination.current_location())), canonic_url, executable)){
              /* failed cache copy is unhandable */
              cache.stop(canonic_url);
              return cache_error;
            };
          };
          cache.stop(canonic_url);
          return success; // Leave here. Rest of code below is for transfer.
        };
        break;
      };
      if(cacheable) {
        if(res != success) continue;
      };
    };
    if(mapped) {
      if((strncasecmp(mapped_url.c_str(),"link:/",6) == 0) ||
         (strncasecmp(mapped_url.c_str(),"file:/",6) == 0)) {
        /* check permissions first */
        odlog(INFO)<<"URL is mapped to local access - checking permissions on original URL"<<std::endl;
        if(!source_h.check()) {
          odlog(ERROR)<<"Permission checking on original URL failed: "<<source<<std::endl;
          source.next_location(); /* try another source */
          odlog(DEBUG)<<"source.next_location"<<std::endl;
          res=read_start_error;
          if(cacheable) cache.stopAndDelete(canonic_url);
          continue;
        };
        odlog(DEBUG)<<"Permission checking passed"<<std::endl;
        if(strncasecmp(mapped_url.c_str(),"link:/",6) == 0) {
          odlog(DEBUG)<<"Linking local file"<<std::endl;
          const char* file_name = get_url_path(mapped_url.c_str());
          const char* link_name = get_url_path(destination.current_location());
          // create directory structure for link_name
          {
            uid_t uid=get_user_id();
            gid_t gid=get_user_group(uid);
            std::string dirpath = link_name;
            std::string::size_type n = dirpath.rfind('/');
            if(n == std::string::npos) n=0;
            if(n == 0) { dirpath="/"; } else { dirpath.resize(n); };
            if(mkdir_recursive(NULL,dirpath.c_str(),S_IRWXU,uid,gid) != 0) {
              if(errno != EEXIST) {
                char* err;
                char errbuf[BUFLEN];
#ifndef _AIX
                err=strerror_r(errno,errbuf,sizeof(errbuf));
#else
                errbuf[0]=0; err=errbuf;
                strerror_r(errno,errbuf,sizeof(errbuf));
#endif
                odlog(ERROR)<<"Failed to create/find directory "<<dirpath<<" : "<<err<<std::endl;
                source.next_location(); /* try another source */
                odlog(DEBUG)<<"source.next_location"<<std::endl;
                res=read_start_error;
                if(cacheable) cache.stopAndDelete(canonic_url);
                continue;
              };
            };
          };
          // make link
          if(symlink(file_name,link_name) == -1) {
            char* err;
            char errbuf[BUFLEN];
#ifndef _AIX
            err=strerror_r(errno,errbuf,sizeof(errbuf));
#else
            errbuf[0]=0; err=errbuf;
            strerror_r(errno,errbuf,sizeof(errbuf));
#endif
            odlog(ERROR)<<"Failed to make symbolic link "<<link_name<<" to "<<file_name<<" : "<<err<<std::endl;
            source.next_location(); /* try another source */
            odlog(DEBUG)<<"source.next_location"<<std::endl;
            res=read_start_error;
            if(cacheable) cache.stopAndDelete(canonic_url);
            continue;
          };
          {
            uid_t uid = get_user_id();
            gid_t gid = get_user_group(uid);
            lchown(link_name,uid,gid);
          };
          if(cacheable) cache.stop(canonic_url);
          return success; // Leave after making a link. Rest moves data.
        };
      };
    };
    std::string churl;
    if(cacheable) {
      /* create new destination for cache file */
      churl="file://"+cache.file(canonic_url);
      odlog(INFO)<<"cache file: "<<churl<<std::endl;
    };
    DataPoint chdest(churl.c_str());
    DataHandle chdest_h(&chdest);
    chdest_h.secure(force_secure);
    chdest_h.passive(force_passive);
    chdest_h.additional_checks(do_checks);
    chdest.meta(destination); // share metadata
    DataPoint& source_url = mapped?mapped_p:source;
    DataHandle& source_channel = mapped?mapped_h:source_h;
    DataHandle& destination_channel = cacheable?chdest_h:destination_h;
    // Disable checks meant to provide meta-information if not needed
    source_h.additional_checks(do_checks & (checks_required | cacheable));
    mapped_h.additional_checks(do_checks & (checks_required | cacheable));
    if(!source_channel.start_reading(buffer)) {
      odlog(ERROR)<<"Failed to start reading from source: "<<source_url<<std::endl;
      res=read_start_error;
      if(source_channel.failure_reason()==
         DataHandle::credentials_expired_failure) res=credentials_expired_error;
      /* try another source */
      if(source.next_location()) odlog(DEBUG)<<"(Re)Trying next source"<<std::endl;
      if(cacheable) 
        cache.stopAndDelete(canonic_url);
      continue;
    };
    if(mapped) destination.meta(mapped_p);
    if(force_registration && destination.meta()) { // at least compare metadata
      if(!destination.meta_compare(source)) {
        odlog(ERROR)<<"Metadata of source and destination are different"<<std::endl;
        source.next_location(); /* not exactly sure if this would help */
        res=preregister_error;
        //if(cacheable) cache.stop(true,false);
        if(cacheable) cache.stopAndDelete(canonic_url);
        continue;
      };
    };
    destination.meta(source); // pass metadata gathered during start_reading()
                              // from source to destination
    if(destination.meta_size_available()) {
      buffer.speed.set_max_data(destination.meta_size());
    };
    if(!destination.meta_preregister(replication,force_registration)) {
      odlog(ERROR)<<"Failed to preregister destination: "<<destination<<std::endl;
      destination.next_location(); /* not exactly sure if this would help */
      odlog(DEBUG)<<"destination.next_location"<<std::endl;
      res=preregister_error;
      // Normally remote destination is not cached. But who knows.
      if(cacheable) cache.stopAndDelete(canonic_url);
      continue;
    };
    buffer.speed.reset();
    DataHandle::failure_reason_t read_failure = DataHandle::common_failure;
    DataHandle::failure_reason_t write_failure = DataHandle::common_failure;
    if(!cacheable) {
      if(!destination_h.start_writing(buffer)) {
        odlog(ERROR)<<"Failed to start writing to destination: "<<destination<<std::endl;
        source_channel.stop_reading();
        if(!destination.meta_preunregister(replication ||
                               destination_meta_initially_stored)) {
          odlog(ERROR)<<"Failed to unregister preregistered lfn, You may need to unregister it manually: "<<destination<<std::endl;
        };
        if(destination.next_location())
          odlog(DEBUG)<<"(Re)Trying next destination"<<std::endl;
        res=write_start_error;
        if(destination_h.failure_reason()==
         DataHandle::credentials_expired_failure) res=credentials_expired_error;
        continue;
      };
    } else { 
      if(!chdest_h.start_writing(buffer)) { // TODO: put callback to clean cache into FileCache
        odlog(ERROR)<<"Failed to start writing to cache"<<std::endl;
        source_channel.stop_reading();
        // hope there will be more space next time
        cache.stopAndDelete(canonic_url);
        if(!destination.meta_preunregister(replication || 
                                  destination_meta_initially_stored)) {
          odlog(ERROR)<<"Failed to unregister preregistered lfn, You may need to unregister it manually"<<std::endl;
        };
        return cache_error;  /* repeating won't help here */
      };
    };
    odlog(DEBUG)<<"Waiting for buffer"<<std::endl;
    for(;(!buffer.eof_read() || !buffer.eof_write()) && !buffer.error();) {
      buffer.wait();
    };
    odlog(INFO)<<"buffer: read eof : "<<buffer.eof_read()<<std::endl;
    odlog(INFO)<<"buffer: write eof: "<<buffer.eof_write()<<std::endl;
    odlog(INFO)<<"buffer: error    : "<<buffer.error()<<std::endl;
    odlog(DEBUG)<<"Closing read channel"<<std::endl;
    source_channel.stop_reading();
    read_failure=source_channel.failure_reason();
    if(cacheable && mapped) {
      source.meta(mapped_p); // pass more metadata (checksum)
    };
    odlog(DEBUG)<<"Closing write channel"<<std::endl;
    destination_channel.stop_writing();
    write_failure=destination_channel.failure_reason();
    // write_failure=destination_h.failure_reason();
    if(cacheable) {
      bool download_error = buffer.error();
      if(!download_error) {
        if(source.meta_validtill_available()) cache.validtill_force(canonic_url, source.meta_validtill());
        bool cache_link_result;
        if (executable) {
          odlog(DEBUG)<<"Copying cached file"<<std::endl;
          cache_link_result = cache.copy_file(std::string(get_url_path(destination.current_location())), canonic_url, true);
        }
        else {
          odlog(DEBUG)<<"Linking/copying cached file"<<std::endl;
          cache_link_result = cache.link_file(std::string(get_url_path(destination.current_location())), canonic_url);
        }
        if (!cache_link_result) {
          buffer.error_write(true);
          cache.stop(canonic_url);
          if(!destination.meta_preunregister(replication || destination_meta_initially_stored)) {
            odlog(ERROR)<<"Failed to unregister preregistered lfn, You may need to unregister it manually"<<std::endl;
          };
          return cache_error;  /* retry won't help */
        };
        cache.stop(canonic_url);
      } else {
        cache.stopAndDelete(canonic_url); // delete in case failed download left corrupted file 
      };
    };
    if(buffer.error()) {
      if(!destination.meta_preunregister(replication || destination_meta_initially_stored)) {
        odlog(ERROR)<<"WARNING: Failed to unregister preregistered lfn, You may need to unregister it manually"<<std::endl;
      };
      // Analyze errors
      // Easy part first - if either read or write part report error
      // go to next endpoint.
      if(buffer.error_read())  {
        if(failure_description) 
          (*failure_description)=source_channel.failure_text();
        if(source.next_location())
          odlog(DEBUG)<<"(Re)Trying next source"<<std::endl;
        if(read_failure == DataHandle::credentials_expired_failure) {
          res=credentials_expired_error;
        } else { res=read_error; };
      } else if(buffer.error_write()) {
        if(failure_description)
          (*failure_description)=destination_channel.failure_text();
        if(destination.next_location())
          odlog(DEBUG)<<"(Re)Trying next destination"<<std::endl;
        if(write_failure == DataHandle::credentials_expired_failure) {
          res=credentials_expired_error;
        } else { res=write_error; };
      } else if(buffer.error_transfer()) {
        // Here is more complicated case - operation timeout
        // Let's first check if buffer was full
        res=transfer_error;
        if(!buffer.for_read()) {
          // No free buffers for 'read' side. Buffer must be full.
          if(failure_description)
            (*failure_description)=destination_channel.failure_text();
          if(destination.next_location())
            odlog(DEBUG)<<"(Re)Trying next destination"<<std::endl;
        } else if(!buffer.for_write()) {
          // Buffer is empty 
          if(failure_description) 
            (*failure_description)=source_channel.failure_text();
          if(source.next_location())
            odlog(DEBUG)<<"(Re)Trying next source"<<std::endl;
        } else {
          // Both endpoints were very slow? Choose randomly.
          odlog(DEBUG)<<"Cause of failure unclear - choosing randomly"<<std::endl;
          if(random() < (RAND_MAX/2)) {
            if(failure_description) 
              (*failure_description)=source_channel.failure_text();
            if(source.next_location())
              odlog(DEBUG)<<"(Re)Trying next source"<<std::endl;
          } else {
            if(failure_description)
              (*failure_description)=destination_channel.failure_text();
            if(destination.next_location())
              odlog(DEBUG)<<"(Re)Trying next destination"<<std::endl;
          };
        };
      };
      continue;
    };
    // check if checksum is specified as a metadata attribute
    if(destination.has_meta_attribute("checksumtype") && destination.has_meta_attribute("checksumvalue")) {
      std::string csum = destination.meta_attribute("checksumtype") + ":" + destination.meta_attribute("checksumvalue");
      source.meta_checksum(csum.c_str());
      odlog(DEBUG)<<"DataMove::Transfer: using supplied checksum "<<csum<<std::endl;
    }
    else if(crc) {
      if(buffer.checksum_valid()) {
        // source.meta_checksum(crc.end());
        char buf[100];
        crc.print(buf,100);
        source.meta_checksum(buf);
        odlog(DEBUG)<<"DataMove::Transfer: have valid checksum"<<std::endl;
      };
    };
    destination.meta(source); // pass more metadata (checksum)
    if(!destination.meta_postregister(replication,buffer.error())) {
      odlog(ERROR)<<"Failed to postregister destination"<<destination<<std::endl;
      if(!destination.meta_preunregister(replication || destination_meta_initially_stored)) {
        odlog(ERROR)<<"Failed to unregister preregistered lfn, You may need to unregister it manually"<<destination<<std::endl;
      };
      destination.next_location(); /* not sure if this can help */
      odlog(DEBUG)<<"destination.next_location"<<std::endl;
      res=postregister_error;
      continue;
    };
    if(buffer.error()) continue; // should never happen - keep just in case
    break;
  };
  return success;
}

void DataMove::secure(bool val) {
  force_secure=val;
}

void DataMove::passive(bool val) {
  force_passive=val;
}

void DataMove::force_to_meta(bool val) {
  force_registration=val;
}


static const char* result_string[] = {
    "success",  // 0
    "bad source URL", // 1
    "bad destination URL", // 2
    "failed to resolve source locations", // 3
    "failed to resolve destination locations", // 4
    "failed to register new destination file", // 5
    "can't start reading from source", // 6
    "can't start writing to destination", // 7
    "can't read from source", // 8
    "can't write to destination", // 9
    "data transfer was too slow", // 10
    "failed while closing connection to source", // 11
    "failed while closing connection to destination", // 12
    "failed to register new location", // 13
    "can't use local cache", // 14
    "system error", // 15
    "delegated credentials expired", // 16
    "error with cache, but can retry" // 20
};

const char* DataMove::get_result_string(result r) {
  if(r == undefined_error) return "unexpected error";
  if((r<0) || (r>16)) return "unknown error";
  return result_string[r];
}

