#include "../std.h"

#include <list>

extern "C" {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
#include <globus_replica_catalog.h>
#endif
}

#include "../config/config_map.h"
#include "../misc/url_options.h"
#include "../misc/checksum.h"
#include "../misc/globus_modules.h"
#include "../misc/log_time.h"

#include "datapoint.h"

#include "../https/fireman/fireman_client.h"

std::list<DataPoint::constructor_t> DataPoint::protocols;
LockSimple DataPoint::protocols_lock;

class DataPointRegistrator {
 public:
  DataPointRegistrator(void) {
    DataPoint::AddProtocol(&(DataPointFile::CreateInstance));
    DataPoint::AddProtocol(&(DataPointFTP::CreateInstance));
    DataPoint::AddProtocol(&(DataPointHTTP::CreateInstance));
#ifndef SRM_MISSING
    DataPoint::AddProtocol(&(DataPointSRM::CreateInstance));
#endif
    DataPoint::AddProtocol(&(DataPointRLS::CreateInstance));
    DataPoint::AddProtocol(&(DataPointRC::CreateInstance));
#ifndef FIREMAN_MISSING
    DataPoint::AddProtocol(&(DataPointFireman::CreateInstance));
#endif
    DataPoint::AddProtocol(&(DataPointLFC::CreateInstance));
  };
};

static DataPointRegistrator registrator;

DataPoint::DataPoint(const char* url):instance(NULL) {
  instance=CreateInstance(url);
  /*
  if((url == NULL) || (url[0]==0)) return;
  if((strcasecmp("-",url)==0) || (strncasecmp("file://",url,7)==0)) {
    instance = new DataPointFile(url);
  } else if((strncasecmp("ftp://",url,6)==0) ||
            (strncasecmp("gsiftp://",url,9)==0)) {
    instance = new DataPointFTP(url);
  } else if((strncasecmp("http://",url,7)==0) ||
            (strncasecmp("https://",url,8)==0) ||
            (strncasecmp("httpg://",url,8)==0) ||
            (strncasecmp("se://",url,5)==0)) {
    instance = new DataPointHTTP(url);
  } else {
    return;
  };
  if(*instance) {
  } else {
    delete instance;
    instance=NULL;
  };
  */
}

bool DataPoint::AddProtocol(constructor_t constructor) {
  protocols_lock.block();
  protocols.push_back(constructor);
  protocols_lock.unblock();
  return true;
}

DataPoint* DataPoint::CreateInstance(const char* url) {
  if((url == NULL) || (url[0]==0)) return NULL;
  DataPoint* point = NULL;
  protocols_lock.block();
  for(std::list<constructor_t>::const_iterator i = protocols.begin();
                                          i!=protocols.end();++i) {
    point = (*(*i))(url);
    if(point) {
      if(*point) break;
      delete point; point=NULL;
    };
  };
  protocols_lock.unblock();
  return point;
}

DataPoint::~DataPoint(void) {
  if(instance) delete instance;
};

DataPointDirect::DataPointDirect(const char* u):
           is_valid(false),url(u),
           meta_size_valid(false),meta_checksum_valid(false),
           meta_created_valid(false),meta_validtill_valid(false),
           tries_left(5) {
  if(u) {
    locations.push_back(u); location=locations.begin();
  } else {
    location=locations.end();
  };
}

DataPointMeta::DataPointMeta(const char* u):
        DataPointDirect(u),is_metaexisting(false),is_resolved(false) {
  locations.clear();
  location=locations.end();
}

bool DataPointMeta::process_meta_url(void) {
  return false;
}

bool DataPointMeta::extract_meta_attributes(std::string& lfn) {
  meta_attributes.clear();
  std::string::size_type attribute_start = lfn.find(':',0);
  if( attribute_start == std::string::npos ) return true;
  std::string allattributes = lfn.substr( attribute_start+1 );
  lfn.erase( attribute_start );
  attribute_start = 0;
  std::string::size_type new_attribute_start;
  do {
    new_attribute_start = allattributes.find(':',attribute_start);
    std::string attribute = allattributes.substr( attribute_start,
      new_attribute_start!=std::string::npos?
        new_attribute_start-attribute_start:std::string::npos
    );
    std::string::size_type findvalue = attribute.find('=');
    if( findvalue != std::string::npos ) {
      std::string value = attribute.substr( findvalue+1 );
      std::string name = attribute.substr( 0, findvalue );
      odlog(DEBUG) << "Attribute: "<< name << " = " << value << std::endl;
      meta_attributes[ name ] = value;
    } else {
      odlog(DEBUG) << "Invalid attribute: " << attribute << std::endl;
    }
    attribute_start=new_attribute_start+1;
  } while( new_attribute_start != std::string::npos );
  return true;
}

bool DataPointMeta::meta_resolve(bool source,const UrlMap &maps) {
  if(is_resolved) return true;
  bool res=meta_resolve(source);
  if(!res) return false;
  sort(maps); location=locations.begin();
  return true;
}

void DataPointMeta::fix_unregistered(bool all) {
  if(all) {
    is_metaexisting=false;
    locations.clear();
    location=locations.end();
  } else {
    location=locations.erase(location);
    if(location == locations.end()) location=locations.begin();
  };
}

bool DataPointMeta::get_info(DataPoint::FileInfo &fi) {
  if(!meta_resolve(true)) { return false; }; // resolution failure
  fi.name=lfn();
  for(std::list<DataPointMeta::Location>::iterator i = locations.begin();
                                              i!=locations.end();++i) {
    fi.urls.push_back(i->url);
  };
  if(meta_size_valid) { fi.size=meta_size_; fi.size_available=true; };
  if(meta_checksum_valid) { fi.checksum=meta_checksum_; fi.checksum_available=true; };
  if(meta_created_valid) { fi.created=meta_created_; fi.created_available=true; };
  if(meta_validtill_valid) { fi.valid=meta_validtill_; fi.valid_available=true; };
  fi.type=DataPoint::FileInfo::file_type_file;
  return true;
}

bool DataPointDirect::have_locations(void) const {
  if(!is_valid) return false;
  return (locations.size() != 0);
}

bool DataPointDirect::have_location(void) const {
  if(!is_valid) return false;
  if(tries_left <= 0) return false;
  std::list<Location>::const_iterator l = location;
  if(l == locations.end()) return false;
  return true;
}

bool DataPointDirect::next_location(void) {
  if(tries_left <= 0) return false;
  if(location == locations.end()) return false;
  ++location;
  if(location == locations.end()) {
    tries_left--;
    if(tries_left <= 0) return false;
    location=locations.begin();
  };
  return true;
}

bool DataPointDirect::remove_location(void) {
  if(location == locations.end()) return false;
  location=locations.erase(location);
  return true;
}

bool DataPointDirect::remove_locations(const DataPoint& p_) {
  if(!(p_.have_locations())) return true;
  const DataPointDirect& p = *((const DataPointDirect*)(p_.constInstance()));
  std::list<Location>::iterator p_int;
  std::list<Location>::const_iterator p_ext;
  for(p_ext=p.locations.begin();p_ext!=p.locations.end();++p_ext) {
    std::string p_ext_s = p_ext->url; ::canonic_url(p_ext_s);
    std::string::size_type p_ext_l = 0;
    if((p_ext_l=p_ext_s.find(':',p_ext_l)) != std::string::npos) {
      if((p_ext_s[++p_ext_l] == '/') && (p_ext_s[++p_ext_l] == '/')) {
        p_ext_l=p_ext_s.find('/',++p_ext_l);
        if(p_ext_l != std::string::npos) p_ext_s.resize(p_ext_l);
      } else { p_ext_l=std::string::npos; };
    };
    for(p_int=locations.begin();p_int!=locations.end();) {
      // Compare protocol+host+port part
      std::string p_int_s = p_int->url; ::canonic_url(p_int_s);
      std::string::size_type p_int_l = 0;
      if((p_int_l=p_int_s.find(':',p_int_l)) != std::string::npos) {
        if((p_int_s[++p_int_l] == '/') && (p_int_s[++p_int_l] == '/')) {
          p_int_l=p_int_s.find('/',++p_int_l);
          if(p_int_l != std::string::npos) p_int_s.resize(p_int_l);
        } else { p_int_l=std::string::npos; };
      };
      if((p_int_l != std::string::npos) &&
         (p_ext_l != std::string::npos) &&
         (p_int_s == p_ext_s)) {
        if(location == p_int) {
          p_int=locations.erase(p_int); location=p_int;
        } else {
          p_int=locations.erase(p_int);
        };
        continue;
      };
      ++p_int;
    };
  };
  if(location == locations.end()) location=locations.begin();
  return true;
}

bool DataPointDirect::add_location(const char* meta_loc,const char* loc) {
  odlog(DEBUG)<<"Add location: metaname: "<<meta_loc<<std::endl;
  odlog(DEBUG)<<"Add location: location: "<<loc<<std::endl;
  for(std::list<Location>::iterator i = locations.begin();i!=locations.end();++i) {
    if(i->meta == meta_loc) return true; // Already exists
  };
  locations.insert(locations.end(),Location(meta_loc,loc,false));
  return true;
}

/* put local urls first and shuffle rest */
bool DataPointDirect::sort(const UrlMap &maps) {
  std::list<Location>::iterator ii = locations.begin();
  int nn = 0;
  for(std::list<Location>::iterator i=locations.begin();i!=locations.end();) {
    std::string c_url = i->url;
    ::canonic_url(c_url);
    if(maps.local(c_url)) {  /* move to first place */
      if(i == ii) { ++ii; ++i; nn++; continue; };
      locations.insert(ii,*i);
      if(i == location) location=locations.begin();
      i=locations.erase(i);
      nn++;
    }
    else {
      ++i;
    };
  };
  nn=locations.size()-nn;
  if(nn<=1) return true;
  srandom(time(NULL));
  for(;nn>1;) {
    int r = random()/2;
    int n=r/((RAND_MAX/2+1)/nn);
    std::list<Location>::iterator i=ii;
    for(;n;n--) { ++i; };
    if(i == locations.end()) { nn--; ++ii; continue; };
    if(i == ii) { nn--; ++ii; continue; };
    locations.insert(ii,*i);
    if(i == location) location=locations.begin();
    i=locations.erase(i);
    nn--;
  };
  return true;
}

bool DataPointDirect::map(const UrlMap &maps) {
  for(std::list<Location>::iterator i=locations.begin();i!=locations.end();) {
    if(maps.map(i->url)) {  /* move to first place */
      locations.push_front(*i);
      if(i == location) location=locations.begin();
      i=locations.erase(i);
    }
    else {
      ++i;
    };
  };
  return true;
}

std::string DataPointDirect::base_url(void) const {
  return url;
}

std::string DataPointDirect::canonic_url(void) const {
  std::string tmp = url;
  if(tmp == "-") { return tmp; };
  if(::canonic_url(tmp) != 0) tmp="";
  return tmp;
}

int DataPointDirect::tries(void) {
  return tries_left;
};

void DataPointDirect::tries(int n) {
  if(n<0) n=0;
  tries_left=n;
  if(n == 0) {
    location=locations.end();
  } else {
    if(location == locations.end()) location=locations.begin();
  };
};

DataPointFile::DataPointFile(const char* u):DataPointDirect(u),is_stdchannel(false) {
  if(u == NULL) return;
  if((u[0] == '-') && (u[1] == 0)) {
    is_stdchannel=true; 
  } else if(strncasecmp("file://",u,7) == 0) {
  } else {
    return;
  };
  is_valid=true;
}

DataPoint* DataPointFile::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("file://",u,7) && strcmp("-",u)) return NULL;
  return new DataPointFile(u);
}
 
DataPointFile::~DataPointFile(void) {
}

DataPointFTP::DataPointFTP(const char* u):DataPointDirect(u) {
  if(strncasecmp("ftp://",u,6) == 0) {
    is_secure=false;
  } else if(strncasecmp("gsiftp://",u,9) == 0) {
    is_secure=true;
  } else {
    return;
  };
  is_valid=true;
}

DataPoint* DataPointFTP::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("ftp://",u,6) && strncasecmp("gsiftp://",u,9)) return NULL;
  return new DataPointFTP(u);
}

DataPointFTP::~DataPointFTP(void) {
}

DataPointHTTP::DataPointHTTP(const char* u):DataPointDirect(u) {
  is_http=false; is_https=false; is_httpg=false; is_se=false;
  if(strncasecmp("http://",u,7) == 0) {
    is_http=true;
  } else if(strncasecmp("https://",u,8) == 0) {
    is_https=true;
  } else if(strncasecmp("httpg://",u,8) == 0) {
    is_httpg=true;
  } else if(strncasecmp("se://",u,5) == 0) {
    is_se=true;
  } else {
    return;
  };
  is_valid=true;
}

DataPoint* DataPointHTTP::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("http://",u,7) && strncasecmp("https://",u,8) &&
     strncasecmp("httpg://",u,8) && strncasecmp("se://",u,5)) return NULL;
  return new DataPointHTTP(u);
}

DataPointHTTP::~DataPointHTTP(void) {
}

DataPointSRM::DataPointSRM(const char* u):DataPointDirect(u) {
  if(strncasecmp("srm://",u,6)) return;
  is_valid=true;
}

DataPoint* DataPointSRM::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("srm://",u,6)) return NULL;
  return new DataPointSRM(u);
}

DataPointSRM::~DataPointSRM(void) {
}

/* TODO: do not use private attributes */
std::ostream& operator<<(std::ostream& o,const DataPoint &point) {
  if(!point) { o<<"<invalid>"; return o; };
  if(point.meta()) {
    if(point.have_location()) {
      o<<point.base_url()<<"["<<point.current_location()<<"]";
      return o;
    };
  };
  o<<point.base_url();
  return o;
}


// Wrappers

bool DataPoint::process_meta_url(void) {
  return false;
}

bool DataPoint::meta_resolve(bool source) {
  if(!instance) return false; return instance->meta_resolve(source);
}

bool DataPoint::meta_resolve(bool source,const UrlMap &maps) {
  if(!instance) return false; return instance->meta_resolve(source,maps);
}

bool DataPoint::meta_preregister(bool replication,bool force) {
  if(!instance) return false; return instance->meta_preregister(replication,force);
}

bool DataPoint::meta_postregister(bool replication,bool failure) {
  if(!instance) return false; return instance->meta_postregister(replication,failure);
}

bool DataPoint::meta_register(bool replication) {
  if(!instance) return false; return instance->meta_register(replication);
}

bool DataPoint::meta_preunregister(bool replication) {
  if(!instance) return false; return instance->meta_preunregister(replication);
}

bool DataPoint::meta_unregister(bool all) {
  if(!instance) return false; return instance->meta_unregister(all);
}

bool DataPoint::list_files(std::list<DataPoint::FileInfo> &files,bool long_list,bool resolve,bool metadata){
  if(!instance) return false; return instance->list_files(files,long_list,resolve,metadata);
}

bool DataPoint::get_info(DataPoint::FileInfo &fi) {
  if(!instance) return false; return instance->get_info(fi);
}

bool DataPoint::meta_size_available(void) const {
  if(!instance) return false; return instance->meta_size_available();
}

void DataPoint::meta_size(unsigned long long int val) {
  if(instance) instance->meta_size(val);
}

void DataPoint::meta_size_force(unsigned long long int val) {
  if(instance) instance->meta_size_force(val);
}

unsigned long long int DataPoint::meta_size(void) const {
  if(!instance) return 0; return instance->meta_size();
}

bool DataPoint::meta_checksum_available(void) const {
  if(!instance) return false; return instance->meta_checksum_available();
}

void DataPoint::meta_checksum(const char* val) {
  if(instance) instance->meta_checksum(val);
}

void DataPoint::meta_checksum_force(const char* val) {
  if(instance) instance->meta_checksum_force(val);
}

const char* DataPoint::meta_checksum(void) const {
  if(!instance) return ""; return instance->meta_checksum();
}

bool DataPoint::meta_created_available(void) const {
  if(!instance) return false; return instance->meta_created_available();
}

void DataPoint::meta_created(time_t val) {
  if(instance) instance->meta_created(val);
}

void DataPoint::meta_created_force(time_t val) {
  if(instance) instance->meta_created_force(val);
}

time_t DataPoint::meta_created(void) const {
  if(!instance) return 0; return instance->meta_created();
}

bool DataPoint::meta_validtill_available(void) const {
  if(!instance) return false; return instance->meta_validtill_available();
}

void DataPoint::meta_validtill(time_t val) {
  if(instance) instance->meta_validtill(val);
}

void DataPoint::meta_validtill_force(time_t val) {
  if(instance) instance->meta_validtill_force(val);
}

time_t DataPoint::meta_validtill(void) const {
  if(!instance) return 0; return instance->meta_validtill();
}

bool DataPoint::has_meta_attribute(std::string attr) const {
  if(!instance) return false; return instance->has_meta_attribute(attr);
}

std::string DataPoint::meta_attribute(std::string name) const {
  if(!instance) return ""; return instance->meta_attribute(name);
}

bool DataPoint::meta(void) const {
  if(!instance) return false; return instance->meta();
}

bool DataPoint::accepts_meta(void) {
  if(!instance) return false; return instance->accepts_meta();
}

bool DataPoint::provides_meta(void) {
  if(!instance) return false; return instance->provides_meta();
}

void DataPoint::meta(const DataPoint &p) {
  if(instance) instance->meta(p);
}

bool DataPoint::meta_compare(const DataPoint &p) const {
  if(!instance) return false; return instance->meta_compare(p);
}

bool DataPoint::meta_stored(void) {
  if(!instance) return false; return instance->meta_stored();
}

bool DataPoint::local(void) const {
  if(!instance) return false; return instance->local();
}

bool DataPoint::map(const UrlMap &maps) {
  if(!instance) return false; return instance->map(maps);
}

bool DataPoint::sort(const UrlMap &maps) {
  if(!instance) return false; return instance->sort(maps);
}

DataPoint::operator bool (void) const {
  if(!instance) return false; return (*instance);
}

bool DataPoint::operator!(void) const {
  if(!instance) return true; return !(*instance);
}

const char* DataPoint::current_location(void) const {
  if(!instance) return ""; return instance->current_location();
}

const char* DataPoint::current_meta_location(void) const {
  if(!instance) return ""; return instance->current_meta_location();
}

bool DataPoint::next_location(void) {
  if(!instance) return false; return instance->next_location();
}

bool DataPoint::have_location(void) const {
  if(!instance) return false; return instance->have_location();
}

bool DataPoint::have_locations(void) const {
  if(!instance) return false; return instance->have_locations();
}

bool DataPoint::remove_location(void) {
  if(!instance) return false; return instance->remove_location();
}

int DataPoint::tries(void) {
  if(!instance) return 0; return instance->tries();
}

void DataPoint::tries(int n) {
  if(instance) instance->tries(n);
}

std::string DataPoint::base_url(void) const {
  if(!instance) return ""; return instance->base_url();
}

std::string DataPoint::canonic_url(void) const {
  if(!instance) return ""; return instance->canonic_url();
}

const char* DataPoint::lfn(void) const {
  if(!instance) return ""; return instance->lfn();
}

bool DataPoint::add_location(const char* meta,const char* loc) {
  if(!instance) return false; return instance->add_location(meta,loc);
}

bool DataPoint::remove_locations(const DataPoint& p) {
  if(!instance) return false; return instance->remove_locations(p);
}

