#include "../std.h"

#include "../misc/guid.h"
#include "../misc/log_time.h"
#include "../misc/url_options.h"
#include "../misc/inttostring.h"

#include <arc/url.h>
/*
#include "../config/config_map.h"
#include "../transfer/rls.h"
#include "../misc/checksum.h"
#include "../misc/stringtoint.h"
#include "../misc/time_utils.h"
*/

#include "datapoint.h"

#ifdef HAVE_LFC_API_H
extern "C" {

/* See http://goc.grid.sinica.edu.tw/gocwiki/UsingLiblfcWithThreads
 * for LFC threading info */
  
/* use POSIX standard threads library */
#  include <pthread.h>
extern int Cthread_init(); /* from <Cthread_api.h> */
#  define thread_t      pthread_t
#  define thread_create(tid_p,func,arg)  pthread_create(tid_p,NULL,func,arg)
#  define thread_join(tid)               pthread_join(tid,NULL)

#include <lfc_api.h>

#ifndef _THREAD_SAFE
#define _THREAD_SAFE
#endif
#include <serrno.h>
}
#endif

DataPointLFC::DataPointLFC(const char* u):DataPointMeta(u), guid("") {

  // set retry env variables (don't overwrite if set already)
  // connection timeout
  setenv("LFC_CONNTIMEOUT", "30", 0);
  // number of retries
  setenv("LFC_CONRETRY", "1", 0);
  // interval between retries
  setenv("LFC_CONRETRYINT", "10", 0);

  if(u == NULL) return;
  if(strncasecmp("lfc://",u,6)) return;
  if(!process_meta_url()) return;
  if(locations.size()) location=locations.begin();
  
  // set host name env var
  setenv("LFC_HOST", meta_service_url.c_str()+6, 0);
  is_valid=true;
}

DataPoint* DataPointLFC::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("lfc://",u,6)) return NULL;
  return new DataPointLFC(u);
}

DataPointLFC::~DataPointLFC(void) {
}

/* perform resolve operation, which can take long time */
bool DataPointLFC::meta_resolve(bool source) {
#ifdef HAVE_LFC_API_H
  
#ifndef WITH_CTHREAD    
  /* Initialize Cthread library - should be called before any LFC-API function */
  if (0 !=  Cthread_init()) {
    odlog(ERROR)<<"Cthread_init() error: "<<sstrerror(serrno)<<std::endl;
    return false;
  }
#endif

  if(lfc_startsess((char*)(meta_service_url.c_str()+6),"ARC") != 0) {
    odlog(ERROR)<<"Error starting session: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  }

  if (source && !resolveGUIDToLFN()) {
    lfc_endsess();
    return false;
  }
  if (!source) {
    // check for guid in the attributes
    std::map<std::string, std::string>::iterator iter = meta_attributes.find("guid"); 
    if (iter != meta_attributes.end()) {
      guid = iter->second;
      odlog(DEBUG)<<"Using supplied guid "<<guid<<std::endl;
    }
  }
  
  is_resolved=false;
  is_metaexisting=false;
  if(source) {
    if(meta_lfn.length() == 0) {
      odlog(INFO)<<"Source must contain LFN"<<std::endl;
      lfc_endsess();
      return false;
    };
  } else {
    if(meta_lfn.length() == 0) {
      odlog(INFO)<<"Destination must contain LFN"<<std::endl;
      lfc_endsess();
      return false;
    };
    if(locations.size() == 0) {
      odlog(INFO)<<"Locations are missing in destination LFC URL"<<std::endl;
      lfc_endsess();
      return false;
    };
  };
  int nbentries = 0;
  struct lfc_filereplica* entries = NULL;
  if(lfc_getreplica(meta_lfn.c_str(),NULL,NULL,&nbentries,&entries) != 0) {
    if(source || ((serrno != ENOENT) && (serrno != ENOTDIR))) {
      odlog(ERROR)<<"Error finding replicas: "<<sstrerror(serrno)<<std::endl;
      lfc_endsess();
      return false;
    };
    nbentries=0; entries=NULL;
  } else {
    is_metaexisting=true;
  };
  if(locations.size() == 0) {
    for(int n=0;n<nbentries;n++) {
      std::list<DataPointDirect::Location>::iterator loc =
           locations.insert(locations.end(),
                DataPointDirect::Location(meta_service_url,entries[n].sfn));
      loc->arg=(void*)1; // marker
      odlog(DEBUG)<<"Adding location: "<<meta_service_url<<" - "<<entries[n].sfn<<std::endl;
    };
  } else {
    std::list<Location>::iterator loc = locations.begin();
    for(;loc!=locations.end();++loc) {
      if(loc->arg != NULL) continue;
      for(int n=0;n<nbentries;n++) {
        if(strncmp(entries[n].sfn,loc->meta.c_str(),loc->meta.length())==0){
          odlog(DEBUG)<<"Adding location: "<<meta_service_url<<" - "<<entries[n].sfn<<std::endl;
          if(source) {
            loc->meta=meta_service_url; loc->url=entries[n].sfn;
          } else {
            loc->meta=meta_service_url;
          };
          loc->arg=(void*)1; break;
        };
      };
    };
  };
  if(entries) free(entries);
  struct lfc_filestatg st;
  if(lfc_statg(meta_lfn.c_str(),NULL,&st) == 0) {
    is_metaexisting=true;
    meta_size(st.filesize);
    meta_created(st.mtime);
    if(st.csumtype[0] && st.csumvalue[0]) {
      std::string csum = st.csumtype;
      if (csum == "MD") csum = "md5"; 
      csum+=":"; csum+=st.csumvalue;
      meta_checksum(csum.c_str());
    };
    if (guid.empty()) guid=st.guid;
  };
  lfc_endsess();
  if(!source) {
    if(locations.size() == 0) {
      odlog(INFO)<<"No locations found for destination"<<std::endl;
      return false;
    };
    // Make pfns
    std::list<Location>::iterator loc = locations.begin();
    for(;loc!=locations.end();) {
      if(loc->arg != NULL) { loc=locations.erase(loc); continue; };
      if(strncasecmp(loc->url.c_str(),"se://",5) == 0) { loc->url+="?"; }
      else if(loc->url.find_last_of("/") != loc->url.length()-1) { loc->url+="/"; };
      // take off leading dirs of LFN
      std::string::size_type slash_index = meta_lfn.rfind("/", meta_lfn.length()+1);
      if (slash_index != std::string::npos) loc->url += meta_lfn.substr(slash_index+1);
      else loc->url+=meta_lfn;
      odlog(DEBUG)<<"Using location: "<<loc->meta<<" - "<<loc->url<<std::endl;
      ++loc;
    };
  };
  odlog(DEBUG)<<"meta_get_data: checksum: "<<meta_checksum()<<std::endl;
  odlog(DEBUG)<<"meta_get_data: size: "<<meta_size()<<std::endl;
  time_t creation_time = meta_created();
  odlog(DEBUG)<<"meta_get_data: created: "<<ctime(&creation_time);
  if(common_url_options.length() != 0) {
    std::list<Location>::iterator loc = locations.begin();
    for(;loc!=locations.end();++loc) {
      add_url_options(loc->url,common_url_options.c_str(),0);
    };
  };
  location=locations.begin();
  is_resolved=true;
  return true;
#else
  return false;
#endif
}

bool DataPointLFC::meta_preregister(bool replication,bool force) {
#ifdef HAVE_LFC_API_H
  
#ifndef WITH_CTHREAD    
  /* Initialize Cthread library - should be called before any LFC-API function */
  if (0 !=  Cthread_init()) {
    odlog(ERROR)<<"Cthread_init() error: "<<sstrerror(serrno)<<std::endl;
    return false;
  }
#endif
  
  if(replication) { /* replicating inside same lfn */
    if(!is_metaexisting) { /* for replication it must be there */
      odlog(ERROR)<<"LFN is missing in LFC (needed for replication)"<<std::endl;
      return false;
    };
    return true;
  };
  if(is_metaexisting) { /* algorithm require this to be new file */
    if(!force) {
      odlog(ERROR)<<"LFN already exists in LFC"<<std::endl;
      return false;
    };
    return true;
  };
  if(lfc_startsess((char*)(meta_service_url.c_str()+6),"ARC") != 0) {
  	odlog(ERROR)<<"Error starting session: "<<sstrerror(serrno)<<std::endl;
  	lfc_endsess();
  	return false;
  }
  if (guid.empty()) {
    GUID(guid);
  }
  else {
    // check for guid in the attributes
    std::map<std::string, std::string>::iterator iter = meta_attributes.find("guid"); 
    if (iter != meta_attributes.end()) { 
      guid = iter->second;
      odlog(DEBUG)<<"Using supplied guid "<<guid<<std::endl;
    }
  }
  if(lfc_creatg(meta_lfn.c_str(),guid.c_str(),
                S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP) != 0) {
    // check error - if it is no such file or directory, try to create the
    // required dirs
    if (serrno == ENOENT) {
      std::string::size_type slashpos = meta_lfn.find("/", 1); // don't create root dir
      while (slashpos != std::string::npos) {
        std::string dirname = meta_lfn.substr(0, slashpos);
        // list dir to see if it exists (we can't tell the difference between
        // dir already exists and permission denied)
        struct lfc_filestat statbuf;
        if (lfc_stat(dirname.c_str(), &statbuf) == 0) {
          slashpos = meta_lfn.find("/", slashpos+1);
          continue;
        };
        
        odlog(DEBUG)<<"Creating LFC directory "<<dirname<<std::endl;
        if (lfc_mkdir(dirname.c_str(), 0775) != 0) {
          if (serrno != EEXIST) {
            odlog(ERROR)<<"Error creating required LFC dirs: "<<sstrerror(serrno)<<std::endl;
            lfc_endsess();
            return false;
          };
        };
        slashpos = meta_lfn.find("/", slashpos+1);
      };
      if(lfc_creatg(meta_lfn.c_str(),guid.c_str(),
                    S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP) != 0) {
        odlog(ERROR)<<"Error creating LFC entry: "<<sstrerror(serrno)<<std::endl;
        lfc_endsess();
        return false;
      };
    }
    else {
      odlog(ERROR)<<"Error creating LFC entry: "<<sstrerror(serrno)<<std::endl;
      lfc_endsess();
      return false;
    };
  };
  if(meta_checksum_valid) {
    std::string ckstype;
    std::string cksumvalue = meta_checksum();
    std::string::size_type p = cksumvalue.find(':');
    if(p == std::string::npos) {
      ckstype="cksum";
    } else {
      ckstype=cksumvalue.substr(0,p);
      cksumvalue=cksumvalue.substr(p+1);
    };
    if(meta_size_valid) {
      lfc_setfsizeg(guid.c_str(),meta_size(),ckstype.c_str(),(char*)(cksumvalue.c_str()));
    } else {
      lfc_setfsizeg(guid.c_str(),0,ckstype.c_str(),(char*)(cksumvalue.c_str()));
    };
  } else if(meta_size_valid) {
    lfc_setfsizeg(guid.c_str(),meta_size(),NULL,NULL);
  };
  lfc_endsess();
  return true;
#else
  return false;
#endif
}

bool DataPointLFC::meta_postregister(bool replication,bool failure) {
#ifdef HAVE_LFC_API_H
  
#ifndef WITH_CTHREAD    
  /* Initialize Cthread library - should be called before any LFC-API function */
  if (0 !=  Cthread_init()) {
    odlog(ERROR)<<"Cthread_init() error: "<<sstrerror(serrno)<<std::endl;
    return false;
  }
#endif
  
  if(guid.empty()) {
    odlog(ERROR)<<"No GUID defined for LFN - probably not preregistered"<<std::endl;
    return false;
  };
  std::string pfn(location->url.c_str());
  ::canonic_url(pfn); // it is always better to register pure url
  std::string server;
  {
    URL u(location->url);
    server=u.Host();
  };
  if(lfc_startsess((char*)(meta_service_url.c_str()+6),"ARC") != 0) {
    odlog(ERROR)<<"Error starting session: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  }
  if(lfc_addreplica(guid.c_str(),NULL,server.c_str(),pfn.c_str(),'-','P',NULL,NULL) != 0) {
    odlog(ERROR)<<"Error adding replica: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  };
  if(meta_checksum_valid) {
    std::string ckstype;
    std::string cksumvalue = meta_checksum();
    std::string::size_type p = cksumvalue.find(':');
    if(p == std::string::npos) {
      ckstype="cksum";
    } else {
      ckstype=cksumvalue.substr(0,p);
      if (ckstype=="md5") ckstype = "MD";
      if (ckstype=="ad") ckstype = "AD";
      cksumvalue=cksumvalue.substr(p+1);
      odlog(DEBUG)<<"Entering checksum type "<<ckstype<<", value "<<cksumvalue<<", file size "<<meta_size()<<std::endl;
    };
    if(meta_size_valid) {
      lfc_setfsizeg(guid.c_str(),meta_size(),ckstype.c_str(),(char*)(cksumvalue.c_str()));
     } else {
      lfc_setfsizeg(guid.c_str(),0,ckstype.c_str(),(char*)(cksumvalue.c_str()));
     };
  } else if(meta_size_valid) {
    lfc_setfsizeg(guid.c_str(),meta_size(),NULL,NULL);
  };
  lfc_endsess();
  return true;
#else
  return false;
#endif
}

bool DataPointLFC::meta_preunregister(bool replication) {
#ifdef HAVE_LFC_API_H
  
#ifndef WITH_CTHREAD    
  /* Initialize Cthread library - should be called before any LFC-API function */
  if (0 !=  Cthread_init()) {
    odlog(ERROR)<<"Cthread_init() error: "<<sstrerror(serrno)<<std::endl;
    return false;
  }
#endif
  
  if(replication) return true;
  if(lfc_startsess((char*)(meta_service_url.c_str()+6),"ARC") != 0) {
    odlog(ERROR)<<"Error starting session: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  }
  if (!resolveGUIDToLFN()) {
    lfc_endsess();
    return false;
  }
  if(lfc_unlink(meta_lfn.c_str()) != 0) {
    if((serrno != ENOENT) && (serrno != ENOTDIR)) {
      odlog(ERROR)<<"Failed to remove LFN in LFC - You may need to do that by hand"<<std::endl;
      lfc_endsess();
      return false;
    };
  };
  lfc_endsess();
  return true;
#else
  return false;
#endif
}

bool DataPointLFC::meta_unregister(bool all) {
#ifdef HAVE_LFC_API_H
  
#ifndef WITH_CTHREAD    
  /* Initialize Cthread library - should be called before any LFC-API function */
  if (0 !=  Cthread_init()) {
    odlog(ERROR)<<"Cthread_init() error: "<<sstrerror(serrno)<<std::endl;
    return false;
  }
#endif
  
  if(!all) {
    if(location == locations.end()) { 
      odlog(ERROR)<<"Location is missing"<<std::endl;
      return false;
    };
    if(strncasecmp(location->url.c_str(),"se://",5) == 0) {
      odlog(DEBUG)<<"SE location will be unregistered automatically"<<std::endl;
      fix_unregistered(all);
      return true;
    };
  };
  if(lfc_startsess((char*)(meta_service_url.c_str()+6),"ARC") != 0) {
    odlog(ERROR)<<"Error starting session: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  }
  if (!resolveGUIDToLFN()) {
    lfc_endsess();
    return false;
  }
  if(all) {
    int nbentries = 0;
    struct lfc_filereplica* entries = NULL;
    if(lfc_getreplica(meta_lfn.c_str(),NULL,NULL,&nbentries,&entries) != 0) {
      lfc_endsess();
      if((serrno == ENOENT) || (serrno == ENOTDIR)) {
        fix_unregistered(all);
        return true;
      };
      odlog(ERROR)<<"Error getting replicas: "<<sstrerror(serrno)<<std::endl;
      return false;
    };
    for(int n=0;n<nbentries;n++) {
      if(lfc_delreplica(guid.c_str(),NULL,entries[n].sfn) != 0) {
        if(serrno == ENOENT) continue;
        lfc_endsess();
        odlog(ERROR)<<"Failed to remove location from LFC: "<<sstrerror(serrno)<<std::endl;
        return false;
      };
    };
    if(lfc_unlink(meta_lfn.c_str()) != 0) {
      if(serrno == EPERM) { // we have a directory
        if(lfc_rmdir(meta_lfn.c_str()) != 0) {
          if(serrno == EEXIST) { // still files in the directory
            odlog(ERROR)<<"Failed to remove LFC directory: directory is not empty"<<std::endl;
            lfc_endsess();
            return false;
          };
          odlog(ERROR)<<"Failed to remove LFC directory: "<<sstrerror(serrno)<<std::endl;
          lfc_endsess();
          return false;
        };
      }
      else if((serrno != ENOENT) && (serrno != ENOTDIR)) {
        odlog(ERROR)<<"Failed to remove LFN in LFC: "<<sstrerror(serrno)<<std::endl;
        lfc_endsess();
        return false;
      };
    };
  } else {
    if(lfc_delreplica(guid.c_str(),NULL,location->url.c_str()) != 0) {
      lfc_endsess();
      odlog(ERROR)<<"Failed to remove location from LFC: "<<sstrerror(serrno)<<std::endl;
      return false;
    };
  };
  lfc_endsess();
  fix_unregistered(all);
  return true;
#else
  return false;
#endif
}

/* lfc://[url[|url[...]]@]server/lfn */
bool DataPointLFC::process_meta_url(void) {
  if(!strncasecmp(url.c_str(),"lfc://",6)) {
    meta_service_url.resize(0); locations.clear(); meta_lfn.resize(0);
    std::string url_(url.c_str());
    /* find out if it contains locations */
    std::string::size_type loc_start=6;
    std::string::size_type loc_end=url_.find('@',loc_start);
    std::string urls("");
    if(loc_end!=std::string::npos) {
      urls=url_.substr(loc_start,loc_end-loc_start);
      url_.erase(loc_start,loc_end-loc_start+1);
    };
    /* get lfn */
    std::string::size_type server_start=6;
    std::string::size_type server_end=url_.find('/',server_start);
    std::string meta_lfn;
    if(server_end==std::string::npos) {
      meta_lfn=""; meta_service_url=url_;
    } else {
      meta_lfn=url_.substr(server_end+1);
      meta_service_url=url_.substr(0,server_end);
    };
    std::string guid_val;
//    if(get_url_option(meta_service_url,"guid",guid_val) == 0) {
//      if((guid_val == "yes") || (guid_val == "")) {
//        guid_enabled=true;
//      };
//    };
    ::canonic_url(meta_service_url);
    extract_meta_attributes(meta_lfn);
    this->meta_lfn=meta_lfn;
    odlog(DEBUG)<<"LFN: "<<meta_lfn<<std::endl;
    odlog(DEBUG)<<"LFC server: "<<meta_service_url<<std::endl;
    odlog(DEBUG)<<"Location urls: "<<urls<<std::endl;
    std::string::size_type n=0;
    for(std::string::size_type nn=0;n<urls.length();) {
      nn=urls.find('|',n); if(nn == std::string::npos) nn=urls.length();
      if(n == nn) { n++; continue; };
      std::string loc(urls.c_str()+n,nn-n);
      if(loc[0] == ';') { common_url_options+=loc; }
      else {
        locations.push_back(DataPointDirect::Location(loc.c_str(),loc.c_str()));
      };
      n=nn+1;
    };
    return true;
  };
  return false;
};

bool DataPointLFC::list_files(std::list<DataPoint::FileInfo> &files,
                              bool long_list,
                              bool resolve,
                              bool metadata) {
#ifdef HAVE_LFC_API_H
  
#ifndef WITH_CTHREAD    
  /* Initialize Cthread library - should be called before any LFC-API function */
  if (0 !=  Cthread_init()) {
    odlog(ERROR)<<"Cthread_init() error: "<<sstrerror(serrno)<<std::endl;
    return false;
  }
#endif
  
  if(lfc_startsess((char*)(meta_service_url.c_str()+6),"ARC") != 0) {
    odlog(ERROR)<<"Error starting session: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  };
  if (!resolveGUIDToLFN()) {
    lfc_endsess();
    return false;
  }
  // first stat the url and see if it is a file or directory
  struct lfc_filestatg st;
  if(lfc_statg(meta_lfn.c_str(),NULL,&st) != 0) {
    odlog(ERROR)<<"Error listing file or directory: "<<sstrerror(serrno)<<std::endl;
    lfc_endsess();
    return false;
  };

  // if it's a directory, list entries
  if((st.filemode & S_IFDIR) && !metadata) {
    odlog(DEBUG)<<"listing dir"<<std::endl;
    lfc_DIR * dir = lfc_opendirxg((char*)(meta_service_url.c_str()+6), (char*)meta_lfn.c_str(), NULL);
    if (dir == NULL) {
      odlog(ERROR)<<"Error opening directory: "<<sstrerror(serrno)<<std::endl;
      lfc_endsess();
      return false;
    };

    // list entries. If not long list use readdir
    // does funny things with pointers, so always use long listing for now
    if (false) { // if (!long_list) {
      struct dirent * direntry;
      while ((direntry = lfc_readdir(dir))) {
        odlog(DEBUG)<<"d_name "<<direntry->d_name<<std::endl;
        std::list<DataPoint::FileInfo>::iterator f = files.insert(files.end(),FileInfo(direntry->d_name));
      };
    }
    else { // for long list use readdirg
      struct lfc_direnstatg * direntry;
      while ((direntry = lfc_readdirg(dir))) {      
        std::list<DataPoint::FileInfo>::iterator f = files.insert(files.end(),FileInfo(direntry->d_name));

        f->size=direntry->filesize; f->size_available=true;
        if(strcmp(direntry->csumtype, "") != 0 && strcmp(direntry->csumvalue, "") != 0) {
          f->checksum=direntry->csumtype; f->checksum+=":"; f->checksum+=direntry->csumvalue;
          f->checksum_available=true;
        };
        f->created=direntry->ctime; f->created_available=true;
        f->type=(direntry->filemode & S_IFDIR) ? DataPoint::FileInfo::file_type_dir:DataPoint::FileInfo::file_type_file;
      };
    };
    if(serrno) {
      odlog(ERROR)<<"Error listing directory: "<<sstrerror(serrno)<<std::endl;
      lfc_closedir(dir);
      lfc_endsess();
      return false;
    };
    
    // if we want to resolve replicas call readdirxr
    if(resolve) {
      lfc_rewinddir(dir);
      struct lfc_direnrep * direntry;

      while ((direntry = lfc_readdirxr(dir, NULL))) {
        for (std::list<DataPoint::FileInfo>::iterator f = files.begin();
             f != files.end();
             f++) {
          if (f->name == direntry->d_name) {
            for(int n=0;n<direntry->nbreplicas;n++) {
              f->urls.push_back(std::string(direntry->rep[n].sfn));
            };
            break;
          };
        };
      };
      if(serrno) {
        odlog(ERROR)<<"Error listing directory: "<<sstrerror(serrno)<<std::endl;
        lfc_closedir(dir);
        lfc_endsess();
        return false;
      };
    };
    lfc_closedir(dir);
  } // if (dir)
  else {
    std::list<DataPoint::FileInfo>::iterator f = files.insert(files.end(),FileInfo(meta_lfn.c_str()));
    f->metadata["path"] = meta_lfn;
    f->size=st.filesize; f->size_available=true;
    f->metadata["size"] = inttostring(st.filesize);
    if(st.csumvalue[0]) {
      std::string checksum = std::string(st.csumtype) + ':' + std::string(st.csumvalue);
      f->checksum=checksum;
      f->checksum_available=true;
      f->metadata["checksum"] = checksum;
    };
    f->created=st.mtime; f->created_available=true;
    const time_t modtime = st.mtime;
    std::string mtimestr = ctime(&modtime);
    f->metadata["mtime"] = mtimestr.erase(mtimestr.length()-1);
    
    f->type=(st.filemode & S_IFDIR)?DataPoint::FileInfo::file_type_dir:DataPoint::FileInfo::file_type_file;
    f->metadata["type"] = (st.filemode & S_IFDIR) ? "dir" : "file";
    if(resolve) {
      int nbentries = 0;
      struct lfc_filereplica* entries = NULL;

      if(lfc_getreplica(meta_lfn.c_str(),NULL,NULL,&nbentries,&entries) != 0) {
        odlog(ERROR)<<"Error listing replicas: "<<sstrerror(serrno)<<std::endl;
        lfc_endsess();
        return false;
      };
      for(int n=0;n<nbentries;n++) {
        f->urls.push_back(std::string(entries[n].sfn));
      };
    };
    // fill some more metadata
    if (st.guid[0] != '\0') f->metadata["guid"] = st.guid;
    if(metadata) {
      char username[256];
      if (lfc_getusrbyuid(st.uid, username) == 0) f->metadata["owner"] = username;
      char groupname[256];
      if (lfc_getgrpbygid(st.gid, groupname) == 0) f->metadata["group"] = groupname;
    };
    mode_t mode = st.filemode;
    std::string perms;
    if (mode & S_IRUSR) perms += 'r'; else perms += '-';
    if (mode & S_IWUSR) perms += 'w'; else perms += '-';
    if (mode & S_IXUSR) perms += 'x'; else perms += '-';
    if (mode & S_IRGRP) perms += 'r'; else perms += '-';
    if (mode & S_IWGRP) perms += 'w'; else perms += '-';
    if (mode & S_IXGRP) perms += 'x'; else perms += '-';
    if (mode & S_IROTH) perms += 'r'; else perms += '-';
    if (mode & S_IWOTH) perms += 'w'; else perms += '-';
    if (mode & S_IXOTH) perms += 'x'; else perms += '-';
    f->metadata["accessperm"] = perms;
    
    const time_t chtime = st.ctime;
    std::string chtimestr = ctime(&chtime);
    f->metadata["ctime"] = chtimestr.erase(chtimestr.length()-1);
    const time_t atime = st.atime;
    std::string atimestr = ctime(&atime);
    f->metadata["atime"] = atimestr.erase(atimestr.length()-1);
  };
  lfc_endsess();
  return true;

#else
  return false;
#endif
}
  
bool DataPointLFC::resolveGUIDToLFN() {

#ifdef HAVE_LFC_API_H

  // check if guid is already defined
  if (!guid.empty()) return true;
    
  // check for guid in the attributes
  std::map<std::string, std::string>::iterator iter = meta_attributes.find("guid"); 
  if (iter != meta_attributes.end()) {
    guid = iter->second;
    lfc_list listp;
    struct lfc_linkinfo * info = lfc_listlinks (NULL, (char*)guid.c_str(), CNS_LIST_BEGIN, &listp);
    if (!info) {
      odlog(ERROR)<<"Error finding LFN from guid "<<guid<<": "<<sstrerror(serrno)<<std::endl;
      return false;
    }
    meta_lfn = std::string(info[0].path);
    odlog(DEBUG)<<"guid "<<guid<<" resolved to LFN "<<meta_lfn<<std::endl;
    lfc_listlinks (NULL, (char*)guid.c_str(), CNS_LIST_END, &listp);
  }
  return true;
#else
  return false;
#endif
}
