#include "../../std.h"
#ifdef HAVE_SETFSUID
#include <sys/fsuid.h>
#endif

#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#define GRIDFTP_PLUGIN
#include "../../config/conf.h"
#include "../../jobs/job.h"
#include "../../jobs/commfifo.h"
#include "../../jobs/plugins.h"
#include "../userspec.h"
#include "../names.h"
#include "../misc.h"
#include "../../config/environment.h"
#include "../../config/config_pre.h"
#include "../../files/info_types.h"
#include "../../files/info_files.h"
#include "../../jobs/job_request.h"
#include "../../misc/inttostring.h"
#include "../../misc/stringtoint.h"
#include "../../misc/delete.h"
#include "../../misc/escaped.h"
#include "../../misc/log_time.h"
#include "../../misc/proxy.h"
#include "../../run/run_parallel.h"
#include "../fileplugin/fileplugin.h"
#include <arc/certificate.h>
#include "jobplugin.h"

RunParallel JobPlugin::run;


typedef struct {
  const JobUser* user;
  const std::string* job;
  const char* reason;
} job_subst_t;

#define IS_ALLOWED_READ  1
#define IS_ALLOWED_WRITE 2
#define IS_ALLOWED_LIST  4
#define IS_ALLOWED_RW    3
#define IS_ALLOWED_ALL   7

#ifdef HAVE_SETFSUID

// Non-portable solution. Needed as long as Linux
// does not support proper setuid in threads
#define SET_USER_UID {  ::setfsuid(user->get_uid()); ::setfsgid(user->get_gid()); }
#define RESET_USER_UID { ::setfsuid(getuid()); ::setfsgid(getgid()); }

#else

// Not sure how this will affect other threads. Most probably 
// not in a best way. Anyway this option is not for linux.
#define SET_USER_UID { setegid(user->get_gid()); seteuid(user->get_uid()); }
#define RESET_USER_UID { seteuid(getuid()); setegid(getgid()); }

#endif

static void job_subst(std::string& str,void* arg) {
  job_subst_t* subs = (job_subst_t*)arg;
  if(subs->job) for(std::string::size_type p = 0;;) {
    p=str.find('%',p);
    if(p==std::string::npos) break;
    if(str[p+1]=='I') {
      str.replace(p,2,subs->job->c_str());
      p+=subs->job->length();
    } else if(str[p+1]=='S') {
      str.replace(p,2,"UNKNOWN"); // WRONG
      p+=7;
    } else if(str[p+1]=='O') {
      str.replace(p,2,subs->reason);
      p+=strlen(subs->reason);
    } else {
      p+=2;
    };
  };
  if(subs->user) subs->user->substitute(str);
}

// run external plugin to acquire non-unix local credentials
// U - user, J - job, O - reason
#define ApplyLocalCred(U,J,O) {                              \
  if(cred_plugin && (*cred_plugin)) {                        \
    job_subst_t subst_arg;                                   \
    subst_arg.user=U;                                        \
    subst_arg.job=J;                                         \
    subst_arg.reason=O;                                      \
    if(!cred_plugin->run(job_subst,&subst_arg)) {            \
      olog << "Failed to run plugin" << std::endl;                \
      return 1;                                              \
    };                                                       \
    if(cred_plugin->result() != 0) {                         \
      olog << "Plugin failed: " << cred_plugin->result() << std::endl;  \
      return 1;                                              \
    };                                                       \
  };                                                         \
}


JobPlugin::JobPlugin(std::istream &cfile,userspec_t &user_s):user_a(user_s.user),job_map(user_s.user) {
  initialized=true;
  rsl_opened=false;
  user=NULL;
  direct_fs=NULL;
  proxy_fname="";
  std::string configfile("");
  cont_plugins=NULL;
  cred_plugin=NULL;
  readonly=false;
  chosenFilePlugin=NULL;
  for(;;) {
    std::string rest;
    std::string command=config_read_line(cfile,rest);
    if(command.length() == 0) { break; } /* end of file - should not be here */
    else if(command == "configfile") {
      input_escaped_string(rest.c_str(),configfile); 
    } else if(command == "allownew") {
      std::string value("");
      input_escaped_string(rest.c_str(),value);
      if(strcasecmp(value.c_str(),"no") == 0) { readonly=true; }
      else if(strcasecmp(value.c_str(),"yes") == 0) { readonly=false; };
    } else if(command == "unixmap") {  /* map to local unix user */
      if(!job_map) job_map.mapname(rest.c_str());
    } else if(command == "unixgroup") {  /* map to local unix user */
      if(!job_map) job_map.mapgroup(rest.c_str());
    } else if(command == "unixvo") {  /* map to local unix user */
      if(!job_map) job_map.mapvo(rest.c_str());
    } else if(command == "remotegmdirs") {
      std::string remotedir = config_next_arg(rest);
      if(remotedir.length() == 0) {
        olog << "empty argument to remotedirs" << std::endl;
        initialized=false;
      };
      struct gm_dirs_ dirs;
      dirs.control_dir = remotedir;
      remotedir = config_next_arg(rest);
      if(remotedir.length() == 0) {
        olog << "bad arguments to remotedirs" << std::endl;
        initialized=false;
      };
      dirs.session_dir = remotedir;
      gm_dirs_info.push_back(dirs);
    } else if(command == "end") {
      break; /* end of section */
    } else {
      olog<<"Warning: unsupported configuration command: "<<command<<std::endl;
    };
  };
  if(!run.is_initialized()) {
    olog<<"Warning: Initialization of signal environment failed"<<std::endl;
  };
  if(!read_env_vars()) { initialized=false; }
  else {
    if(configfile.length()) nordugrid_config_loc=configfile;
    const char* uname = user_s.get_uname();
    if((bool)job_map) uname=job_map.unix_name();
    user=new JobUser(uname);
    if(!user->is_valid()) { initialized=false; }
    else {
      /* read configuration */
      if(nordugrid_loc.length() != 0) {
        std::vector<std::string> session_roots;
        std::string control_dir;
        std::string default_lrms;
        std::string default_queue;
        cont_plugins = new ContinuationPlugins;
        cred_plugin = new RunPlugin;
        std::string allowsubmit;
        bool strict_session;
        if(!configure_user_dirs(uname,
          control_dir,session_roots,
          default_lrms,default_queue,avail_queues,*cont_plugins,*cred_plugin,
          allowsubmit,strict_session)) {
          olog<<"Failed processing grid-manager configuration"<<std::endl;
          initialized=false;
        } else {
          if(default_queue.empty() && (avail_queues.size() == 1)) {
            default_queue=*(avail_queues.begin());
          };
          user->SetControlDir(control_dir);
          user->SetSessionRoot(session_roots);
          user->SetLRMS(default_lrms,default_queue);
          user->SetStrictSession(strict_session);
          for(;allowsubmit.length();) {
            std::string group = config_next_arg(allowsubmit);
            if(user_a.check_group(group)) { readonly=false; break; };
          };
          if(readonly) olog<<"This user is denied to submit new jobs."<<std::endl;
          if(!session_roots.empty()) {
            struct gm_dirs_ dirs;
            dirs.control_dir = user->ControlDir();
            dirs.session_dir = user->SessionRoot();
            gm_dirs_info.push_back(dirs);
          };
          session_dirs = user->SessionRoots();
          /* link to the class for direct file access - creating one object per set of GM dirs */
          // choose whether to use multiple session dirs or remote GM dirs
          if (session_dirs.size() > 1) {
            for (std::vector<std::string>::iterator i = session_dirs.begin(); i != session_dirs.end(); i++) {
              std::string direct_config = "";
              direct_config += "mount "+(*i)+"\n";
              direct_config+="dir / nouser read cd dirlist delete append overwrite";
              direct_config+=" create "+
                  inttostring(user->get_uid())+":"+inttostring(user->get_gid())+
                  " 600:600";
              direct_config+=" mkdir "+
                  inttostring(user->get_uid())+":"+inttostring(user->get_gid())+
                  " 700:700\n";
              direct_config+="end\n";
  #ifdef HAVE_SSTREAM
              std::stringstream fake_cfile(direct_config);
  #else
              std::strstream fake_cfile;
              fake_cfile<<direct_config;
  #endif
              DirectFilePlugin * direct_fs = new DirectFilePlugin(fake_cfile,user_s);
              file_plugins.push_back(direct_fs);
            }
          }
          else {
            for (std::vector<struct gm_dirs_>::iterator i = gm_dirs_info.begin(); i != gm_dirs_info.end(); i++) {
              std::string direct_config = "";
              direct_config += "mount "+(*i).session_dir+"\n";
              direct_config+="dir / nouser read cd dirlist delete append overwrite";
              direct_config+=" create "+
                  inttostring(user->get_uid())+":"+inttostring(user->get_gid())+
                  " 600:600";
              direct_config+=" mkdir "+
                  inttostring(user->get_uid())+":"+inttostring(user->get_gid())+
                  " 700:700\n";
              direct_config+="end\n";
  #ifdef HAVE_SSTREAM
              std::stringstream fake_cfile(direct_config);
  #else
              std::strstream fake_cfile;
              fake_cfile<<direct_config;
  #endif
              DirectFilePlugin * direct_fs = new DirectFilePlugin(fake_cfile,user_s);
              file_plugins.push_back(direct_fs);
            }
          }
          if((bool)job_map) {
            olog<<"Job submission user: "<<uname<<
                  " ("<<user->get_uid()<<":"<<user->get_gid()<<")"<<std::endl;
          };
        };
      } else {
        initialized=false;
      };
    };
  };
  if(!initialized) if(user) { delete user; user=NULL; };
  if((!user_a.is_proxy()) ||
     (user_a.proxy() == NULL) || (user_a.proxy()[0] == 0)) {
    olog<<"WARNING: no delegated credentials were passed"<<std::endl;
  } else {
    proxy_fname=user_a.proxy();
  };
  subject=user_a.DN();
  port=user_s.get_port();
  memcpy(host,user_s.get_host(),sizeof(host));
  job_id="";
#if 0
  if(user) {
    if(user->StrictSession()) {
      // Changing unix user. That means control directory must be
      // writable for every serviced user.
      user->SwitchUser(true);
    };
  };
#endif
}

JobPlugin::~JobPlugin(void) {
  delete_job_id();
  if(proxy_fname.length() != 0) { remove(proxy_fname.c_str()); };
  if(cont_plugins) delete cont_plugins;
  if(cred_plugin) delete cred_plugin;
//  olog << "JobPlugin: destructor " << count << std::endl;
};
 

int JobPlugin::makedir(std::string &dname) {
  if(!initialized) return 1;
  std::string id;
  bool spec_dir;
  if((dname == "new") || (dname == "info")) return 0;
  if(is_allowed(dname.c_str(),true,&spec_dir,&id) & IS_ALLOWED_WRITE) {
    if(spec_dir) {
      error_description="Can't create subdirectory in a special directory.";
      return 1;
    };
    ApplyLocalCred(user,&id,"write");
    DirectFilePlugin * fp = selectFilePlugin(id);
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=fp->makedir(dname);
      RESET_USER_UID;
      return r;
    };
    return fp->makedir(dname);
  };
  error_description="Not allowed for this job.";
  return 1;
}

int JobPlugin::removefile(std::string &name) {
  if(!initialized) return 1;
  if(name.find('/') == std::string::npos) { /* request to cancel the job */
    if((name == "new") || (name == "info")) {
      error_description="Special directory can't be mangled.";
      return 1;
    };
    if(is_allowed(name.c_str()) & IS_ALLOWED_WRITE) {  /* owner of the job */
      JobId id(name); JobDescription job_desc(id,"");
      user->SetControlDir(selectControlDir(id));
      if(job_cancel_mark_put(job_desc,*user)) return 0;
    };
    error_description="Not allowed to cancel this job.";
    return 1;
  };
  const char* logname;
  std::string id;
  bool spec_dir;
  if(is_allowed(name.c_str(),false,&spec_dir,&id,&logname) & IS_ALLOWED_WRITE) {
    if(logname) {
      if((*logname) != 0) return 0; /* pretend status file is deleted */
    };
    if(spec_dir) {
      error_description="Special directory can't be mangled.";
      return 1; /* can delete status directory */
    };
    ApplyLocalCred(user,&id,"write");
    DirectFilePlugin * fp = selectFilePlugin(id);
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=fp->removefile(name);
      RESET_USER_UID;
      return r;
    };
    return fp->removefile(name);
  };
  error_description="Not allowed for this job.";
  return 1;
}

int JobPlugin::removedir(std::string &dname) {
  if(!initialized) return 1;
  if(dname.find('/') == std::string::npos) { /* request to clean the job */
    if((dname == "new") || (dname == "info")) {
      error_description="Special directory can't be mangled.";
      return 1;
    };
    if(is_allowed(dname.c_str()) & IS_ALLOWED_WRITE) {  /* owner of the job */
      /* check the status */
      JobId id(dname); 
      user->SetControlDir(selectControlDir(id));
      user->SetSessionRoot(selectSessionDir(id));
      job_state_t status=job_state_read_file(id,*user);
      if((status == JOB_STATE_FINISHED) ||
         (status == JOB_STATE_DELETED)) { /* remove files */
        if(job_clean_final(JobDescription(id,user->SessionRoot()+"/"+id),
                           *user)) return 0;
      }
      else { /* put marks */
        JobDescription job_desc(id,"");
        bool res = job_cancel_mark_put(job_desc,*user);
        res &= job_clean_mark_put(job_desc,*user);
        if(res) return 0;
      };
      error_description="Failed to clean job.";
      return 1;
    };
    error_description="Not allowed for this job.";
    return 1;
  };
  std::string id;
  bool spec_dir;
  if(is_allowed(dname.c_str(),false,&spec_dir,&id) & IS_ALLOWED_WRITE) {
    if(spec_dir) {
      error_description="Special directory can't be mangled.";
      return 1;
    };
    ApplyLocalCred(user,&id,"write");
    DirectFilePlugin * fp = selectFilePlugin(id);
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=fp->removedir(dname);
      RESET_USER_UID;
      return r;
    };
    return fp->removedir(dname);
  };
  error_description="Not allowed for this job.";
  return 1;
}

int JobPlugin::open(const char* name,open_modes mode,unsigned long long int size) {
  if(!initialized) return 1;
  if(rsl_opened) {  /* unfinished request - cancel */
    olog<<"Request to open file with storing in progress"<<std::endl;
    rsl_opened=false;
    delete_job_id();
    error_description="Job submission is still in progress.";
    return 1; 
  };
  /* check if acl request */
  if((strncmp(name,".gacl-",6) == 0) && (strchr(name,'/') == NULL)) {
    std::string newname(name+6);
    newname="info/"+newname+"/acl";
    return open(newname.c_str(),mode,size);
  };
  if( mode == GRIDFTP_OPEN_RETRIEVE ) {  /* open for reading */
    std::string fname;
    const char* logname;
    /* check if reading status files */
    bool spec_dir;
    if(is_allowed(name,false,&spec_dir,&fname,&logname) & IS_ALLOWED_READ) {
      user->SetControlDir(selectControlDir(fname));
      chosenFilePlugin = selectFilePlugin(fname);
      if(logname) {
        if((*logname) != 0) {
          if(strncmp(logname,"proxy",5) == 0) {
            error_description="Not allowed for this file.";
            chosenFilePlugin = NULL;
            return 1;
          }; 
          fname=user->ControlDir()+"/job."+fname+"."+logname;
          return chosenFilePlugin->open_direct(fname.c_str(),mode);
        };
      };
      if(spec_dir) {
        error_description="Special directory can't be mangled.";
        return 1;
      };
      ApplyLocalCred(user,&fname,"read");
      if((getuid()==0) && (user) && (user->StrictSession())) {
        SET_USER_UID;
        int r=chosenFilePlugin->open(name,mode);
        RESET_USER_UID;
        return r;
      };
      return chosenFilePlugin->open(name,mode);
    };
    error_description="Not allowed for this job.";
    return 1;
  }
  else if( mode == GRIDFTP_OPEN_STORE ) {
    std::string name_f(name);
    std::string::size_type n = name_f.find('/');
    if((n != std::string::npos) && (n != 0)) {
      if(((n==3) && 
          (strncmp(name_f.c_str(),"new",n) == 0)
         ) ||
         ((n==job_id.length()) && 
          (strncmp(name_f.c_str(),job_id.c_str(),n) == 0)
         )) {
        if(name_f.find('/',n+1) != std::string::npos) { // should not contain subdirs
          error_description="Can't create subdirectory here.";
          return 1;
        };
        if(job_id.length() == 0) {
          if(readonly) {
            error_description="You are not allowed to submit new jobs to this service.";
            olog<<error_description<<std::endl;
            return 1;
          };
          if(!make_job_id()) {
            error_description="Failed to allocate ID for job.";
            olog<<error_description<<std::endl;
            return 1;
          };
        };
        olog<<"Accepting submission of new job: "<<job_id<<std::endl;
        memset(job_rsl,0,sizeof(job_rsl));
        rsl_opened=true;
        chosenFilePlugin = selectFilePlugin(job_id);
        return 0;
      };
    };
    std::string id;
    bool spec_dir;
    const char* logname;
    if(is_allowed(name,true,&spec_dir,&id,&logname) & IS_ALLOWED_WRITE) {
      user->SetControlDir(selectControlDir(id));
      chosenFilePlugin = selectFilePlugin(id);
      if(spec_dir) {
        // It is allowed to modify ACL
        if(logname) {
          if(strcmp(logname,"acl") == 0) {
            std::string fname=user->ControlDir()+"/job."+id+"."+logname;
            return chosenFilePlugin->open_direct(fname.c_str(),mode);
          };
        };
        error_description="Special directory can't be mangled.";
        chosenFilePlugin = NULL;
        return 1;
      };
      ApplyLocalCred(user,&id,"write");
      if((getuid()==0) && (user) && (user->StrictSession())) {
        SET_USER_UID;
        int r=chosenFilePlugin->open(name,mode,size);
        RESET_USER_UID;
        return r;
      };
      return chosenFilePlugin->open(name,mode,size);
    };
    error_description="Not allowed for this job.";
    return 1;
  }
  else {
    olog << "ERROR: unknown open mode " << mode << std::endl;
    error_description="Unknown/unsupported request.";
    return 1;
  };
}

int JobPlugin::close(bool eof) {
  if(!initialized || chosenFilePlugin == NULL) return 1;
  if(!rsl_opened) {
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=chosenFilePlugin->close(eof);
      RESET_USER_UID;
      return r;
    };
    return chosenFilePlugin->close(eof);
  };
  rsl_opened=false;
  if(job_id.length() == 0) {
    error_description="There is no job ID defined.";
    return 1;
  };
  if(!eof) { delete_job_id(); return 0; }; /* download was canceled */
  user->SetControlDir(selectControlDir(job_id));
  /* *************************************************
   * store RSL (description)                         *
   ************************************************* */
  std::string rsl_fname=user->ControlDir()+"/job."+job_id+".description";
  std::string acl("");
  {
    int l;
    /* int h=::open(rsl_fname.c_str(),O_WRONLY | O_CREAT | O_EXCL,0600); */
    int h=::open(rsl_fname.c_str(),O_WRONLY,0600);
    fix_file_owner(rsl_fname,*user);
    if(h == -1) { l=-1; }
    else {
      const char* s;
      for(l=0,s=job_rsl;(*s) && (l!=-1);s+=l) l=::write(h,s,strlen(s));
      ::close(h);
    };
    if(l == -1) { 
      olog<<"Failed writing RSL"<<std::endl;
      error_description="Failed to store job RSL description.";
      delete_job_id(); return 1;
    };
  };
  /* analyze rsl (checking, substituting, etc)*/
  JobLocalDescription job_desc;
  if(!parse_job_req(rsl_fname.c_str(),job_desc,&acl)) {
    error_description="Failed to parse job/action description.";
    olog<<error_description<<std::endl;
    delete_job_id();
    return 1;
  };
  if(job_desc.action.length() == 0) job_desc.action="request";
  if(job_desc.action == "cancel") {
    delete_job_id();
    if(job_desc.jobid.length() == 0) {
      error_description="Missing ID in request to cancel job.";
      olog<<error_description<<std::endl;
      return 1;
    };
    return removefile(job_desc.jobid);
  };
  if(job_desc.action == "clean") {
    delete_job_id();
    if(job_desc.jobid.length() == 0) {
      error_description="Missing ID in request to clean job.";
      olog<<error_description<<std::endl;
      return 1;
    };
    return removedir(job_desc.jobid);
  };
  if(job_desc.action == "renew") {
    delete_job_id();
    if(job_desc.jobid.length() == 0) {
      error_description="Missing ID in request to renew credentials.";
      olog<<error_description<<std::endl;
      return 1;
    };
    return checkdir(job_desc.jobid);
  };
  if(job_desc.action == "restart") {
    delete_job_id();
    if(job_desc.jobid.length() == 0) {
      error_description="Missing ID in request to clean job.";
      olog<<error_description<<std::endl;
      return 1;
    };
    const char* logname;
    std::string id;
    if(!(is_allowed(job_desc.jobid.c_str(),false,NULL,&id,&logname) & 
                                                       IS_ALLOWED_LIST)) {
      error_description="Not allowed for this job.";
      olog<<error_description<<std::endl;
      return 1;
    };
    if(job_desc.jobid != id) {
      error_description="Wrong ID specified.";
      olog<<error_description<<std::endl;
      return 1;
    };
    JobLocalDescription job_desc;
    if(!job_local_read_file(id,*user,job_desc)) {
      error_description="Job is probably corrupted: can't read internal information.";
      olog<<error_description<<std::endl;
      return 1;
    };
    if(job_desc.failedstate.length() == 0) {
      error_description="Job can't be restarted.";
      olog<<error_description<<std::endl;
      return 1;
    };
    if(job_desc.reruns <= 0) {
      error_description="Job run out number of allowed retries.";
      olog<<error_description<<std::endl;
      return 1;
    };
    if(!job_restart_mark_put(JobDescription(id,""),*user)) {
      error_description="Failed to report restart request.";
      olog<<error_description<<std::endl;
      return 1;
    };
    return 0;
  };
  if(job_desc.action != "request") {
    olog<<"action("<<job_desc.action<<") != request"<<std::endl;
    error_description="Wrong action in job RSL description.";
    delete_job_id();
    return 1;
  };
  if((job_desc.jobid.length() != 0) && (job_desc.jobid != job_id)) {
    delete_job_id(); 
    if(readonly) {
      remove(rsl_fname.c_str());
      error_description="New jobs are not allowed.";
      olog<<error_description<<std::endl;
      return 1;
    };
    if(!make_job_id(job_desc.jobid)) {
      remove(rsl_fname.c_str());
      error_description="Failed to allocate requested job ID: "+job_desc.jobid;
      olog<<error_description<<std::endl;
      return 1;
    };
    rsl_fname=user->ControlDir()+"/job."+job_id+".description";
    {
      int l;
      int h=::open(rsl_fname.c_str(),O_WRONLY,0600);
      fix_file_owner(rsl_fname,*user);
      if(h == -1) { l=-1; }
      else {
        const char* s;
        for(l=0,s=job_rsl;(*s) && (l!=-1);s+=l) l=::write(h,s,strlen(s));
        ::close(h);
      };
      if(l == -1) {
        olog<<"Failed writing RSL"<<std::endl;
        remove(rsl_fname.c_str());
        error_description="Failed to store job RSL description.";
        delete_job_id(); return 1;
      };
    };
  };
  // Check for proper LRMS name in request. If there is no LRMS name 
  // in user configuration that means service is opaque frontend and
  // accepts any LRMS in request.
  if((!job_desc.lrms.empty()) && (!user->DefaultLRMS().empty())) {
    if(job_desc.lrms != user->DefaultLRMS()) {
      error_description="Request for LRMS "+job_desc.lrms+" is not allowed.";
      olog<<error_description<<std::endl;
      delete_job_id(); 
      return 1;
    };
  };
  if(job_desc.lrms.empty()) job_desc.lrms=user->DefaultLRMS();
  // Check for proper queue in request. 
  if(job_desc.queue.empty()) job_desc.queue=user->DefaultQueue();
  if(job_desc.queue.empty()) {
    error_description="Request has no queue defined.";
    olog<<error_description<<std::endl;
    delete_job_id(); 
    return 1;
  };
  if(avail_queues.size() > 0) { // If no queues configured - service takes any
    for(std::list<std::string>::iterator q = avail_queues.begin();;++q) {
      if(q == avail_queues.end()) {
        error_description="Requested queue "+job_desc.queue+" does not match any of available queues.";
        olog<<error_description<<std::endl;
        delete_job_id(); 
        return 1;
      };
      if(*q == job_desc.queue) break; 
    };
  };
  user->SetSessionRoot(selectSessionDir(job_id));
  if(!preprocess_job_req(rsl_fname,user->SessionRoot()+"/"+job_id,job_id)) {
    error_description="Failed to preprocess job description.";
    olog<<error_description<<std::endl;
    delete_job_id(); 
    return 1;
  };
  /* ****************************************
   * Start local file                       *
   **************************************** */
  /* !!!!! some parameters are unchecked here - rerun,diskspace !!!!! */
  job_desc.jobid=job_id;
  job_desc.starttime=time(NULL);
  job_desc.DN=subject;
  if(port != 0) {
    job_desc.clientname=
       inttostring(host[0])+"."+inttostring(host[1])+"."+
       inttostring(host[2])+"."+inttostring(host[3])+":"+
       inttostring(port);
  };
  /* ***********************************************
   * Try to create proxy                           *
   *********************************************** */
  if(proxy_fname.length() != 0) {
    std::string fname=user->ControlDir()+"/job."+job_id+".proxy";
    int h=::open(fname.c_str(),O_WRONLY | O_CREAT | O_EXCL,0600);
    if(h == -1) {
      error_description="Failed to store credentials.";
      return 1;
    };
    int hh=::open(proxy_fname.c_str(),O_RDONLY);
    if(hh == -1) {
      ::close(h);
      ::remove(fname.c_str());
      error_description="Failed to read credentials.";
      return 1;
    };
    fix_file_owner(fname,*user);
    int l,ll;
    const char* s;
    char buf[256]; 
    for(;;) {
      ll=::read(hh,buf,sizeof(buf));
      if((ll==0) || (ll==-1)) break;
      for(l=0,s=buf;(ll>0) && (l!=-1);s+=l,ll-=l) l=::write(h,s,ll);
      if(l==-1) break;
    };
    ::close(h);
    ::close(hh);
    try {
      Certificate ci(PROXY,fname);
      job_desc.expiretime = ci.Expires().GetTime();
    } catch (std::exception) {
      job_desc.expiretime = time(NULL);
    };
  };
  /* ******************************************
   * Write local file                         *
   ****************************************** */
  JobDescription job(job_id,"",JOB_STATE_ACCEPTED);
  if(!job_local_write_file(job,*user,job_desc)) {
    olog<<"Failed writing local description"<<std::endl;
    delete_job_id(); 
    error_description="Failed to create job description.";
    return 1;
  };
  if(acl.length() != 0) {
    if(!job_acl_write_file(job_id,*user,acl)) {
      olog<<"Failed writing ACL"<<std::endl;
      delete_job_id(); 
      error_description="Failed to process/store job ACL.";
      return 1;
    };
  };
  /* ***********************************************
   * Call authentication/authorization plugin/exec *
   *********************************************** */
  int result;
  std::string response;
  /* talk to external plugin to ask if we can proceed */
  ContinuationPlugins::action_t act =
                 cont_plugins->run(job,*user,result,response);
  // analyze result
  if(act == ContinuationPlugins::act_fail) {
    olog<<"Failed to run external plugin: "<<response<<std::endl;
    delete_job_id();
    error_description="Job is not allowed by external plugin: "+response;
    return 1;
  } else if(act == ContinuationPlugins::act_log) {
    // Scream but go ahead
    olog<<"Failed to run external plugin: "<<response<<std::endl;
  } else if(act == ContinuationPlugins::act_pass) {
    // Just continue
    if(response.length()) olog<<"Plugin response: "<<response<<std::endl;
  } else {
    olog<<"Failed to run external plugin"<<std::endl;
    delete_job_id();
    error_description="Failed to pass external plugin.";
    return 1;
  };
  /* ************************************************************
   * From here code accesses filesystem on behalf of local user *
   ************************************************************ */
  if(cred_plugin && (*cred_plugin)) {
    job_subst_t subst_arg;
    subst_arg.user=user;
    subst_arg.job=&job_id;
    subst_arg.reason="new";
    // run external plugin to acquire non-unix local credentials
    if(!cred_plugin->run(job_subst,&subst_arg)) {
      olog << "Failed to run plugin" << std::endl;
      delete_job_id();
      error_description="Failed to obtain external credentials.";
      return 1;
    };
    if(cred_plugin->result() != 0) {
      olog << "Plugin failed: " << cred_plugin->result() << std::endl;
      delete_job_id();
      error_description="Failed to obtain external credentials.";
      return 1;
    };
  };
  /* *******************************************
   * Create session directory                  *
   ******************************************* */
  std::string dir=user->SessionRoot()+"/"+job_id;
  if((getuid()==0) && (user) && (user->StrictSession())) {
    SET_USER_UID;
  };
  if(mkdir(dir.c_str(),0700) != 0) {
    if((getuid()==0) && (user) && (user->StrictSession())) {
      RESET_USER_UID;
    };
    olog<<"Failed to create session directory"<<std::endl;
    delete_job_id();
    error_description="Failed to create session directory.";
    return 1;
  };
  if((getuid()==0) && (user) && (user->StrictSession())) {
    RESET_USER_UID;
  };
  fix_file_owner(dir,*user);
  /* **********************************************************
   * Create status file (do it last so GM picks job up here)  *
   ********************************************************** */
  if(!job_state_write_file(job,*user,JOB_STATE_ACCEPTED)) {
    olog<<"Failed writing status"<<std::endl;
    delete_job_id(); 
    error_description="Failed registering job in grid-manager.";
    return 1;
  };
  SignalFIFO(*user);
  job_id.resize(0);
  chosenFilePlugin = NULL;
  return 0;
}

int JobPlugin::read(unsigned char *buf,unsigned long long int offset,unsigned long long int *size) {
  if(!initialized || chosenFilePlugin == NULL) {
    error_description="Transfer is not initialised.";
    return 1;
  };
  error_description="Failed to read from disc.";
  if((getuid()==0) && (user) && (user->StrictSession())) {
    SET_USER_UID;
    int r=chosenFilePlugin->read(buf,offset,size);
    RESET_USER_UID;
    return r;
  };
  return chosenFilePlugin->read(buf,offset,size);
}

int JobPlugin::write(unsigned char *buf,unsigned long long int offset,unsigned long long int size) {
  if(!initialized || chosenFilePlugin == NULL) {
    error_description="Transfer is not initialised.";
    return 1;
  };
  error_description="Failed to write to disc.";
  if(!rsl_opened) {
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=chosenFilePlugin->write(buf,offset,size);
      RESET_USER_UID;
      return r;
    };
    return chosenFilePlugin->write(buf,offset,size);
  };
  /* write to rsl */
  if(job_id.length() == 0) {
    error_description="No job ID defined.";
    return 1;
  };
  if((offset >= (sizeof(job_rsl)-1)) ||
     (size >= (sizeof(job_rsl)-1)) ||
     ((offset+size) >= (sizeof(job_rsl)-1)) ) {
    error_description="Job description is too big.";
    return 1;
  };
  memcpy(job_rsl+offset,buf,size);
  return 0;
}

int JobPlugin::readdir(const char* name,std::list<DirEntry> &dir_list,DirEntry::object_info_level mode) {
  if(!initialized) {
    error_description="Plugin is not initialised.";
    return 1;
  };
  if((name[0] == 0) || (strcmp("info",name) == 0)) { /* root jobs directory or jobs' info */
    if(name[0] == 0) {
      DirEntry dent_new(false,"new");
      DirEntry dent_info(false,"info");
      dent_new.may_dirlist=true;
      dent_info.may_dirlist=true;
      dir_list.push_back(dent_new);
      dir_list.push_back(dent_info);
    };
    struct dirent file_;
    struct dirent *file;
    // loop through all control dirs
    for (std::vector<struct gm_dirs_>::iterator i = gm_dirs_info.begin(); i != gm_dirs_info.end(); i++) {
      std::string cdir=(*i).control_dir;
      DIR *dir=opendir(cdir.c_str());
      if(dir != NULL) for(;;) {
        readdir_r(dir,&file_,&file);
        if(file == NULL) break;
        int l=strlen(file->d_name);
        if(l>15) {
          if(!strncmp(file->d_name,"job.",4)) {
            if(!strncmp((file->d_name)+(l-6),".local",6)) { /* can only process those with .local information */
              JobLocalDescription job_desc;
              std::string fname=cdir+'/'+file->d_name;
              if(job_local_read_file(fname,job_desc)) {
                if(job_desc.DN == subject) {
                  JobId id((file->d_name)+4,l-6-4);
                  dir_list.push_back(DirEntry(false,id.c_str()));
                };
              };
            };
          };
        };
      };
      closedir(dir);
    };
    return 0;
  };
  if(strcmp(name,"new") == 0) { /* directory /new is always empty */
    return 0;
  };
  /* check for allowed job directory */
  const char* logname;
  std::string id;
  std::string log;
  if(is_allowed(name,false,NULL,&id,&logname,&log) & IS_ALLOWED_LIST) {
    if(logname) {
      user->SetControlDir(selectControlDir(id));
      if((*logname) != 0) {
        if(strchr(logname,'/') != NULL) return 1; /* no subdirs */
        if(strncmp(logname,"proxy",5) == 0) return 1;
        id=user->ControlDir()+"/job."+id+"."+logname;
        struct stat st;
        if(::stat(id.c_str(),&st) != 0) return 1;
        if(!S_ISREG(st.st_mode)) return 1;
        DirEntry dent(true,logname);
        if(strncmp(logname,"proxy",5) != 0) dent.may_read=true;
        dir_list.push_back(dent);
        return -1;
      };
      DIR* d=::opendir(user->ControlDir().c_str());
      if(d == NULL) { return 1; }; /* maybe return ? */
      struct dirent *de;
      struct dirent de_;
      id="job."+id+".";
      int idl = id.length();
      for(;;) {
        readdir_r(d,&de_,&de);
        if(de == NULL) break;
        if((!strcmp(de->d_name,".")) || (!strcmp(de->d_name,".."))) continue;
        if(strncmp(id.c_str(),de->d_name,idl) != 0) continue;
        DirEntry dent(true,(de->d_name)+idl);
        //if(strncmp((de->d_name)+idl,"proxy",5) != 0) dent.may_read=true;
        if(strncmp((de->d_name)+idl,"proxy",5) == 0) continue;
        dir_list.push_back(dent);
      };
      ::closedir(d);
      return 0;
    };
    if(log.length() > 0) {
      char* s = strchr(name,'/');
      if((s == NULL) || (s[1] == 0)) {
        DirEntry dent(false,log.c_str());
        dent.may_dirlist=true;
        dir_list.push_back(dent);
      };
    };
    /* allowed - pass to file system */
    ApplyLocalCred(user,&id,"read");
    chosenFilePlugin = selectFilePlugin(id);
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=chosenFilePlugin->readdir(name,dir_list,mode);
      RESET_USER_UID;
      return r;
    };
    return chosenFilePlugin->readdir(name,dir_list,mode);
  };
  error_description="Not allowed for this job.";
  return 1;
}

int JobPlugin::checkdir(std::string &dirname) {
  if(!initialized) return 1;
  /* chdir to /new will create new job */
  if(dirname.length() == 0) return 0; /* root */
  if(dirname == "new") { /* new job */
    if(readonly) {
      error_description="New jobs are not allowed.";
      olog<<error_description<<std::endl;
      return 1;
    };
    if(!make_job_id()) {
      error_description="Failed to allocate ID for job.";
      olog<<error_description<<std::endl;
      return 1;
    };
    dirname=job_id;
    return 0;
  };
  if(dirname == "info") { /* always allowed */
    return 0;
  };
  const char* logname;
  std::string id;
  if(is_allowed(dirname.c_str(),false,NULL,&id,&logname) & IS_ALLOWED_LIST) {
    user->SetControlDir(selectControlDir(id));
    if(logname) {
      if((*logname) != 0) {
        error_description="There is no such special subdirectory.";
        return 1; /* log directory has no subdirs */
      };
      return 0;
    };
    if((dirname == id) && (proxy_fname.length())) {  /* cd to session directory - renew proxy request */
      JobLocalDescription job_desc;
      if(!job_local_read_file(id,*user,job_desc)) {
        error_description="Job is probably corrupted: can't read internal information.";
        olog<<error_description<<std::endl;
        return 1;
      };
      /* check if new proxy is better than old one */
      std::string old_proxy_fname=user->ControlDir()+"/job."+id+".proxy";
      time_t new_proxy_expires = time(NULL);
      time_t old_proxy_expires = time(NULL);
      try { 
        Certificate new_ci(PROXY,proxy_fname);
        new_proxy_expires=new_ci.Expires().GetTime();
      } catch (std::exception) { };
      try { 
        Certificate old_ci(PROXY,old_proxy_fname);
        old_proxy_expires=old_ci.Expires().GetTime();
      } catch (std::exception) { };
      if(((int)(new_proxy_expires-old_proxy_expires)) > 0) {
        /* try to renew proxy */
        if(renew_proxy(old_proxy_fname.c_str(),proxy_fname.c_str()) == 0) {
          fix_file_owner(old_proxy_fname,*user);
          olog<<"New proxy expires at "<<mds_time(new_proxy_expires)<<std::endl;
          JobDescription job(id,"",JOB_STATE_ACCEPTED);
          job_desc.expiretime=new_proxy_expires;
          if(!job_local_write_file(job,*user,job_desc)) {
            olog<<"Failed to write 'local' information"<<std::endl;
          };
          error_description="Applying external credentials locally failed.";
          ApplyLocalCred(user,&id,"renew");
          error_description="";
          /* Cause restart of job if it potentially failed 
             because of expired proxy */
          if((((int)(old_proxy_expires-time(NULL))) <= 0) && (
              (job_desc.failedstate == 
                    JobDescription::get_state_name(JOB_STATE_PREPARING)) ||
              (job_desc.failedstate == 
                    JobDescription::get_state_name(JOB_STATE_FINISHING))
             )
            ) {
            olog<<"Job could have died due to expired proxy: restarting"<<std::endl;
            if(!job_restart_mark_put(JobDescription(id,""),*user)) {
              olog<<"Failed to report renewed proxy to job"<<std::endl;
            };
          };
        } else {
          olog<<"Failed to renew proxy"<<std::endl;
        };
      };
    };
    ApplyLocalCred(user,&id,"read");
    chosenFilePlugin = selectFilePlugin(id);
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=chosenFilePlugin->checkdir(dirname);
      RESET_USER_UID;
      return r;
    };
    return chosenFilePlugin->checkdir(dirname);
  };
  error_description="Not allowed for this job.";
  return 1;
}

int JobPlugin::checkfile(std::string &name,DirEntry &info,DirEntry::object_info_level mode) {
  if(!initialized) return 1;
  if(name.length() == 0) {
    info.name=""; info.is_file=false;
    return 0; 
  };
  if((name == "new") || (name == "info")) {
    info.name=""; info.is_file=false;
    return 0; 
  };
  const char* logname;
  std::string id;
  if(is_allowed(name.c_str(),false,NULL,&id,&logname) & IS_ALLOWED_LIST) {
    user->SetControlDir(selectControlDir(id));
    if(logname) {
      if((*logname) == 0) { /* directory itself */
        info.is_file=false; info.name=""; info.may_dirlist=true;
      }
      else {
        if(strncmp(logname,"proxy",5) == 0) {
          error_description="There is no such special file.";
          return 1;
        };
        id=user->ControlDir()+"/job."+id+"."+logname;
        struct stat st;
        if(::stat(id.c_str(),&st) != 0) {
          error_description="There is no such special file.";
          return 1;
        };
        if(!S_ISREG(st.st_mode)) {
          error_description="There is no such special file.";
          return 1;
        };
        info.is_file=true; info.name="";
        info.may_read=true; info.size=st.st_size;
      };
      return 0;
    };
    ApplyLocalCred(user,&id,"read");
    chosenFilePlugin = selectFilePlugin(id);
    if((getuid()==0) && (user) && (user->StrictSession())) {
      SET_USER_UID;
      int r=chosenFilePlugin->checkfile(name,info,mode);
      RESET_USER_UID;
      return r;
    };
    return chosenFilePlugin->checkfile(name,info,mode);
  };
  error_description="Not allowed for this job.";
  return 1;
}

bool JobPlugin::delete_job_id(void) {
  if(job_id.length() != 0) {
    user->SetSessionRoot(selectSessionDir(job_id));
    user->SetControlDir(selectControlDir(job_id));
    job_clean_final(JobDescription(job_id,user->SessionRoot()+"/"+job_id),*user);
    job_id="";
  };
  return true;
}

bool JobPlugin::make_job_id(const std::string &id) {
  if((id.find('/') != std::string::npos) || (id.find('\n') != std::string::npos)) {
    olog<<"ID contains forbidden characters"<<std::endl;
    return false;
  };
  if((id == "new") || (id == "info")) return false;
  // claim id by creating empty description file
  user->SetControlDir(selectControlDir(id));
  std::string fname=user->ControlDir()+"/job."+id+".description";
  struct stat st;
  if(stat(fname.c_str(),&st) == 0) return false;
  int h = ::open(fname.c_str(),O_RDWR | O_CREAT | O_EXCL,S_IRWXU);
  // So far assume control directory is on local fs.
  // TODO: add locks or links for NFS
  if(h == -1) return false;
  fix_file_owner(fname,*user);
  close(h);
  delete_job_id();
  job_id=id;
  return true;
}

bool JobPlugin::make_job_id(void) {
  int i;
  bool found = false;
  delete_job_id();
  for(i=0;i<100;i++) {
    job_id=inttostring((unsigned int)getpid())+
           inttostring((unsigned int)time(NULL))+
           inttostring(rand(),1);
    // loop through all control dirs to find if job_id exists
    for (std::vector<gm_dirs_>::iterator i = gm_dirs_info.begin(); i != gm_dirs_info.end(); i++) {
      std::string fname=(*i).control_dir+"/job."+job_id+".description";
      struct stat st;
      if(stat(fname.c_str(),&st) == 0) {
        found = true;
        break;
      }
    }
    if (found) {
      found = false;
      continue;
    }
    user->SetControlDir(selectControlDir(job_id));
    std::string fname=user->ControlDir()+"/job."+job_id+".description";
    int h = ::open(fname.c_str(),O_RDWR | O_CREAT | O_EXCL,0600);
    // So far assume control directory is on local fs.
    // TODO: add locks or links for NFS
    if(h == -1) {
      if(errno == EEXIST) continue;
      olog << "Failed to create file in " << user->ControlDir()<< std::endl;
      return false;
    };
    fix_file_owner(fname,*user);
    close(h);
    break;
  };
  if(i>=100) {
    olog << "Out of tries while allocating new job id" << std::endl;
    job_id=""; return false;
  };
  return true;
}

/*
  name - name of file to access
  locked - true if job already running
  jobid - returns id extracted from name
  logname - name of log file (errors, status, etc.)
  log - stdlog of job
  spec_dir - if file belogs to virtual directory 
  returns access rights. For special files superset of all  rights is
  returned. Distinction between files is processed at higher levels.
*/
int JobPlugin::is_allowed(const char* name,bool locked,bool* spec_dir,std::string* jobid,char const ** logname,std::string* log) {
  if(logname) (*logname) = NULL;
  if(log) (*log)="";
  if(spec_dir) (*spec_dir)=false;
  JobId id(name);
  if(id == "info") { // directory which contains list of jobs-directories
    if(spec_dir) (*spec_dir)=false;
    return (IS_ALLOWED_READ | IS_ALLOWED_LIST);
  };
  if(strncmp(id.c_str(),"info/",5) == 0) {
    if(spec_dir) (*spec_dir)=true;
    name+=5; id=name;
    std::string::size_type n=id.find('/'); if(n != std::string::npos) id.erase(n);
    if(jobid) (*jobid)=id;
    if(id.length() == 0) return 0;
    const char* l_name = name+id.length();
    if(l_name[0] == '/') l_name++;
    if(logname) { (*logname)=l_name; };
    JobLocalDescription job_desc;
    user->SetControlDir(selectControlDir(id));
    if(!job_local_read_file(id,*user,job_desc)) return false;
    if(job_desc.DN != subject) {
      // Not an owner. Check acl.
      std::string acl_file = user->ControlDir()+"/job."+id+".acl";
      struct stat st;
      if(stat(acl_file.c_str(),&st) == 0) {
        if(S_ISREG(st.st_mode)) {
          GACLacl* acl = GACLloadAcl((char*)(acl_file.c_str()));
          if(acl) {
            GACLperm perm = AuthUserGACLTest(acl,user_a);
            int res = 0;
            if(GACLhasList(perm))
              res|=IS_ALLOWED_LIST;
            if(GACLhasRead(perm) | GACLhasWrite(perm))
              res|=(IS_ALLOWED_READ | IS_ALLOWED_LIST);
            if(GACLhasAdmin(perm))
              res|=(IS_ALLOWED_READ | IS_ALLOWED_WRITE | IS_ALLOWED_LIST);
            //if(strncmp(l_name,"proxy",5) == 0) res&=IS_ALLOWED_LIST;
            //if(strncmp(l_name,"acl",3) != 0) res&=~IS_ALLOWED_WRITE;
            return res;
          };
        };
      };
      return 0;
    };
    //if(strncmp(l_name,"proxy",5) == 0) return (IS_ALLOWED_LIST);
    //if(strncmp(l_name,"acl",3) != 0) return (IS_ALLOWED_READ | IS_ALLOWED_LIST);;
    return (IS_ALLOWED_READ | IS_ALLOWED_WRITE | IS_ALLOWED_LIST);
  };
  std::string::size_type n=id.find('/'); if(n != std::string::npos) id.erase(n);
  if(jobid) (*jobid)=id;
  JobLocalDescription job_desc;
  user->SetControlDir(selectControlDir(id));
  if(job_local_read_file(id,*user,job_desc)) {
    int res = 0;
    bool spec = false;
    //bool proxy = false;
    //bool acl = false;
    if(log) (*log)=job_desc.stdlog;
    if(n != std::string::npos) {
      int l = job_desc.stdlog.length();
      if(l != 0) {
        if(strncmp(name+n+1,job_desc.stdlog.c_str(),l) == 0) {
          if(name[n+1+l] == 0) {
            if(spec_dir) (*spec_dir)=true;
            if(logname) (*logname)=name+n+1+l;
            spec=true;
          } else if(name[n+1+l] == '/') {
            if(spec_dir) (*spec_dir)=true;
            if(logname) (*logname)=name+n+1+l+1;
            spec=true;
            //if(strncmp(name+n+1+l+1,"proxy",5) == 0) proxy=true;
            //if(strncmp(name+n+1+l+1,"acl",3) == 0) acl=true;
          };
        };
      };
    };
    if(job_desc.DN == subject) {
      res|=(IS_ALLOWED_READ | IS_ALLOWED_WRITE | IS_ALLOWED_LIST);
    } else {
      // Not an owner. Check acl.
      std::string acl_file = user->ControlDir()+"/job."+id+".acl";
      struct stat st;
      if(stat(acl_file.c_str(),&st) == 0) {
        if(S_ISREG(st.st_mode)) {
          GACLacl* acl = GACLloadAcl((char*)(acl_file.c_str()));
          if(acl) {
            GACLperm perm = AuthUserGACLTest(acl,user_a);
            if(spec) {
              if(GACLhasList(perm))
                res|=IS_ALLOWED_LIST;
              if(GACLhasRead(perm) | GACLhasWrite(perm))
                res|=(IS_ALLOWED_READ | IS_ALLOWED_LIST);
              if(GACLhasAdmin(perm))
                res|=(IS_ALLOWED_READ | IS_ALLOWED_WRITE | IS_ALLOWED_LIST);
            } else {
              if(GACLhasList(perm)) res|=IS_ALLOWED_LIST;
              if(GACLhasRead(perm)) res|=IS_ALLOWED_READ;
              if(GACLhasWrite(perm)) res|=IS_ALLOWED_WRITE;
              if(GACLhasAdmin(perm))
                res|=(IS_ALLOWED_READ | IS_ALLOWED_WRITE | IS_ALLOWED_LIST);
            };
          } else {
            olog << "Failed to read job's ACL for job " << id <<
                    " from " << user->ControlDir() << std::endl;
          };
        };
      };
    };
    if(spec) return res;
    if(res) {
      if(!locked) return res;
      job_state_t status=job_state_read_file(id,*user);
      if((status != JOB_STATE_ACCEPTED) && (status != JOB_STATE_PREPARING)) {
        if(!job_desc.fullaccess) res&=(~IS_ALLOWED_WRITE);
      };
    };
    return res; 
  } else {
    olog << "Failed to read job's local description for job " << id <<
            " from " << user->ControlDir() << std::endl;
  };
  return 0;
}

/**
 * algorithm for picking control dir and session dir based on job id
 */
int JobPlugin::selectDirFromID(std::string id, int dirs) {
  if (dirs < 2) return 0;
  int id_int;
  // take only the last 4 digits from job_id
  std::string short_job_id = id.length() < 4 ? id : id.substr(id.length() - 4); 
  if (!stringtoint(short_job_id, id_int)) return 0;
  return id_int % dirs;
}

DirectFilePlugin * JobPlugin::selectFilePlugin(std::string id) {
  int index = 0;
  if (session_dirs.size() > 1) index = selectDirFromID(id, session_dirs.size());
  else index = selectDirFromID(id, gm_dirs_info.size());
  return file_plugins.at(index);
}

std::string JobPlugin::selectControlDir(std::string id) {
  // the 'main' control dir is last in the list
  if (session_dirs.size() > 2) return gm_dirs_info.at(gm_dirs_info.size()-1).control_dir;
  int index = selectDirFromID(id, gm_dirs_info.size());
  return gm_dirs_info.at(index).control_dir;
}

std::string JobPlugin::selectSessionDir(std::string id) {
  // if multiple session dirs are defined, don't use remote dirs
  if (session_dirs.size() > 1) {
    int index = selectDirFromID(id, session_dirs.size());
    olog << "Using session dir " << session_dirs.at(index) <<std::endl;
    return session_dirs.at(index);
  }
  int index = selectDirFromID(id, gm_dirs_info.size());
  return gm_dirs_info.at(index).session_dir;
}
