#include "../../std.h"

#include "../../misc/condition.h"
#include "../../misc/url_options.h"
#include <arc/globuserrorutils.h>
#include "../../misc/proxy.h"
#include "../../misc/log_time.h"
#include <arc/stringconv.h>


#include "../../misc/inttostring.h"
// #include "../../auth/auth.h"

#include "client.h"


HTTP_Client::HTTP_Client(const char* base,bool heavy_encryption,bool gssapi_server, int timeout, bool check_host_cert) try: base_url(base), timeout(timeout*1000){
  c=NULL;
  cred=GSS_C_NO_CREDENTIAL;
  valid=false; connected=false;
  /* initialize globus io connection related objects */
  if(strcasecmp(base_url.Protocol().c_str(),"http") == 0) {
    const char * proxy = getenv("ARC_HTTP_PROXY");
    if(!proxy) proxy = getenv("NORDUGRID_HTTP_PROXY");
    if(proxy) {
      proxy_hostname = proxy;
      proxy_port = 8000;
      std::string::size_type n = proxy_hostname.find(':');
      if(n != std::string::npos) {
        proxy_port=atoi(proxy_hostname.c_str()+n+1);
        proxy_hostname.resize(n);
      };
    };
  };
  if(proxy_hostname.empty()) {
    if(!gssapi_server) {
      c=new HTTP_Client_Connector_Globus(base,heavy_encryption);
    } else {
      c=new HTTP_Client_Connector_GSSAPI(base,heavy_encryption, timeout*1000, cred, check_host_cert);
    };
  } else {
    std::string u = "http://"+proxy_hostname+":"+tostring<int>(proxy_port);
    if(!gssapi_server) {
      c=new HTTP_Client_Connector_Globus(u.c_str(),heavy_encryption);
    } else {
      c=new HTTP_Client_Connector_GSSAPI(u.c_str(),heavy_encryption, timeout*1000, cred, check_host_cert);
    };
  };
  valid=true;
} catch(std::exception e) {
  valid=false; connected=false;
}

bool HTTP_Client::credentials(const char* filename) {
  if(!filename) return false;
  gss_cred_id_t cred_new = read_proxy(filename);
  if(cred_new == GSS_C_NO_CREDENTIAL) return false;
  if(!c->credentials(cred_new)) {
    free_proxy(cred_new);
    return false;
  };
  if(cred != GSS_C_NO_CREDENTIAL) {
    free_proxy(cred);  
  };
  cred=cred_new;
  return true;
}

int HTTP_Client::connect(void) {
  if(connected) return 0;
  if(!valid) return -1;
  if(!c->connect()) return -1;
  connected=true;
  return 0;
}

int HTTP_Client::disconnect(void) {
  if(!connected) return 0;
  c->disconnect();
  connected=false;
  return 0;
}

HTTP_Client::~HTTP_Client(void) {
  if(!valid) return;
  disconnect();
  if(c) delete c;
  if(cred != GSS_C_NO_CREDENTIAL) {
    free_proxy(cred);
  };
}

void HTTP_Client::clear_input(void) {
  if(!valid) return;
  char buf[256];
  unsigned int l;
  bool isread,iswritten;
  for(;;) {
    l=sizeof(buf);
    if(!c->read(buf,&l)) return;
    if(!c->transfer(isread,iswritten,0)) { c->read(); return; };
    if(!isread) { c->read(); return; };
    odlog(VERBOSE)<<"clear_input: ";
    for(int n=0;n<l;n++) odlog_(VERBOSE)<<buf[n];
    odlog_(VERBOSE)<<std::endl;
  };
}

int HTTP_Client::analyze_response_line(char* line) {
  for(;*line;line++) if(!isspace(*line)) break;
  int len = strlen(line);
  if(len<2) return -1; // too short
  if(answer_count==0) {  // first line
    // odlog(DEBUG)<<"analyze_response_line: first line: "<<line<<std::endl;
    bool answer_keep_alive = true;
    answer_code=0;
    char* p = line;
    char* http_version = p;
    for(;*p;p++) if(isspace(*p)) break; (*p)=0; p++;
    for(;*p;p++) if(!isspace(*p)) break;
    char* code = p;
    for(;*p;p++) if(isspace(*p)) break; (*p)=0; p++;
    for(;*p;p++) if(!isspace(*p)) break;
    char* reason = p;
    char *e;
    answer_code=strtoul(code,&e,10);
    if((*e) != 0) return -1;
    answer_reason=reason;
    answer_count++;
    if(strcmp(http_version,"HTTP/1.1") == 0) { answer_keep_alive=true; }
    else { answer_keep_alive=false; };
    fields.reset(answer_keep_alive);
    // odlog(DEBUG)<<"analyze_response_line: answer_code:"<<answer_code<<std::endl;
    // odlog(DEBUG)<<"analyze_response_line: http_version:"<<http_version<<std::endl;
    // odlog(DEBUG)<<"analyze_response_line: reason:"<<reason<<std::endl;
  } else  {
    // odlog(DEBUG)<<"analyze_response_line: line: "<<line<<std::endl;
    char* token = line; for(;*token;token++) if(isspace(*token)) break;
    int l = token-line;
    if(*token) {
      (*token)=0; token++;
      for(;*token;token++) if(!isspace(*token)) break;
    };
    // odlog(DEBUG)<<"analyze_response_line: name: "<<line<<std::endl;
    // odlog(DEBUG)<<"analyze_response_line: value: "<<token<<std::endl;
    fields.set(line,token);
  };
}

// read server response
int HTTP_Client::read_response_header(void) {
  int r;
  bool isread,iswritten;
  answer_count=0; // line counter
  // let external procedures manage timeouts
  if(!c->transfer(isread,iswritten,-1)) {
    disconnect();
    return -1;
  };
  if(answer_size != 0) isread=true;
  if((!isread) && (!iswritten)) {
    disconnect();
    return -1;
  };
  char line_buf[256];
  int line_p = 0;
  for(;;) {
    answer_buf[answer_size]=0;
    char* p = strchr(answer_buf,'\n');
    int l = answer_size;
    if(p) l=(p-answer_buf)+1; // available data
    int ll = (sizeof(line_buf)-1-line_p);
    if(ll>l) ll=l; // available space
    memcpy(line_buf+line_p,answer_buf,ll); line_p+=ll; line_buf[line_p]=0;
    if(l<answer_size) { memmove(answer_buf,answer_buf+l,answer_size-l); };
    answer_size-=l;
    if(p) {  // eol 
      for(;line_p;) {
        if((line_buf[line_p-1] != '\r') && (line_buf[line_p-1] != '\n')) break;
        line_p--;
      };
      line_buf[line_p]=0;
      if(line_p == 0) break; // end of header
      odlog(VERBOSE)<<"read_response_header: line: "<<line_buf<<std::endl;
      analyze_response_line(line_buf);
      line_p=0;
    };
    if(answer_size>0) continue;
    // request new portion
    answer_size=sizeof(answer_buf)-1; 
    if(isread && (!c->read(answer_buf,&answer_size))) {
      disconnect();
      return -1;
    };
    if(!c->transfer(isread,iswritten,timeout)) {
      olog<<"Timeout while reading response header"<<std::endl;
      disconnect();
      return -1;
    };
    if(!isread) {
      olog<<"Error while reading response header"<<std::endl;
      disconnect();
      return -1;
    };
  };
  odlog(VERBOSE)<<"read_response_header: header finished"<<std::endl;
  return 0;
}

int HTTP_Client::skip_response_entity(void) {
  odlog(VERBOSE)<<"skip_response_entity"<<std::endl;
  if(fields.haveContentLength() || fields.haveContentRange()) {
    unsigned long long int size = fields.ContentLength();
    odlog(VERBOSE)<<"skip_response_entity: size: "<<size<<std::endl;
    if(size<=answer_size) {
      memmove(answer_buf,answer_buf+size,answer_size-size);
      answer_size-=size;
      odlog(VERBOSE)<<"skip_response_entity: already have all"<<std::endl;
      return 0;
    };
    size-=answer_size;
    odlog(VERBOSE)<<"skip_response_entity: size left: "<<size<<std::endl;
    // read size bytes
    char buf[1024];
    for(;size;) {
      odlog(VERBOSE)<<"skip_response_entity:  to read: "<<size<<std::endl;
      answer_size=sizeof(buf);
      if(!c->read(buf,&answer_size)) {
        disconnect(); return -1;
      };
      bool isread,iswritten;
      if(!c->transfer(isread,iswritten,timeout)) {
        odlog(VERBOSE)<<"skip_response_entity: timeout"<<size<<std::endl;
        disconnect(); return -1; // timeout
      };
      if(!isread) { disconnect(); return -1; }; // failure
      size-=answer_size;
      odlog(VERBOSE)<<"skip_response_entity: read: "<<answer_size<<" ("<<size<<")"<<std::endl;
    };
    odlog(VERBOSE)<<"skip_response_entity: read all"<<std::endl;
    return 0;
  };
  if(fields.KeepAlive()) {
    odlog(VERBOSE)<<"skip_response_entity: no entity"<<std::endl;
    // no entity passed at all
    return 0;
  };
  odlog(VERBOSE)<<"skip_response_entity: unknown size"<<std::endl;
  // can't handle unknow size - let connection be closed
  return 0;
}

//int HTTP_Client::put(const char* path,
//    unsigned long long int offset,unsigned long long int size,
//    int fd,unsigned long long int fd_offset,unsigned long long int fd_size) {

/*
  HTTP PUT method
  Content of 'buf' with size 'size' is sent to 'path'. 
  Treated as part size 'fd_size' starting from 'offset'.
*/
int HTTP_Client::make_header(const char* path,
      unsigned long long int offset,unsigned long long int size,
      unsigned long long int fd_size,std::string& header) {
  // create header
  if(!valid) return -1;
  if(path[0] == '/') path++;
  header = "PUT ";
  std::string url_path;
  if(proxy_hostname.length() == 0) {
    url_path=base_url.Path();
  } else {
    url_path=base_url.Protocol()+"://"+base_url.Host()+":"+inttostring(base_url.Port())+base_url.Path();
  };
  if(path[0]) {
    if(url_path[url_path.length()-1] != '/') url_path+="/";
    url_path+=path;
  };
  if(!base_url.HTTPOptions().empty()) {
    url_path+='?'+URL::OptionString(base_url.HTTPOptions());
  };
  std::string url_host = base_url.Host()+":"+inttostring(base_url.Port());
  header+=url_path; header+=" HTTP/1.1\r\n";
  header+="Host: "+url_host+"\r\n";
  header+="Connection: keep-alive\r\n";
  header+="Content-Length: "+inttostring(size)+"\r\n";
  header+="Content-Range: bytes "+inttostring(offset)+"-"+inttostring(offset+size-1);
  if(fd_size>=size) {
    header+="/"+inttostring(fd_size);
  };
  header+="\r\n";
  header+="\r\n";
  return 0;
}

/*
  HTTP PUT method
*/
int HTTP_Client::PUT(const char* path,
      unsigned long long int offset,unsigned long long int size,
      const unsigned char* buf,unsigned long long int fd_size,bool wait) {
  if(!connected) {
    olog<<"Not connected"<<std::endl;
    return -1;
  };
  // send header
  // create header
  std::string header;
  make_header(path,offset,size,fd_size,header);
  int r;
  c->clear();
  answer_size=sizeof(answer_buf)-1;
  if(!c->read(answer_buf,&answer_size)) {
    disconnect(); return -1;
  };
  // send header
  if(!c->write(header.c_str(),header.length())) {
    disconnect(); return -1;
  };
  bool isread,iswritten;
  if(!c->transfer(isread,iswritten,timeout)) {
    olog<<"Timeout sending header"<<std::endl;
    disconnect(); return -1;
  };
  if(!iswritten) { // Server responded too early
    olog<<"Early response from server"<<std::endl;
    disconnect(); return -1;
  };
  // send body
  if(!c->write((const char*)buf,size)) {
    disconnect(); return -1;
  };
  // read server response
  if(read_response_header() != 0) {
    olog<<"No response from server received"<<std::endl;
    disconnect(); return -1;
  };
  if(!c->eofwrite()) {
    olog<<"Failed to send body"<<std::endl;
    disconnect(); return -1;
  };
  if(fields.KeepAlive()) {  // skip entity only if trying to keep connection
    if(skip_response_entity() != 0) {
      olog<<"Failure while receiving entity"<<std::endl;
      disconnect();
      return -1;
    };
    c->read(); // just in case
  } else {
    disconnect();
  };
  if(answer_code != HTTP_OK) return -1;
  return 0;
}

/*
  HTTP GET method
*/
int HTTP_Client::GET_header(const char* path,
      unsigned long long int offset,unsigned long long int size) {
  // create header
  if(!valid) return -1;
  // *** Generate header
  if(path[0] == '/') path++;
  std::string header = "GET ";
  std::string url_path;
  if(proxy_hostname.length() == 0) {
    url_path=base_url.Path();
  } else {
    url_path=base_url.Protocol()+"://"+base_url.Host()+":"+inttostring(base_url.Port())+base_url.Path();
  };
  if(path[0]) {
    if(url_path[url_path.length()-1] != '/') url_path+="/"; 
    url_path+=path;
  };
  if(!base_url.HTTPOptions().empty()) {
    url_path+='?'+URL::OptionString(base_url.HTTPOptions());
  };
  std::string url_host = base_url.Host()+":"+inttostring(base_url.Port());
  header+=url_path; header+=" HTTP/1.1\r\n";
  header+="Host: "+url_host+"\r\n";
  header+="Connection: keep-alive\r\n";
  header+="Range: bytes="+inttostring(offset)+"-"+inttostring(offset+size-1)+"\r\n";
  header+="\r\n";
  odlog(VERBOSE)<<"header: "<<header<<std::endl;
  // *** Send header 
  c->clear();
  // get ready for any answer
  answer_size=sizeof(answer_buf);
  if(!c->read(answer_buf,&answer_size)) {
    disconnect();
    return -1;
  };
  // send
  if(!c->write(header.c_str(),header.length())) {
    disconnect();
    return -1;
  };
  bool isread, iswritten;
  for(;;) {
    if(!c->transfer(isread,iswritten,timeout)) {
      olog<<"Timeout while sending header"<<std::endl;
      disconnect();
      return -1;
    };
    if(iswritten) break;
    if(isread) continue;
    olog<<"Failed to send header"<<std::endl;
    disconnect();
    return -1;
  };
  return 0;
}

int HTTP_Client::GET(const char* path,
      unsigned long long int offset,unsigned long long int size,
      get_callback_t callback,void* arg,
      unsigned char* buf,unsigned long long int bufsize) {
  if(!connected) {
    olog<<"Not connected"<<std::endl;
    return -1;
  };
  // send header
  if(GET_header(path,offset,size)) {
    // There are servers which close connection even if they shouldn't - retry here
    if(connect()) return -1;
    if(GET_header(path,offset,size)) return -1;
  };
  // read server response
  if(read_response_header() != 0) {
    olog<<"No response from server received"<<std::endl;
    disconnect();
    return -1;
  };
  if(answer_code == 416) { // out of range
    if(skip_response_entity() != 0) {
      disconnect(); return -1;
    };
    if(!fields.KeepAlive()) {
      odlog(DEBUG)<<"GET: connection to be closed"<<std::endl;
      disconnect();
    };
    return 0;
  };
  if((answer_code != 200) && (answer_code != 206)) {
    if(skip_response_entity() != 0) {
      disconnect(); return -1;
    };
    if(!fields.KeepAlive()) {
    odlog(DEBUG)<<"GET: connection to be closed"<<std::endl;
      disconnect();
    };
    return -1;
  };
  odlog(DEBUG)<<"GET: header is read - rest: "<<answer_size<<std::endl;
  unsigned long long c_offset = 0;
  if(fields.haveContentRange()) c_offset=fields.ContentStart();
  bool have_length = fields.haveContentLength() || fields.haveContentRange();
  unsigned long long length = fields.ContentLength();
  // take rest of already read data
  if(answer_size) {
    if(have_length) if(answer_size > length) answer_size=length;
    odlog(VERBOSE)<<"GET: calling callback(rest): content: "<<answer_buf<<std::endl;
    odlog(VERBOSE)<<"GET: calling callback(rest): size: "<<answer_size<<std::endl;
    odlog(VERBOSE)<<"GET: calling callback(rest): offset: "<<c_offset<<std::endl;
    unsigned char* in_buf = (unsigned char*)answer_buf;
    unsigned long  in_size = answer_size;
    for(;;) {
      if(in_size == 0) break;
      if(buf) {
        unsigned long l = in_size;
        if(l>bufsize) l=bufsize;
        memcpy(buf,in_buf,l);
        if(callback(c_offset,l,&buf,&bufsize,arg) != 0) {
          olog<<"GET callback returned error"<<std::endl;
          disconnect(); // ?????????????????
          return -1;
        };
        in_buf+=l; c_offset+=l; in_size-=l;
      } else {
        unsigned char* in_buf_ = in_buf;
        if(callback(c_offset,in_size,&in_buf_,&bufsize,arg) != 0) {
          olog<<"GET callback returned error"<<std::endl;
          disconnect(); // ?????????????????
          return -1;
        };
        if(in_buf_ != in_buf) buf=in_buf_;
        in_buf+=in_size; c_offset+=in_size; in_size=0;
      };
    };
    if(have_length) length-=answer_size;
  };
  unsigned char* in_buf = NULL;
  for(;;) {
    if(have_length) if(length == 0) break; 
    if(buf == NULL) {
      if(in_buf == NULL) in_buf=(unsigned char*)malloc(65536);
      if(in_buf == NULL) {
        olog<<"Failed to allocate memory"<<std::endl;
        disconnect();
        return -1;
      };
      buf=in_buf; bufsize=65536;
    };
    /// cond_read.reset();
    answer_size=bufsize;
    if(!c->read((char*)buf,&answer_size)) {
      olog<<"Failed while reading response content"<<std::endl;
      disconnect();
      if(in_buf) free(in_buf);
      return -1;
    };
    bool isread,iswritten;
    if(!c->transfer(isread,iswritten,timeout)) {
      olog<<"Timeout while reading response content: "<<std::endl;
      disconnect();
      if(in_buf) free(in_buf);
      return -1; // timeout
    };
    if(!isread) { // failure
      if(c->eofread()) { // eof - ok if length not specified
        if(!have_length) { disconnect(); break; };
      };
      olog<<"Error while reading response content: "<<std::endl;
      disconnect();
      if(in_buf) free(in_buf);
      return -1;
    };
    odlog(VERBOSE)<<"GET: calling callback: content: "<<buf<<std::endl;
    odlog(VERBOSE)<<"GET: calling callback: size: "<<answer_size<<std::endl;
    odlog(VERBOSE)<<"GET: calling callback: offset: "<<c_offset<<std::endl;
    if(callback(c_offset,answer_size,&buf,&bufsize,arg) != 0) {
      olog<<"GET callback returned error"<<std::endl;
      disconnect();
      if(in_buf) free(in_buf);
      return -1;
    };
    c_offset+=answer_size;
    if(have_length) length-=answer_size;
  };
  // cancel? - just in case
  if(in_buf) free(in_buf);
  if(!fields.KeepAlive()) {
    odlog(DEBUG)<<"GET: connection to be closed"<<std::endl;
    disconnect();
  };
  return 0;
}

// -------------------------------------------------

bool HTTP_Client_Connector::connect(void) { return false; }

bool HTTP_Client_Connector::disconnect(void)  { return false; }

bool HTTP_Client_Connector::clear(void) { return false; }

bool HTTP_Client_Connector::read(char* buf,unsigned int* size) { return false; }

bool HTTP_Client_Connector::write(const char* buf,unsigned int size) { return false; }

bool HTTP_Client_Connector::transfer(bool& read,bool& write,int timeout) { return false; }

bool HTTP_Client_Connector::eofread(void) { return false; }

bool HTTP_Client_Connector::eofwrite(void) { return false; }

HTTP_Client_Connector::HTTP_Client_Connector(void) { }

HTTP_Client_Connector::~HTTP_Client_Connector(void) { }

bool HTTP_Client_Connector::credentials(gss_cred_id_t cred) { return false; }



