#include "../../std.h"

#include <sys/timeb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <globus_io.h>

#include "../../misc/condition.h"
#include "../../misc/url_options.h"
#include <arc/globuserrorutils.h>
#include "../../misc/proxy.h"
#include "../../misc/log_time.h"
//#include "../../misc/inttostring.h"
// #include "../../auth/auth.h"

#include "client.h"

// ------------------ Globus -------------------------------

GlobusModuleIO HTTP_Client_Connector_Globus::mod_io;

HTTP_Client_Connector_Globus::HTTP_Client_Connector_Globus(const char* base,bool heavy_encryption,int timeout_,gss_cred_id_t cred_) try: base_url(base) {
  valid=false; connected=false;
  read_registered=false;
  write_registered=false;
  read_size=NULL;
  cred=cred_;
  timeout=timeout_; // 1 min.
  /* initialize globus io connection related objects */
  globus_io_tcpattr_init(&attr);
  globus_io_secure_authorization_data_initialize(&auth);
  globus_io_secure_authorization_data_set_callback(&auth,&authorization_callback,NULL);
  if(strcasecmp(base_url.Protocol().c_str(),"http") == 0) {
    globus_io_attr_set_secure_authentication_mode(&attr,
              GLOBUS_IO_SECURE_AUTHENTICATION_MODE_NONE,GSS_C_NO_CREDENTIAL);
    globus_io_attr_set_secure_authorization_mode(&attr,
              GLOBUS_IO_SECURE_AUTHORIZATION_MODE_NONE,GLOBUS_NULL);
    globus_io_attr_set_secure_channel_mode(&attr,
                                 GLOBUS_IO_SECURE_CHANNEL_MODE_CLEAR);
    globus_io_attr_set_secure_protection_mode(&attr,
                                 GLOBUS_IO_SECURE_PROTECTION_MODE_NONE);
    globus_io_attr_set_secure_delegation_mode(&attr,
                                 GLOBUS_IO_SECURE_DELEGATION_MODE_NONE);
  } else if(strcasecmp(base_url.Protocol().c_str(),"https") == 0) {
    globus_io_attr_set_secure_authentication_mode(&attr,
              GLOBUS_IO_SECURE_AUTHENTICATION_MODE_MUTUAL,cred);
    globus_io_attr_set_secure_authorization_mode(&attr,
              GLOBUS_IO_SECURE_AUTHORIZATION_MODE_HOST,GLOBUS_NULL);
    globus_io_attr_set_secure_channel_mode(&attr,
                                 GLOBUS_IO_SECURE_CHANNEL_MODE_SSL_WRAP);
    if(heavy_encryption) {
      globus_io_attr_set_secure_protection_mode(&attr,
                                 GLOBUS_IO_SECURE_PROTECTION_MODE_PRIVATE);
    } else {
      globus_io_attr_set_secure_protection_mode(&attr,
                                 GLOBUS_IO_SECURE_PROTECTION_MODE_SAFE);
    };
    globus_io_attr_set_secure_delegation_mode(&attr,
                                 GLOBUS_IO_SECURE_DELEGATION_MODE_NONE);
  } else if(strcasecmp(base_url.Protocol().c_str(),"httpg") == 0) {
    globus_io_attr_set_secure_authentication_mode(&attr,
              GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,cred);
    globus_io_attr_set_secure_authorization_mode(&attr,
              GLOBUS_IO_SECURE_AUTHORIZATION_MODE_HOST,GLOBUS_NULL);
    globus_io_attr_set_secure_channel_mode(&attr,
                                 GLOBUS_IO_SECURE_CHANNEL_MODE_GSI_WRAP);
    if(heavy_encryption) {
      globus_io_attr_set_secure_protection_mode(&attr,
                                 GLOBUS_IO_SECURE_PROTECTION_MODE_PRIVATE);
    } else {
      globus_io_attr_set_secure_protection_mode(&attr,
                                 GLOBUS_IO_SECURE_PROTECTION_MODE_SAFE);
    };
    globus_io_attr_set_secure_delegation_mode(&attr,
                                 GLOBUS_IO_SECURE_DELEGATION_MODE_FULL_PROXY);
  } else {
    return;
  };
  globus_io_attr_set_secure_proxy_mode(&attr,GLOBUS_IO_SECURE_PROXY_MODE_NONE);
  valid=true;
} catch(std::exception e) {
  timeout=timeout_; // 1 min.
  valid=false; connected=false;
  /* initialize globus io connection related objects */
  globus_io_tcpattr_init(&attr);
  globus_io_secure_authorization_data_initialize(&auth);
  globus_io_secure_authorization_data_set_callback(&auth,&authorization_callback,NULL);
}

HTTP_Client_Connector_Globus::~HTTP_Client_Connector_Globus(void) {
  disconnect();
  globus_io_secure_authorization_data_destroy(&auth);
  globus_io_tcpattr_destroy(&attr);
}

bool HTTP_Client_Connector_Globus::credentials(gss_cred_id_t cred_) {
  if(cred_ == GSS_C_NO_CREDENTIAL) return false;
  gss_cred_id_t cred_old;
  globus_io_secure_authentication_mode_t mode;
  if(globus_io_attr_get_secure_authentication_mode(&attr,&mode,&cred_old) !=
                                       GLOBUS_SUCCESS) return false;
  if(globus_io_attr_set_secure_authentication_mode(&attr,mode,cred_) !=
                                       GLOBUS_SUCCESS) return false;
  cred=cred_;
  return true;
}

bool HTTP_Client_Connector_Globus::read(char* buf,unsigned int* size) {
  if(!connected) return false;
  globus_result_t res;
  unsigned int size_=size?*size:0;
  if(size) *size=0;
  if((buf == NULL) || (size_ == 0)) {
    // cancel request
    if(read_registered) {
      res=globus_io_cancel(&s,GLOBUS_FALSE);
      if(res != GLOBUS_SUCCESS) {
        olog<<"globus_io_cancel failed: "<<GlobusResult(res)<<std::endl;
        return false;
      };
      read_registered=false;
      write_registered=false;
    };
    return true;
  };
  if(read_registered) return false;
  read_size=size;
  read_registered=true;
  read_done=-1; cond.reset();
  res=globus_io_register_read(&s,(globus_byte_t*)buf,size_,1,
                                                     &read_callback,this);
  if(res != GLOBUS_SUCCESS) {
    read_registered=false;
    olog<<"globus_io_register_read failed: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  return true;
}

bool HTTP_Client_Connector_Globus::write(const char* buf,unsigned int size) {
  if(!connected) return false;
  globus_result_t res;
  if((buf == NULL) || (size == 0)) {
    // cancel request
    if(write_registered) {
      res=globus_io_cancel(&s,GLOBUS_FALSE);
      if(res != GLOBUS_SUCCESS) {
        olog<<"globus_io_cancel failed: "<<GlobusResult(res)<<std::endl;
        return false;
      };
      read_registered=false;
      write_registered=false;
    };
    return true;
  };
  if(write_registered) return false;
  write_registered=true;
  write_done=-1; cond.reset();
  res=globus_io_register_write(&s,(globus_byte_t*)buf,size,&write_callback,this);
  if(res != GLOBUS_SUCCESS) {
    write_registered=false;
    olog<<"globus_io_register_write failed: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  return true;
}

bool HTTP_Client_Connector_Globus::connect(void) {
  if(!valid) return false;
  if(connected) return true;
  globus_result_t res;
  read_registered=false; write_registered=false;
  read_done=-1; write_done=-1;
  cond.reset();
  if((res=globus_io_tcp_register_connect(
                    (char*)(base_url.Host().c_str()),base_url.Port(),
                    &attr,&general_callback,this,&s)) != GLOBUS_SUCCESS) {
    olog<<"Connect to "<<base_url<<" failed: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  int r;
  globus_thread_blocking_will_block(); // ????
  if(!cond.wait(r,timeout)) {  // timeout
    olog<<"Connection to "<<base_url<<" timed out after "<<timeout/1000<<" seconds"<<std::endl;
    globus_io_cancel(&s,GLOBUS_FALSE);
    globus_io_close(&s);
    return false;
  };
  if(r != 0) {
    globus_io_close(&s); // just in case
    olog<<"Connection to "<<base_url<<" failed"<<std::endl;
    return false;
  };
  connected=true;
  return true;
}

bool HTTP_Client_Connector_Globus::disconnect(void) {
  if(!connected) return true;
  globus_io_cancel(&s,GLOBUS_FALSE);
  globus_io_close(&s);
  connected=false;
  return true;
}

bool HTTP_Client_Connector_Globus::transfer(bool& read,bool& write,int timeout) {
  read=false; write=false;
  if((!read_registered) && (!write_registered)) return true;
  for(;;) {
    if(read_registered && (read_done!=-1)) {
      read_registered=false; read=(read_done==0); break;
    };
    if(write_registered && (write_done!=-1)) {
      write_registered=false; write=(write_done==0); break;
    };
    int r;
    if(!cond.wait(r,timeout)) return false; // timeout
  };
  return true;
}


bool HTTP_Client_Connector_Globus::clear(void) {
  if(!valid) return false;
  globus_byte_t buf[256];
  globus_size_t l;
  for(;;) {
    if(globus_io_read(&s,buf,256,0,&l) != GLOBUS_SUCCESS) return false;
    if(l == 0) break;
    odlog(VERBOSE)<<"clear_input: ";
    for(int n=0;n<l;n++) odlog_(VERBOSE)<<buf[n];
    odlog_(VERBOSE)<<std::endl;
  };
  return true;
}

void HTTP_Client_Connector_Globus::general_callback(void *arg,globus_io_handle_t *handle,globus_result_t  result) {
  HTTP_Client_Connector_Globus* it = (HTTP_Client_Connector_Globus*)arg;
  if(result != GLOBUS_SUCCESS) {
    olog<<"Globus error: "<<GlobusResult(result)<<std::endl;
    it->cond.signal(-1);
  } else {
    it->cond.signal(0);
  };
}

void HTTP_Client_Connector_Globus::read_callback(void *arg,globus_io_handle_t *handle,globus_result_t result,globus_byte_t *buf,globus_size_t nbytes) {
  HTTP_Client_Connector_Globus* it = (HTTP_Client_Connector_Globus*)arg;
  int res = 0;
  if(result != GLOBUS_SUCCESS) {
    globus_object_t* err = globus_error_get(result);
    char* tmp=globus_object_printable_to_string(err);
    if(strstr(tmp,"end-of-file") != NULL) {
      odlog(VERBOSE)<<"Connection closed"<<std::endl;
      res=2; // eof
    } else {
      olog<<"Globus error (read): "<<tmp<<std::endl;
      res=1;
    };
    free(tmp); globus_object_free(err);
  } else {
    odlog(VERBOSE)<<"*** Server response: ";
    for(globus_size_t n=0;n<nbytes;n++) odlog_(VERBOSE)<<(char)(buf[n]);
    odlog_(VERBOSE)<<std::endl;
    if(it->read_size) *(it->read_size)=nbytes;
  };
  it->cond.block();
  it->read_done=res; it->cond.signal_nonblock(0);
  it->cond.unblock();
}

void HTTP_Client_Connector_Globus::write_callback(void *arg,globus_io_handle_t *handle,globus_result_t result,globus_byte_t *buf,globus_size_t nbytes) {
  HTTP_Client_Connector_Globus* it = (HTTP_Client_Connector_Globus*)arg;
  int res = 0;
  if(result != GLOBUS_SUCCESS) {
    olog<<"Globus error (write): "<<GlobusResult(result)<<std::endl;
    res=1;
  } else {
    odlog(VERBOSE)<<"*** Client request: ";
    for(globus_size_t n=0;n<nbytes;n++) odlog_(VERBOSE)<<(char)(buf[n]);
    odlog_(VERBOSE)<<std::endl;
  };
  it->cond.block();
  it->write_done=res; it->cond.signal_nonblock(0);
  it->cond.unblock();
}

#if GLOBUS_IO_VERSION<4
globus_bool_t HTTP_Client_Connector_Globus::authorization_callback(void* arg,globus_io_handle_t* h,globus_result_t result,char* identity,gss_ctx_id_t* context_handle) {
#else
globus_bool_t HTTP_Client_Connector_Globus::authorization_callback(void* arg,globus_io_handle_t* h,globus_result_t result,char* identity,gss_ctx_id_t context_handle) {
#endif
  odlog(VERBOSE)<<"Authenticating: "<<identity<<std::endl;
  return GLOBUS_TRUE;
}

bool HTTP_Client_Connector_Globus::eofread(void) {
  return (read_done==2);
}

bool HTTP_Client_Connector_Globus::eofwrite(void) {
  return (!write_registered);
}

// --------------- GSSAPI ---------------

GlobusModuleGSIGSSAPI HTTP_Client_Connector_GSSAPI::mod_gssapi;

static std::string gss_error_string(OM_uint32 maj_status,OM_uint32 min_status) {
  std::string message;
  OM_uint32 major_status = 0;
  OM_uint32 minor_status = 0;
  OM_uint32 m_context = 0;
  gss_buffer_desc buf;
  for(;;) {
    buf.length=0; buf.value=NULL;
    major_status=gss_display_status(&minor_status,maj_status,
                   GSS_C_GSS_CODE,GSS_C_NULL_OID,&m_context,&buf);
    if(buf.value != NULL) {
      if(!message.empty()) message+="; ";
      message+=(char*)(buf.value);
      gss_release_buffer(&minor_status,&buf);
    };
    if(m_context == 0) break;
  };
  for(;;) {
    buf.length=0; buf.value=NULL;
    major_status=gss_display_status(&minor_status,min_status,
                   GSS_C_MECH_CODE,GSS_C_NULL_OID,&m_context,&buf);
    if(buf.value != NULL) {
      if(!message.empty()) message+="; ";
      message+=(char*)(buf.value);
      gss_release_buffer(&minor_status,&buf);
    };
    if(m_context == 0) break;
  };
  return message;
}

HTTP_Client_Connector_GSSAPI::HTTP_Client_Connector_GSSAPI(const char* base,bool heavy_encryption,int timeout_,gss_cred_id_t cred_,bool check_host) try: base_url(base), check_host_cert(check_host) {
  s=-1;
  cred=cred_;
  timeout=timeout_;
  context=GSS_C_NO_CONTEXT;
  valid=true;
} catch(std::exception e) {
  valid=false;
}

bool HTTP_Client_Connector_GSSAPI::connect(void) {
  if(!valid) return false;
  if(s != -1) return true;
  read_buf=NULL; read_size=0; read_size_result=NULL;
  write_buf=NULL; write_size=0;
  read_eof_flag=false;

  struct hostent* host;
  struct hostent  hostbuf;
  int    errcode;
#ifndef _AIX
  char   buf[BUFSIZ];
  if(gethostbyname_r(base_url.Host().c_str(),&hostbuf,buf,sizeof(buf),
                                        &host,&errcode) != 0) return false;
#else
  struct hostent_data buf[BUFSIZ];
  if((errcode=gethostbyname_r(base_url.Host().c_str(),
                                  (host=&hostbuf),buf)) != 0) return false;
#endif
  if( (host == NULL) ||
      (host->h_length < sizeof(struct in_addr)) ||
      (host->h_addr_list[0] == NULL) ) {
    odlog(ERROR)<<"Host not found: "<<base_url.Host()<<std::endl;
    return false;
  };
  struct sockaddr_in addr;
  memset(&addr,0,sizeof(addr));
  addr.sin_family=AF_INET;
  addr.sin_port=htons(base_url.Port());
  memcpy(&addr.sin_addr,host->h_addr_list[0],sizeof(struct in_addr));

  s=::socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
  if(s==-1) {
    char buf[1024];
    char* str = strerror_r(errno,buf,sizeof(buf));
    odlog(ERROR)<<"Socket creation failed: "<<(str?str:"")<<std::endl;
    return false;
  };

  if(::connect(s,(struct sockaddr *)&addr,sizeof(addr))==-1) {
    char buf[1024];
    char* str = strerror_r(errno,buf,sizeof(buf));
    odlog(ERROR)<<"Connection to server failed: "<<(str?str:"")<<std::endl;
    ::close(s); s=-1;
    return false;
  };

  OM_uint32 major_status = 0;
  OM_uint32 minor_status = 0;

  //if(cred == GSS_C_NO_CREDENTIAL) {
  //  major_status = gss_acquire_cred(&minor_status,
  //                                  GSS_C_NO_NAME,
  //                                  0,
  //                                  GSS_C_NULL_OID_SET,
  //                                  GSS_C_ACCEPT,
  //                                  &cred,
  //                                  NULL,
  //                                  NULL);
  //  if (major_status != GSS_S_COMPLETE) {
  //    odlog(ERROR)<<"Failed to acquire local credentials"<<std::endl;
  //    ::close(s); s=-1;
  //    return false;
  //  };
  //};

  OM_uint32 init_sec_min_stat;
  OM_uint32 ret_flags =  0;
  gss_name_t remote_name = GSS_C_NO_NAME;
  gss_OID oid = GSS_C_NO_OID; // ??????????
  gss_buffer_desc recv_tok;
  gss_buffer_desc send_tok;
  gss_buffer_desc name;
  int context_flags = GSS_C_CONF_FLAG | GSS_C_MUTUAL_FLAG | GSS_C_INTEG_FLAG | GSS_C_DELEG_FLAG; // GSS_C_GLOBUS_SSL_COMPATIBLE
#ifdef HAVE_GLOBUS_GSS_ASSIST_AUTHORIZATION_HOST_NAME
  globus_gss_assist_authorization_host_name((char*)(base_url.Host().c_str()),&remote_name);
#else
  {
    gss_buffer_desc name_tok;
    name_tok.value = (void*)(base_url.Host().c_str());
    name_tok.length = base_url.Host().length() + 1;
    major_status=gss_import_name(&minor_status,&name_tok,
                                 GSS_C_NT_HOSTBASED_SERVICE,&remote_name);
    // if(GSS_ERROR(major_status))
  };
#endif
  // check if we want to do service host cert checks
  if(!check_host_cert) {
    remote_name=GSS_C_NO_NAME;
    // can't do delegation with no target host
    context_flags = GSS_C_CONF_FLAG | GSS_C_MUTUAL_FLAG | GSS_C_INTEG_FLAG;
  };

  recv_tok.length=0; recv_tok.value=NULL;
  send_tok.length=0; send_tok.value=NULL;
  for(;;) {
    major_status = gss_init_sec_context(&init_sec_min_stat,
                                        cred,
                                        &context,
                                        remote_name,
                                        oid,
                                        context_flags,
                                        0,
                                        NULL,
                                    recv_tok.value?&recv_tok:GSS_C_NO_BUFFER,
                                        NULL,
                                        &send_tok,
                                        &ret_flags,
                                        NULL);
    if(recv_tok.value) { free(recv_tok.value); recv_tok.value=NULL; };
    if((major_status!=GSS_S_COMPLETE) &&
       (major_status!=GSS_S_CONTINUE_NEEDED)) {
      odlog(ERROR)<<"Failed to authenticate: "<<gss_error_string(major_status,init_sec_min_stat)<<std::endl;
      ::close(s); s=-1; break;
    };
    if(context == NULL) {
      odlog(ERROR)<<"Failed to create GSI context: "<<gss_error_string(major_status,init_sec_min_stat)<<std::endl;
      ::close(s); s=-1; break;
    };
    if(send_tok.length != 0) {
      int to = timeout;
      if(do_write((char*)(send_tok.value),send_tok.length,to) == -1) {
        ::close(s); s=-1; break;
      };
      gss_release_buffer(&minor_status,&send_tok); send_tok.length=0;
    };
    if(major_status==GSS_S_COMPLETE) break;
    int l=read_SSL_token(&(recv_tok.value),timeout);
    if(l <= 0) {
      odlog(ERROR)<<"Failed to read SSL token during authentication"<<std::endl;
      if(context != GSS_C_NO_CONTEXT)
        gss_delete_sec_context(&minor_status,&context,GSS_C_NO_BUFFER);
      context=GSS_C_NO_CONTEXT;
      ::close(s); s=-1;
      return false;
    };
    recv_tok.length=l;
  };
  if(s == -1) {
    if(context != GSS_C_NO_CONTEXT) {
      gss_delete_sec_context(&minor_status,&context,GSS_C_NO_BUFFER);
      context=GSS_C_NO_CONTEXT;
    };
  };
  if(recv_tok.value) { free(recv_tok.value); recv_tok.value=NULL; };
  if(send_tok.length != 0) gss_release_buffer(&minor_status,&send_tok);
  if(remote_name != GSS_C_NO_NAME) gss_release_name(&minor_status,&remote_name);
  if(s == -1) {
    /// ??????????????
  };
  return (s != -1);
}

bool HTTP_Client_Connector_GSSAPI::disconnect(void) {
  if(s == -1) return true;
  ::close(s); s=-1;
  OM_uint32 minor_status = 0;
  if(context != GSS_C_NO_CONTEXT)
    gss_delete_sec_context(&minor_status,&context,GSS_C_NO_BUFFER);
  context=GSS_C_NO_CONTEXT;
}  

bool HTTP_Client_Connector_GSSAPI::read(char* buf,unsigned int* size) {
  if(s == -1) return false;
  read_size=size?*size:0;
  read_size_result=size;
  if(size) *size=0;
  read_buf=buf;
  return true;
}

bool HTTP_Client_Connector_GSSAPI::write(const char* buf,unsigned int size) {
  if(s == -1) return false;
  write_size=size;
  write_buf=buf;
  return true;
}

bool HTTP_Client_Connector_GSSAPI::transfer(bool& read,bool& write,int timeout) {
  read=false; write=false;
  if(write_buf) {
    OM_uint32 major_status;
    OM_uint32 minor_status;
    gss_buffer_desc send_tok;
    gss_buffer_desc data_tok;
    int conf_state;
    data_tok.length=write_size;
    data_tok.value=(void*)write_buf;
    odlog(VERBOSE)<<"*** Client request: ";
    for(globus_size_t n=0;n<data_tok.length;n++) odlog_(VERBOSE)<<((char*)(data_tok.value))[n];
    odlog_(VERBOSE)<<std::endl;
    major_status = gss_wrap(&minor_status,
                            context,
                            0,
                            GSS_C_QOP_DEFAULT,
                            &data_tok,
                            &conf_state,
                            &send_tok);
    if(major_status != GSS_S_COMPLETE) {
      odlog(ERROR)<<"Failed wraping GSI token: "<<gss_error_string(major_status,minor_status)<<std::endl;
      return false;
    };
    int to = timeout;
    int r = do_write((char*)(send_tok.value),send_tok.length,to);
    gss_release_buffer(&minor_status,&send_tok);
    write_buf=NULL; write_size=0; write=(r!=-1);
    return true;
  };
  if(read_buf) {
    gss_buffer_desc recv_tok;
    gss_buffer_desc data_tok = GSS_C_EMPTY_BUFFER;
    OM_uint32 major_status;
    OM_uint32 minor_status;
    int ll = read_SSL_token(&(recv_tok.value),timeout);
    if(ll == 0) { read_eof_flag=true; read=false; return true; };
    if(ll == -1) { read=false; return true; };
    recv_tok.length=ll;
    major_status = gss_unwrap(&minor_status,
                              context,
                              &recv_tok,
                              &data_tok,
                              NULL,
                              NULL);
    free(recv_tok.value);
    if(major_status != GSS_S_COMPLETE) {
      odlog(ERROR)<<"Failed unwraping GSI token: "<<gss_error_string(major_status,minor_status)<<std::endl;
      return false;
    };
    odlog(VERBOSE)<<"*** Server response: ";
    for(globus_size_t n=0;n<data_tok.length;n++) odlog_(VERBOSE)<<((char*)(data_tok.value))[n];
    odlog_(VERBOSE)<<std::endl;

    if(data_tok.length > read_size) {
      odlog(ERROR)<<"Unwrapped data does not fit into buffer"<<std::endl;
      return false;
    };
    memcpy(read_buf,data_tok.value,data_tok.length);
    if(read_size_result) (*read_size_result)=data_tok.length;
    gss_release_buffer(&minor_status,&data_tok);
    read_buf=NULL; read_size=0; read_size_result=NULL; read=true;
    return true;
  };
  return true;
}

bool HTTP_Client_Connector_GSSAPI::eofread(void) {
  return read_eof_flag;
}

bool HTTP_Client_Connector_GSSAPI::eofwrite(void) {
  return (write_buf == NULL);
}

static unsigned int timems(void) {
  struct timeval tv;
  struct timezone tz;
  if(gettimeofday(&tv,&tz) != 0) return (time(NULL)*1000);
  return (tv.tv_sec*1000+tv.tv_usec/1000);
}

static bool waitsocket(int r,int w,int& timeout) {
  unsigned int t_start = timems();
  unsigned int dt = 0;
  if(timeout == -1) return true; // for blocking operation
  for(;;) {
    fd_set rs; FD_ZERO(&rs); if(r>=0) FD_SET(r,&rs);
    fd_set ws; FD_ZERO(&ws); if(w>=0) FD_SET(w,&ws);
    struct timeval t;
    t.tv_sec=(timeout-dt)/1000; t.tv_usec=(timeout-dt-t.tv_sec*1000)*1000;
    int n = ::select((r>w?r:w)+1,&rs,&ws,NULL,&t);
    if(n > 0) break;
    if((n == -1) && (errno != EINTR)) break; // to cause error on read/write
    dt=timems()-t_start;
    if(dt >= timeout) { timeout=0; return false; };
  };
  dt=timems()-t_start; 
  if(dt > timeout) dt=timeout; timeout-=dt;
  return true;
}

int HTTP_Client_Connector_GSSAPI::do_read(char* buf,int size,int& timeout) {
  int n = size;
  for(;n;) {
    if(!waitsocket(s,-1,timeout)) return -1;
    int l = ::recv(s,buf,n,0);
    if((l == -1) && (errno != EINTR)) return -1;
    if(l == 0) {
      if(n == size) return 0;
      return -1;
    };
    buf+=l; n-=l;
  };
  return size;
}

int HTTP_Client_Connector_GSSAPI::do_write(char* buf,int size,int& timeout) {
  int n = size;
  for(;n;) {
    if(!waitsocket(-1,s,timeout)) return -1;
    int l = ::send(s,buf,n,0);
    if((l == -1) && (errno != EINTR)) return -1;
    buf+=l; n-=l;
  };
  return size;
}

int HTTP_Client_Connector_GSSAPI::read_SSL_token(void** val,int timeout) {
  unsigned char header[5]; // SSL V3 header
  (*val)=NULL;
  int l = do_read((char*)header,sizeof(header),timeout);
  if(l == 0) return 0;
  if(l < 0) return -1;
  if(header[0] == (unsigned char)0x80) {
    /* SSL V2 - 2 byte header */
    l=((unsigned int)(header[1]))-3;
  } else if((header[0] >= 20) &&
            (header[0] <= 26) &&
            (header[1] == 3) &&
            (header[2] == 0 || header[2] == 1)) {
         /* SSL V3 */
    l=(((unsigned int)(header[3])) << 8) | ((unsigned int)(header[4]));
  } else {
    odlog(ERROR)<<"Urecognized SSL token received"<<std::endl;
    return -1;
  };
  unsigned char* data = (unsigned char*)malloc(l+5);
  if(data == NULL) return -1;
  memcpy(data,header,5);
  if(l) {
    int ll = do_read((char*)(data+5),l,timeout);
    if(ll <= 0) {
      free(data);
      return -1;
    };
  };
  (*val)=data;
  return (l+5);
}

bool HTTP_Client_Connector_GSSAPI::clear(void) {
  gss_buffer_desc recv_tok;
  int l;
  while(true) {
    l=read_SSL_token(&(recv_tok.value),0);
    if(l <= 0) return true;
    if(recv_tok.value) free(recv_tok.value);
  };
}

bool HTTP_Client_Connector_GSSAPI::credentials(gss_cred_id_t cred_) {
  cred=cred_;
  return true;
}

HTTP_Client_Connector_GSSAPI::~HTTP_Client_Connector_GSSAPI(void) {
  disconnect();
}

