#include "../../std.h"

#include "../../misc/stringtoint.h"
#include "../../misc/escaped.h"
#include "../../misc/time_utils.h"
#include "../../misc/log_time.h"
#include "se_replicator.h"
#include "files.h"


SEFileHandle::SEFileHandle(SEFile& f,uint64_t o,bool for_read):file(f) {
  active=true;
  offset=o;
  open_read=for_read;
  if(file.open(open_read) != 0) active=false;
  odlog(VERBOSE)<<"SEFileHandle constructor: active: "<<active<<std::endl;
}

SEFileHandle::~SEFileHandle(void) {
  if(active) file.close(open_read);
}

uint64_t SEFileHandle::read(void* buf,uint64_t size) {
  odlog(VERBOSE)<<"SEFileHandle::read - size: "<<size<<std::endl;
  if(!active) return 0;
  uint64_t l = file.read(buf,offset,size);
  offset+=l;
  odlog(VERBOSE)<<"SEFileHandle::read - l: "<<l<<std::endl;
  return l;
}

uint64_t SEFileHandle::write(void* buf,uint64_t size) {
  if(!active) return 0;
  uint64_t l = file.write(buf,offset,size);
  offset+=l;
  return l;
}

SEFileHandle* SEFileHandle::open(const char* id,uint64_t o,bool for_read,SEFiles& files) {
  SEFile* file = files.get(id);
  if(!file) return NULL;
  SEFileHandle* f = new SEFileHandle(*file,o,for_read);
  if(!(*f)) { delete f; return NULL; };
  return f;
}

SEFiles::SEFiles(void) {
//  files=NULL;
  ns=NULL;
//  nfiles=0;
  valid=false;
  reg_type=registration_immediate;
  timeout_collecting=DEFAULT_TIMEOUT_COLLECTING;
  timeout_downloading=DEFAULT_TIMEOUT_DOWNLOADING;
  retries_downloading=DEFAULT_RETRIES_DOWNLOADING;
  timeout_complete=DEFAULT_TIMEOUT_COMPLETE;
  timeout_failed=DEFAULT_TIMEOUT_FAILED;
}

SEFiles::SEFiles(const char* dirpath):path(dirpath),space(dirpath) {
  valid=false;
//  files=NULL;
  ns=NULL;
  reg_type=registration_immediate;
  timeout_collecting=DEFAULT_TIMEOUT_COLLECTING;
  timeout_downloading=DEFAULT_TIMEOUT_DOWNLOADING;
  retries_downloading=DEFAULT_RETRIES_DOWNLOADING;
  timeout_complete=DEFAULT_TIMEOUT_COMPLETE;
  timeout_failed=DEFAULT_TIMEOUT_FAILED;
//  nfiles=0;
  struct dirent file_;
  struct dirent *file;
  DIR *dir=opendir(dirpath);
  if(dir == NULL) {
    odlog(ERROR)<<"Failed opening directory: "<<dirpath<<std::endl; return;
  };
  for(;;) {
    readdir_r(dir,&file_,&file);
    if(file == NULL) break;
    int l=strlen(file->d_name);
    if(l>5) {
      if(!strcmp(file->d_name+l-5,".attr")) {
        std::string name(file->d_name); name.resize(l-5);
        name="/"+name; name=dirpath+name;
        odlog(DEBUG)<<"SEFiles: creating SEFile: "<<name<<std::endl;
        SEFile* f = new SEFile(name.c_str(),space);
        if(!(*f)) {
          odlog(ERROR)<<"SEFiles: failed to acquire SEFile: "<<name<<std::endl;
        } else {
          odlog(DEBUG)<<"SEFiles: adding SEFile: "<<name<<std::endl;
          add(*f);
          odlog(INFO)<<"Added file: "<<f->id()<<std::endl;
        };
      };
    };
  };
  closedir(dir);
  valid=true;
}

SEFiles::~SEFiles() {
  lock.block();
//  for(int i=0;i<nfiles;i++) {
//    if(files[i]) delete files[i];
//  };
//  free(files); files=NULL;
  valid=false;
  lock.unblock();
}

SEFiles::iterator SEFiles::add(SEFile& file) {
  odlog(DEBUG)<<"SEFiles::add"<<std::endl;
  lock.block(); // TODO: to remove ?!
  if(files) {
    //SEFile* f = find_by_id(file.id());
    SEFiles::iterator f = begin();
    for(;f!=end();++f) if(f->check_id(file.id())) break;
    if(f.exists()) {
      // File with this id already exists. Allow (re)creation if 
      // owner is a same person (DN) and other attributes are same
      // too (if set).
      bool same = true;
      if(file.creator() != f->creator()) same=false;
      if(same && file.size_available() && f->size_available()) {
        if(file.size() != f->size()) same=false;
      };
      if(same && file.checksum_available() && f->checksum_available()) {
        if(file.checksum() != f->checksum()) same=false;
      };
      if(same && file.created_available() && f->created_available()) {
        if(memcmp(file.created(),f->created(),sizeof(struct tm))!=0)same=false;
      };
      if(!same) {
        odlog(ERROR)<<"SEFiles::add: file already exists: "<<file.id()<<std::endl;
        file.destroy();
        lock.unblock();
        return end();
      };
      // files look identical, destroy new and replace it by old
      file.destroy();
      lock.unblock();
      return f;
    };
  };
  odlog(DEBUG)<<"SEFiles::add: new file: "<<file.id()<<std::endl;
  SEFiles::iterator f = files.add(file);
  lock.unblock();
  return f;
}

bool SEFiles::remove(SEFile& file) {
  for(SEFiles::iterator i=begin();i!=end();++i) {
    if(&(*i) == &file) {
      return remove(i);
    };
  };
  return false;
}

bool SEFiles::remove(SEFiles::iterator& file) {
   odlog(INFO)<<"SEFiles::remove: "<<file->id()<<std::endl;
  if(!file) return true;
  lock.block(); // TODO: to remove
  file->destroy(); 
  files.remove(file);
  lock.unblock();
  return true;
}

SEFile* SEFiles::find_by_id(const char* id) {
  for(SEFiles::iterator i = begin();i!=end();++i) {
    if(i->check_id(id)) {
      SEFile* f = &(*i);
      return f;
    };
  };
  return NULL;
}

SEFile* SEFiles::get(const char* id) {
  lock.block(); // TODO: to remove
  SEFile* f = find_by_id(id);
  lock.unblock();
  return f;
}

// Register all files, which require registration
int SEFiles::Register(void) {
  odlog(DEBUG)<<"SEFiles::Register"<<std::endl;
  // TODO: Be careful with NS - it can be non-thread-safe
  int failures = 0;
  if(files && ns) {
    for(SEFiles::iterator f = files.begin();f!=files.end();++f) {
      // analyze situation
      f->acquire();
      if((f->state_reg() == REG_STATE_UNREGISTERING) ||
         (f->state_reg() == REG_STATE_REGISTERING)) {
        f->release(); continue; // file is alredy claimed
      };
      if(f->state_reg()!=REG_STATE_LOCAL) {  // already/yet registered 
        f->release(); continue;
      };
      if(!register_retry()) { // this function only serves retries
        f->release(); continue;
      };
      if(!(
           (f->state_file()==FILE_STATE_VALID) ||
           (register_immediately() && (f->state_file()==FILE_STATE_COLLECTING))
          )) { // not proper for register
        f->release(); continue;
      };
      // claim file being registered
      odlog(INFO)<<"Registering: "<<f->id()<<std::endl;
      if(!f->state_reg(REG_STATE_REGISTERING)) {
        odlog(ERROR)<<"Registering: "<<f->id()<<" - change state failed"<<std::endl;
        failures++;
        f->release(); continue;
      };
      f->release();
      ns->Connect();
      if(ns->Register(*f) != 0) {
        failures++;
        if(!f->state_reg(REG_STATE_LOCAL)) {
          // ???????????
        };
        continue;
      };
      if(f->state_reg(REG_STATE_ANNOUNCED)) {
        // ???????????
        continue;
      };
    };
    ns->Disconnect();
  };
  return failures;
}

// Unregister all files being deleted or failed.
// Also deletes unregistered files.
int SEFiles::Unregister(void) {
  odlog(VERBOSE)<<"SEFiles::Unregister"<<std::endl;
  int failures = 0;
  if(files && ns) {
    for(SEFiles::iterator f = files.begin();f!=files.end();++f) {
      // analyze situation
      f->acquire(); // lock file to prevent race while analyzing state
      if((f->state_reg() == REG_STATE_UNREGISTERING) ||
         (f->state_reg() == REG_STATE_REGISTERING)) {
        f->release(); continue; // file is already claimed
      };
      if((f->state_file()!=FILE_STATE_DELETING) &&
         (f->state_file()!=FILE_STATE_FAILED)) { // not proper for unregister
        f->release(); continue;
      };
      if(f->state_reg()==REG_STATE_LOCAL) {  // stuck file 
        if((f->state_file()!=FILE_STATE_DELETING) ||
           ((time(NULL) - f->state_file_changed()) > timeout_failed)) {
          f->release(); f->destroy(); files.remove(f); continue;
        };
      };
      if(f->state_reg()!=REG_STATE_ANNOUNCED) {  // already/yet not registered 
        f->release(); f->destroy(); files.remove(f); continue;
      };
      if(!register_retry()) { // this function only serves retries
        f->release(); continue;
      };
      // claim file being unregistered
      odlog(INFO)<<"Unregistering: "<<f->id()<<std::endl;
      if(!f->state_reg(REG_STATE_UNREGISTERING)) {
        odlog(ERROR)<<"Unregistering: "<<f->id()<<" - change state failed"<<std::endl;
        failures++;
        f->release(); continue;
      };
      f->release();
      ns->Connect();
      if(ns->Unregister(*f) != 0) {
        odlog(ERROR)<<"Unregistering - Unregister failed"<<std::endl;
        failures++;
        if(!f->state_reg(REG_STATE_ANNOUNCED)) {
          // ???????????
        };
        continue;
      };
      odlog(DEBUG)<<"Unregistering - Unregister succeeded"<<std::endl;
      if(!f->state_reg(REG_STATE_LOCAL)) {
        failures++;
        // ???????????
        continue;
      };
      // unregistered file in deleting state  - destroy and remove
      if((f->state_file()==FILE_STATE_DELETING) ||
         ((time(NULL) - f->state_file_changed()) > timeout_failed)) {
        f->destroy(); files.remove(f);
      };
    };
    ns->Disconnect();
  };
  return failures;
}

// There MUST be only one thread which uses this function.
// Verify content of file (and do immediate (re)registration)
int SEFiles::Verify(void) {
  odlog(VERBOSE)<<"SEFiles::Verify"<<std::endl;
  int failures = 0;
  if(files) {
    for(SEFiles::iterator f = files.begin();f!=files.end();++f) {
      // Verification
      f->acquire();
      if(f->state_file()!=FILE_STATE_COMPLETE) { f->release(); continue; };
      if(!f->complete()) {
        // Check if this state is not for too long
        if((time(NULL) - f->state_file_changed()) > timeout_complete) {
          // Keep it in FAILED so user can see what happened to file
          f->state_file_description("Timeout waiting for file to be verified");
          if(!f->state_file(FILE_STATE_FAILED)) {
            odlog(ERROR)<<"SEFiles::Verify: failed to set FILE_STATE_FAILED"
                    <<std::endl;
          };
          f->destroy_content();
        };
        f->release(); continue;
      }; // metadata not complete
      f->release();
      int res = f->verify();
      if(res == 1) {
        odlog(ERROR)<<"File has incomplete (meta)data, validation failed: "<<f->id()<<std::endl;
      } else if(res != 0) {
        odlog(ERROR)<<"Failed to validate file: "<<f->id()<<std::endl;
      };
      f->acquire();
      if(res != 0) { // make it FAILED and destroy content
        f->state_file_description("Timeout waiting for file to be verified");
        if(!f->state_file(FILE_STATE_FAILED)) {
          odlog(ERROR)<<"Failed to change file state: "<<f->id()<<std::endl;
        };
        f->destroy_content();
        f->release(); continue;
      };
      // Attempt of immediate registration
      if(ns) {
        if(f->state_reg(REG_STATE_REGISTERING)) {
          f->release();
          if(ns->Register(*f) != 0) {
            if(register_retry()) {
              odlog(ERROR)<<"Failed to register (will retry)"<<std::endl;
            } else {
              odlog(ERROR)<<"Failed to register (what to do ?)"<<std::endl;
            };
            f->state_reg(REG_STATE_LOCAL);
          } else {
            f->state_reg(REG_STATE_ANNOUNCED);
          };
          f->acquire();
        } else {
          if(register_retry()) {
            odlog(ERROR)<<"Failed to set REGISTERING state (will retry)"<<std::endl;
          } else {
            odlog(ERROR)<<"Failed to set REGISTERING state (what to do ?)"<<std::endl;
          };
        };
      };
      if(!f->state_file(FILE_STATE_VALID)) {
        odlog(ERROR)<<"Verify: "<<f->id()<<" - change state failed"<<std::endl;
        failures++; continue;
      };
      f->release();
    };
  };
  return failures;
}

bool SEFiles::try_unregister(SEFiles::iterator f) {
  if(!NS()) return true;
  if(f->state_reg() == REG_STATE_LOCAL) return true;
  if(f->state_reg() != REG_STATE_ANNOUNCED) return false;
  if(!f->state_reg(REG_STATE_UNREGISTERING)) {
    odlog(ERROR)<<"SEFiles::try_unregister: failed to set REG_STATE_UNREGISTERING"
            <<std::endl;
    return false; 
  };
  f->release();
  if(NS()->Unregister(*f) != 0) {
    f->acquire(); f->state_reg(REG_STATE_ANNOUNCED);
    odlog(ERROR)<<"SEFiles::try_unregister: failed to unregister"<<std::endl;
    return false;
  };
  f->acquire(); f->state_reg(REG_STATE_LOCAL);
  return true;
}

bool SEFiles::try_register(SEFiles::iterator f) {
  if(!NS()) return true;
  if(f->state_reg() == REG_STATE_ANNOUNCED) return true;
  if(!(f->state_reg(REG_STATE_REGISTERING))) {
    if(register_retry()) {
      odlog(ERROR)<<"Failed to set REGISTERING state (will retry)"<<std::endl;
    } else {
      odlog(ERROR)<<"Failed to set REGISTERING state (what to do?)"<<std::endl;
    };
    return false;
  };
  f->release();
  if(NS()->Register(*f) != 0) {
    if(register_retry()) {
      odlog(ERROR)<<"Failed to register (will retry)"<<std::endl;
    } else {
      odlog(ERROR)<<"Failed to register (what to do ?)"<<std::endl;
    };
    f->state_reg(REG_STATE_LOCAL);
    f->acquire();
    return false;
  };
  f->state_reg(REG_STATE_ANNOUNCED);
  f->acquire();
  return true;
}

int SEFiles::RemoveStuck(void) {
  odlog(VERBOSE)<<"SEFiles::RemoveStuck"<<std::endl;
  int failures = 0;
  if(files) {
    for(SEFiles::iterator f = files.begin();f!=files.end();++f) {
      // analyze situation
      f->acquire();
      if(f->state_file() == FILE_STATE_COLLECTING) {
        uint64_t file_size = DEFAULT_SIZE_COLLECTING;
        if(f->size_available()) file_size=f->size()/1024/1024 + 1;
        int file_timeout = file_size * timeout_collecting;
        if(((time(NULL) - f->state_file_changed()) > file_timeout) &&
           ((time(NULL) - f->data_changed()) > (timeout_collecting*10))) {
          // file was not touched for long time - probably forgotten
          odlog(ERROR)<<"Removing file which stayed too long in COLLECTING state: "<<f->id()<<std::endl;
          bool unregistered = try_unregister(f);
          // Keep it in FAILED so user can see what happened to file
          f->state_file_description("Timeout waiting for file to be uploaded");
          if(!f->state_file(FILE_STATE_FAILED)) {
            odlog(ERROR)<<"SEFiles::RemoveStuck: failed to set FILE_STATE_FAILED"
                    <<std::endl;
          };
          f->destroy_content();
          // if(unregistered) { remove(*f); };
        };
      };
      f->release();
    };
  };
  return failures;
}

int SEFiles::Replicate(void) {
  int failures = 0;
  if(files) {
    for(SEFiles::iterator f = files.begin();f!=files.end();++f) {
      // analyze situation
      f->acquire();
      if(f->state_file() == FILE_STATE_REQUESTED) {
        if(!f->state_tries()) f->state_tries(retries_downloading);
        // claim file being downloaded
        odlog(ERROR)<<"Downloading: "<<f->id()<<std::endl;
        if(!f->state_file(FILE_STATE_DOWNLOADING)) {
          odlog(ERROR)<<"Downloading: "<<f->id()<<" - change state failed"<<std::endl;
          failures++; f->release(); continue;
        };
        f->release();
        bool downloaded = true;
        SEReplicator replicator;
        odlog(VERBOSE)<<"Downloading - calling Obtain"<<std::endl;
        if(replicator.Obtain(*f) != 0) {
          odlog(ERROR)<<"Downloading - Obtain failed"<<std::endl;
          failures++; downloaded=false;
        };
        if(downloaded) { // validate file if possible
          int res;
          if(f->checksum_available()) {
            res = f->verify();
            if(res == 1) {
              odlog(ERROR)<<"File downloaded but has incomplete (meta)data: "<<f->id()<<std::endl;
              f->state_file_description(
                          "File has incomplete metadata after download");
            } else if(res != 0) {
              odlog(ERROR)<<"Failed to validate downloaded file: "<<f->id()<<std::endl;
              f->state_file_description(
                          "File failed to validate after download");
            };
          } else { // Can't validate - at least compute checksum
            odlog(ERROR)<<"WARNING: Checksum for file is not known, computing for later usage."<<std::endl;
            res=f->checksum_compute("md5");
            if(res != 0) {
              odlog(ERROR)<<"Failed to compute checksum."<<std::endl;
              f->state_file_description("Failed to compute checksum");
            } else if(f->write_attr() != 0) {
              odlog(ERROR)<<"Failed to write attributes."<<std::endl;
              f->state_file_description("Internal failure");
              res=-1;
            };
          };
          if(res != 0) { // mark file FAILED and go to next
            f->acquire();
            if(!f->state_file(FILE_STATE_FAILED)) {
              odlog(ERROR)<<"Failed to change file state: "<<f->id()<<std::endl;
            };
            f->destroy_content();
            f->release();
            continue;
          };
        };
        // File either not downloaded or downloaded AND validated.
        f->acquire();
        if(downloaded) { // here means validated
          if(!f->state_file(FILE_STATE_VALID)) {
            odlog(ERROR)<<"Downloading: "<<f->id()<<" - change state failed"<<std::endl;
            failures++; f->release(); continue;
          };
          // Try immediate registration
          try_register(f);
        } else {  // download failed
          if(f->state_tries_dec()) { // have tries left
            if(!f->state_file(FILE_STATE_REQUESTED)) {
              odlog(ERROR)<<"Downloading: "<<f->id()<<" - change state failed"
              <<std::endl;
              failures++;
            };
          } else {
            f->state_file_description(
                              "All allowed attempts to download file used");
            if(!f->state_file(FILE_STATE_FAILED)) {
              odlog(ERROR)<<"Failed to change file state: "<<f->id()<<std::endl;
            };
            f->destroy_content();
          };
        };
      };
      f->release();
    };
  };
  return failures;
}

void SEFiles::Maintain(void) {
  if(files) {
    for(SEFiles::iterator f = files.begin();f!=files.end();++f) {
      f->acquire();
      f->Maintain();
      f->release();
    };
  };
}

