#include "../../std.h"
#include <dlfcn.h>

#include <fstream>
#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#include <globus_gsi_credential.h>
#include <globus_gss_assist.h>
#include <globus_openssl.h>

#include "../../auth/auth.h"
#include "../../run/run.h"
#include "../../misc/escaped.h"
#include "../../misc/inttostring.h"
#include "../../misc/condition.h"
#include "../../misc/log_time.h"
#include <arc/globuserrorutils.h>
#include "../../config/environment.h"
#include "../../config/config_sections.h"
#include "../../config/config_vo.h"
#include "../../config/daemon.h"

#include "../httpsd.h"

unsigned int conections_processed = 0;

//  Attention - ugly hack
static int curr_connections = 0;
static int pre_connections = 2;
static int max_connections = 100;

globus_cond_t conn_cond;
globus_mutex_t conn_lock;

HTTP_Services services;

std::list<AuthEvaluator*> auths;
std::list<AuthVO> vos;

typedef struct {
  const char* name;
  service_creator      creat;
  service_configurator config;
} service_list_t;

service_item_t service_list_internal[] = {
#ifdef BUILD_LOGGER
 { "logger", &logger_service_creator, &logger_service_configurator },
#endif
#ifdef BUILD_SE
 { "se", &se_service_creator, &se_service_configurator },
#endif
 { NULL, NULL, NULL }
};

std::list<service_item_t> service_list;

LockSimple service_lock;

void add_services(service_item_t* s_list) {
  for(service_item_t* s=s_list;s->name;++s) {
    std::list<service_item_t>::iterator p=service_list.insert(service_list.end(),*s);    odlog(ERROR)<<"Loaded service: "<<p->name<<std::endl;
  };
}

bool add_library(const char* libname) {
  service_lock.block();
  void* handle = dlopen(libname,RTLD_NOW | RTLD_GLOBAL);
  if(handle == NULL) {
    odlog(ERROR)<<"Library '"<<libname<<"' could not be loaded: "<<dlerror()<<std::endl;
    service_lock.unblock();
    return false;
  };
  service_item_t* s_list = (service_item_t*)dlsym(handle,"service_list");
  if(s_list == NULL) {
    odlog(ERROR)<<"Library '"<<libname<<"' has no services"<<std::endl;
    dlclose(handle);
    service_lock.unblock();
    return false;
  };
  add_services(s_list);
  service_lock.unblock();
  return true;
}

service_item_t* find_service(const char* name) {
  service_lock.block();
  // look among already loaded services
  std::list<service_item_t>::iterator s = service_list.begin();
  service_item_t* s_;
  for(;s!=service_list.end();++s) if(strcmp(s->name,name) == 0) break;
  if(s!=service_list.end()) {
    s_=&(*s); service_lock.unblock();
    return s_;
  };
  // look for library with <name>.so
  std::string libname = std::string(name)+".so";
  void* handle = dlopen(libname.c_str(),RTLD_NOW | RTLD_GLOBAL);
  if(handle == NULL) {
    // look for library with lib<name>.so
    libname = std::string("lib")+name+".so";
    handle = dlopen(libname.c_str(),RTLD_NOW | RTLD_GLOBAL);
  };
  if(handle == NULL) {
    odlog(ERROR)<<"Library '"<<libname<<"' could not be loaded: "<<dlerror()<<std::endl;
    service_lock.unblock();
    return NULL;
  };
  service_item_t* s_list = (service_item_t*)dlsym(handle,"service_list");
  if(s_list == NULL) {
    odlog(ERROR)<<"Library '"<<libname<<"' has no services"<<std::endl;
    dlclose(handle);
    service_lock.unblock();
    return NULL;
  };
  for(s_=s_list;s_->name;++s_) {
    if(strcmp(s_->name,name) == 0) break;
  };
  if(!(s_->name)) {
    odlog(ERROR)<<"Library '"<<libname<<"' has no service named "<<name<<std::endl;
    dlclose(handle);
    service_lock.unblock();
    return NULL;
  };
  // Add new services
  s=service_list.end();
  for(service_item_t* s__=s_list;s__->name;++s__) {
    std::list<service_item_t>::iterator p = 
                  service_list.insert(service_list.end(),*s__);
    odlog(ERROR)<<"Loaded service: "<<p->name<<std::endl;
    if(s_ == s__) s=p;
  };
  if(s==service_list.end()) {
    service_lock.unblock();
    return NULL;
  };
  s_=&(*s); service_lock.unblock();
  return s_;
}

#define COMMON_PORT_N 0
#define GSI_PORT_N    1
#define SSL_PORT_N    2
#define GSSAPI_PORT_N 3
#define PLAIN_PORT_N  4

#define CONFIGURE_PORT_NC(cmdlen,cmd,portn,portname) \
  if((command_len==cmdlen) && (strncmp(command,cmd,command_len)==0)) { \
    if(!ports[portn]) { \
      if(sscanf(command+command_len,"%u",&(ports[portn])) != 1) { \
        olog<<"Wrong "<<portname<<" port number in configuration\n"; \
        f.close(); return 1; \
      }; \
    }; \
  }

#define CONFIGURE_PORT_C(cmd,portn,portname) \
  if(command == cmd) { \
    if(!ports[portn]) if(sscanf(rest.c_str(),"%u",&(ports[portn])) != 1) { \
      olog<<"Wrong "<<portname<<" port number in configuration\n"; \
      f.close(); return 1; \
    }; \
  }


int configure_common(const char* fname,Daemon& daemon,unsigned int ports[5]) {
  std::ifstream f(fname);
  if(!f.is_open()) {
    olog<<"Failed to read configuration file "<<fname<<std::endl;
    return -1;
  };
  if(!central_configuration) {
    for(;!f.eof();) {
      char buf[1024];
      istream_readline(f,buf,sizeof(buf));
      char* p = buf;
      for(;*p;p++) if(!isspace(*p)) break;
      char* command = p;
      if(*command == '#') continue;
      for(;*p;p++) if(isspace(*p)) break;
      int command_len = p-command;
      if(!command_len) continue;
      int r;
      {
        std::string cmd(command); cmd.resize(command_len);
        std::string rest(command+command_len);
        r=daemon.config(cmd,rest);
        if(r==0) continue;
        if(r==-1) { f.close(); return -1; };
      };
      CONFIGURE_PORT_NC(7,"gsiport",GSI_PORT_N,"GSI")
      else CONFIGURE_PORT_NC(7,"sslport",SSL_PORT_N,"SSL")
      else CONFIGURE_PORT_NC(9,"plainport",PLAIN_PORT_N,"HTTP")
      else CONFIGURE_PORT_NC(10,"gssapiport",GSSAPI_PORT_N,"GSSAPI")
      else CONFIGURE_PORT_NC(4,"port",COMMON_PORT_N,"")
      else {
      };
    };
  } else {
    ConfigSections cf(f);
    cf.AddSection("common");
    cf.AddSection("httpsd");
    for(;;) {
      std::string rest;
      std::string command;
      cf.ReadNext(command,rest);
      if((command.length() == 0) && (rest.length() == 0)) break; /* EOF */
      if(cf.SubSection()[0]) continue;
      int r;
      {
        r=daemon.config(command,rest);
        if(r==0) continue;
        if(r==-1) { f.close(); return -1; };
      };
      CONFIGURE_PORT_C("gsiport",GSI_PORT_N,"GSI") 
      else CONFIGURE_PORT_C("sslport",SSL_PORT_N,"SSL") 
      else CONFIGURE_PORT_C("gssapiport",GSSAPI_PORT_N,"GSSAPI") 
      else CONFIGURE_PORT_C("plainport",PLAIN_PORT_N,"HTTP") 
      else CONFIGURE_PORT_C("port",COMMON_PORT_N,"") 
      else {
      };
    };
  };
  f.close();
  return 0;
}

int configure_services(const char* fname) {
  std::ifstream f(fname);
  if(!f.is_open()) {
    olog<<"Failed to read configuration file "<<fname<<std::endl;
    return -1;
  };
  // Add internal services
  service_lock.block();
  add_services(service_list_internal);
  service_lock.unblock();
  if(!central_configuration) {
    for(;!f.eof();) {
      char buf[1024];
      istream_readline(f,buf,sizeof(buf));
      char* p = buf;
      for(;*p;p++) if(!isspace(*p)) break;
      char* command = p;
      if(*command == '#') continue;
      for(;*p;p++) if(isspace(*p)) break;
      int command_len = p-command;
      if(!command_len) continue;
      {
        std::string cmd(command); cmd.resize(command_len);
        if(Daemon::skip_config(cmd) == 0) continue;
      };
      if((command_len==5) && (strncmp(command,"group",command_len)==0)) {
        std::string name("");
        int n = input_escaped_string(p,name);
        if(n==0) {
          odlog(ERROR)<<"Group name is missing"<<std::endl; f.close(); return -1;
        };
        p+=n;
        AuthEvaluator* auth = new AuthEvaluator(name.c_str());
        if(!auth) {
          odlog(ERROR)<<"Failed to create AuthEvaluator"<<std::endl; f.close();
          if(auth) delete auth;
          return -1;
        };
        auths.push_back(auth);
        for(;!f.eof();) {
          istream_readline(f,buf,sizeof(buf));
          p = buf;
          for(;*p;p++) if(!isspace(*p)) break;
          command = p;
          if(*command == '#') continue;
          for(;*p;p++) if(isspace(*p)) break;
          command_len = p-command;
          if(!command_len) continue;
          if((command_len==3) && (strncmp(command,"end",command_len)==0)) break;
          auth->add(command);
        };
        odlog(ERROR)<<"Added authorization group: "<<name<<std::endl;
      } else if((command_len==2) && (strncmp(command,"vo",command_len)==0)) {
        if(config_vo(vos,command,p) != 0) {
          odlog(ERROR)<<"Error: couldn't process VO configuration"<<std::endl;
          return -1;
        };
      } else if((command_len==6) && (strncmp(command,"plugin",command_len)==0)) {
        std::string libname("");
        int n;
        for(;;) {
          n=input_escaped_string(p,libname);
          if(n==0) break;
          if(libname.length() == 0) continue;
          p+=n;
          add_library(libname.c_str());
        };
      } else if((command_len==7) && (strncmp(command,"service",command_len)==0)) {
        /* service name path */
        std::string name("");
        int n = input_escaped_string(p,name);
        if(n==0) {
          odlog(ERROR)<<"Service name is missing"<<std::endl; f.close(); return -1;
        };
        p+=n;
        std::string path("");
        n = input_escaped_string(p,path);
        if(path[path.length()-1] == '/') path.resize(path.length()-1);
        service_item_t* s = find_service(name.c_str());
        if(!s) {
          odlog(ERROR)<<"Service '"<<name<<"' is not supported"<<std::endl; f.close();
          return -1;
        };
        HTTP_Service_Properties prop;
        if(s->config(f,path.c_str(),prop)) {
          services.add(path.c_str(),s->creat,prop);
          odlog(ERROR)<<"Added service "<<s->name<<" at "<<path<<std::endl;
        } else {
          odlog(ERROR)<<"Service '"<<s->name<<"' at "<<path<<" failed to configure"<<std::endl;
        };
      }
      else if((command_len==7) && (strncmp(command,"gsiport",command_len)==0)) {
      }
      else if((command_len==7) && (strncmp(command,"sslport",command_len)==0)) {
      }
      else if((command_len==4) && (strncmp(command,"port",command_len)==0)) {
      } else {
        *p=0;
        odlog(ERROR)<<"Unknown command: "<<command<<std::endl; f.close(); return -1;
      };
    };
  } else {
    typedef enum {
      conf_state_single,
      conf_state_group,
      conf_state_service
    } config_state_t;
    config_state_t st = conf_state_single;
    ConfigSections cf(f);
    cf.AddSection("common");
    cf.AddSection("group");
    cf.AddSection("httpsd");
    cf.AddSection("vo");
    std::string group_name;
    std::list<std::string> group_rules;
    std::string service_name;
    std::string service_path;
    std::string service_config;
    for(;;) {
      std::string rest;
      std::string command;
      cf.ReadNext(command,rest);
      int r = config_vo(vos,cf,command,rest);
      if(r == 0) continue;
      if(cf.SectionNew()) {
        switch(st) {
          case conf_state_group: {
            if(group_name.length() == 0) {
              odlog(ERROR)<<"Group name is missing"<<std::endl; f.close();
              return -1;
            };
            AuthEvaluator* auth = new AuthEvaluator(group_name.c_str());
            if(!auth) {
              odlog(ERROR)<<"Failed to create AuthEvaluator"<<std::endl; f.close();
              if(auth) delete auth;
              return -1;
            };
            auths.push_back(auth);
            for(std::list<std::string>::iterator a = group_rules.begin();
                                a!=group_rules.end();++a) {
              auth->add(a->c_str());
            };
            odlog(ERROR)<<"Added authorization group: "<<group_name<<std::endl;
          }; break;
          case conf_state_service: {
            if(service_name.length() == 0) {
              odlog(ERROR)<<"Service name is missing"<<std::endl; f.close(); return -1;
            };
            if(service_path.length() == 0) {
              odlog(ERROR)<<"Service path/url is missing"<<std::endl;
              f.close(); return -1;
            };
            if(service_path[service_path.length()-1] == '/') 
                        service_path.resize(service_path.length()-1);
            service_item_t* s = find_service(service_name.c_str());
            if(!s) {
              odlog(ERROR)<<"Service '"<<service_name<<"' is not supported"<<std::endl; f.close();
              f.close(); return -1;
            };
            /* find service */
            HTTP_Service_Properties prop;
#ifdef HAVE_SSTREAM
            std::stringstream fake_f(service_config);
#else
            std::strstream fake_f;
            fake_f<<service_config;
#endif
            if(s->config(fake_f,service_path.c_str(),prop)) {
              services.add(service_path.c_str(),s->creat,prop);
              odlog(ERROR)<<"Added service "<<s->name<<" at "<<service_path<<std::endl;
            } else {
              odlog(ERROR)<<"Service '"<<s->name<<"' at "<<service_path<<
                                               " failed to configure"<<std::endl;
            };
          }; break;
        };
        // prepare for new section
        service_name="";
        service_path="";
        service_config="";
        group_name="";
        group_rules.resize(0);
        // 0 - common, 1 - group, 2 - httpsd
        if(cf.SectionNum() == 1) {
          st=conf_state_group;
          group_name=cf.SubSection();
        } else if(cf.SubSection()[0] == 0) {
          st=conf_state_single;
        } else {
          service_name=cf.SubSection();
          st=conf_state_service;
        };
      };
      if((command.length() == 0) && (rest.length() == 0)) break; /* EOF */
      switch(st) {
        case conf_state_single: {
          if(Daemon::skip_config(command) == 0) {
          } else if(command == "gsiport") {
          } else if(command == "sslport") {
          } else if(command == "port") {
          } else if(command == "plugin") {
            std::string libname("");
            int n;
            const char* p = rest.c_str();
            for(;;) {
              n=input_escaped_string(p,libname);
              if(n==0) break;
              if(libname.length() == 0) continue;
              p+=n;
              add_library(libname.c_str());
            };
          };
        }; break;
        case conf_state_group: {
          if(command == "name") {
            group_name=rest;
          } else {
            rest=command+" "+rest;
            group_rules.push_back(rest);
          };
        }; break;
        case conf_state_service: {
          if(command == "name") {
            service_name=rest;
          } else if(command == "path") {
            service_path=rest;
          } else {
            service_config+=command;
            service_config+=" ";
            service_config+=rest;
            service_config+="\n";
          };
        }; break;
      };
    };
  };
  f.close();
  return 0;
}

class base_url_t {
 private:
  std::string type_;
  std::string url_;
 public:
  base_url_t(const char* type,const char* url);
  bool operator==(const char* type);
  operator std::string(void);
  operator const char*(void);
  bool match(const char* url);
};

std::list<base_url_t> base_urls;

base_url_t::base_url_t(const char* type,const char* url):type_(type),url_(url) {

}

bool base_url_t::operator==(const char* type) {
  return (strcmp(type,type_.c_str()) == 0);
}

base_url_t::operator std::string(void) {
  return url_;
}

base_url_t::operator const char*(void) {
  return url_.c_str();
}

bool base_url_t::match(const char* url) {
  if(!url) return false;
  int l = strlen(url);
  int l_ = url_.length();
  return (strncmp(url,url_.c_str(),l<l_?l:l_) == 0);
}

const char* base_url_by_type(const char* type) {
  for(std::list<base_url_t>::iterator i = base_urls.begin();
                                         i!=base_urls.end();++i) {
    if((*i) == type) return ((const char*)(*i));
  };
  return NULL;
}

const char* base_url_by_url(const char* url) {
  for(std::list<base_url_t>::iterator i = base_urls.begin();
                                         i!=base_urls.end();++i) {
    if(i->match(url)) return ((const char*)(*i));
  };
  return NULL;
}

#define DEFAULT_GSI_PORT 8000
#define DEFAULT_SSL_PORT 8001
#define DEFAULT_GSSAPI_PORT 8002
#define DEFAULT_PLAIN_PORT 80

#define DEFAULT_LOG_FILE "/var/log/httpsd.log"
#define DEFAULT_PID_FILE "/var/run/httpsd.pid"

int main(int argc,char* argv[]) {
  globus_module_deactivate_all(); // to fight VOMS
  unsigned short server_port = 0;
  int handle;
  int handle_ssl;
  struct sockaddr_in myaddr;
  LogTime::Active(true);
  LogTime::Level(ERROR);

  // nordugrid_config_loc="httpsd.conf";
  // read_env_vars(true);
  std::string config_file;
  int n;
  Daemon daemon;
  while((n=daemon.getopt(argc,argv,"hp:c:n:")) != -1) {
    switch(n) {
      case ':': { olog<<"Missing argument\n"; return 1; };
      case 'h': {
        std::cout<<"httpsd [-p port_to_listen] [-c config_file] [-n maximal_connections] "<<daemon.short_help()<<std::endl;
         return 0;
      };
      case 'p': {
        if(sscanf(optarg,"%hu",&server_port) != 1) {
          olog<<"Wrong port number\n";
          return 1;
        };
      }; break;
      case 'c': {
        config_file=optarg;
      }; break;
      case 'n': {
        if((sscanf(optarg,"%i",&max_connections) != 1) ||
           (max_connections < 0)) {
          olog<<"Wrong number of connections\n";
          return 1;
        };
      }; break;
      default: break;
    };
  };
  if(config_file.length()) nordugrid_config_loc=config_file;
  nordugrid_config_basename="httpsd.conf";
  read_env_vars(true);
  config_file=nordugrid_config_loc;
  if(config_file.length() == 0) {
  //  if(!central_configuration) {
  //    config_file=nordugrid_loc+"/etc/httpsd.conf";
  //  } else {
  //    config_file="/etc/nordugrid.conf";
  //  };
    olog<<"Missing configuration file"<<std::endl;
    return 1;
  };
  //unsigned int gsiport=0;
  //unsigned int sslport=0;
  //unsigned int port=0;
  unsigned int ports[5] = { 0, 0, 0, 0, 0 };
  if(configure_common(config_file.c_str(),daemon,ports) != 0) {
    return 1;
  };
  if(server_port != 0) {
    ports[GSI_PORT_N]=server_port; ports[SSL_PORT_N]=server_port+1;
  };
  if((ports[GSI_PORT_N] == 0) &&
     (ports[GSSAPI_PORT_N] == 0)) ports[GSI_PORT_N]=DEFAULT_GSI_PORT;
  if(ports[SSL_PORT_N] == 0) ports[SSL_PORT_N]=DEFAULT_SSL_PORT;

  daemon.logfile(DEFAULT_LOG_FILE);
  daemon.pidfile(DEFAULT_PID_FILE);
  if(daemon.daemon() != 0) {
    perror("daemonization failed");
  };

  // Initialize children signal handling environemnt
  Run run;

  if(globus_module_activate(GLOBUS_IO_MODULE) != GLOBUS_SUCCESS) {
    olog<<"Failed to activate Globus IO module"<<std::endl;
    return 1;
  };
  if(globus_module_activate(GLOBUS_GSI_CREDENTIAL_MODULE) != GLOBUS_SUCCESS) {
    olog<<"Failed to activate Globus GSI Credential module"<<std::endl;
    return 1;
  };
  if(globus_module_activate(GLOBUS_GSI_GSS_ASSIST_MODULE) != GLOBUS_SUCCESS) {
    olog<<"Failed to activate Globus GSI GSS Assist module"<<std::endl;
    return 1;
  };
  if(globus_module_activate(GLOBUS_OPENSSL_MODULE) != GLOBUS_SUCCESS) {
    olog<<"Failed to activate Globus OpenSSL module"<<std::endl;
    return 1;
  };
  run.reinit(false);
  char hbuf[1024];
  hbuf[sizeof(hbuf)-1]=0;
  if(globus_libc_gethostname(hbuf,sizeof(hbuf)-1) != 0) {
    strcpy(hbuf,"localhost");
  };

  if(ports[GSI_PORT_N]) base_urls.push_back(base_url_t("gsi",
  (std::string("httpg://")+hbuf+":"+inttostring(ports[GSI_PORT_N])).c_str()));
  if(ports[SSL_PORT_N]) base_urls.push_back(base_url_t("ssl",
  (std::string("https://")+hbuf+":"+inttostring(ports[SSL_PORT_N])).c_str()));
  if(ports[GSSAPI_PORT_N]) base_urls.push_back(base_url_t("gssapi",
  (std::string("httpg://")+hbuf+":"+inttostring(ports[GSSAPI_PORT_N])).c_str()));
  if(ports[PLAIN_PORT_N]) base_urls.push_back(base_url_t("plain",
  (std::string("http://")+hbuf+":"+inttostring(ports[PLAIN_PORT_N])).c_str()));

  if(configure_services(config_file.c_str()) != 0) {
    olog<<"Configuration failed. Exiting"<<std::endl;
    globus_module_deactivate(GLOBUS_IO_MODULE);
    exit(1);
  };

  globus_io_attr_t attr;
  globus_result_t res;

  globus_cond_init(&conn_cond,GLOBUS_NULL);
  globus_mutex_init(&conn_lock,GLOBUS_NULL);

  HTTP_Globus_Listener* gsi_listener = NULL;
  HTTP_SSL_Listener* ssl_listener = NULL;
  HTTP_GSSAPI_Listener* gssapi_listener = NULL;
  HTTP_Plain_Listener* plain_listener = NULL;
  

  if(ports[GSI_PORT_N]) {
    globus_io_tcpattr_init(&attr);
    globus_io_attr_set_secure_authentication_mode(&attr,GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,GSS_C_NO_CREDENTIAL);
    globus_io_attr_set_secure_channel_mode(&attr,GLOBUS_IO_SECURE_CHANNEL_MODE_GSI_WRAP);
    globus_io_attr_set_secure_protection_mode(&attr,GLOBUS_IO_SECURE_PROTECTION_MODE_PRIVATE);
    globus_io_attr_set_secure_delegation_mode(&attr,GLOBUS_IO_SECURE_DELEGATION_MODE_NONE);
    globus_io_attr_set_secure_proxy_mode(&attr,GLOBUS_IO_SECURE_PROXY_MODE_MANY);
    gsi_listener=new HTTP_Globus_Listener(ports[GSI_PORT_N],&attr,base_url_by_type("gsi"),services,auths,vos);
    if(!(*gsi_listener)) {
      olog<<"Failed to create GSI listener: "<<res<<std::endl;
      return 1; 
    };
    odlog(ERROR)<<"Running on: "<<base_url_by_type("gsi")<<std::endl;
  };
  // globus_io_attr_set_secure_channel_mode(&attr,GLOBUS_IO_SECURE_CHANNEL_MODE_SSL_WRAP);
  // globus_io_attr_set_secure_protection_mode(&attr,GLOBUS_IO_SECURE_PROTECTION_MODE_SAFE);  // allow SSL connection without encryption (decision by client)
  // HTTP_Globus_Listener ssl_listener(sslport,&attr,base_ssl_url.c_str(),services,auths);
  if(ports[SSL_PORT_N]) {
    ssl_listener=new HTTP_SSL_Listener(ports[SSL_PORT_N],base_url_by_type("ssl"),services,auths,vos);
    if(!(*ssl_listener)) {
      olog<<"Failed to create SSL listener: "<<res<<std::endl;
      return 1; 
    };
    odlog(ERROR)<<"Running on: "<<base_url_by_type("ssl")<<std::endl;
  };
  if(ports[GSSAPI_PORT_N]) {
    gssapi_listener=new HTTP_GSSAPI_Listener(ports[GSSAPI_PORT_N],base_url_by_type("gssapi"),services,auths,vos);
    if(!(*gssapi_listener)) {
      olog<<"Failed to create GSSAPI listener: "<<res<<std::endl;
      return 1;
    };
    odlog(ERROR)<<"Running on: "<<base_url_by_type("gssapi")<<std::endl;
  };
  if(ports[PLAIN_PORT_N]) {
    plain_listener=new HTTP_Plain_Listener(ports[PLAIN_PORT_N],base_url_by_type("plain"),services,auths,vos);
    if(!(*plain_listener)) {
      olog<<"Failed to create HTTP listener: "<<res<<std::endl;
      return 1;
    };
    odlog(ERROR)<<"Running on: "<<base_url_by_type("plain")<<std::endl;
  };

  globus_abstime_t timeout;
  globus_mutex_lock(&conn_lock);
  while(1) {  // forever
    GlobusTimeAbstimeSet(timeout,10,0);
    globus_cond_timedwait(&conn_cond,&conn_lock,&timeout);
//    odlog(ERROR)<<"conections_processed: "<<conections_processed<<std::endl;
//    if(conections_processed >= 10) break;
//    globus_cond_wait(&conn_cond,&conn_lock);
//    globus_thread_blocking_will_block();
  };
  globus_mutex_unlock(&conn_lock);

  globus_io_tcpattr_destroy(&attr);

  globus_cond_destroy(&conn_cond);
  globus_mutex_destroy(&conn_lock);

  globus_module_deactivate(GLOBUS_IO_MODULE);

  return 0;
}

