#ifndef __CONDITION_H__
#define __CONDITION_H__

#include <pthread.h>
#include <errno.h>
#include <sys/time.h>

/*
  Specifies few objects, which make it a bit simpler to use 
  locks and conditions.
*/
/*
  Common purpose condition. Can pass arbitrary type variable
  from 'signal' to 'wait'. Good for using in callbacks.
*/
template <typename T> class Condition {
 private:
  int timeout;
  pthread_mutex_t lock;
  pthread_cond_t  cond;
  T value;
  bool signaled;
 public:
  /* create object with default timeout of 20s */
  Condition(int t = 20000):signaled(false) {
    pthread_mutex_init(&lock,NULL);
    pthread_cond_init(&cond,NULL);
    timeout=t;
  };
  ~Condition(void) {
    pthread_cond_broadcast(&cond);
    pthread_cond_destroy(&cond);
    pthread_mutex_destroy(&lock);
  };
  void reset(void) { signaled=false; };
  void block(void) { pthread_mutex_lock(&lock); };
  void unblock(void) { pthread_mutex_unlock(&lock); };
  void signal_nonblock(T val) {
    if(!signaled) {
      value=val;
      signaled=true;
      pthread_cond_signal(&cond);
    };
  };
  void signal(T val) {
    block();
    signal_nonblock(val);
    unblock();
  };
  /* wait specified amount of time 
     return false if timed out */
  bool wait(T &val) {
    return wait(val,timeout);
  };
  bool wait(T &val,int t) {
    pthread_mutex_lock(&lock);
    if(t<0) {
      while(!signaled) {
        int err=pthread_cond_wait(&cond,&lock);
        if(err == EINTR) continue;
        if(err != 0) { pthread_mutex_unlock(&lock); return false; };
      };
    }
    else {
      struct timeval stime;
      gettimeofday(&stime,NULL);
      struct timespec etime;
      etime.tv_sec = stime.tv_sec + (t/1000);
      etime.tv_nsec = ((t%1000)*1000 + stime.tv_usec) * 1000;
      etime.tv_sec += etime.tv_nsec/1000000000;
      etime.tv_nsec = etime.tv_nsec%1000000000;
      while(!signaled) {
        int err=pthread_cond_timedwait(&cond,&lock,&etime);
        if(err == EINTR) continue;
        if(err != 0) { pthread_mutex_unlock(&lock); return false; };
      };
    };
    val=value;
    signaled=false;  /* ????????? */
    pthread_mutex_unlock(&lock);
    return true;
  };
  bool check(void) { return signaled; };
};

/*
  Simple triggered condition. Does not pass anything.
*/
class CondSimple {
 private:
  pthread_cond_t cond;
  pthread_mutex_t lock;
  bool flag;
 public:
  CondSimple(void) { 
    flag=false;
    pthread_cond_init(&cond,NULL);
    pthread_mutex_init(&lock,NULL);
  };
  ~CondSimple(void) {
    /* race condition ? */
    broadcast();
    pthread_cond_destroy(&cond);
    pthread_mutex_destroy(&lock);
  };
  void block(void) {
    pthread_mutex_lock(&lock); 
  };
  void unblock(void) {
    pthread_mutex_unlock(&lock); 
  };
  void signal(void) {
    pthread_mutex_lock(&lock); 
    flag=true;
    pthread_cond_signal(&cond);
    pthread_mutex_unlock(&lock);
  };
  void signal_nonblock(void) {
    flag=true;
    pthread_cond_signal(&cond);
  };
  void broadcast(void) {
    pthread_mutex_lock(&lock);
    flag=true;
    pthread_cond_broadcast(&cond);
    pthread_mutex_unlock(&lock);
  };
  void wait(void) {
    pthread_mutex_lock(&lock);
    while(!flag) {
      if(pthread_cond_wait(&cond,&lock) != EINTR) break;
    };
    flag=false;
    pthread_mutex_unlock(&lock);
  };
  void wait_nonblock(void) {
    while(!flag) {
      if(pthread_cond_wait(&cond,&lock) != EINTR) break;
    };
    flag=false;
  };
  void wait(int t) {
    struct timeval stime;
    struct timespec etime;
    pthread_mutex_lock(&lock);
    gettimeofday(&stime,NULL);
    etime.tv_sec = stime.tv_sec + (t/1000);
    etime.tv_nsec = ((t%1000)*1000 + stime.tv_usec) * 1000;
    etime.tv_sec += etime.tv_nsec/1000000000;
    etime.tv_nsec = etime.tv_nsec%1000000000;
    while(!flag) {
      int err=pthread_cond_timedwait(&cond,&lock,&etime);
      if(err == EINTR) continue;
      if(err != 0) break;
    };
    flag=false;
    pthread_mutex_unlock(&lock);
  };
  void reset(void) {
    pthread_mutex_lock(&lock);
    flag=false;
    pthread_mutex_unlock(&lock);
  };
};

class LockSimple {
 private:
  pthread_mutex_t lock;
 public:
  LockSimple(void) {
    pthread_mutex_init(&lock,NULL);
  };
  ~LockSimple(void) {
    pthread_mutex_destroy(&lock);
  };
  void block(void) {
    pthread_mutex_lock(&lock);
  };
  void unblock(void) {
    pthread_mutex_unlock(&lock);
  };
};

/*
  Counter secured by lock.
*/
class CounterSimple {
 private:
  int counter;
  pthread_mutex_t lock;
 public:
  CounterSimple(void) {
    counter=0;
    pthread_mutex_init(&lock,NULL);
  };
  ~CounterSimple(void) {
    pthread_mutex_destroy(&lock);
  };
  int inc(void) {
    pthread_mutex_lock(&lock);
    counter++;
    int tmp = counter;
    pthread_mutex_unlock(&lock);
    return tmp;
  };
  int dec(void) {
    pthread_mutex_lock(&lock);
    counter--; if(counter == -1) counter=0;
    int tmp = counter;
    pthread_mutex_unlock(&lock);
    return tmp;
  };
  void reset(void) {
    pthread_mutex_lock(&lock);
    counter=0;
    pthread_mutex_unlock(&lock);
  };
};

#endif // __CONDITION_H__
