#include "std.h"

#include <string>
#include <list>
#include <iostream>
#include <fstream>

#include "datamove/datapoint.h"
#include "datamove/datahandle.h"
#include "misc/log_time.h"

#include "ngdata.h"

static bool get_url_list(const char* file,std::list<std::string>& urls) {
  std::ifstream f(file);
  if(!f.is_open()) return false;
  for(;!f.eof();) {
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    if(!(buf[0])) continue;
    urls.push_back(buf);
  };
  if(urls.size() == 0) return false;
  return true;
}

void arcrm(const std::string& file_url_,
           bool errcont,
           int timeout) {
  LogTime::Active(false);
  bool verbose = false;

  LogTime::Level(GetNotifyLevel());
  if(LogTime::Level() > FATAL) verbose=true;

  if(file_url_[0] == '@') {
    std::list<std::string> files;
    if(!get_url_list(file_url_.c_str()+1,files)) {
      throw ARCCLIDataError(std::string("Can't read list of locations "
                      "from file ") + (file_url_.c_str()+1));
    };
    std::list<std::string>::iterator file_ = files.begin();
    for(;file_!=files.end();) {
      arcrm(*file_,errcont,timeout);
      ++file_;
    };
    return;
  };

  std::string file_url = file_url_;
  if(file_url != "-") {
    std::string::size_type p = file_url.find("://");
    if((p == std::string::npos) || (file_url.find('/') < p)) {
      if(file_url[0] == '/') {
        file_url="file://"+file_url;
      } else {
        char buf[1024]; buf[0]=0;
        getcwd(buf,sizeof(buf));
        file_url=std::string("file://")+buf+"/"+file_url;
      };
    };
  };
  DataPoint url(file_url.c_str());
  if(!url) {
    throw ARCCLIDataError("Unsupported url given");
  };
  bool remove_lfn = true;
  if(url.have_locations()) { /* pfn or plain url */
    remove_lfn=false;
  };
  if(!url.meta_resolve(true)) {
    if(remove_lfn) {  
      odlog(INFO)<<"No locations found - probably no more physical instances"
              <<std::endl;
    };
  };
  /* go through all locations and try to remove files
     physically and their metadata */
  std::list<std::string> removed_urls; // list of physical removed urls
  if(url.have_locations()) for(;url.have_location();) {
    if(verbose) odlog(ERROR)<<"Removing "<<url.current_location()<<std::endl;
    // It can happen that after resolving list contains duplicated
    // physical locations obtained from different meta-data-services.
    // Because not all locations can reliably say if files does not exist
    // or access is not allowed, avoid duplicated delete attempts.
    bool url_was_deleted = false;
    for(std::list<std::string>::iterator u = removed_urls.begin();
                                           u!=removed_urls.end();++u) {
      if(url.current_location() == (*u)) { url_was_deleted=true; break; };
    };
    if(url_was_deleted) {
      odlog(ERROR)<<"This instance was already deleted"<<std::endl;
    } else {
      DataHandle handle(&url);
      handle.secure(false);
      if(!handle.remove()) {
        odlog(ERROR)<<"Failed to delete physical file"<<std::endl;
        if(!errcont) { 
          url.next_location(); continue;
        };
      } else {
        removed_urls.push_back(url.current_location());
      };
    };
    if(!url.meta()) {
      url.remove_location();
    } else {
      if(verbose) odlog(ERROR)<<"Removing metadata in "
                          <<url.current_meta_location()<<std::endl;
      if(!url.meta_unregister(false)) {
        odlog(ERROR)<<"Failed to delete meta-information"<<std::endl;
        url.next_location();
      };
    };
  };
  if(url.have_locations()) {
    throw ARCCLIDataError("Failed to remove all physical instances");
  };
  if(url.meta()) {
    if(remove_lfn) {
      if(verbose) odlog(ERROR)<<"Removing logical file from metadata "
                          <<url.canonic_url()<<std::endl;
      if(!url.meta_unregister(true)) {
        throw ARCCLIDataError("Failed to delete logical file");
      };
    };
  };
  //  if(verbose) { std::cout<<"Remove succeeded"<<std::endl; };
  return;
}

extern "C"
int ngrmxx (
    const std::string& file_url,
    bool errcont,
    int verbosity_level,
    int timeout
) {
  try {
    SetNotifyLevel(NotifyLevel(FATAL+verbosity_level));
    arcrm(file_url,errcont,timeout);
  } catch(ARCCLIDataError& e) {
    return -1;
  } catch(std::exception& e) {
    return -1;
  };
  return 0;
}

