/* Utility to duplicte file or directory from remote gridftp server */
#include "std.h"
#include <stdlib.h>
#include <string>
#include <list>
extern "C" {
#include <globus_common.h>
#include <globus_error.h>
#include <globus_ftp_client.h>
#include <globus_object.h>
#include "misc/mkdir_recursive.h"
}
#include "misc/canonical_dir.h"
#include "misc/globus_modules.h"
#include "files/info_types.h"
#include "files/info_files.h"
#include "misc/log_time.h"
//#include "transfer/check.h"
#include "transfer/lister.h"
#include "datamove/datamove.h"
#include "datamove/datapoint.h"
#include "datamove/datahandle.h"
#include "datamove/databufferpar.h"
#include "ui_downloader.h"

#define GLOBUS_ERROR GLOBUS_ERROR_NO_INFO

static globus_mutex_t c_lock;
static globus_cond_t c_cond;
static int c_done;

static int url_to_path(std::string &s) {
  std::string::size_type n = s.find(':');
  if( n == std::string::npos) return -1;
  n++; if(n >= s.length()) return -1; if(s[n] != '/') return -1;
  n++; if(n >= s.length()) return -1; if(s[n] != '/') return -1;
  n++; n=s.find('/',n); if(n == std::string::npos) n=s.length();
  s.erase(0,n);
  return 0;
}

static int list_recursively(const std::string &url,std::list<std::string> &names, const std::string &add_path = "") {
  static Lister *lister = NULL;
  static int depth;
  if(add_path.length() == 0) {
    depth=0;
    lister = new Lister;
    if(!lister) {
      odlog(ERROR)<<"Problem creating Lister"<<std::endl;
      return 1;
    };
    if(!(*lister)) {
      odlog(ERROR)<<"Problem initializing Lister"<<std::endl;
      delete lister;
      return 1;
    };
  }
  else {
    depth++;
    if(depth >= 20) {
      odlog(ERROR)<<"Directories are too deep"<<std::endl;
      depth--;
      return -1;
    };
  };
  if(lister->retrieve_dir(url+add_path) != 0) {
    odlog(ERROR)<<"List not retrieved: "<<url+add_path<<std::endl;
    depth--;
    if(add_path.length() == 0) { delete lister; };
    return -1;
  };
  /* check if requested object is directory or file */
  if(lister->size() == 1) {
    if((lister->begin()->GetType() == ListerFile::file_type_unknown) ||
       (lister->begin()->GetType() == ListerFile::file_type_file)) {
      std::string path=url+add_path;
      url_to_path(path);
      if(path == lister->begin()->GetName()) { /* most probably file */
        names.push_back(add_path);
        depth--;
        if(add_path.length() == 0) { delete lister; };
        return 0;
      };
    };
  };
  /* directory(ies) */
  {
    std::list<std::string> tnames;
    for(std::list<ListerFile>::iterator i=lister->begin();i!=lister->end();++i) {
      if(i->GetType() == ListerFile::file_type_file) {
        names.push_back(add_path+'/'+i->GetLastName());
      } else {
        tnames.push_back(i->GetLastName());
      };
    };
    for(std::list<std::string>::iterator i=tnames.begin();i!=tnames.end();++i) {
      std::string add_path_=add_path+'/'+(*i);
      if(list_recursively(url,names,add_path_) != 0) { 
        depth--;
        if(add_path.length() == 0) { delete lister; };
        return -1;
      };
    };
  };
  depth--;
  if(add_path.length() == 0) { delete lister; };
  return 0;
}

static void deleted_callback(void* arg,globus_ftp_client_handle_t *handle,globus_object_t *error) {
  globus_mutex_lock(&c_lock);
  if(error != GLOBUS_SUCCESS) {
    odlog(ERROR) << "Delete failed"<<std::endl;
    char* tmp = globus_object_printable_to_string(error);
    odlog(ERROR)<<tmp<<std::endl; free(tmp);
    c_done=2;
  }
  else {
    odlog(DEBUG) << "File deleted"<<std::endl;
    c_done=1;
  };
  globus_cond_signal(&c_cond);
  globus_mutex_unlock(&c_lock);
}

#ifndef __CREATE_EXEC__
int ui_downloader(const char* url,bool recursive,const char* path,const std::vector<std::string> &filenames,bool download_files,bool remove_files,int debug_level,int timeout) {
  char* session_url = url?strdup(url):NULL;
  LogTime::Level(NotifyLevel(FATAL+debug_level));
#else
int main(int argc,char* argv[]) {
  char* session_url = NULL;
  bool remove_files = true;
  bool download_files = true;
  int timeout = -1;
#endif
  LogTime::Active(false);
  int res = 1;
  GlobusModuleCommon mod_common;
  if(!mod_common.active()) {
    odlog(ERROR)<<"COMMON module activation failed\n";
    if(session_url) free(session_url); return 1;
  };

  int l;
  char* path_ = NULL;
  bool download_successful = true;
#ifdef __CREATE_EXEC__
  const char* path = NULL;
  bool recursive = false;
  for(;;) {
    opterr=0;
    int optc=getopt(argc,argv,"+hkrp:");
    if(optc == -1) break;
    switch(optc) {
      case 'h': {
        cerr<<"ui-downloader [-r] [-k] [-p local_path] session_dir_url [list]"<<std::endl;
        exit(1);
      }; break;
      case 'p': {
        if(path != NULL) {
          cerr<<"Only one path can be given"<<std::endl;
          exit(1);
        };
        path=optarg;
      }; break;
      case 'r': {
        recursive=true;
      }; break;
      case 'k': {
        remove_files=false;
      }; break;
      case '?': {
        cerr<<"Unsupported option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      case ':': {
        cerr<<"Missing parameter for option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      default: {
        cerr<<"Undefined processing error"<<std::endl;
        exit(1);
      };
    };
  };
  session_url = strdup(argv[optind]);
#endif
  if(!session_url) {
    odlog(ERROR) << "Missing session directory url" << std::endl; return 1;
  };
  l = strlen(session_url);
  if(l == 0) {
    odlog(ERROR) << "Wrong session url" << std::endl; free(session_url); return 1;
  };
  if(session_url[l-1] == '/') session_url[l-1]=0;
#ifdef __CREATE_EXEC__
  if(argv[optind+1] && recursive) {
    cerr << "Recursive and listed retrieval can't be requested at same time" << std::endl;
    free(session_url); return 1;
  };
#endif
  if(!path) path="";
  l = strlen(path);
  if(path[0] != '/') {
    path_=(char*)malloc(1024+l+1); 
    if(path_ == NULL) { odlog(ERROR)<<"Memory allocation error"<<std::endl; return 1; };
    memset(path_,0,1024); getcwd(path_,1023); 
    strcat(path_,"/"); strcat(path_,path); path=path_;
  };
  l = strlen(path);
  if(path[l-1] == '/') ((char*)path)[l-1]=0;
  odlog(INFO)<<"Remote base url: "<<session_url<<std::endl;
  odlog(INFO)<<"Local path: "<<path<<std::endl;
  
  std::list<FileData> job_files;

  // Create directory
  if(mkdir_recursive(NULL,path,S_IRWXU,getuid(),getgid()) != 0) {
    odlog(ERROR)<<"Can't create directory for storing results: "<<path<<std::endl;
    res=1; goto exit;
  };
  // check permissions on destination directory
  {
    struct stat st;
    if(stat(path,&st) != 0) {
      odlog(ERROR)<<"Can't check directory for storing results: "<<path<<std::endl;
      res=1; goto exit;
    };
    if( !((st.st_uid == getuid()) && (st.st_mode & S_IWUSR)) &&
        !((st.st_gid == getgid()) && (st.st_mode & S_IWGRP)) &&
        !(st.st_mode & S_IWOTH) ) {
      odlog(ERROR)<<"Have no permission to write to directory for storing results: "<<path<<std::endl;
      res=1; goto exit;
    };
  };

  if(recursive) {
    std::list<std::string> names;
    GlobusModuleFTPControl mod_ftp_control;
    if(!mod_ftp_control.active()) {
      odlog(ERROR)<<"FTP CONTROL module activation failed\n";
      res=1; goto exit;
    };
    odlog(INFO)<<"Listing available files: "<<std::endl;
    if(list_recursively(std::string(session_url),names) != 0) {
      odlog(ERROR)<<"List failed"<<std::endl; res=1;
      goto exit;
    }
    else {
      for(std::list<std::string>::iterator i=names.begin();i!=names.end();++i) {
        job_files.push_back(FileData(i->c_str(),NULL));
        odlog(INFO)<<"  "<<(*i)<<std::endl;
      };
    };
  }
  else {
#ifdef __CREATE_EXEC__
    char* file_list=argv[optind+1];
    if(file_list) {
      if(!job_Xput_read_file(std::string(file_list),job_files)) {
        cerr << "Can't read list of files from " << file_list << std::endl;
        res=1; goto exit;
      };
    }
    else {
      if(!job_Xput_read_file(job_files)) {
        cerr << "Can't parse list of files" << std::endl;
        res=1; goto exit;
      };
    };
#else
    for(std::vector<std::string>::const_iterator iv=filenames.begin();
                                     iv!=filenames.end();++iv) {
      std::string f1 = *iv;
      std::string f2;
      canonical_dir(f1);
      ++iv; if(iv==filenames.end()) {
        job_files.push_back(FileData(f1.c_str(),NULL));
        break;
      };
      f2 = *iv;
      job_files.push_back(FileData(f1.c_str(),f2.c_str()));
    };
#endif
  };

  if((job_files.size() == 0) && (download_files)) {
    odlog(INFO)<<"No files to download"<<std::endl; res=0; goto exit;
  };
  globus_mutex_init(&c_lock,GLOBUS_NULL);
  globus_cond_init(&c_cond,GLOBUS_NULL);
  if(download_files) {
    /* download */
    std::string rurl;
    std::string lurl;
    std::string lfile;
    std::string ldir;
    c_done=0;
    for(FileData::iterator i=job_files.begin();i!=job_files.end();) {
      if(i->lfn.length() != 0) { lfile=i->lfn; } else { lfile=i->pfn; };
      if( lfile[0] != '/' ) { lfile=std::string(path)+"/"+lfile; }
      else { lfile=std::string(path)+lfile; };
      lurl="file://"+lfile;
      if(i->pfn[0] != '/') { rurl=std::string(session_url)+'/'+i->pfn; }
      else { rurl=std::string(session_url)+i->pfn; };
      ldir=lfile; std::string::size_type n = ldir.rfind('/');
      if(n == std::string::npos) n=0;
      ldir.erase(n,std::string::npos);
      DataPoint source(rurl.c_str());
      DataPoint destination(lurl.c_str());
      DataMove mover;
      if(timeout > 0) {
        mover.set_default_min_speed(0,timeout);
        mover.set_default_max_inactivity_time(timeout);
      };
      if(debug_level > 0) mover.verbose(true);
      FileCache cache;

      mover.secure(false);
      mover.passive(true);  /* make it work even from behind firewall */
      DataMove::result res = mover.Transfer(source,destination,cache,UrlMap());
      fix_file_permissions(lfile,false);
      if(res == DataMove::success) {
        odlog(INFO)<<"Downloaded "<<lfile<<std::endl;
        ++i;
      }
      else {
        odlog(INFO)<<"Failed to download "<<lfile<<std::endl;
        i=job_files.erase(i);
        download_successful = false;
      };
    };
  };
  if(remove_files) {
    /* delete. TODO - through common datamove interface */
    std::string rurl;
    GlobusModuleFTPClient mod_ftp_client;
    if(!mod_ftp_client.active()) {
      odlog(ERROR)<<"FTP CLIENT module activation failed\n";
      res=1; goto exit;
    };
    globus_ftp_client_handle_t c_h;
    globus_ftp_client_handle_init(&c_h,GLOBUS_NULL);
    c_done=0;
    for(FileData::iterator i=job_files.begin();i!=job_files.end();++i) {
      if(i->pfn[0] != '/') { rurl=std::string(session_url)+'/'+i->pfn; }
      else { rurl=std::string(session_url)+i->pfn; };
      odlog(INFO)<<"Deleting "<<rurl<<std::endl;
      if(globus_ftp_client_delete(&c_h,rurl.c_str(),NULL,&deleted_callback,NULL)
                != GLOBUS_SUCCESS) {
        odlog(ERROR) << "Probaly bad url: "<<rurl<< std::endl;
      }
      else {
        globus_mutex_lock(&c_lock);
        while(!c_done) {
          globus_cond_wait(&c_cond,&c_lock);
        };
        c_done=0;
        globus_mutex_unlock(&c_lock);
      };
    };
    globus_ftp_client_handle_destroy(&c_h);
  };
  globus_cond_destroy(&c_cond);
  globus_mutex_destroy(&c_lock);
  if(download_successful) res = 0;
exit:
  free(session_url); if(path_) free(path_);
  return res;
}

/*
int ui_follow(const char* url,const char* filename,unsigned int size,int fd) {
  if(url == NULL) return -1;
  if(url[0] == 0) return -1;
  if(filename == NULL) return -1;
  std::string rurl = url;
  if(rurl[rurl.length()-1] != '/') rurl+="/";
  for(;filename[0] == '/';filename++) { };
  rurl+=filename;
  if(session_url[l-1] == '/') session_url[l-1]=0;

}
*/

int get_url_to_string(const char* url,std::string& content) {
  DataPoint source(url);
  if(!source.meta_resolve(true)) {
    odlog(ERROR)<<"Failed to resolve source: "<<source<<std::endl;
    return -1;
  };
  if(!source.have_locations()) {
    odlog(ERROR)<<"No locations for source found: "<<source<<std::endl;
    return -1;
  };
  DataHandle source_h(&source);
  source_h.additional_checks(false);
  source_h.secure(false);
  source_h.passive(true);
  long int bufsize = 65536;
  int bufnum = 1;
  // DataHandle::analyze_t hint_source;
  // source_h.analyze(hint_read);
  // if(hint_source.bufsize  > bufsize) bufsize=hint_source.bufsize;
  // if(hint_source.bufnum  > bufnum)  bufnum=hint_source.bufnum;
  // bufnum=bufnum*2;
  DataBufferPar buffer(bufsize,bufnum);
  if(!source_h.start_reading(buffer)) {
    odlog(ERROR)<<"Failed to start reading from source: "<<source<<std::endl;
    return -1;
  };
  content="";
  for(;;) {
    int handle;
    unsigned long long int offset;
    unsigned int length;
    odlog(DEBUG)<<"Waiting for buffer"<<std::endl;
    if(!buffer.for_write(handle,length,offset,true)) break;
    content.append((char*)(buffer[handle]),length);
    buffer.is_written(handle);
  };
  odlog(INFO)<<"buffer: read eof : "<<buffer.eof_read()<<std::endl;
  odlog(INFO)<<"buffer: write eof: "<<buffer.eof_write()<<std::endl;
  odlog(INFO)<<"buffer: error    : "<<buffer.error()<<std::endl;
  odlog(DEBUG)<<"Closing read channel"<<std::endl;
  source_h.stop_reading();
  if(buffer.error()) return -1;
  std::string::size_type pos;
  while((pos = content.find("\n")) != std::string::npos)
    if(pos == content.size()-1)
      content.erase(pos);
    else
      content.replace(pos,1," ");
  return 0;
}

int ui_state(const char* url,std::string& state,std::string& failure) {
  std::string u(url);
  if(url == NULL) return -1;
  if(*url == 0) return -1;
  std::string::size_type p = u.length()-1;
  for(;p;--p) if(u[p] != '/') break;
  u.resize(p+1);
  p=u.rfind('/');
  if(p == std::string::npos) return -1;
  u.insert(p,"/info");
  std::string u_;

  u_=u; u_+="/status";
  if(get_url_to_string(u_.c_str(),state) != 0) return -1;

  u_=u; u_+="/failure";
  get_url_to_string(u_.c_str(),failure);

  return 0;
}

