#include "ninfg_arc_is.h"

#include <iostream>
#include <map>

#include <arc/mdsquery.h>
#include <arc/stringconv.h>
#include <arc/xrsl.h>
#include <arc/jobsubmission.h>
#include <arc/standardbrokers.h>
#include <arc/jobftpcontrol.h>
#include <arc/notify.h>

// -------------------------------------------------------------------

std::string NinfGJob::rte_parallel("ENV/PARALLEL");
std::string NinfGJob::rte_mpi("ENV/MPI");
std::string NinfGJob::rte_blacs("");
std::string NinfGJob::default_hostname("");
int NinfGJob::default_port = 2135;
std::string NinfGJob::basename = "Mds-Vo-name=local,o=Grid";


bool NinfGJob::operator==(const std::string& id) {
  return (id == id_);
}

// -------------------------------------------------------------------

NinfGJobs::JobWrap::JobWrap(NinfGJob& job) {
  job_=&job;
  acquired_=false;
  destroyed_=false;
  iterator_count_=0;
}

NinfGJobs::NinfGJobs(void) {
}

NinfGJobs::~NinfGJobs(void) {
}

std::ostream& operator<<(std::ostream& o,const NinfGJob& job) {
  o<<job.ID()<<" ("<<job.Status()<<")";
  return o;
}

void NinfGJobs::Add(NinfGJob& job) {
  lock_.Block();
  JobWrap j(job);
  jobs_.push_back(j);
  lock_.Unblock();
}

void NinfGJobs::Wait(std::list<JobWrap>::iterator i) {
  (i->iterator_count_)++;
  while(i->acquired_) {
    bool res;
    lock_.Unblock();
    lock_.Wait(res);
    lock_.Block();
  };
  (i->iterator_count_)--;
}

NinfGJob* NinfGJobs::Get(const std::string& id) {
  lock_.Block();
  for(std::list<JobWrap>::iterator i = jobs_.begin();
                                     i != jobs_.end();++i) {
    if((*(i->job_)) == id) {
      NinfGJob* j = NULL;
      Wait(i);
      i->acquired_=true;
      j=i->job_;
      lock_.Unblock();
      return j;
    };
  };
  lock_.Unblock();
  return NULL;
}

NinfGJob* NinfGJobs::First(void) {
  lock_.Block();
  NinfGJob* j = NULL;
  if(jobs_.size() > 0) {
    std::list<JobWrap>::iterator i = jobs_.begin();
    Wait(i);
    i->acquired_=true;
    j=i->job_;
  };
  lock_.Unblock();
  return j;
}

NinfGJob* NinfGJobs::Next(NinfGJob& job) {
  lock_.Block();
  NinfGJob* j = NULL;
  for(std::list<JobWrap>::iterator i = jobs_.begin();
                                     i != jobs_.end();++i) {
    if(i->job_ == &job) {
      ++i;
      if(i != jobs_.end()) {
        Wait(i);
        i->acquired_=true;
        j=i->job_;
      };
      break;
    };
  };
  lock_.Unblock();
  return j;
}

void NinfGJobs::Release(NinfGJob& job) {
  lock_.Block();
  for(std::list<JobWrap>::iterator i = jobs_.begin();
                                     i != jobs_.end();++i) {
    if(i->job_ == &job) {
      i->acquired_=false;
      if((i->destroyed_) && (i->iterator_count_ == 0)) { delete (i->job_); jobs_.erase(i); };
      lock_.SignalNonBlock(true);
      break;
    };
  };
  lock_.Unblock();
  return;
}

void NinfGJobs::Destroy(NinfGJob& job) {
  lock_.Block();
  for(std::list<JobWrap>::iterator i = jobs_.begin();
                                     i != jobs_.end();++i) {
    if(i->job_ == &job) {
      i->destroyed_=true;
      i->acquired_=false;
      if(i->iterator_count_ == 0) { delete (i->job_); jobs_.erase(i); };
      break;
    };
  };
  lock_.Unblock();
  return;
}

// -------------------------------------------------------------------

static bool read_attr_line(std::istream& s,std::string& name,std::string& value) {
  if(s.eof()) return false;
  if(s.fail()) return false;
  char buf[1024];
  s.get(buf,sizeof(buf),'\n'); if(s.fail()) s.clear();
  std::cin.ignore(INT_MAX,'\n'); if(s.fail()) s.clear();
  buf[sizeof(buf)-1]=0;
  int l = strlen(buf); if(l>0) if(buf[l-1] == '\r') buf[l-1]=0;
  char* p = strchr(buf,' ');
  if(p == NULL) {
    name=buf; value="";
    return true;
  };
  *p=0; ++p; for(;*p;++p) if((*p) != ' ') break;
  name=buf; value=p;
  return true;
}

static std::string empty_string;

NinfGRequest::Attr::Attr(void) {
}

NinfGRequest::NinfGRequest(void) {
  type_=NONE;
  std::string command;
  std::string arguments;
  if(!read_attr_line(std::cin,command,arguments)) {
    notify(WARNING)<<"Communication channel closed - exiting\n"<<std::endl;
    type_=EXIT;
    return;
  };
  if(command == "JOB_CREATE") {
    Attr attr;
    for(;;) {
      if(!read_attr_line(std::cin,attr.name,attr.value)) return;
      if(attr.name == "JOB_CREATE_END") break;
      attributes_.push_back(attr);
    };
    type_=CREATE; id_=arguments;
    notify(WARNING)<<"JOB_CREATE request with ID "<<id_<<std::endl;
  } else if(command == "JOB_STATUS") {
    type_=STATUS; id_=arguments;
    notify(WARNING)<<"JOB_STATUS request for job "<<id_<<std::endl;
  } else if(command == "JOB_DESTROY") {
    type_=DESTROY; id_=arguments;
    notify(WARNING)<<"JOB_DESTROY request for job "<<id_<<std::endl;
  } else if(command == "EXIT") {
    type_=EXIT; id_=arguments;
    notify(WARNING)<<"EXIT request"<<std::endl;
  };
  return;
}

NinfGRequest::~NinfGRequest(void) {
}

const std::string& NinfGRequest::ID(void) const {
  return id_;
}
 
const std::string& NinfGRequest::Attribute(const std::string& name) const {
  for(std::list<Attr>::const_iterator i = attributes_.begin();
                                          i != attributes_.end();++i) {
    if(i->name == name) return i->value;
  };
  return empty_string;
}

std::list<std::string> NinfGRequest::Attributes(const std::string& name) const {
  std::list<std::string> values;
  for(std::list<Attr>::const_iterator i = attributes_.begin();
                                          i != attributes_.end();++i) {
    if(i->name == name) values.push_back(i->value);
  };
  return values;
}

std::list<std::string> NinfGRequest::AttrNames(void) const {
  std::list<std::string> names;
  std::string last_name;
  for(std::list<Attr>::const_iterator i = attributes_.begin();
                                          i != attributes_.end();++i) {
    if(i->name != last_name) {
      last_name=i->name; names.push_back(last_name);
    };
  };
  return names;
}

NinfGRequest::operator bool(void) {
  return (type_ != NONE);
}

bool NinfGRequest::operator!(void) {
  return (type_ == NONE);
}

std::string NinfGRequest::SType(void) const {
  switch(type_) {
    case CREATE:  return "JOB_CREATE";
    case STATUS:  return "JOB_STATUS";
    case DESTROY: return "JOB_DESTROY";
    case EXIT:    return "EXIT";
    default: break;
  };
  return "";
}

std::ostream& operator<<(std::ostream& o,const NinfGRequest& req) {
  o<<req.SType()<<" "<<req.ID()<<std::endl;
  std::list<std::string> names = req.AttrNames();
  for(std::list<std::string>::iterator name = names.begin();
                                       name != names.end(); ++name) {
    o<<*name<<":"<<std::endl;
    std::list<std::string> values = req.Attributes(*name);
    for(std::list<std::string>::iterator value = values.begin();
                                         value != values.end(); ++value) {
      o<<"\t"<<*value<<std::endl;
    };
  };
  return o;
}

// -------------------------------------------------------------------

NinfGRequests::NinfGRequests(void) {
}

NinfGRequests::~NinfGRequests(void) {
}

void NinfGRequests::Add(NinfGRequest& req) {
  lock_.Block();
  reqs_.push_back(&req);
  lock_.SignalNonBlock(true);
  lock_.Unblock();
}

NinfGRequest* NinfGRequests::Get(const std::string& id) {
  lock_.Block();
  for(std::list<NinfGRequest*>::iterator i = reqs_.begin();
                                     i != reqs_.end();++i) {
    if((*i)->ID() == id) {
      NinfGRequest* r = *i;
      reqs_.erase(i);
      lock_.Unblock();
      return r;
    };
  };
  lock_.Unblock();
  return NULL;
}

NinfGRequest* NinfGRequests::Get(void) {
  lock_.Block();
  if(reqs_.size() > 0) {
    NinfGRequest* r = *(reqs_.begin());
    reqs_.erase(reqs_.begin());
    lock_.Unblock();
    return r;
  };
  bool res;
  lock_.Unblock(); lock_.Wait(res); lock_.Block();
  if(reqs_.size() > 0) {
    NinfGRequest* r = *(reqs_.begin());
    reqs_.erase(reqs_.begin());
    lock_.Unblock();
    return r;
  };
  lock_.Unblock();
  return NULL;
}

// -------------------------------------------------------------------

NinfGResponse::NinfGResponse(bool positive,const std::string& description) {
  char c = positive?'S':'F';
  if(description.empty()) {
    std::cout<<c<<"\r\n";
  } else {
    std::cout<<c<<" "<<description<<"\r\n";
  };
  std::cout.flush();
}
 
NinfGResponse::~NinfGResponse(void) {
}

NinfGResponse::operator bool(void) {
  return (!(std::cout.fail()));
}

bool NinfGResponse::operator!(void) {
  return std::cout.fail();
}

// -------------------------------------------------------------------

NinfGNotification::NinfGNotification(const NinfGJob& job,const std::string& description) {
  if(job.status_.empty()) return;
  if(description.empty()) {
    std::cerr<<"STATUS_NOTIFY "<<job.id_<<" "<<job.status_<<"\r\n";
  } else {
    std::cerr<<"STATUS_NOTIFY "<<job.id_<<" "<<job.status_<<" "<<description<<"\r\n";
  };
  std::cout.flush();
}

NinfGNotification::NinfGNotification(const NinfGRequest& req,const NinfGJob& job,const std::string& description) {
  if(job.id_.empty()) {
    if(description.empty()) {
      std::cerr<<"CREATE_NOTIFY "<<req.ID()<<" F\r\n";
    } else {
      std::cerr<<"CREATE_NOTIFY "<<req.ID()<<" F "<<description<<"\r\n";
    };
  } else {
    std::cerr<<"CREATE_NOTIFY "<<req.ID()<<" S "<<job.id_<<"\r\n";
  };
  std::cout.flush();
}

NinfGNotification::~NinfGNotification(void) {
}

NinfGNotification::operator bool(void) {
  return (!(std::cerr.fail()));
}

bool NinfGNotification::operator!(void) {
  return std::cerr.fail();
}

// -------------------------------------------------------------------

NinfGRequestProcessor::NinfGRequestProcessor(NinfGRequests& reqs,NinfGJobs& jobs):reqs_(reqs),jobs_(jobs) {
  attention();
}

void NinfGRequestProcessor::func(void) {
  for(;;) {
    NinfGRequest* req = new NinfGRequest;
    if(!req) break;
    switch(req->RType()) {
      case NinfGRequest::CREATE: {
        reqs_.Add(*req); req=NULL;
        NinfGResponse resp(true,"");
      }; break;
      case NinfGRequest::STATUS: {
        NinfGJob* j = jobs_.Get(req->ID());
        NinfGResponse resp(j != NULL,j?j->Status():std::string("No such job"));
        if(j) jobs_.Release(*j);
      }; break;
      case NinfGRequest::DESTROY: {
        NinfGJob* j = jobs_.Get(req->ID());
        NinfGResponse resp(j != NULL,j?"":"No such job");
        // Destroy job in list. This will also send notification DONE.
        if(j) jobs_.Destroy(*j);
      }; break;
      case NinfGRequest::EXIT: {
        NinfGResponse resp(true,"");
        for(;;) {
          NinfGJob* j = jobs_.First();
          if(!j) break;
          jobs_.Destroy(*j);
        };
        ::exit(0);
      }; break;
      default: {
        NinfGResponse resp(false,"Unsupported request");
      }; break;
    };
    if(req) delete req;
  };
}

// -------------------------------------------------------------------

NinfGJobSubmitter::NinfGJobSubmitter(NinfGRequests& reqs,NinfGJobs& jobs):reqs_(reqs),jobs_(jobs) {
  attention();
}

void NinfGJobSubmitter::func(void) {
  for(;;) {
    NinfGRequest* req = reqs_.Get();
    if(req == NULL) continue;
    // Must be creation request
    if(req->RType() != NinfGRequest::CREATE) {
      delete req;
      continue;
    };
    // Convert request to job :)
    NinfGJob* job = new NinfGJob(*req);
    NinfGNotification(*req,*job,"");
    if(*job) {
      jobs_.Add(*job);
    } else {
      delete job;
    };
    delete req;
  };
}

// -------------------------------------------------------------------
//  Job submission
// -------------------------------------------------------------------

static bool add_rsl(std::string& rsl,NinfGRequest& req,const char* req_name,const char* rsl_name) {
  std::string value = req.Attribute(req_name);
  if(value.empty()) return false;
  rsl+="(";
  rsl+=rsl_name;
  rsl+="=\"";  
  rsl+=value;
  rsl+="\")\n";
  return true;
}

static bool add_rsl(std::string& rsl,const char* rsl_name,const char* rsl_value) {
  rsl+="(";
  rsl+=rsl_name;
  rsl+="=\"";  
  rsl+=rsl_value;
  rsl+="\")\n";
  return true;
}

static bool add_rsl_single(std::string& rsl,NinfGRequest& req,const char* req_name,const char* rsl_name) {
  std::list<std::string> values = req.Attributes(req_name);
  if(values.empty()) return false;
  std::string rsl_ = "("; rsl_+=rsl_name;  rsl_+="=";
  for(std::list<std::string>::iterator value = values.begin();
                                        value != values.end();++value) {
    rsl_+="\""+(*value)+"\" ";
  };
  rsl_+=")\n";
  rsl+=rsl_;
  return true;
}

/*
static bool add_rsl_multi(std::string& rsl,NinfGRequest& req,const char* req_name,const char* rsl_name) {
  std::list<std::string> values = req.Attributes(req_name);
  if(values.empty()) return false;
  for(std::list<std::string>::iterator value = values.begin();
                                        value != values.end();++value) {
    std::string rsl_ = "("; rsl_+=rsl_name;  rsl_+="=";
    rsl_+="\""+(*value)+"\")\n";
    rsl+=rsl_;
  };
  return true;
}
*/

static bool add_rsl_pairs(std::string& rsl,NinfGRequest& req,const char* req_name
,const char* rsl_name) {
  std::list<std::string> values = req.Attributes(req_name);
  if(values.empty()) return false;
  std::string rsl_ = "("; rsl_+=rsl_name;  rsl_+="=";
  for(std::list<std::string>::iterator value = values.begin();
                                        value != values.end();++value) {
    std::string v = *value;
    std::string::size_type p = v.find("=");
    if(p == std::string::npos) {
      rsl_+="(\""+v+"\" \"\") ";
    } else {
      rsl_+="(\""+v.substr(0,p)+"\" \""+v.substr(p+1)+"\") ";
    };
  };
  rsl_+=")\n";
  rsl+=rsl_;
  return true;
}

NinfGJob::NinfGJob(NinfGRequest& req) {
  // Build RSL
  std::string value;
  std::string hostname = req.Attribute("hostname");
  int port = 0;
  bool staging = false;
  bool output_redirect = false;
  std::multimap<std::string,std::string> inputfiles;
  std::multimap<std::string,std::string> outputfiles;
  int count;
  std::string rsl("&\n");
  value=req.Attribute("port"); port=atoi(value.c_str());
  value=req.Attribute("staging"); staging=(value == "true");
  value=req.Attribute("redirect_enable"); output_redirect=(value == "true");
  add_rsl(rsl,req,"client_name","hostname");
  value=req.Attribute("executable_path"); if(value.empty()) {
    notify(WARNING)<<"Request "<<req.ID()<<" is missing 'executable_path'"<<std::endl;
    return;
  };
  if(staging) {
    std::string local_name = value;
    std::string remote_name = value;
    std::string::size_type p = remote_name.rfind('/');
    if(p != std::string::npos) remote_name.erase(0,p+1);
    inputfiles.insert(std::make_pair(remote_name,local_name));
    add_rsl(rsl,"executable",remote_name.c_str());
    add_rsl(rsl,"stdout",(remote_name+".stdout").c_str());
    add_rsl(rsl,"stderr",(remote_name+".stderr").c_str());
    outputfiles.insert(std::make_pair(remote_name+".stdout",std::string("")));
    outputfiles.insert(std::make_pair(remote_name+".stderr",std::string("")));
  } else {
    add_rsl(rsl,"executable",value.c_str());
    add_rsl(rsl,"stdout","NinfG.stdout");
    add_rsl(rsl,"stderr","NinfG.stderr");
    outputfiles.insert(std::make_pair(std::string("NinfG.stdout"),std::string("")));
    outputfiles.insert(std::make_pair(std::string("NinfG.stderr"),std::string("")));
  };
  value=req.Attribute("count"); if(value.empty()) {
    notify(WARNING)<<"Request "<<req.ID()<<" is missing 'count'"<<std::endl;
    return;
  };
  count=atoi(value.c_str());
  value=req.Attribute("backend"); if(value.empty()) {
    notify(WARNING)<<"Request "<<req.ID()<<" is missing 'backend'"<<std::endl;
    return;
  };
  if(value == "NORMAL") {
    if(count > 1) {
      if(rte_parallel.empty()) {
        notify(WARNING)<<"Request "<<req.ID()<<" asks for parallel submission but no RTE is specified"<<std::endl;
        return;
      };
      add_rsl(rsl,"runtimeenvironment",rte_parallel.c_str());
    };
  } else if(value == "MPI") {
    if(rte_mpi.empty()) {
      notify(WARNING)<<"Request "<<req.ID()<<" asks for MPI but no RTE is specified"<<std::endl;
      return;
    };
    add_rsl(rsl,"runtimeenvironment",rte_mpi.c_str());
  } else if(value == "BLACS") {
    if(rte_blacs.empty()) {
      notify(WARNING)<<"Request "<<req.ID()<<" asks for BLACS but no RTE is specified"<<std::endl;
      return;
    };
    add_rsl(rsl,"runtimeenvironment",rte_blacs.c_str());
  } else {
    notify(WARNING)<<"Request "<<req.ID()<<" asks for unsupported backend "<<value<<std::endl;
    return;
  };
  if(count>0) {
    add_rsl(rsl,"count",tostring<int>(count).c_str());
  };
  add_rsl_single(rsl,req,"argument","arguments");
  add_rsl_pairs(rsl,req,"environment","environment");
  add_rsl(rsl,req,"max_wall_time","walltime");
  if(!add_rsl(rsl,req,"max_cpu_time","cputime"))
    add_rsl(rsl,req,"max_time","cputime");
  add_rsl(rsl,req,"queue_name","queue");
  if(!add_rsl(rsl,req,"max_memory","memory"))
    add_rsl(rsl,req,"min_memory","memory");
  if(!inputfiles.empty()) {
    std::string rsl_ = "(inputfiles=";
    for(std::multimap<std::string,std::string>::iterator i = inputfiles.begin();
                                         i != inputfiles.end();++i) {
      rsl_+="(\""+i->first+"\" \""+i->second+"\") ";
    };
    rsl_+=")\n";
    rsl+=rsl_;
  };
  if(!outputfiles.empty()) {
    std::string rsl_ = "(outputfiles=";
    for(std::multimap<std::string,std::string>::iterator i =outputfiles.begin();
                                         i != outputfiles.end();++i) {
      rsl_+="(\""+i->first+"\" \""+i->second+"\") ";
    };
    rsl_+=")\n";
    rsl+=rsl_;
  };

  // Submit RSL (suboptimal)
  if(hostname.empty()) hostname=default_hostname;
  if(port<=0) port=default_port;
  if(hostname.empty()) return;
  try {
    Xrsl xrsl(rsl);
    URL ldapurl = "ldap://"+hostname+(port>0?(":"+tostring<int>(port)):"")+
                  "/"+basename;
    std::list<Queue> queues = GetQueueInfo(ldapurl);
    if(queues.empty()) return;
    std::list<Target> targets = ConstructTargets(queues,xrsl);
    if(targets.empty()) return;
    PerformStandardBrokering(targets);
    id_ = SubmitJob(xrsl,targets);
    if(id_.empty()) return;
    status_="PENDING";
  } catch (ARCLibError e) {
    notify(WARNING)<<"Job submission for request "<<req.ID()<<" failed due to "<<e.what()<<std::endl;
    return;
  };
  notify(WARNING)<<"Job "<<id_<<" for request "<<req.ID()<<" submitted"<<std::endl;
  return;
}

NinfGJob::~NinfGJob(void) {
  if(id_.empty()) return;
  notify(WARNING)<<"Job "<<id_<<" being destroyed"<<std::endl;
  try {
    JobFTPControl jobc;
    jobc.Cancel(id_,TIMEOUT,false);
    jobc.Clean(id_,TIMEOUT,false);
  } catch (ARCLibError e) {
    notify(WARNING)<<"Job "<<id_<<" destrusction failed (ignored) due to "<<e.what()<<std::endl;
  };
  Status("DONE");
}

void NinfGJob::Status(const std::string& status) {
  if(status == status_) return;
  status_=status;
  notify(WARNING)<<"Job "<<id_<<" status updated to "<<status_<<std::endl;
  NinfGNotification(*this,"");
}

void NinfGJobMonitor::func(void) {
  for(;;) {
    sleep(60);
    std::list<std::string> jobids;
    for(NinfGJob* job = jobs_.First();job;) {
      jobids.push_back(job->ID());
      NinfGJob* job_n = jobs_.Next(*job);
      jobs_.Release(*job); job=job_n;
    };
    if(jobids.empty()) continue;
    std::list<Job> arc_jobs = GetJobInfo(jobids);
    for(std::list<Job>::iterator arc_job = arc_jobs.begin();
                                 arc_job != arc_jobs.end();++arc_job) {
      NinfGJob* job = jobs_.Get(arc_job->id);
      if(!job) continue;
      std::string new_status;
      if(arc_job->status == "INLRMS:R") {
         new_status="ACTIVE";
      } else if((arc_job->status == "EXECUTED") ||
                (arc_job->status == "FINISHING") ||
                (arc_job->status == "FINISHED") ||
                (arc_job->status == "DELETED")) {
         new_status="DONE";
      } else if((arc_job->status == "KILLING") ||
                (arc_job->status == "CANCELLING") ||
                (arc_job->status == "KILLED") ||
                (arc_job->status == "FAILED")) {
         new_status="FAILED";
      };
      if(!new_status.empty()) job->Status(new_status);
      jobs_.Release(*job);
    }; 
  }; 
}

NinfGJobMonitor::NinfGJobMonitor(NinfGJobs& jobs):jobs_(jobs) {
  attention();
}

