#include <string>
#include <list>
#include <pthread.h>
#include <arc/condition.h>


class NinfGNotification;
class NinfGRequest;

class NinfGJob {
 friend class NinfGNotification;
 private:
  std::string id_;
  std::string status_;
 public:
  static std::string rte_parallel;
  static std::string rte_mpi;
  static std::string rte_blacs;
  static std::string default_hostname;
  static int default_port;
  static std::string basename;
  bool operator==(const std::string& id);
  const std::string& ID() const { return id_; };
  const std::string& Status() const { return status_; };
  void Status(const std::string& status);
  NinfGJob(NinfGRequest& req);
  ~NinfGJob(void);
  operator bool(void) { return !(id_.empty()); };
  bool operator!(void) { return id_.empty(); };
};

class NinfGJobs {
 private:
  class JobWrap {
   public:
    NinfGJob* job_;
    bool acquired_;
    bool destroyed_;
    int iterator_count_;
    JobWrap(NinfGJob& job);
  };
  std::list<JobWrap> jobs_;
  Condition<bool> lock_;
  void Wait(std::list<JobWrap>::iterator i);
 public:
  NinfGJobs(void);
  ~NinfGJobs(void);
  void Add(NinfGJob& job);
  NinfGJob* Get(const std::string& id);
  NinfGJob* First(void);
  NinfGJob* Next(NinfGJob& job);
  void Release(NinfGJob& job);
  void Destroy(NinfGJob& job);
};

class NinfGRequest {
 public:
  typedef enum {
    NONE = 0,
    CREATE = 1,
    STATUS= 2,
    DESTROY = 3,
    EXIT = 4
  } Type;
  class Attr {
   public:
    std::string name;
    std::string value;
    Attr(void);
  };
 private:
  Type type_;
  std::string id_;
  std::list<Attr> attributes_;
 public:
  NinfGRequest(void);
  ~NinfGRequest(void);
  const std::string& ID(void) const; 
  const std::string& Attribute(const std::string& name) const;
  std::list<std::string> Attributes(const std::string& name) const;
  std::list<std::string> AttrNames(void) const;
  Type RType(void) const { return type_; };
  std::string SType(void) const;
  operator bool(void);
  bool operator!(void);
};

class NinfGRequests {
 private:
  std::list<NinfGRequest*> reqs_;
  Condition<bool> lock_;
 public:
  NinfGRequests(void);
  ~NinfGRequests(void);
  void Add(NinfGRequest& req);
  NinfGRequest* Get(const std::string& id);
  NinfGRequest* Get(void);
};

class NinfGResponse {
 public:
  NinfGResponse(bool positive,const std::string& description);
  ~NinfGResponse(void);
  operator bool(void);
  bool operator!(void);
};

class NinfGNotification {
 public:
  typedef enum {
    CREATE = 1,
    STATUS = 2
  } Type;
  NinfGNotification(const NinfGJob& job,const std::string& description);
  NinfGNotification(const NinfGRequest& req,const NinfGJob& job,const std::string& description);
  ~NinfGNotification(void);
  operator bool(void);
  bool operator!(void);
};

class NinfGThread {
 protected:
  pthread_t thr_;
  bool stop;
  bool valid;
  Condition<bool> kicker;
  static void* run(void* it) {
    ((NinfGThread*)it)->thr_ = pthread_self();
    ((NinfGThread*)it)->idle(-1);
    ((NinfGThread*)it)->func();
    ((NinfGThread*)it)->exit();
    return NULL;
  };
  virtual void func(void) {
    //std::cerr<<"Pure virtual thread started and finished"<<std::endl;
  };
 public:
  NinfGThread(void):stop(false) {
    valid=false;
    if(pthread_create(&thr_,NULL,&run,(void*)this) != 0) return;
    pthread_detach(thr_);
    valid=true;
  };
  virtual ~NinfGThread(void) {
    exit();
  };
  void idle(int t = 0) {
    if(pthread_equal(pthread_self(),thr_)) {
      if(stop) exit();
      bool res;
      if(t<0) { kicker.Wait(res); } else { kicker.Wait(res,t); };
      if(stop) exit();
    };
  };
  void attention(void) {
    kicker.Signal(true);
  };
  void exit(void) {
    if(pthread_equal(pthread_self(),thr_)) {
      kicker.Block();
      valid=false;
      kicker.SignalNonBlock(true);
      kicker.Unblock();
      pthread_exit(NULL);
    };
    stop=true;
    bool res;
    while(valid) kicker.Wait(res);
  };
};


class NinfGRequestProcessor: public NinfGThread {
 protected:
  NinfGRequests& reqs_;
  NinfGJobs& jobs_;
  virtual void func(void);
 public:
  NinfGRequestProcessor(NinfGRequests& reqs,NinfGJobs& jobs);
};

class NinfGJobSubmitter: public NinfGThread {
 protected:
  NinfGRequests& reqs_;
  NinfGJobs& jobs_;
  virtual void func(void);
 public:
  NinfGJobSubmitter(NinfGRequests& reqs,NinfGJobs& jobs);
};

class NinfGJobMonitor: public NinfGThread {
 private:
  NinfGJobs& jobs_;
  virtual void func(void);
 public:
  NinfGJobMonitor(NinfGJobs& jobs);
};

