#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <arc/certificate.h>
#include <arc/common.h>
#include <arc/mdsparser.h>
#include <arc/notify.h>
#include <arc/resource.h>
#include <arc/stringconv.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


std::map<long,int> parse_user_free_cpus(const std::string&);
std::map<int,int> parse_cpu_map(const std::string&);


void SetJobAttribute(void* job,
                     const std::string& attribute,
                     const std::string& value) {

	if (!job) return;
	Job* current_job = (Job*) job;

	try {
		if (attribute == "nordugrid-job-globalid") {
			current_job->id = value;
		}
		else if (attribute == "nordugrid-job-globalowner") {
			current_job->owner = Certificate::ConvertSN(value);
		}
		else if (attribute == "nordugrid-job-execcluster") {
			current_job->cluster = value;
		}
		else if (attribute == "nordugrid-job-execqueue") {
			current_job->queue = value;
		}
		else if (attribute == "nordugrid-job-stdout") {
			current_job->sstdout = value;
		}
		else if (attribute == "nordugrid-job-stderr") {
			current_job->sstderr = value;
		}
		else if (attribute == "nordugrid-job-stdin") {
			current_job->sstdin = value;
		}
		else if (attribute == "nordugrid-job-reqcput") { // backwards compat
			current_job->requested_cpu_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-job-reqcputime") {
			current_job->requested_cpu_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-job-reqwalltime") {
			current_job->requested_wall_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-job-rerunable") {
			current_job->rerunable = value;
		}
		else if (attribute == "nordugrid-job-status") {
			current_job->status = value;
			// backward compatibility for pre-0.5.14 servers
			std::string::size_type pos = value.find(" at: ");
			if (pos != std::string::npos) {
				current_job->status = value.substr(0, pos);
				current_job->completion_time = value.substr(pos + 5, 15);
			}
			if (current_job->status == "FINISHED" &&
			    !current_job->errors.empty()) {
				if (current_job->errors.find
				    ("User requested to cancel the job") != std::string::npos)
					current_job->status = "KILLED";
				else
					current_job->status = "FAILED";
			}
			// map some obsolete states to new ones
			if (current_job->status == "PENDING: ACCEPTED" ||
			    current_job->status == "PENDING:ACCEPTED")
				current_job->status = "ACCEPTED";
			if (current_job->status == "PENDING: PREPARING" ||
			    current_job->status == "PENDING:PREPARING")
				current_job->status = "PREPARED";
			if (current_job->status == "PENDING: INLRMS" ||
			    current_job->status == "PENDING:INLRMS")
				current_job->status = "EXECUTED";
			if (current_job->status == "SUBMIT")
				current_job->status = "SUBMITTING";
			if (current_job->status == "CANCELING" ||
			    current_job->status == "CANCELLING")
				current_job->status = "KILLING";
			if (current_job->status == "CANCELED" ||
			    current_job->status == "CANCELLED")
				current_job->status = "KILLED";
			if (current_job->status.substr(0, 7) == "INLRMS:") {
				if (current_job->status[7] == ' ')
					current_job->status.erase(7, 1);
				current_job->status = current_job->status.substr(0, 8);
			}
		}
		else if (attribute == "nordugrid-job-queuerank") {
			current_job->queue_rank = stringtoi(value);
		}
		else if (attribute == "nordugrid-job-lrmscomment") {
			current_job->comment = value; // backwards compatibility
		}
		else if (attribute == "nordugrid-job-comment") {
			current_job->comment = value;
		}
		else if (attribute == "nordugrid-job-submissionui") {
			current_job->submission_ui = value;
		}
		else if (attribute == "nordugrid-job-submissiontime") {
			current_job->submission_time = value;
		}
		else if (attribute == "nordugrid-job-usedcputime") {
			current_job->used_cpu_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-job-usedwalltime") {
			current_job->used_wall_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-job-sessiondirerasetime") {
			current_job->erase_time = value;
		}
		else if (attribute == "nordugrid-job-usedmem") {
			current_job->used_memory = stringtoi(value);
		}
		else if (attribute == "nordugrid-job-errors") {
			current_job->errors = value;
			// backward compatibility for pre-0.5.14 servers
			if (current_job->status == "FINISHED") {
				if (value.find("User requested to cancel the job") !=
				    std::string::npos)
					current_job->status = "KILLED";
				else
					current_job->status = "FAILED";
			}
			std::string::size_type pos = value.find("Job exit code is ");
			if (pos != std::string::npos) {
				pos += 17;
				std::string::size_type pos2 = value.find(" != 0", pos);
				if (pos2 != std::string::npos) {
					current_job->exitcode =
						stringtoi(value.substr(pos, pos2 - pos));
				}
			}
		}
		else if (attribute == "nordugrid-job-exitcode") {
			current_job->exitcode = stringtoi(value);
		}
		else if (attribute == "nordugrid-job-jobname") {
			current_job->job_name = value;
		}
		else if (attribute == "nordugrid-job-runtimeenvironment") {
			current_job->runtime_environments.push_back(value);
		}
		else if (attribute == "nordugrid-job-cpucount") {
			current_job->cpu_count = stringtoi(value);
		}
		else if (attribute == "nordugrid-job-executionnodes") {
			current_job->execution_nodes.push_back(value);
		}
		else if (attribute == "nordugrid-job-gmlog") {
			current_job->gmlog = value;
		}
		else if (attribute == "nordugrid-job-clientsoftware") {
			current_job->client_software = value;
		}
		else if (attribute == "nordugrid-job-proxyexpirationtime") {
			current_job->proxy_expire_time = value;
		}
		else if (attribute == "nordugrid-job-completiontime") {
			current_job->completion_time = value;
		}
		else if (attribute == "mds-validfrom") {
			current_job->mds_validfrom = value;
		}
		else if (attribute == "mds-validto") {
			current_job->mds_validto = value;
		}
		else {
			notify(DEBUG) << _("Unhandled job attribute") << ": " << attribute
			              << std::endl;
		}
	}
	catch (StringConvError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
	catch (TimeError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
}


void SetUserAttribute(void* user,
                      const std::string& attribute,
                      const std::string& value) {

	if (!user) return;
	User* current_user = (User*) user;

	try {
		if (attribute == "nordugrid-authuser-name") {
			current_user->name = Certificate::ConvertSN(value);
		}
		else if (attribute == "nordugrid-authuser-sn") {
			current_user->subject_name = Certificate::ConvertSN(value);
		}
		else if (attribute == "nordugrid-authuser-freecpus") {
			current_user->free_cpus = parse_user_free_cpus(value);
		}
		else if (attribute == "nordugrid-authuser-diskspace") {
			current_user->free_diskspace = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-authuser-queuelength") {
			current_user->queue_length = stringtoi(value);
		}
		else if (attribute == "mds-validfrom") {
			current_user->mds_validfrom = value;
		}
		else if (attribute == "mds-validto") {
			current_user->mds_validto = value;
		}
		else {
			notify(DEBUG) << _("Unhandled user attribute") << ": "
			              << attribute << std::endl;
		}
	}
	catch (StringConvError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
	catch (TimeError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
}


void SetQueueAttribute(void* queue,
                       const std::string& attribute,
                       const std::string& value) {

	if (!queue) return;
	Queue* current_queue = (Queue*) queue;

	try {
		if (attribute == "nordugrid-queue-name") {
			current_queue->name = value;
		}
		else if (attribute == "nordugrid-queue-status") {
			current_queue->status = value;
		}
		else if (attribute == "nordugrid-queue-running") {
			current_queue->running = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-queued") {
			current_queue->queued = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-maxrunning") {
			current_queue->max_running = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-maxqueuable") {
			current_queue->max_queuable = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-maxuserrun") {
			current_queue->max_user_run = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-maxcputime") {
			current_queue->max_cpu_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-queue-mincputime") {
			current_queue->min_cpu_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-queue-defaultcputime") {
			current_queue->default_cpu_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-queue-maxwalltime") {
			current_queue->max_wall_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-queue-minwalltime") {
			current_queue->min_wall_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-queue-defaultwalltime") {
			current_queue->default_wall_time = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-queue-schedulingpolicy") {
			current_queue->scheduling_policy = value;
		}
		else if (attribute == "nordugrid-queue-totalcpus") {
			current_queue->total_cpus = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-nodecpu") {
			current_queue->node_cpu = value;
			std::string::size_type pos = value.find(" MHz");
			if (pos != std::string::npos) {
				std::string::size_type numpos = value.rfind(' ', pos - 1) + 1;
				current_queue->benchmarks["gridtime"] =
					current_queue->cpu_freq =
					stringtof(value.substr(numpos, pos - numpos));
			}
		}
		else if (attribute == "nordugrid-queue-nodememory") {
			std::string::size_type pos = value.find(" MB");
			current_queue->node_memory = stringtoi(value.substr(0, pos));
		}
		else if (attribute == "nordugrid-queue-architecture") {
			current_queue->architecture = value;
		}
		else if (attribute == "nordugrid-queue-opsys") {
			current_queue->operating_systems.push_back(value);
		}
		else if (attribute == "nordugrid-queue-gridrunning") {
			current_queue->grid_running = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-gridqueued") {
			current_queue->grid_queued = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-localqueued") {
			current_queue->local_queued = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-prelrmsqueued") {
			current_queue->prelrms_queued = stringtoi(value);
		}
		else if (attribute == "nordugrid-queue-homogeneity") {
			if (value == "false" || value == "False" || value == "FALSE") {
				current_queue->homogeneity = false;
			}
			else if (value == "true" || value == "True" || value == "TRUE") {
				current_queue->homogeneity = true;
			}
			else {
				notify(DEBUG) << _("Bad value for homogeneity") << ": "
				              << value << std::endl;
			}
		}
		else if (attribute == "nordugrid-queue-comment") {
			current_queue->comment = value;
		}
		else if (attribute == "nordugrid-queue-benchmark") {
			std::string::size_type pos = value.find(" @ ");
			if (pos != std::string::npos) {
				std::string type = value.substr(0, pos);
				float speed = stringtof(value.substr(pos + 3));
				if (speed != 0.) current_queue->benchmarks[type] = speed;
			} else {
				notify(DEBUG) << _("Illegal benchmark") << ": " << value 
				              << std::endl;
			}
		}
		else if (attribute == "nordugrid-queue-runtimeenvironment") {
			current_queue->runtime_environments.push_back(value);
		}
		else if (attribute == "nordugrid-queue-middleware") {
			current_queue->middlewares.push_back(value);
		}
		else if (attribute == "mds-validfrom") {
			current_queue->mds_validfrom = value;
		}
		else if (attribute == "mds-validto") {
			current_queue->mds_validto = value;
		}
		else {
			notify(DEBUG) << _("Unhandled queue attribute") << ": "
			              << attribute << std::endl;
		}
	}
	catch (StringConvError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
	catch (TimeError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
}


void SetClusterAttribute(void* cluster,
                         const std::string& attribute,
                         const std::string& value) {

	if (!cluster) return;
	Cluster* current_cluster = (Cluster*) cluster;

	try {
		if (attribute == "nordugrid-cluster-name") {
			current_cluster->hostname = value;
		}
		else if (attribute == "nordugrid-cluster-aliasname") {
			current_cluster->alias = value;
		}
		else if (attribute == "nordugrid-cluster-owner") {
			current_cluster->owners.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-contactstring") {
			current_cluster->contact = value;
		}
		else if (attribute == "nordugrid-cluster-interactive-contactstring") {
			current_cluster->interactive_contact = value;
		}
		else if (attribute == "nordugrid-cluster-support") {
			current_cluster->support.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-lrms-type") {
			current_cluster->lrms_type = value;
		}
		else if (attribute == "nordugrid-cluster-lrms-version") {
			current_cluster->lrms_version = value;
		}
		else if (attribute == "nordugrid-cluster-lrms-config") {
			current_cluster->lrms_config = value;
		}
		else if (attribute == "nordugrid-cluster-architecture") {
			current_cluster->architecture = value;
		}
		else if (attribute == "nordugrid-cluster-opsys") {
			current_cluster->operating_systems.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-homogeneity") {
			if (value == "false" || value == "False" || value == "FALSE") {
				current_cluster->homogeneity = false;
			}
			else if (value == "true" || value == "True" || value == "TRUE") {
				current_cluster->homogeneity = true;
			}
			else {
				notify(DEBUG) << _("Bad value for homogeneity") << ": "
				              << value << std::endl;
			}
		}
		else if (attribute == "nordugrid-cluster-nodecpu") {
			current_cluster->node_cpu = value;
			std::string::size_type pos = value.find(" MHz");
			if (pos != std::string::npos) {
				std::string::size_type numpos = value.rfind(' ', pos - 1) + 1;
				current_cluster->benchmarks["gridtime"] =
					current_cluster->cpu_freq =
					stringtof(value.substr(numpos, pos - numpos));
			}
		}
		else if (attribute == "nordugrid-cluster-nodememory") {
			std::string::size_type pos = value.find(" MB");
			current_cluster->node_memory = stringtoi(value.substr(0, pos));
		}
		else if (attribute == "nordugrid-cluster-totalcpus") {
			current_cluster->total_cpus = stringtoi(value);
		}
		else if (attribute == "nordugrid-cluster-cpudistribution") {
			current_cluster->cpu_distribution = parse_cpu_map(value);
		}
		else if (attribute == "nordugrid-cluster-sessiondir-free") {
			current_cluster->session_dir_free = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-cluster-sessiondir-total") {
			current_cluster->session_dir_total = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-cluster-sessiondir-lifetime") {
			current_cluster->session_dir_lifetime = stringtol(value) * 60;
		}
		else if (attribute == "nordugrid-cluster-cache-free") {
			current_cluster->cache_free = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-cluster-cache-total") {
			current_cluster->cache_total = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-cluster-runtimeenvironment") {
			current_cluster->runtime_environments.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-middleware") {
			current_cluster->middlewares.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-localse") {
			current_cluster->local_se.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-totaljobs") {
			current_cluster->total_jobs = stringtoi(value);
		}
		else if (attribute == "nordugrid-cluster-usedcpus") {
			current_cluster->used_cpus = stringtoi(value);
		}
		else if (attribute == "nordugrid-cluster-queuedjobs") {
			current_cluster->queued_jobs = stringtoi(value);
		}
		else if (attribute == "nordugrid-cluster-prelrmsqueued") {
			current_cluster->prelrms_queued = stringtoi(value);
		}
		else if (attribute == "nordugrid-cluster-location") {
			current_cluster->location = value;
		}
		else if (attribute == "nordugrid-cluster-issuerca") {
			current_cluster->issuer_ca = Certificate::ConvertSN(value);
		}
		else if (attribute == "nordugrid-cluster-issuerca-hash") {
			current_cluster->issuer_ca_hash = value;
		}
		else if (attribute == "nordugrid-cluster-trustedca") {
			current_cluster->trusted_ca.push_back(Certificate::ConvertSN(value));
		}
		else if (attribute == "nordugrid-cluster-acl") {
			current_cluster->cluster_acl.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-nodeaccess") {
			current_cluster->node_access.push_back(value);
		}
		else if (attribute == "nordugrid-cluster-comment") {
			current_cluster->comment = value;
		}
		else if (attribute == "nordugrid-cluster-benchmark") {
			std::string::size_type pos = value.find(" @ ");
			if (pos != std::string::npos) {
				std::string type = value.substr(0, pos);
				float speed = stringtof(value.substr(pos + 3));
				if (speed != 0.) current_cluster->benchmarks[type] = speed;
			}
			else {
				notify(DEBUG) << _("Illegal benchmark") << ": " << value
				              << std::endl;
			}
		}
		else if (attribute == "nordugrid-cluster-opsysdistribution") {
			// obsolete attribute
			// treated as nordugrid-cluster-opsys for backward compatibility
			current_cluster->operating_systems.push_back(value);
		}
		else if (attribute == "mds-validfrom") {
			current_cluster->mds_validfrom = value;
		}
		else if (attribute == "mds-validto") {
			current_cluster->mds_validto = value;
		}
		else {
			notify(DEBUG) << _("Unhandled cluster attribute") << ": "
			              << attribute << std::endl;
		}
	}
	catch (StringConvError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
	catch (TimeError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
}


void SetStorageElementAttribute(void* se,
                                const std::string& attribute,
                                const std::string& value) {

	if (!se) return;
	StorageElement* current_se = (StorageElement*) se;

	try {
		if (attribute == "nordugrid-se-name") {
			current_se->name = value;
		}
		else if (attribute == "nordugrid-se-aliasname") {
			current_se->alias = value;
		}
		else if (attribute == "nordugrid-se-type") {
			current_se->type = value;
		}
		else if (attribute == "nordugrid-se-freespace") {
			current_se->free_space = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-se-baseurl") {
			current_se->url = value; // will be deprecated soon
		}
		else if (attribute == "nordugrid-se-url") {
			current_se->url = value;
		}
		else if (attribute == "nordugrid-se-authuser") {
			current_se->auth_users.push_back(value);
		}
		else if (attribute == "nordugrid-se-location") {
			current_se->location = value;
		}
		else if (attribute == "nordugrid-se-owner") {
			current_se->owners.push_back(value);
		}
		else if (attribute == "nordugrid-se-issuerca") {
			current_se->ca_issuer = Certificate::ConvertSN(value);
		}
		else if (attribute == "nordugrid-se-issuerca-hash") {
			current_se->ca_issuer_hash = value;
		}
		else if (attribute == "nordugrid-se-trustedca") {
			current_se->trusted_ca.push_back(Certificate::ConvertSN(value));
		}
		else if (attribute == "nordugrid-se-acl") {
			current_se->se_acl.push_back(value);
		}
		else if (attribute == "nordugrid-se-accesscontrol") {
			current_se->access_control = value;
		}
		else if (attribute == "nordugrid-se-totalspace") {
			current_se->total_space = stringtoll(value) * 1024 * 1024;
		}
		else if (attribute == "nordugrid-se-middleware") {
			current_se->middlewares.push_back(value);
		}
		else if (attribute == "nordugrid-se-comment") {
			current_se->comment = value;
		}
		else if (attribute == "mds-validfrom") {
			current_se->mds_validfrom = value;
		}
		else if (attribute == "mds-validto") {
			current_se->mds_validto = value;
		}
		else {
			notify(DEBUG) << _("Unhandled storage element attribute") << ": "
			              << attribute << std::endl;
		}
	}
	catch (StringConvError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
	catch (TimeError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
}


void SetReplicaCatalogAttribute(void* rc,
                                const std::string& attribute,
                                const std::string& value) {

	if (!rc) return;
	ReplicaCatalog* current_rc = (ReplicaCatalog*) rc;

	try {
		if (attribute == "nordugrid-rc-name") {
			current_rc->name = value;
		}
		else if (attribute == "nordugrid-rc-aliasname") {
			current_rc->alias = value;
		}
		else if (attribute == "nordugrid-rc-baseurl") {
			current_rc->base_url = value;
		}
		else if (attribute == "nordugrid-rc-authuser") {
			current_rc->auth_users.push_back(value);
		}
		else if (attribute == "nordugrid-rc-location") {
			current_rc->location = value;
		}
		else if (attribute == "nordugrid-rc-owner") {
			current_rc->owners.push_back(value);
		}
		else if (attribute == "nordugrid-rc-issuerca") {
			current_rc->ca_issuer = Certificate::ConvertSN(value);
		}
		else if (attribute == "nordugrid-rc-issuerca-hash") {
			current_rc->ca_issuer_hash = value;
		}
		else if (attribute == "mds-validfrom") {
			current_rc->mds_validfrom = value;
		}
		else if (attribute == "mds-validto") {
			current_rc->mds_validto = value;
		}
		else {
			notify(DEBUG) << _("Unhandled replica catalog attribute") << ": "
			              << attribute << std::endl;
		}
	}
	catch (StringConvError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
	catch (TimeError e) {
		notify(DEBUG) << attribute << ": " << e.what() << std::endl;
	}
}


std::map<long,int> parse_user_free_cpus(const std::string& value) {

	std::map<long,int> user_free_cpus;

	if (value.empty())
		return user_free_cpus;

	std::string::size_type pos = 0;

	do {
		std::string::size_type spacepos = value.find(' ', pos);
		std::string entry;
		if (spacepos == std::string::npos)
			entry = value.substr(pos);
		else
			entry = value.substr(pos, spacepos - pos);
		int num_cpus;
		long seconds;
		std::string::size_type colonpos = entry.find(':');
		if (colonpos == std::string::npos) {
			num_cpus = stringtoi(entry);
			seconds = LONG_MAX;
		}
		else {
			num_cpus = stringtoi(entry.substr(0, colonpos));
			seconds = stringtol(entry.substr(colonpos + 1)) * 60;
		}
		user_free_cpus[seconds] = num_cpus;
		pos = spacepos;
		if (pos != std::string::npos) pos++;
	} while (pos != std::string::npos);

	return user_free_cpus;
}


std::map<int,int> parse_cpu_map(const std::string& value) {

	std::map<int,int> cpu_distribution;

	if (value.empty())
		return cpu_distribution;

	std::string::size_type pos = 0;

	do {
		std::string::size_type spacepos = value.find(' ', pos);
		std::string entry;
		if (spacepos == std::string::npos)
			entry = value.substr(pos);
		else
			entry = value.substr(pos, spacepos - pos);
		std::string::size_type colonpos = entry.find ("cpu:");
		if (colonpos != std::string::npos) {
			int ncpu = stringtoi(entry.substr(0, colonpos));
			int count = stringtoi(entry.substr(colonpos + 4));
			cpu_distribution[ncpu] = count;
		}
		else {
			notify(DEBUG) << _("Bad format for CPU distribution")
			              << ": " << entry << std::endl;
		}
		pos = spacepos;
		if (pos != std::string::npos) pos++;
	} while (pos != std::string::npos);

	return cpu_distribution;
}
