#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <algorithm>

#include <arc/certificate.h>
#include <arc/common.h>
#include <arc/ldapquery.h>
#include <arc/mdsdiscovery.h>
#include <arc/mdsparser.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/stringconv.h>
#include <arc/url.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif

#define MDS_DEFAULT_PORT 2135

std::list<Queue> ExtractQueueInfo(std::list<Cluster> clusterinfo);

class MDSQueryCallback {

	public:
		MDSQueryCallback();

		void SetJobList(std::list<std::string> joblist);
		std::list<Job> GetJobList();

		void SetClusterList(std::list<URL> clusterlist);
		std::list<Cluster> GetClusterList();

		std::list<StorageElement> GetSEList();
		std::list<ReplicaCatalog> GetRCList();

		static void Callback(const std::string& attribute,
		                     const std::string& value,
		                     void* ref);
		void ProcessCallback(const std::string& attribute,
		                     const std::string& value);

	private:
		/** callbackparser is a pointer to a member-function that
		 *  parses the callback. */
		void (*callbackparser)(void*, const std::string&, const std::string&);

		std::list<StorageElement> ses;
		std::list<ReplicaCatalog> rcs;
		std::list<Cluster> clusters;
		std::list<Queue> queues;
		std::list<User> users;
		std::list<Job> jobs;

		StorageElement* current_se;
		ReplicaCatalog* current_rc;
		Cluster* current_cluster;
		Queue* current_queue;
		User* current_user;
		Job* current_job;

		void* current_object;
		/** This parameter specifies if information should be gathered about
		 *  all jobs encountered in the query of only those whose jobid is
		 *  already specified in the joblist above.
		 *  By default getalljobs is true but is set false if a joblist is
		 *  specified through the SetJobList call
		 */
		bool addinfo;
};

MDSQueryCallback::MDSQueryCallback() {
	current_job = NULL;
	current_user = NULL;
	current_queue = NULL;
	current_cluster = NULL;
	current_rc = NULL;
	current_se = NULL;

	current_object = NULL;
	addinfo = true;
}

void MDSQueryCallback::SetJobList(std::list<std::string> jobslist) {
	jobs.clear();
	std::list<std::string>::iterator it;
	for (it = jobslist.begin(); it != jobslist.end(); it++) {
		Job job;
		job.id = *it;
		jobs.push_back(job);
	}

	if (!jobslist.empty())
		addinfo = false; // Only get information about specified jobs
}

std::list<Job> MDSQueryCallback::GetJobList() {
	return jobs;
}

void MDSQueryCallback::SetClusterList(std::list<URL> clusterlist) {
	clusters.clear();
	std::list<URL>::iterator it;
	for (it = clusterlist.begin(); it != clusterlist.end(); it++) {
		Cluster cluster;
		cluster.hostname = it->Host();
		clusters.push_back(cluster);
	}

	if (!clusterlist.empty())
		addinfo = false; // Only get information about specified clusters
}

std::list<Cluster> MDSQueryCallback::GetClusterList() {
	return clusters;
}

std::list<StorageElement> MDSQueryCallback::GetSEList() {
	return ses;
}

std::list<ReplicaCatalog> MDSQueryCallback::GetRCList() {
	return rcs;
}

void MDSQueryCallback::Callback(const std::string& attribute,
                                const std::string& value,
                                void* ref) {
	notify(VERBOSE) << "MDSQueryCallback: " << attribute << "  " << value << std::endl;
	MDSQueryCallback* object = (MDSQueryCallback*)ref;
	object->ProcessCallback(attribute, value);
}

void MDSQueryCallback::ProcessCallback(const std::string& attribute,
                                       const std::string& value) {

	std::string attr(attribute);
	std::string val(value);
	transform(attr.begin(), attr.end(), attr.begin(), tolower);
	transform(value.begin(), value.end(), val.begin(), tolower);

	if (attr=="dn" && val.substr(0, 13)=="nordugrid-job") {
		current_job = NULL;
		callbackparser = &SetJobAttribute;

		if (addinfo==false) {
			std::string::size_type pos = val.find("nordugrid-job-globalid=");
			if (pos==std::string::npos)
				throw MDSQueryError(_("nordugrid-job-globalid not found in dn"));
			pos += 23;
			std::string jobid = val.substr(pos, val.find(",",pos+1)-pos);

			std::list<Job>::iterator it;
			for (it = jobs.begin(); it != jobs.end(); it++) {
				if ((*it).id==jobid) {
					current_job = &(*it);
					break;
				}
			}

			current_object = (void*)current_job;
			return;

		} else {

			Job newjob;
			if (current_queue!=NULL) {
				current_queue->jobs.push_back(newjob);
				current_job = &current_queue->jobs.back();
			} else {
				jobs.push_back(newjob);
				current_job = &jobs.back();
			}

			current_job->id = value;
			current_object = (void*)current_job;
			return;
		}
	}

	if (attr=="dn" && val.substr(0, 17)=="nordugrid-cluster") {
		current_cluster = NULL;
		callbackparser = &SetClusterAttribute;

		if (addinfo==false) {
			std::string::size_type pos = val.find("nordugrid-cluster-name=");
			if (pos==std::string::npos)
				throw MDSQueryError(_("nordugrid-cluster-name not found in dn"));
			pos += 23;
			std::string name = val.substr(pos, val.find(",",pos+1)-pos);

			std::list<Cluster>::iterator it;
			for (it = clusters.begin(); it != clusters.end(); it++) {
				if ((*it).hostname==name) {
					current_cluster = &(*it);
					break;
				}
			}
			current_object = (void*)current_cluster;
			return;
		} else {

			Cluster newcluster;

			clusters.push_back(newcluster);
			current_cluster = &clusters.back();
			current_cluster->hostname = value;

			current_object = (void*)current_cluster;
			return;
		}
	}

	if (attr=="dn" && val.substr(0, 15)=="nordugrid-queue") {
		Queue newqueue;
		callbackparser = &SetQueueAttribute;

		if (current_cluster!=NULL) {
			current_cluster->queues.push_back(newqueue);
			current_queue = &current_cluster->queues.back();
		} else {
			queues.push_back(newqueue);
			current_queue = &queues.back();
		}

		current_queue->name = value;
		current_object = (void*)current_queue;
		return;
	}

	if (attr=="dn" && val.substr(0, 18)=="nordugrid-authuser") {
		User newuser;
		callbackparser = &SetUserAttribute;

		if (current_queue!=NULL) {
			current_queue->users.push_back(newuser);
			current_user = &current_queue->users.back();
		} else {
			users.push_back(newuser);
			current_user = &users.back();
		}

		current_user->name = value;
		current_object = (void*)current_user;
		return;
	}

	if (attr=="dn" && val.substr(0, 12)=="nordugrid-se") {
		current_se = NULL;
		callbackparser = &SetStorageElementAttribute;

		StorageElement newse;
		ses.push_back(newse);
		current_se = &ses.back();
		current_se->name = value;

		current_object = (void*)current_se;
		return;
	}

	if (attr=="dn" && val.substr(0, 12)=="nordugrid-rc") {
		current_rc = NULL;
		callbackparser = &SetReplicaCatalogAttribute;

		ReplicaCatalog newrc;
		rcs.push_back(newrc);
		current_rc = &rcs.back();
		current_rc->name = value;

		current_object = (void*)current_rc;
		return;
	}

	if (attr=="objectclass") return;

	if (callbackparser==NULL)
		throw MDSQueryError(_("No parser to parse callback defined"));

	(*callbackparser)(current_object, attr, value);
}


void FilterSubstitution(std::string& filter) {

	std::string sn1;
	std::string sn2;
	try {
		sn1 = GetEffectiveSN(Certificate::LDAP1);
		sn2 = GetEffectiveSN(Certificate::LDAP2);
	} catch(CertificateError e) {
		notify(DEBUG) << _("Warning") << ": " << e.what() << std::endl;
	}

	std::string::size_type pos = filter.find("%s");
	if (pos == std::string::npos) return;
	if (sn1 == sn2) {
		filter.replace(pos, 2, sn1);
	} else {
		std::string::size_type posleft = filter.rfind('(', pos);
		std::string::size_type posright = filter.find(')', pos) + 1;
		filter.insert(posright, ")");
		filter.insert(posleft, filter, posleft, posright - posleft);
		filter.insert(posleft, "(|");
		pos = filter.find("%s");
		filter.replace(pos, 2, sn1);
		pos = filter.find("%s");
		filter.replace(pos, 2, sn2);
	}
}


URL JobIDToClusterURL(const std::string& jobid) {
	unsigned int port = MDS_DEFAULT_PORT;  // ugly hardcoding
	std::string basedn = MDS_BASEDN;

	URL joburl(jobid);
	if (joburl.Protocol()!="gsiftp")
		throw MDSQueryError(_("Invalid jobid") + std::string(": ") + jobid);

	URL newurl = "ldap://" + joburl.Host() + ":" + tostring(port) 
	           + "/" + basedn;

    return newurl;
}


std::list<URL> JobIDsToClusterURLs(const std::list<std::string>& jobids) {

	std::list<URL> hosts;

	std::list<std::string>::const_iterator it;
	for (it = jobids.begin(); it != jobids.end(); it++) {
		URL newurl = JobIDToClusterURL(*it);
		if (find(hosts.begin(), hosts.end(), newurl)==hosts.end())
			hosts.push_back(newurl);
	}

	return hosts;
}


std::list<Job> GetJobInfo(std::list<std::string> jobids,
                          std::string filter,
                          const bool& anonymous,
                          const std::string& usersn,
                          unsigned int timeout) {

	FilterSubstitution(filter);

	std::list<std::string>::iterator it;
	std::list<URL> hosts = JobIDsToClusterURLs(jobids);

	std::vector<std::string> attrs;

	MDSQueryCallback JobInfo;
	JobInfo.SetJobList(jobids);

	ParallelLdapQueries pldapq(hosts,
	                           filter,
	                           attrs,
	                           &MDSQueryCallback::Callback,
	                           (void*)&JobInfo,
	                           LdapQuery::subtree,
	                           usersn,
	                           anonymous,
	                           timeout);
	pldapq.Query();

	return JobInfo.GetJobList();
}


Job GetJobInfo(std::string job,
               std::string filter,
               const bool& anonymous,
               const std::string& usersn,
               unsigned int timeout) {

	std::list<std::string> jobids;
	jobids.push_back(job);

	std::list<Job> joblist = GetJobInfo(jobids,
	                                    filter,
	                                    anonymous,
	                                    usersn,
	                                    timeout);
	return *(joblist.begin());
}


std::list<Job> GetAllJobs(std::list<URL> clusters,
                          bool anonymous,
                          const std::string& usersn,
                          unsigned int timeout) {

	std::string filter = MDS_FILTER_JOBINFO;
	FilterSubstitution(filter);

	if (clusters.empty()) clusters = GetResources();
	std::list<Job> jobs;

	std::vector<std::string> attrs;
	MDSQueryCallback JobInfo;

	ParallelLdapQueries pldapq(clusters,
	                           filter,
	                           attrs,
	                           &MDSQueryCallback::Callback,
	                           (void*)&JobInfo,
	                           LdapQuery::subtree,
	                           usersn,
	                           anonymous,
	                           timeout);
	pldapq.Query();

	return JobInfo.GetJobList();
}


std::list<Job> GetAllJobs(const URL& cluster,
                          bool anonymous,
                          const std::string& usersn,
                          unsigned int timeout) {

	std::list<URL> clusters;
	clusters.push_back(cluster);
	return GetAllJobs(clusters, anonymous, usersn, timeout);
}


std::list<Job> GetClusterJobs(std::list<URL> clusters,
                              bool anonymous,
                              const std::string& usersn,
                              unsigned int timeout) {

	std::string filter = MDS_FILTER_CLUSTER_JOBS;
	FilterSubstitution(filter);

	if (clusters.empty()) clusters = GetResources();
	std::list<Job> jobs;

	std::vector<std::string> attrs;
	MDSQueryCallback JobInfo;

	ParallelLdapQueries pldapq(clusters,
	                           filter,
	                           attrs,
	                           &MDSQueryCallback::Callback,
	                           (void*)&JobInfo,
	                           LdapQuery::subtree,
	                           usersn,
	                           anonymous,
	                           timeout);
	pldapq.Query();

	return JobInfo.GetJobList();
}


std::list<Job> GetClusterJobs(const URL& cluster,
                              bool anonymous,
                              const std::string& usersn,
                              unsigned int timeout) {

	std::list<URL> clusters;
	clusters.push_back(cluster);
	return GetClusterJobs(clusters, anonymous, usersn, timeout);
}


Cluster GetClusterInfo(const URL& cluster,
                       std::string filter,
                       const bool& anonymous,
                       const std::string& usersn,
                       unsigned int timeout) {

	std::list<URL> clusters;
	clusters.push_back(cluster);

	std::list<Cluster> clusterinfo = GetClusterInfo(clusters,
	                                                filter,
	                                                anonymous,
	                                                usersn,
	                                                timeout);

	if (clusterinfo.empty())
		throw MDSQueryError(_("No information returned from cluster"));

	return clusterinfo.front();
}


std::list<Cluster> GetClusterInfo(std::list<URL> clusters,
                                  std::string filter,
                                  const bool& anonymous,
                                  const std::string& usersn,
                                  unsigned int timeout) {

	FilterSubstitution(filter);

	if (clusters.empty()) clusters = GetResources();

	std::vector<std::string> attrs;
	MDSQueryCallback ClusterInfo;
	ClusterInfo.SetClusterList(clusters);

	ParallelLdapQueries pldapq(clusters,
	                           filter,
	                           attrs,
	                           &MDSQueryCallback::Callback,
	                           (void*)&ClusterInfo,
	                           LdapQuery::subtree,
	                           usersn,
	                           anonymous,
	                           timeout);
	pldapq.Query();

	return ClusterInfo.GetClusterList();
}


std::list<Queue> GetQueueInfo(const URL& cluster,
                              std::string filter,
                              const bool& anonymous,
                              const std::string& usersn,
                              unsigned int timeout) {

	std::list<URL> clusters;
	clusters.push_back(cluster);

	std::list<Cluster> clusterinfo = GetClusterInfo(clusters,
	                                                filter,
	                                                anonymous,
	                                                usersn,
	                                                timeout);

	return ExtractQueueInfo(clusterinfo);
}


std::list<Queue> GetQueueInfo(std::list<URL> clusters,
                              std::string filter,
                              const bool& anonymous,
                              const std::string& usersn,
                              unsigned int timeout) {

	std::list<Cluster> clusterinfo = GetClusterInfo(clusters,
	                                                filter,
	                                                anonymous,
	                                                usersn,
	                                                timeout);

	return ExtractQueueInfo(clusterinfo);
}


std::list<StorageElement> GetSEInfo(const URL& url,
                                    std::string filter,
                                    const bool& anonymous,
                                    const std::string& usersn,
                                    unsigned int timeout) {
	std::list<URL> urls;
	urls.push_back(url);

	std::list<StorageElement> seinfo = GetSEInfo(urls,
	                                             filter,
	                                             anonymous,
	                                             usersn,
	                                             timeout);

	if (seinfo.empty())
		throw MDSQueryError(_("No information returned from cluster"));

	return seinfo;
}


std::list<StorageElement> GetSEInfo(std::list<URL> urls,
                                    std::string filter,
                                    const bool& anonymous,
                                    const std::string& usersn,
                                    unsigned int timeout) {

	FilterSubstitution(filter);

	if (urls.empty()) urls = GetSEResources();

	std::vector<std::string> attrs;
	MDSQueryCallback SEInfo;

	ParallelLdapQueries pldapq(urls,
	                           filter,
	                           attrs,
	                           &MDSQueryCallback::Callback,
	                           (void*)&SEInfo,
	                           LdapQuery::subtree,
	                           usersn,
	                           anonymous,
	                           timeout);
	pldapq.Query();

	return SEInfo.GetSEList();
}


std::list<ReplicaCatalog> GetRCInfo(const URL& url,
                                    std::string filter,
                                    const bool& anonymous,
                                    const std::string& usersn,
                                    unsigned int timeout) {

	std::list<URL> urls;
	urls.push_back(url);

	std::list<ReplicaCatalog> rcinfo = GetRCInfo(urls,
	                                             filter,
	                                             anonymous,
	                                             usersn,
	                                             timeout);

	if (rcinfo.empty())
		throw MDSQueryError(_("No information returned from cluster"));

	return rcinfo;
}


std::list<ReplicaCatalog> GetRCInfo(std::list<URL> urls,
                                    std::string filter,
                                    const bool& anonymous,
                                    const std::string& usersn,
                                    unsigned int timeout) {

	FilterSubstitution(filter);

	if (urls.empty()) urls = GetRCResources();

	std::vector<std::string> attrs;
	MDSQueryCallback RCInfo;

	ParallelLdapQueries pldapq(urls,
	                           filter,
	                           attrs,
	                           &MDSQueryCallback::Callback,
	                           (void*)&RCInfo,
	                           LdapQuery::subtree,
	                           usersn,
	                           anonymous,
	                           timeout);
	pldapq.Query();

	return RCInfo.GetRCList();
}


std::list<Queue> ExtractQueueInfo(std::list<Cluster> clusterinfo) {

	std::list<Queue> allqueues;

	std::list<Cluster>::iterator it;
	for (it = clusterinfo.begin(); it != clusterinfo.end(); it++) {

		std::list<Queue>::iterator queueit;
		for (queueit = it->queues.begin(); queueit != it->queues.end();
		     queueit++) {

			queueit->cluster = *it; // Copy cluster to Queue::cluster element
			queueit->cluster.queues.clear();
			// Erase queues in Queue::cluster. Not needed here.

			allqueues.push_back(*queueit);
		}
	}

	return allqueues;
}
