#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstdlib>

#include <arc/common.h>
#include <arc/notify.h>
#include <arc/standardbrokers.h>
#include <arc/databroker.h>
#include <arc/stringconv.h>
#include <arc/configcore.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


void ClusterBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("cluster", targets);
}


bool ClusterBroker::RelationCheck(Target& target, XrslRelation& rel) {

	std::string clustername = rel.GetSingleValue();
	xrsl_operator op = rel.GetOperator();

	if (op == operator_eq) {
		if (target.cluster.hostname != clustername)
			return false;
	}
	else if (op == operator_neq) {
		if (target.cluster.hostname == clustername)
			return false;
	}

	return true;
}


void QueueBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("queue", targets);
}


bool QueueBroker::RelationCheck(Target& target, XrslRelation& rel) {

	std::string queuename = rel.GetSingleValue();
	xrsl_operator op = rel.GetOperator();

	if (op == operator_eq) {
		if (target.name != queuename)
			return false;
	} else if (op == operator_neq) {
		if (target.name == queuename)
			return false;
	}

	return true;
}


void CountBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("count", targets);
}


bool CountBroker::RelationCheck(Target& target, XrslRelation& rel) {

	int countval = stringtoi(rel.GetSingleValue());

	int totalcpus = target.total_cpus;
	if (totalcpus == UNDEFINED) totalcpus = target.cluster.total_cpus;

	if (totalcpus<countval) return false;

	return true;
}


void MemoryBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("memory", targets);
}


bool MemoryBroker::RelationCheck(Target& target, XrslRelation& rel) {

	int memoryval = stringtoi(rel.GetSingleValue());

	int targetmemory = target.node_memory;
	if (targetmemory == UNDEFINED) targetmemory = target.cluster.node_memory;

	if (targetmemory == UNDEFINED) return false;
	if (targetmemory < memoryval) return false;

	return true;
}


void ArchitectureBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("architecture", targets);
}


bool ArchitectureBroker::RelationCheck(Target& target, XrslRelation& rel) {

	std::string archval = rel.GetSingleValue();
	xrsl_operator op = rel.GetOperator();

	std::string targetarch = target.architecture;
	if (targetarch.empty()) targetarch = target.cluster.architecture;
	
	if (op==operator_eq) {
		if (archval != targetarch) return false;
		return true;
	} else if (op==operator_neq) {
		if (archval == targetarch) return false;
		return true;
	}
	return false;
}


void NodeAccessBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("nodeaccess", targets);
}


bool NodeAccessBroker::RelationCheck(Target& target, XrslRelation& rel) {

	std::string accessval = rel.GetSingleValue();
	std::list<std::string> nodeaccess = target.cluster.node_access;

	std::list<std::string>::iterator it;
	for (it = nodeaccess.begin(); it != nodeaccess.end(); it++) {
		if (*it == accessval) return true;
	}

	return false;
}


void MiddlewareBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("middleware", targets);
}


bool MiddlewareBroker::RelationCheck(Target& target, XrslRelation& rel) {

	std::string middleval = rel.GetSingleValue();
	xrsl_operator op = rel.GetOperator();
	RuntimeEnvironment middleenv(middleval);

	std::list<RuntimeEnvironment> mws = target.middlewares;
	std::list<RuntimeEnvironment> clustermws = target.cluster.middlewares;
	if (mws.size()>0) {
		mws.insert(mws.end(), clustermws.begin(), clustermws.end());
	} else {
		mws = clustermws;
	}

	mws.sort();
	mws.unique();

	std::list<RuntimeEnvironment>::iterator it;
	for (it = mws.begin(); it != mws.end(); it++) {
		if (op != operator_neq && it->Name() != middleenv.Name()) continue; // names don't match
		
		if (op == operator_eq && *it == middleenv) return true;	
		if (op == operator_neq && *it != middleenv) return true;	
		if (op == operator_gt && *it > middleenv) return true;	
		if (op == operator_lt && *it < middleenv) return true;	
		if (op == operator_gteq && *it >= middleenv) return true;	
		if (op == operator_lteq && *it <= middleenv) return true;	
	}

	return false;
}


void RuntimeEnvironmentBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("runtimeenvironment", targets);
}


bool RuntimeEnvironmentBroker::RelationCheck(Target& target,
                                             XrslRelation& rel) {

	std::string runtimeval = rel.GetSingleValue();
	xrsl_operator op = rel.GetOperator();
	RuntimeEnvironment runtimeenv(runtimeval);

	std::list<RuntimeEnvironment> res = target.runtime_environments;
	std::list<RuntimeEnvironment> clusterres =
	    target.cluster.runtime_environments;
	if (!res.empty()) {
		res.insert(res.end(), clusterres.begin(), clusterres.end());
	}
	else {
		res = clusterres;
	}

	res.sort();
	res.unique();

	std::list<RuntimeEnvironment>::iterator it;
	for (it = res.begin(); it != res.end(); it++) {
		if (it->Name()!=runtimeenv.Name()) continue; // names don't match

		if (op==operator_eq && *it==runtimeenv) return true;	
		if (op==operator_neq && *it!=runtimeenv) return true;	
		if (op==operator_gt && *it>runtimeenv) return true;	
		if (op==operator_lt && *it<runtimeenv) return true;	
		if (op==operator_gteq && *it>=runtimeenv) return true;	
		if (op==operator_lteq && *it<=runtimeenv) return true;	
	}

	return false;
}


void OpsysBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("opsys", targets);
}


bool OpsysBroker::RelationCheck(Target& target, XrslRelation& rel) {

	std::string opsysval = rel.GetSingleValue();
	xrsl_operator op = rel.GetOperator();
	RuntimeEnvironment opsysenv(opsysval);

	std::list<RuntimeEnvironment> oss = target.operating_systems;
	std::list<RuntimeEnvironment> clusteross =
	    target.cluster.operating_systems;

	if (!oss.empty()) {
		oss.insert(oss.end(), clusteross.begin(), clusteross.end());
	} else {
		oss = clusteross;
	}

	oss.sort();
	oss.unique();

	std::list<RuntimeEnvironment>::iterator it;
	for (it = oss.begin(); it != oss.end(); it++) {
		if (it->Name()!=opsysenv.Name()) continue; // names don't match
		
		if (op==operator_eq && *it==opsysenv) return true;	
		if (op==operator_neq && *it!=opsysenv) return true;	
		if (op==operator_gt && *it>opsysenv) return true;	
		if (op==operator_lt && *it<opsysenv) return true;	
		if (op==operator_gteq && *it>=opsysenv) return true;	
		if (op==operator_lteq && *it<=opsysenv) return true;	
	}

	return false;
}


void CpuTimeBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("cputime", targets);
	SetupAttributeBrokering("walltime", targets);
	SetupAttributeBrokering("gridtime", targets);
	SetupAttributeBrokering("benchmarks", targets);
}


bool CpuTimeBroker::RelationCheck(Target& target, XrslRelation& rel) {

	Xrsl xrsl;
	xrsl.AddRelation(rel);
	long cputimesecs = target.GetCputime(xrsl);

	if (target.max_cpu_time!=UNDEFINED && target.max_cpu_time<cputimesecs)
		return false;
	if (target.min_cpu_time!=UNDEFINED && target.min_cpu_time>cputimesecs)
		return false;
	if (target.max_wall_time!=UNDEFINED && target.max_wall_time<cputimesecs)
		return false;
	if (target.min_wall_time!=UNDEFINED && target.min_wall_time>cputimesecs)
		return false;

	return true;
}


void FreeCpusSortBroker::DoBrokering(std::list<Target>& targets) {
	if (targets.size()<=1) return;
	SetupSorting(targets);
}


bool FreeCpusSortBroker::Compare(Target& target1, Target& target2) {

	notify(VERBOSE) << _("Comparing") << " " << target1.name << "@"
	                << target1.cluster.hostname << " " << _("with") << " "
	                << target2.name << "@" << target2.cluster.hostname
	                << std::endl;

	int queued1 = std::max(target1.queued, target1.grid_queued +
	                       target1.local_queued + target1.prelrms_queued);
	if (queued1<0) queued1 = 0;

	double frac1 = 0.0;
	if (target1.total_cpus==UNDEFINED) {
		frac1 = (double)queued1/target1.cluster.total_cpus;
	} else {
		frac1 = (double)queued1/target1.total_cpus;
	}

	int queued2 = std::max(target2.queued, target2.grid_queued +
	                       target2.local_queued + target2.prelrms_queued);
	if (queued2<0) queued2 = 0;

	double frac2 = 0.0;
	if (target2.total_cpus==UNDEFINED) {
		frac2 = (double)queued2/target2.cluster.total_cpus;
	} else {
		frac2 = (double)queued2/target2.total_cpus;
	}

	/* Read the allowed percentage of queued jobs on a site from
	   the configuration file. */
	double allowqueued = 0.2;

	std::string home = GetEnv("HOME");
	Config conf = ReadConfig(home + "/.arc/client.conf");
	std::string allow = conf.FirstConfValue("client/allowqueued");
	try {
		allowqueued = stringtod(allow);
	} catch(StringConvError e) { };

	int cputime1 = (int)target1.GetCputime(target1.GetXrsls().front());
	int userfreecpus1 = 0;
	if (cputime1!=UNDEFINED && target1.users.size()>0 && frac1<=allowqueued) {
		User user = target1.users.front();
		std::map<long, int>::iterator it =
		    user.free_cpus.lower_bound(cputime1);
		if (it != user.free_cpus.end()) userfreecpus1 = it->second;
	}

	int cputime2 = (int)target2.GetCputime(target2.GetXrsls().front());
	int userfreecpus2 = 0;
	if (cputime2!=UNDEFINED && target2.users.size()>0 && frac2<=allowqueued) {
		User user = target2.users.front();
		std::map<long, int>::iterator it =
		    user.free_cpus.lower_bound(cputime2);
		if (it != user.free_cpus.end()) userfreecpus2 = it->second;
	}

	int countval = 1;
	Xrsl axrsl = target1.GetXrsls().front();
	if (axrsl.IsRelation("count")) {
		XrslRelation count = target1.GetXrsls().front().GetRelation("count");
		countval = stringtoi(count.GetSingleValue());
	}

	if (userfreecpus1>=countval && userfreecpus2>=countval) {
		// Both have enough free CPU's. Let's choose the one with the fastest.
		float cpu_freq1 = target1.cpu_freq;
		if (cpu_freq1==UNDEFINED) cpu_freq1 = target1.cluster.cpu_freq;
		// scale it manually
		if (target1.node_cpu.find("Opteron")!=std::string::npos ||
		    target1.cluster.node_cpu.find("Opteron")!=std::string::npos)
			cpu_freq1 *= 2.9;
		if (target1.node_cpu.find("Xeon")!=std::string::npos ||
		    target1.cluster.node_cpu.find("Xeon")!=std::string::npos)
			cpu_freq1 *= 1.13;

		float cpu_freq2 = target2.cpu_freq;
		if (cpu_freq2==UNDEFINED) cpu_freq2 = target2.cluster.cpu_freq;
		// scale it manually
		if (target2.node_cpu.find("Opteron")!=std::string::npos ||
		    target2.cluster.node_cpu.find("Opteron")!=std::string::npos)
			cpu_freq2 *= 2.9;
		if (target2.node_cpu.find("Xeon")!=std::string::npos ||
		    target2.cluster.node_cpu.find("Xeon")!=std::string::npos)
			cpu_freq2 *= 1.13;

		if (cpu_freq1>=cpu_freq2) {
			notify(VERBOSE) << _("Choosing target") << ": "
			                << target1.name << "@"
			                << target1.cluster.hostname << std::endl;
			return false;
		} else {
			notify(VERBOSE) << _("Choosing target") << ": "
			                << target2.name << "@"
			                << target2.cluster.hostname << std::endl;
			return true;
		}
	}

	if (userfreecpus1>=countval) {
		notify(VERBOSE) << _("Choosing target") << ": "
		                << target1.name << "@"
		                << target1.cluster.hostname << std::endl;
		return false;
	}

	if (userfreecpus2>=countval) {
		notify(VERBOSE) << _("Choosing target") << ": "
		                << target2.name << "@"
		                << target2.cluster.hostname << std::endl;
		return true;
	}

	if (frac1<=frac2) {
		notify(VERBOSE) << _("Choosing target") << ": "
		                << target1.name << "@"
		                << target1.cluster.hostname << std::endl;
		return false;
	}

	if (frac1>frac2) {
		notify(VERBOSE) << _("Choosing target") << ": "
		                << target2.name << "@"
		                << target2.cluster.hostname << std::endl;
		return true;
	}

	return false;
}


void RandomSortBroker::DoBrokering(std::list<Target>& targets) {
	if (targets.size()<=1) return;
	
	srand(time(NULL));
	SetupSorting(targets);
}


bool RandomSortBroker::Compare(Target& target1, Target& target2) {
	return ((rand() % 2)==0);
}


void LifeTimeBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("lifetime", targets);
}


bool LifeTimeBroker::RelationCheck(Target& target, XrslRelation& rel) {

	int sessiondirlife = Seconds(rel.GetSingleValue());
	if (target.cluster.session_dir_lifetime != UNDEFINED &&
	    target.cluster.session_dir_lifetime<sessiondirlife) return false;

	return true;
}


void DiskBroker::DoBrokering(std::list<Target>& targets) {
	SetupAttributeBrokering("disk", targets);
}


bool DiskBroker::RelationCheck(Target& target, XrslRelation& rel) {

	if (target.users.begin()->free_diskspace == UNDEFINED) return true;

	long long reqdiskspace = stringtoll(rel.GetSingleValue())*1024*1024;
	xrsl_operator op = rel.GetOperator();

	long long diskspace = target.users.begin()->free_diskspace;

	if (op == operator_eq && diskspace > reqdiskspace) return true;	
	if (op == operator_neq && diskspace != reqdiskspace) return true;	
	if (op == operator_gt && diskspace > reqdiskspace) return true;	
	if (op == operator_lt && diskspace < reqdiskspace) return true;	
	if (op == operator_gteq && diskspace >= reqdiskspace) return true;	
	if (op == operator_lteq && diskspace <= reqdiskspace) return true;	

	return false;
}


void PerformStandardBrokering(std::list<Target>& targets) {

	std::list<Broker*> brokers;

	ClusterBroker broker1;
	QueueBroker broker2;
	CountBroker broker3;
	MemoryBroker broker4;
	ArchitectureBroker broker5;
	NodeAccessBroker broker6;
	MiddlewareBroker broker7;
	RuntimeEnvironmentBroker broker8;
	OpsysBroker broker9;
	CpuTimeBroker broker10;
	LifeTimeBroker broker11;

	FreeCpusSortBroker broker12;
	RandomSortBroker broker13;

	DiskBroker broker14;

	DataBrokerWrapper broker15;

	brokers.push_back(&broker1);
	brokers.push_back(&broker2);
	brokers.push_back(&broker3);
	brokers.push_back(&broker4);
	brokers.push_back(&broker5);
	brokers.push_back(&broker6);
	brokers.push_back(&broker7);
	brokers.push_back(&broker8);
	brokers.push_back(&broker9);
	brokers.push_back(&broker10);
	brokers.push_back(&broker11);
	brokers.push_back(&broker14);

	std::string home = GetEnv("HOME");
	Config conf = ReadConfig(home + "/.arc/client.conf");
	std::string brokername = conf.FirstConfValue("client/broker");
	if (brokername.empty()) brokername = "FastestCpus";

	Broker* pbroker = &broker12;
	if (brokername=="RandomSort") {
		pbroker = &broker13;
	} else if (brokername=="DataBroker") {
		pbroker = &broker15;
	} else if (brokername!="FastestCpus") {
		notify(DEBUG) << _("Warning: Illegal broker specificed in "
		                   "configuration file. Using default broker.")
		              << std::endl;
	}

	brokers.push_back(pbroker);
	notify(DEBUG) << _("Using broker") << " < " << brokername << " > "
	              << _("for finding the optimal target.") << std::endl;

	PerformBrokering(brokers, targets);
}
