#include "arccli.h"

#include <fstream>
#include <iostream>
#include <list>
#include <string>
#include <algorithm>

#include <arc/jobftpcontrol.h>
#include <arc/ftpcontrol.h>
#include <arc/joblist.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/resource.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


void arccat(const std::list<std::string>& jobs,
            const std::list<std::string>& clusterselect,
            const std::list<std::string>& clusterreject,
            const std::list<std::string>& status,
            const int whichfile,
            const bool follow,
            const int timeout,
            const bool anonymous) {

	std::list<std::string> jobids =
		GetJobIDsList(jobs, clusterselect, clusterreject);

	if (jobs.empty() && clusterselect.empty() && jobids.empty())
		throw ARCCLIError(_("No jobs"));

	if (jobids.empty())
		throw ARCCLIError(_("No valid jobnames/jobids given"));

	std::list<Job> joblist =
		GetJobInfo(jobids, MDS_FILTER_JOBINFO, anonymous, "", timeout);

	std::list<Queue> queues;
	FTPControl ctrl;

	for (std::list<Job>::iterator jli = joblist.begin();
	     jli != joblist.end(); jli++) {

		if (jli->status.empty()) {
			notify(WARNING) << _("Job information not found")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (!status.empty() &&
		    (std::find(status.begin(), status.end(), jli->status) == status.end()))
			continue;

		if (whichfile == STDOUT || whichfile == STDERR) {
			if (jli->status == "DELETED") {
				notify(WARNING) << _("Job has already been deleted")
				                << ": " << jli->id << std::endl;
				continue;
			}

			if (jli->status == "ACCEPTING" || jli->status == "ACCEPTED" ||
			    jli->status == "PREPARING" || jli->status == "PREPARED" ||
			    jli->status == "INLRMS:Q") {
				notify(WARNING) << _("Job has not started yet")
				                << ": " << jli->id << std::endl;
				continue;
			}
		}

		if (whichfile == STDOUT && jli->sstdout.empty()) {
			notify(ERROR) << _("Can not determine the stdout location")
			              << ": " << jli->id << std::endl;
			continue;
		}

		if (whichfile == STDERR && jli->sstderr.empty()) {
			notify(ERROR) << _("Can not determine the stderr location")
			              << ": " << jli->id << std::endl;
			continue;
		}

		std::string url;
		if (whichfile == STDOUT)
			url = jli->id + '/' + jli->sstdout;
		else if (whichfile == STDERR)
			url = jli->id + '/' + jli->sstderr;
		else if (whichfile == GMLOG) {
			url = jli->id;
			url.insert(url.rfind('/'), "/info");
			url += "/errors";
		}

		std::string filename("/tmp/arccat.XXXXXX");
		int tmp_h = mkstemp((char*)filename.c_str());
		if (tmp_h == -1)
			throw ARCCLIError(_("Could not create temporary file") + std::string(" '") + filename + "'");
		close(tmp_h);

		try {
			if (!follow) {
				ctrl.Download(url, filename, timeout, false);

				std::ifstream is(filename.c_str());
				char c;
				while (is.get(c))
					std::cout.put(c);
				is.close();
				unlink(filename.c_str());
			}
			else {
				unsigned long long offset = ctrl.Size(url, timeout, false);

				if(offset < 1024)
					offset = 0;
				else
					offset -= 1024;

				time_t last = time(NULL);
				while (true) {

					int tmp_h = open(filename.c_str(), O_WRONLY | O_TRUNC);
					if (tmp_h == -1)
						throw ARCCLIError(_("Could not truncate temporary file"));
					close(tmp_h);

					ctrl.Download(url, offset, (size_t)(-1), filename, timeout, false);

					std::ifstream is(filename.c_str());
					char c;
					while (is.get(c))
						std::cout.put(c);
					is.close();

					struct stat st;
					if (stat(filename.c_str(), &st) != 0)
						throw ARCCLIError(_("Could not stat temporary file"));
					offset += st.st_size;

					while (true) {
						time_t rest = time(NULL) - last;
						if (rest < 5)
							sleep(5 - rest);
						last = time(NULL);
						unsigned long long fsize = ctrl.Size(url, timeout, false);
						if ((!fsize) || (fsize > offset))
							break;
					}
				}
			}
		}
		catch (FTPControlError e) {
			std::string queuestatus;

			if(WARNING <= GetNotifyLevel()) 
				queuestatus = QueueStatusByJob(*jli, queues, anonymous, timeout);

			if (queuestatus.find("gridftp is down") == std::string::npos) {
				notify(WARNING) << e.what() << std::endl;;
			}
			else {
				notify(WARNING) << _("Gridftp server of job is down at the "
				                     "moment") << ": " << jli->id << std::endl;
			}
		}
	}
}
