#include "arccli.h"

#include <iostream>
#include <list>
#include <string>
#include <algorithm>

#include <arc/jobftpcontrol.h>
#include <arc/joblist.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/resource.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


void arcclean (const std::list<std::string>& jobs,
               const std::list<std::string>& clusterselect,
               const std::list<std::string>& clusterreject,
               const std::list<std::string>& status,
               const bool force,
               const int timeout,
               const bool anonymous) {

	std::list<Queue> queues;
	std::list<std::string> jobids =
		GetJobIDsList(jobs, clusterselect, clusterreject);

	if (jobs.empty() && clusterselect.empty() && jobids.empty())
		throw ARCCLIError(_("No jobs"));

	if (jobids.empty())
		throw ARCCLIError(_("No valid jobnames/jobids given"));

	std::list<Job> joblist =
		GetJobInfo(jobids, MDS_FILTER_JOBINFO, anonymous, "", timeout);

	JobFTPControl ctrl;

	joblist.sort(SortByHost());

	int jobs_processed = 0;
	int jobs_cleaned = 0;

	for (std::list<Job>::iterator jli = joblist.begin();
	     jli != joblist.end(); jli++) {

		jobs_processed++;

		if (force && jli->status.empty() && status.empty()) {
			notify(INFO) << _("Deleting job (from local list only)")
			             << ": " << jli->id << std::endl;
			jobs_cleaned++;
			RemoveJobID(jli->id);
			continue;
		}

		if (jli->status.empty()) {
			notify(WARNING) << _("Job information not found")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (!status.empty() &&
		    std::find(status.begin(), status.end(), jli->status) == status.end())
			continue;

		if (jli->status != "FINISHED" && jli->status != "FAILED" &&
		    jli->status != "KILLED" && jli->status != "DELETED") {
			notify(WARNING) << _("Job has not finished yet")
			                << ": " << jli->id << std::endl;
			continue;
		}

		notify(INFO) << _("Deleting job")
		                << ": " << jli->id << std::endl;

		try {
			ctrl.Clean(jli->id, timeout, false);
			jobs_cleaned++;
		} catch(ARCLibError& e) {
			std::string queuestatus;

			if(WARNING <= GetNotifyLevel()) 
				queuestatus = QueueStatusByJob(*jli, queues, anonymous, timeout);

			if (queuestatus.find("gridftp is down") == std::string::npos) {
				notify(WARNING) << e.what() << std::endl;
			} else {
				notify(WARNING) << _("Gridftp server of job is down at the "
				                     "moment") << ": " << jli->id << std::endl;
			}

			continue;
		}

		RemoveJobID(jli->id);
	}

	notify(WARNING) << _("Jobs processed") << ": " << jobs_processed << ", "
	                << _("deleted") << ": " << jobs_cleaned
	                << std::endl;
}
