#include "arccli.h"

#include <iostream>
#include <list>
#include <string>
#include <algorithm>

#include <arc/ftpcontrol.h>
#include <arc/jobftpcontrol.h>
#include <arc/joblist.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/resource.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


void arcget(const std::list<std::string>& jobs,
            const std::list<std::string>& clusterselect,
            const std::list<std::string>& clusterreject,
            const std::list<std::string>& status,
            const std::string& directory,
            const bool usejobname,
            const bool keep,
            const int timeout,
            const bool anonymous) {

	std::list<std::string> jobids =
		GetJobIDsList(jobs, clusterselect, clusterreject);

	if (jobs.empty() && clusterselect.empty() && jobids.empty())
		throw ARCCLIError(_("No jobs"));

	if (jobids.empty())
		throw ARCCLIError(_("No valid jobnames/jobids given"));

	std::list<Job> joblist =
		GetJobInfo(jobids, MDS_FILTER_JOBINFO, anonymous, "", timeout);

	std::list<Queue> queues;
	JobFTPControl jctrl;

	joblist.sort(SortByHost());

	int jobs_processed = 0;
	int jobs_downloaded = 0;

	for (std::list<Job>::iterator jli = joblist.begin();
	     jli != joblist.end(); jli++) {

		jobs_processed++;

		if (jli->status.empty()) {
			notify(WARNING) << _("Job information not found")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (!status.empty() &&
		    std::find(status.begin(), status.end(), jli->status) == status.end())
			continue;

		if (jli->status == "DELETED") {
			notify(WARNING) << _("Job has already been deleted")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (jli->status != "FINISHED" && jli->status != "FAILED" &&
		    jli->status != "KILLED") {
			notify(WARNING) << _("Job has not finished yet")
			                << ": " << jli->id << std::endl;
			continue;
		}

		std::string dir;
		if (usejobname)
			dir = directory + '/' + jli->job_name;
		else
			dir = directory + '/' + jli->id.substr(jli->id.rfind('/') + 1);

		try {
			RmDir(dir, true);
		} catch (ARCLibError& e) {
		}
		try {
			MkDir(dir, true);
		} catch (ARCLibError& e) {
			notify(WARNING) << e.what() << std::endl;
			notify(WARNING) << _("Skipping job")
			                << ": " << jli->id << std::endl;
			continue;
		}

		try {
			jctrl.DownloadDirectory(jli->id, dir, timeout, false);
			jobs_downloaded++;
			notify(WARNING) << _("Results stored at ") << dir << std::endl;

			if (!keep) {
				notify(INFO) << _("Deleting job")
				                << ": " << jli->id << std::endl;

				jctrl.Clean(jli->id, timeout, false);
				RemoveJobID(jli->id);
			}

		} catch(FTPControlError e) {
			try {
				RmDir(dir, true);
				notify(INFO) << _("Local job directory is deleted") 
				             << std::endl;
			} catch (ARCLibError e) {
				notify(INFO) << _("Although job download failed there are "
				                  "some files in local job directory") 
				             << std::endl;
			}

			std::string queuestatus;

			if(WARNING <= GetNotifyLevel())
				queuestatus = QueueStatusByJob(*jli, queues, anonymous, timeout);

			if (queuestatus.find("gridftp is down") == std::string::npos) {
				notify(WARNING) << e.what() << std::endl;;
			} else {
				notify(WARNING) << _("Gridftp server of job is down at the "
				                     "moment") << ": " << jli->id << std::endl;
			}

		} catch (ARCLibError& e) {
			notify(WARNING) << e.what() << std::endl;
		}
	}

	notify(WARNING) << _("Jobs processed") << ": " << jobs_processed << ", "
	                << _("successfuly downloaded") <<": " << jobs_downloaded
	                << std::endl;
}
