#include "../std.h"

#include <globus_gsi_credential.h>

#include <vector>

#include "auth.h"

#ifdef EDG_VOMS
#include <voms_api.h>
#endif

#ifdef NG_VOMS
#include <Client.h>
#include <api_util.h>
#endif

#include "../misc/escaped.h"
#include "../misc/stringtoint.h"
#include <arc/globuserrorutils.h>
#include "../misc/log_time.h"


#ifdef HAVE_VOMS
const char* voms_error(verror_type err);
int renew_voms_cert(const std::string &contact,const std::string &voms_dir,const std::string &uri);
int process_vomsproxy(const char* filename,std::vector<struct voms> &data,bool auto_cert = false);
#endif

int AuthUser::process_voms(void) {
#ifdef HAVE_VOMS
  if(!voms_extracted) {
    if(filename.length() > 0) {
      int err = process_vomsproxy(filename.c_str(),*voms_data);
      voms_extracted=true;
      odlog(DEBUG)<<"VOMS proxy processing returns: "<<err<<std::endl;
      if(err != AAA_POSITIVE_MATCH) return err;
    };
  };
  return AAA_POSITIVE_MATCH;
#else
  return AAA_FAILURE;
#endif
};

int AuthUser::match_voms(const char* line) {
#ifdef HAVE_VOMS
  // parse line
  std::string vo("");
  std::string group("");
  std::string role("");
  std::string capabilities("");
  std::string auto_c("");
  bool auto_cert = false;
  int n;
  n=input_escaped_string(line,vo,' ','"');
  if(n == 0) {
    odlog(ERROR)<<"Missing VO in configuration"<<std::endl; 
    return AAA_FAILURE;
  };
  line+=n;
  n=input_escaped_string(line,group,' ','"');
  if(n == 0) {
    odlog(ERROR)<<"Missing group in configuration"<<std::endl; 
    return AAA_FAILURE;
  };
  line+=n;
  n=input_escaped_string(line,role,' ','"');
  if(n == 0) {
    odlog(ERROR)<<"Missing role in configuration"<<std::endl; 
    return AAA_FAILURE;
  };
  line+=n;
  n=input_escaped_string(line,capabilities,' ','"');
  if(n == 0) {
    odlog(ERROR)<<"Missing capabilities in configuration"<<std::endl; 
    return AAA_FAILURE;
  };
  n=input_escaped_string(line,auto_c,' ','"');
  if(n != 0) {
    if(auto_c == "auto") auto_cert=true;
  };
  odlog(VERBOSE)<<"VOMS config: vo: "<<vo<<std::endl;
  odlog(VERBOSE)<<"VOMS config: group: "<<group<<std::endl;
  odlog(VERBOSE)<<"VOMS config: role: "<<role<<std::endl;
  odlog(VERBOSE)<<"VOMS config: capabilities: "<<capabilities<<std::endl;
  // extract info from voms proxy
  // if(voms_data->size() == 0) {
  process_voms();
  if(voms_data->size() == 0) return AAA_NO_MATCH;
  // analyse permissions
  for(std::vector<struct voms>::iterator v = voms_data->begin();v!=voms_data->end();++v) {
    odlog(DEBUG)<<"match vo: "<<v->voname<<std::endl;
    if((vo == "*") || (vo == v->voname)) {
      for(std::vector<struct data>::iterator d=v->std.begin();d!=v->std.end();++d) {
        odlog(VERBOSE)<<"match group: "<<d->group<<std::endl;
        odlog(VERBOSE)<<"match role: "<<d->role<<std::endl;
        odlog(VERBOSE)<<"match capabilities: "<<d->cap<<std::endl;
        if(((group == "*") || (group == d->group)) &&
           ((role == "*") || (role == d->role)) &&
           ((capabilities == "*") || (capabilities == d->cap))) {
          odlog(VERBOSE)<<"VOMS matched"<<std::endl;
          default_voms_=v->server.c_str();
          default_vo_=v->voname.c_str();
          default_role_=d->role.c_str();
          default_capability_=d->cap.c_str();
          default_vgroup_=d->group.c_str();
          return AAA_POSITIVE_MATCH;
        };
      };
    };
  };
  odlog(VERBOSE)<<"VOMS matched nothing"<<std::endl;
  return AAA_NO_MATCH;
#else
  odlog(ERROR)<<"VOMS ACL is not supported"<<std::endl; 
  return AAA_FAILURE;
#endif
}

#ifdef NG_VOMS
int process_vomsproxy(const char* filename,std::vector<struct voms> &data,bool auto_cert) {
  globus_result_t res;
  globus_gsi_cred_handle_t cred;
  X509 * cert;
  STACK_OF(X509) * cert_chain;
  std::string voms_dir = "/etc/grid-security/vomsdir";
  std::string cert_dir = "/etc/grid-security/certificates";
  {
    char *v;
    if((v = getenv("X509_VOMS_DIR"))) voms_dir = v;
    if((v = getenv("X509_CERT_DIR"))) cert_dir = v;
  };
  if((res=globus_gsi_cred_handle_init(&cred,GLOBUS_NULL)) != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Globus error: "<<res<<std::endl;
    return AAA_FAILURE;
  };
  if((res=globus_gsi_cred_read_proxy(cred,(char*)filename)) != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Globus error: "<<res<<std::endl;
    globus_gsi_cred_handle_destroy(cred);
    return AAA_FAILURE;
  };
  if((res=globus_gsi_cred_get_cert(cred,&cert)) != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Globus error: "<<res<<std::endl;
    globus_gsi_cred_handle_destroy(cred);
    return AAA_FAILURE;
  };
  if((res=globus_gsi_cred_get_cert_chain(cred,&cert_chain)) != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Globus error: "<<res<<std::endl;
    if(cert) X509_free(cert);
    globus_gsi_cred_handle_destroy(cred);
    return AAA_FAILURE;
  };
  std::string message;
  std::string vo;
  std::string file;
  std::string subject;
  std::string ca;
  verror_type err;
  if(!retrieve(cert,cert_chain,RECURSE_CHAIN,message,vo,file,subject,ca,err)) {
    odlog(ERROR)<<"VOMS error: "<<err<<" - "<<voms_error(err)<<std::endl;
    if(cert) X509_free(cert);
    if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
    globus_gsi_cred_handle_destroy(cred);
    if(err == VERR_NOEXT) return AAA_NO_MATCH;
    return AAA_FAILURE;
  };
  // validate information
  // first stage - parse data
  if(auto_cert) {
    struct vomsdata vdata;
    if(!verify(message,vdata,err,voms_dir,cert_dir,subject,ca,false)) {
      odlog(ERROR)<<"VOMS parsing error: "<<err<<" - "<<voms_error(err)<<std::endl;
      if(cert) X509_free(cert);
      if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
      globus_gsi_cred_handle_destroy(cred);
      return AAA_FAILURE;
    };
    for(int n = 0;n<vdata.data.size();n++) {
      if(vdata.data[n].uri.empty() || vdata.data[n].server.empty()) {
        odlog(ERROR)<<"Missing URI or server subject"<<std::endl;
        if(cert) X509_free(cert);
        if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
        globus_gsi_cred_handle_destroy(cred);
        return AAA_FAILURE;
      };
      if(renew_voms_cert(vdata.data[n].server,voms_dir,vdata.data[n].uri) != AAA_POSITIVE_MATCH) {
        odlog(ERROR)<<"Certificate renewal failed"<<std::endl;
        if(cert) X509_free(cert);
        if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
        globus_gsi_cred_handle_destroy(cred);
        return AAA_FAILURE;
      };
    };
  };
  // second stage - verify data
  {
    struct vomsdata vdata;
    if(!verify(message,vdata,err,voms_dir,cert_dir,subject,ca,false)) {
      odlog(ERROR)<<"VOMS verification error: "<<err<<" - "<<voms_error(err)<<std::endl;
      if(cert) X509_free(cert);
      if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
      globus_gsi_cred_handle_destroy(cred);
      return AAA_FAILURE;
    };
    for(int n = 0;n<vdata.data.size();n++) {
      data.push_back(vdata.data[n]);
    };
  };
  if(cert) X509_free(cert);
  if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
  globus_gsi_cred_handle_destroy(cred);
  return AAA_POSITIVE_MATCH;
}

int renew_voms_cert(const std::string &contact,const std::string &voms_dir,const std::string &uri) {
  // try to load certificate 
//  std::string contact = voms.data[0].server;
  bool voms_cert_cached = false;
  // Is that certificate already cached ? 
  BIO *in = BIO_new(BIO_s_file());
  if(in == NULL) {
    odlog(ERROR)<<"Failure in SSL (BIO_new)"<<std::endl;
    return AAA_FAILURE;
  };
  std::string voms_cert = contact;
  for(std::string::size_type p = voms_cert.find('/');p != std::string::npos;p=voms_cert.find('/',p)) {
    if(p == 0) { voms_cert.erase(0,1); }
    else { voms_cert.replace(p,1,","); };
  };
  voms_cert=voms_dir+"/"+voms_cert;
  // olog<<"Cached certificate: "<<voms_cert<<std::endl;
  if(BIO_read_filename(in,(void*)(voms_cert.c_str())) > 0) {
    X509 *x = PEM_read_bio_X509(in, NULL, 0, NULL);
    if(x)  {
      /* Is it still valid ? */
      if(X509_get_notBefore(x) && X509_get_notAfter(x)) {
        if((X509_cmp_current_time(X509_get_notBefore(x)) < 0) && 
           (X509_cmp_current_time(X509_get_notAfter(x)) > 0)) {
          voms_cert_cached=true;
        };
      };
      X509_free(x);
    };
  };
  BIO_free(in);
  if(!voms_cert_cached) {
    /* Reload certificate */
    std::string host = uri;
    int port = 15000;
    std::string::size_type p = host.find(':'); 
    if(p != std::string::npos) {
      stringtoint(host.substr(p+1,std::string::npos),port);
      host.erase(p,std::string::npos);
    };
    // olog<<"Host: "<<host<<std::endl;
    // olog<<"Port: "<<port<<std::endl;
    GSISocketClient sock(host,port,22);
    sock.RedirectGSIOutput(stderr);
    sock.ServerContact(contact);
    sock.SetFlags(GSS_C_MUTUAL_FLAG | GSS_C_CONF_FLAG | GSS_C_INTEG_FLAG);
    if (!sock.Open()) {
      olog<<"Failed to open connection to "<<host<<":"<<port<<std::endl;
      return AAA_FAILURE;
    };
    if(sock.peer_cert == NULL) {
      odlog(ERROR)<<"Failed to obtain peer certificate from "<<host<<":"<<port<<std::endl;
      return AAA_FAILURE;
    };
    BIO *out = BIO_new(BIO_s_file());
    if(out == NULL) {
      odlog(ERROR)<<"Failure in SSL (BIO_new)"<<std::endl;
      return AAA_FAILURE;
    };
    if(BIO_write_filename(out,(void*)(voms_cert.c_str())) <= 0) {
      odlog(ERROR)<<"Failed to write file "<<voms_cert<<std::endl;
      return AAA_FAILURE;
    };
    PEM_write_bio_X509(out,sock.peer_cert);
    BIO_free(out);
    sock.Close();
  }; 
  return AAA_POSITIVE_MATCH;
};

#endif
#ifdef EDG_VOMS

int process_vomsproxy(const char* filename,std::vector<struct voms> &data,bool auto_cert) {
  X509 * cert = NULL;
  STACK_OF(X509) * cert_chain = NULL;
  EVP_PKEY * key = NULL;
  int n = 0;
  std::vector<struct voms>::iterator i;
  std::string voms_dir = "/etc/grid-security/vomsdir";
  std::string cert_dir = "/etc/grid-security/certificates";
  {
    char *v;
    if((v = getenv("X509_VOMS_DIR"))) voms_dir = v;
    if((v = getenv("X509_CERT_DIR"))) cert_dir = v;
  };
  vomsdata v(voms_dir,cert_dir);
  BIO *bio = NULL;
  if((bio = BIO_new_file(filename, "r")) == NULL) {
    odlog(ERROR)<<"Failed to open file "<<filename<<std::endl;
    goto error_exit;
  };
  if(!PEM_read_bio_X509(bio,&cert, NULL, NULL)) {
    odlog(ERROR)<<"Failed to read PEM from file "<<filename<<std::endl;
    goto error_exit;
  };
  key=PEM_read_bio_PrivateKey(bio,NULL,NULL,NULL);
  if(!key) {
    odlog(ERROR)<<"Failed to read private key from file "<<filename<<" - probably no delegation was done"<<std::endl;
    // goto error_exit;
  };
  if((cert_chain = sk_X509_new_null()) == NULL) {
    odlog(ERROR)<<"Failed in SSL (sk_X509_new_null)"<<std::endl;
    goto error_exit;
  };
  while(!BIO_eof(bio)) {
    X509 * tmp_cert = NULL;
    if(!PEM_read_bio_X509(bio, &tmp_cert, NULL, NULL)) {
      break;
    };
    if(n == 0) {
      X509_free(cert); cert=tmp_cert;
    } else { 
      if(!sk_X509_insert(cert_chain, tmp_cert, n-1)) {
        odlog(ERROR)<<"failed in SSL (sk_X509_insert)"<<std::endl;
        goto error_exit;
      };
    };
    ++n;
  };

  v.SetVerificationType(VERIFY_FULL);
  // v.SetVerificationType((verify_type)(VERIFY_NONE));
  if(!v.Retrieve(cert,cert_chain)) {
    odlog(ERROR)<<"Failed to retrieve VOMS information"<<std::endl;
    odlog(ERROR)<<"Error: "<<v.error<<" - "<<v.ErrorMessage()<<std::endl;
    goto error_exit;
  };
  X509_free(cert);
  EVP_PKEY_free(key);
  sk_X509_pop_free(cert_chain, X509_free);
  BIO_free(bio);
  for(i=v.data.begin();i!=v.data.end();++i) data.push_back(*i);
  ERR_clear_error();
  return AAA_POSITIVE_MATCH;
error_exit:
  if(cert) X509_free(cert);
  if(key) EVP_PKEY_free(key);
  if(cert_chain) sk_X509_pop_free(cert_chain, X509_free);
  if(bio) BIO_free(bio);
  ERR_clear_error();
  return AAA_FAILURE;
}
#endif

#ifdef HAVE_VOMS
const char* voms_errors[] = {
  "Success",
  "Socket problem",
  "Cannot identify itself (certificate problem)",
  "Server problem",
  "Wrong parameters",
  "VOMS extension missing",
  "Initialization error",
  "Error in time checking",
  "User data in extension different from the real ones",
  "VO name and URI missing",
  "Wrong data format",
  "Empty extension",
  "Parse error",
  "Directory error",
  "Signature error",
  "Unidentifiable VOMS server",
  "Memory problems",
  "Generic verification error",
  "Returned data of unknown type"
};

const char* voms_error(verror_type err) {
  if((VERR_NONE < 0) || (err > VERR_TYPE)) return "Unknown";
  return voms_errors[err];
}
#endif

