#ifndef FILECACHE_H_
#define FILECACHE_H_

#include "../std.h"
#include "../misc/log_time.h"
#include "../misc/inttostring.h"
#include "../misc/stringtoint.h"

#include <exception>
#include <sstream>
#include <vector>
#include <sys/utsname.h>

#include "file_cache_hash.h"

/**
 * Contains data on the parameters of a cache.
 */
struct CacheParameters {
  std::string cache_path;
  std::string cache_link_path;
};

/**
 * Exception thrown by any FileCache method or constructor. A description
 * of the exception can optionally be given, and retrieved using
 * description().
 */
class FileCacheException : public std::exception {
  
private:
  std::string _desc;
  
public:
  FileCacheException(std::string desc = ""): _desc(desc) {};
  virtual ~FileCacheException() throw() {};
  std::string what() {return _desc;};
};

/**
 * FileCache provides an interface to all cache operations
 * to be used by external classes. An instance should be created
 * per job, and all files within the job are managed by that
 * instance. When it is decided a file should be downloaded to the
 * cache, start() should be called, so that the cache file can be
 * prepared and locked. When a transfer has finished successfully,
 * link() or copy() should be called to create a hard link to
 * a per-job directory in the cache and then soft link, or copy
 * the file directly to the session directory so it can be accessed from
 * the user's job. stop() must then be called to release any
 * locks on the cache file.
 * 
 * The cache directory(ies) and the optional directory to link to
 * when the soft-links are made are set in the global configuration
 * file. The names of cache files are formed from a hash of the URL 
 * specified as input to the job. To ease the load on the file system,
 * the cache files are split into subdirectories based on the first
 * two characters in the hash. For example the file with hash
 * 76f11edda169848038efbd9fa3df5693 is stored in
 * 76/f11edda169848038efbd9fa3df5693. A cache filename can be found
 * by passing the URL to find(). For more information on the
 * structure of the cache, see the Grid Manager Administration Guide.
 * 
 * A metadata file with the '.meta' suffix is stored next to each
 * cache file. This contains the URL corresponding to the cache
 * file and the expiry time, if it is available. For example
 * lfc://lfc1.ndgf.org//grid/atlas/test/test1 1208352933
 * 
 * While cache files are downloaded, they are locked by creating a
 * lock file with the '.lock' suffix next to the cache file. Calling
 * start() creates this lock and stop() releases it. All processes
 * calling start() must wait until they successfully obtain the lock
 * before downloading can begin.
 */
class FileCache {
 private:
  /**
   * Vector of caches. Each entry defines a cache and specifies 
   * a cache directory, and optional link path.
   */
  std::vector<struct CacheParameters> _caches;
  /**
   * Vector of remote caches. Each entry defines a cache and specifies 
   * a cache directory, per-job directory and link/copy information.
   */
  std::vector<struct CacheParameters> _remote_caches;
  /**
   * identifier used to claim files, ie the job id 
   */
  std::string _id;
  /**
   * owner:group corresponding to the user running the job.
   * The directory with hard links to cached files will be searchable only by this user
   */
  uid_t _uid;
  gid_t _gid;
  /**
   * Our hostname (same as given by uname -n)
   */
  std::string _hostname;
  /**
   * Our pid
   */
  std::string _pid;
  /**
   * The sub-dir of the cache for data
   */
  static const std::string CACHE_DATA_DIR;
  /**
   * The sub-dir of the cache for per-job links
   */
  static const std::string CACHE_JOB_DIR;
  /**
   * The length of each cache subdirectory
   */
  static const int CACHE_DIR_LENGTH;
  /**
   * The number of levels of cache subdirectories
   */
  static const int CACHE_DIR_LEVELS;
  /** 
   * The suffix to use for lock files
   */
  static const std::string CACHE_LOCK_SUFFIX;
  /** 
   * The suffix to use for meta files
   */
  static const std::string CACHE_META_SUFFIX;
  /**
   * Default validity time of cached DNs
   */
  static const int CACHE_DEFAULT_AUTH_VALIDITY;
  /**
   * Common code for constuctors
   */
  void _init(std::vector<std::string> caches,
             std::vector<std::string> remote_caches,
             std::string id,
             uid_t job_uid,
             gid_t job_gid) throw(FileCacheException);
  /**
   * Return the filename of the lock file associated to the given url
   */
  std::string _getLockFileName(std::string url);
  /**
   * Return the filename of the meta file associated to the given url
   */
  std::string _getMetaFileName(std::string url);
  /**
   * Return true if the lock on the cache file corresponding to
   * this url exists and is owned by this process
   */
  bool _checkLock(std::string url);
  /**
   * Generic method to make directories
   * @param dir directory to create
   * @param all_read if true, make the directory readable by all users,
   * if false, it is readable only by the user who created it.
   */
  bool _cacheMkDir(std::string dir, bool all_read);
  /**
   * Choose a cache to use from the list, based on the first character
   * of the url hash mod number of caches.
   * Returns the index of the cache to use in the list.
   */
  int _chooseCache(std::string hash);
 public:
  /**
   * Create a new FileCache instance.
   * @param cache_path The format is "cache_dir[ link_path]".
   * path is the path to the cache directory and the optional
   * link_path is used to create a link in case the
   * cache directory is visible under a different name during actual
   * usage. When linking from the session dir this path is used
   * instead of cache_path.
   * @param id the job id. This is used to create the per-job dir
   * which the job's cache files will be hard linked from
   * @param job_uid owner of job. The per-job dir will only be
   * readable by this user
   * @param job_gid owner group of job
   */
  FileCache(std::string cache_path,
            std::string id,
            uid_t job_uid,
            gid_t job_gid) throw(FileCacheException);

  /**
   * Create a new FileCache instance with remote cache directories.
   * @param cache_path The format is "cache_dir[ link_path]".
   * path is the path to the cache directory and the optional
   * link_path is used to create a link in case the
   * cache directory is visible under a different name during actual
   * usage. When linking from the session dir this path is used
   * instead of cache_path.
   * @param remote_cache_path Same format as cache_path. This is the
   * path to a cache which is under the control of another Grid
   * Manager and is read-only for this process.
   * @param id the job id This is used to create the per-job dir
   * which the job's cache files will be hard linked from
   * @param job_uid owner of job. The per-job dir will only be
   * readable by this user
   * @param job_gid owner group of job
   */
  FileCache(std::string cache_path,
            std::string remote_cache_path,
            std::string id,
            uid_t job_uid,
            gid_t job_gid) throw(FileCacheException);

  /**
   * Create a new FileCache instance with multiple cache dirs
   * @param caches a vector of strings describing caches. The format
   * of each string is "cache_dir[ link_path]".
   * @param id the job id. This is used to create the per-job dir
   * which the job's cache files will be hard linked from
   * @param job_uid owner of job. The per-job dir will only be
   * readable by this user
   * @param job_gid owner group of job
   */
  FileCache(std::vector<std::string> caches,
            std::string id,
            uid_t job_uid,
            gid_t job_gid) throw(FileCacheException);
  /**
   * Create a new FileCache instance with multiple cache dirs and
   * remote cache directories.
   * @param caches a vector of strings describing caches. The format
   * of each string is "cache_dir[ link_path]".
   * @param remote_caches Same format as caches. These are the
   * paths to caches which are under the control of other Grid
   * Managers and are read-only for this process.
   * @param id the job id. This is used to create the per-job dir
   * which the job's cache files will be hard linked from
   * @param job_uid owner of job. The per-job dir will only be
   * readable by this user
   * @param job_gid owner group of job
   */
  FileCache(std::vector<std::string> caches,
            std::vector<std::string> remote_caches,
            std::string id,
            uid_t job_uid,
            gid_t job_gid) throw(FileCacheException);
  /**
   * Copy constructor
   */
  FileCache(const FileCache& cache);
  /**
   * Default constructor. Invalid cache.
   */
  FileCache() {_caches.clear();};
  /**
   * Destructor
   */
  virtual ~FileCache(void);
  /**
   * Prepare cache for downloading file, and lock the cached file.
   * On success returns true. If there is another process downloading
   * the same url, false is returned and is_locked is set to true.
   * In this case the client should wait and retry later. If the lock has
   * expired this process will take over the lock and the method will
   * return as if no lock was present, ie available and is_locked are
   * false. 
   *
   * @param url url that is being downloaded
   * @param available true on exit if the file is already in cache
   * @param is_locked true on exit if the file is already locked, ie
   * cannot be used by this process
   * @param use_remote whether to search remote caches if the file is
   * not available on a local cache
   */
  bool start(std::string url,bool &available, bool &is_locked, bool use_remote = true);
  /**
   * This method (or stopAndDelete) must be called after file was
   * downloaded or download failed, to release the lock on the
   * cache file. stop() does not delete the cache file. It returns
   * false if the lock file does not exist, or another pid was found
   * inside the lock file (this means another process took over the
   * lock so this process must go back to start()), or if it fails
   * to delete the lock file.
   * @param url the url of the file that was downloaded
   */
  bool stop(std::string url);
  /**
   * Release the cache file and delete it, because for example a
   * failed download left an incomplete copy, or it has expired.
   * This method also deletes the meta file which contains the url
   * corresponding to the cache file. The logic of the return
   * value is the same as stop().
   * @param url the url corresponding to the cache file that has 
   * to be released and deleted
   */
  bool stopAndDelete(std::string url);
  /**
   * Returns the full pathname of the file in the cache which
   * corresponds to the given url.
   */
  std::string file(std::string url);
  /**
   * Create a hard-link to the per-job dir from
   * the cache dir, and then a soft-link from here to the
   * session directory. This is effectively 'claiming' the file
   * for the job, so even if the original cache file is deleted,
   * eg by some external process, the hard link still exists
   * until it is explicitly released by calling release().
   * 
   * If cache_link_path is set to "." then files will be
   * copied directly to the session directory rather than via
   * the hard link.
   * @param link_path path to the session dir for soft-link or new file
   * @param url url of file to link to or copy
   */
  bool link_file(std::string link_path, std::string url);
  /**
   * Copy the cache file corresponding to url to the dest_path 
   */
  bool copy_file(std::string dest_path, std::string url, bool executable = false);
  /**
   * Remove some amount of oldest information from cache. 
   * Returns true on success. Not implemented.
   * @param size amount to be removed (bytes)
   */
  bool clean(unsigned long long int size = 1) {return false;};
  /**
   * Release claims on input files for the job specified by id.
   * For each cache directory the per-job directory with the
   * hard-links will be deleted.
   */
  bool release();
  /**
   * Check if there is an information about creation time. Returns
   * true if the file exists in the cache, since the creation time
   * is the creation time of the cache file.
   * @param url the url corresponding to the cache file for which we
   * want to know if the creation date exists
   */
  bool created_available(std::string url);
  /**
   * Add the given DN to the list of cached DNs with the given expiry time
   * @param url the url corresponding to the cache file to which we
   * want to add a cached DN
   * @param DN the DN of the user
   * @param expiry_time the expiry time of this DN in the DN cache
   */
  bool addDN(std::string url, std::string DN, time_t expiry_time);
  /**
   * Check if the given DN is cached for authorisation.
   * @param url the url corresponding to the cache file for which we
   * want to check the cached DN
   * @param DN the DN of the user 
   */
  bool checkDN(std::string url, std::string DN);
  /**
   * Get the creation time of a cached file. If the cache file does
   * not exist, 0 is returned.
   * @param url the url corresponding to the cache file for which we
   * want to know the creation date
   */
  time_t created(std::string url);
  /**
   * Check if there is an information about expiry time.
   * @param url the url corresponding to the cache file for which we
   * want to know if the expiration time exists
   */
  bool validtill_available(std::string url);
  /**
   * Get expiry time of a cached file. If the time is not available,
   * a time equivalent to 0 is returned.
   * @param url the url corresponding to the cache file for which we
   * want to know the expiry time
   */
  time_t validtill(std::string url);
  /**
   * Set expiry time (even if already set).
   * @param url the url corresponding to the cache file for which we
   * want to set the expiry time
   * @param val expiry time
   */
  bool validtill_force(std::string url, time_t val);
  /**
   * Set expiry time (if not already set).
   * @param url the url corresponding to the cache file for which we
   * want to set the expiry time
   * @param val expiry time
   */
  bool validtill(std::string url, time_t val);
  /** 
   * Returns true if object is useable.
   */
  operator bool(void) { return (_caches.size() != 0); };
  /**
   * Return true if all attributes are equal
   */
  bool operator ==(const FileCache& a);

};

#endif /*FILECACHE_H_*/
