#include "../std.h"
#include <string>
#include <list>
#include <iostream>

#include "../misc/condition.h"
#include "../misc/log_time.h"
#include "../misc/checksum.h"
#include "../misc/inttostring.h"
#include "../https/se/file_soapH.h"
extern SOAP_NMAC struct Namespace file_soap_namespaces[];
#include "../https/client/client.h"
#include "../misc/time_utils.h"
#include "../misc/url_options.h"
#include <arc/certificate.h>
#include "databufferpar.h"
#include "datapoint.h"
#include "datahandle_httpg.h"


class httpg_info_t;
static int read_callback(unsigned long long offset,unsigned long long size,unsigned char** buf,unsigned long long* bufsize,void* arg);
static void* read_thread(void *arg);
static void* write_thread(void *arg);

typedef struct {
  httpg_info_t* stat;
  unsigned long long offset;
//  unsigned long long end;
  char* buffer;
  int buffer_h;
  unsigned int buffer_size;
  unsigned int buffer_used;
  pthread_t thr;
  HTTP_Client* s;
} httpg_state_t;

class httpg_info_t {
 friend int read_callback(unsigned long long offset,unsigned long long size,unsigned char** buf,unsigned long long* bufsize,void* arg);
 friend void* read_thread(void *arg);
 friend void* write_thread(void *arg);
 friend class DataHandleHTTPg;
 private:
  int streams;
  unsigned int chunk_size;
  unsigned long long int requested_size;
  int threads;
  bool have_threads;
  CondSimple lock;
  DataBufferPar* buffer;
  const char* url;
  httpg_state_t* channels;
  bool cancel;
  unsigned long long int content_size;
  bool have_content_size;
  DataHandle* handle;
  DataPoint* point;
  DataHandle::failure_reason_t failure_code;
 public:
  httpg_info_t(void);
  ~httpg_info_t(void);
};

httpg_info_t::httpg_info_t(void) {
  url=NULL;
  buffer=NULL;
  threads=0; have_threads=false;
  channels=NULL;
  cancel=false;
  have_content_size=false;
  content_size=0;
}

httpg_info_t::~httpg_info_t(void) {
  if(channels) { free(channels); channels=NULL; };
}

DataHandle* DataHandleHTTPg::CreateInstance(DataPoint* url_) {
  if((!url_) || (!*url_)) return NULL;
  const char* cur_url = url_->current_location();
  if(strncasecmp("http://",cur_url,7) &&
     strncasecmp("https://",cur_url,8) &&
     strncasecmp("httpg://",cur_url,8) &&
     strncasecmp("se://",cur_url,5)) return NULL;
  return new DataHandleHTTPg(url_);
}

DataHandleHTTPg::DataHandleHTTPg(DataPoint* url_):DataHandleCommon(url_) {
  httpg_stat=NULL;
}

DataHandleHTTPg::~DataHandleHTTPg(void) {
  stop_reading();
  stop_writing();
  deinit_handle();
}

bool DataHandleHTTPg::deinit_handle(void) {
  return DataHandleCommon::deinit_handle();
}

bool DataHandleHTTPg::init_handle(void) {
  if(!DataHandleCommon::init_handle()) return false;
  const char* cur_url = url->current_location();
  if((!strncasecmp("http://",cur_url,7)) ||
     (!strncasecmp("https://",cur_url,8)) ||
     (!strncasecmp("httpg://",cur_url,8)) ||
     (!strncasecmp("se://",cur_url,5))) {
    if(!strncasecmp("se://",cur_url,5)) {
      c_url.replace(0,2,"httpg");
      std::string::size_type n = c_url.find('?');
      if(n != std::string::npos) c_url.replace(n,1,"/");
    };
    return true;
  };
  return false;
}

bool DataHandleHTTPg::analyze(analyze_t &arg) {
  // for HTTP(S/G) bufsize directly affects transaction's length
  // Hence it is better to have it bigger
  arg.bufsize=MAX_BLOCK_SIZE;
  return DataHandleCommon::analyze(arg);
}

bool DataHandleHTTPg::out_of_order(void) {
  return true;
}

static bool CHECK_PROXY(const char* msg,DataHandle::failure_reason_t& code) {
  try {
    Certificate ci(PROXY);
    if(!ci.IsExpired()) return true;
  } catch (std::exception) { // proxy not available
  };
  try {
    Certificate ci(USERCERT);
    if(!ci.IsExpired()) return true;
  } catch (std::exception) { // proxy not available
  };
  odlog(INFO)<<msg<<": proxy expired"<<std::endl;
  code=DataHandle::credentials_expired_failure;
  return false;
}

static int read_callback(unsigned long long offset,unsigned long long size,unsigned char** buf,unsigned long long* bufsize,void* arg) {
  httpg_state_t* tstat = (httpg_state_t*)arg;
  if(tstat->buffer == NULL) return -1;
  if(tstat->buffer_used == 0) {
    tstat->offset=offset;
  };
  // move data from buf to buffer
  for(;;) {
    if(size==0) break;
    unsigned long long l = (tstat->buffer_size - tstat->buffer_used);
    if(l>size) l=size; 
    memcpy(tstat->buffer+tstat->buffer_used,*buf,l);
    tstat->buffer_used+=size;
    size-=l; offset+=l; 
    if(tstat->buffer_used == tstat->buffer_size) { // full buffer
      if(!tstat->stat->buffer->is_read(tstat->buffer_h,
			            tstat->buffer_used,tstat->offset)) {
        tstat->buffer=NULL; tstat->buffer_h=-1;
        tstat->stat->buffer->error_read(true);
        return -1;

      };
      tstat->offset+=tstat->buffer_used;
      // new buffer needed
      if(!tstat->stat->buffer->for_read(tstat->buffer_h,
			                   tstat->buffer_size,true)) {
        tstat->buffer=NULL; tstat->buffer_h=-1;
        return -1;
      };
      tstat->buffer=(*(tstat->stat->buffer))[tstat->buffer_h];
      tstat->buffer_used=0;
    };
  };
  return 0;
}

static void* read_thread(void *arg) {
  httpg_info_t* istat = (httpg_info_t*)arg;
  istat->lock.block();
  if(istat->cancel) { istat->lock.unblock(); return NULL; };
  httpg_state_t* tstat = istat->channels+istat->threads;
  istat->threads++; istat->have_threads=true;
  bool encryption = true;
  if(strncasecmp(istat->url,"https://",8) == 0) encryption=false;
  HTTP_Client s(istat->url,encryption); // use relaxed encryption (integrity only)
  tstat->s=&s;
  istat->lock.unblock();
  tstat->stat=istat;
  bool failed = true;
  if(s) for(;;) {
    // get first buffer
    if(!istat->buffer->for_read(tstat->buffer_h,tstat->buffer_size,true)) break;
    tstat->buffer=(*(istat->buffer))[tstat->buffer_h];
    tstat->buffer_used=0;
    // connect
    odlog(DEBUG)<<"read_thread: calling connect"<<std::endl;
    if(s.connect() != 0) break;
    // acquire chunk
    istat->lock.block();
    tstat->offset = istat->requested_size;
    // check for eof and cancel
    if((istat->cancel) || 
       (istat->have_content_size && (tstat->offset >= istat->content_size))) {
      odlog(DEBUG)<<"read_thread: exiting due to eof or cancel: "<<tstat->offset<<" - "<<istat->content_size<<std::endl;
      istat->lock.unblock();
      istat->buffer->is_read(tstat->buffer_h,0,tstat->offset);
      if(!istat->cancel) failed=false;
      break;
    };
    istat->requested_size+=istat->chunk_size;
    istat->lock.unblock();
    unsigned long long int offset = tstat->offset;
    // recieve chunk
    odlog(DEBUG)<<"read_thread: calling GET: offset: "<<tstat->offset<<std::endl;
    odlog(DEBUG)<<"read_thread: calling GET: size: "<<istat->chunk_size<<std::endl;
    int get_res = s.GET("",tstat->offset,istat->chunk_size,&read_callback,tstat);
    // accept last received data
    if(tstat->buffer) {
      odlog(DEBUG)<<"read_thread: calling is_read: "<<tstat->buffer_used<<std::endl;
      istat->buffer->is_read(tstat->buffer_h,tstat->buffer_used,tstat->offset);
    };
    tstat->offset+=tstat->buffer_used;

    if(get_res != 0) { // failed
    odlog(DEBUG)<<"read_thread: GET failed"<<std::endl;
      istat->buffer->error_read(true);
      break;
    } else {
      unsigned long long int cs = s.response().ContentSize();
      if(cs) {
        istat->point->meta_size(cs);
        istat->content_size=cs; istat->have_content_size=true;
        istat->buffer->speed.set_max_data(cs); // little hack
      };
      if(s.response().haveLastModified()) {
        istat->point->meta_created(s.response().LastModified());
      };
      // if 0 bytes transfered - behind eof
      odlog(DEBUG)<<"read_thread: check for eof: "<<offset<<" - "<<tstat->offset<<std::endl;
      if(offset == tstat->offset) { failed=false; break; }; // eof
    };
  };
  odlog(DEBUG)<<"read_thread: loop exited"<<std::endl;
  istat->lock.block();
  istat->threads--;
  if(istat->threads==0) {
  odlog(DEBUG)<<"read_thread: last thread: failure: "<<failed<<std::endl;
    if(failed) {
      istat->buffer->error_read(true);
      CHECK_PROXY("read_thread",istat->failure_code);
    };
    istat->buffer->eof_read(true);
  };
  tstat->s=NULL;
  istat->lock.signal_nonblock();
  istat->lock.unblock();
  return NULL;  
}

bool DataHandleHTTPg::start_reading(DataBufferPar &buf) {
  if(!DataHandleCommon::start_reading(buf)) return false;
  std::string store_url = c_url;
  // Try to obtain faster URL if can choose
  if(strncasecmp(url->current_location(),"se://",5) == 0) {
    odlog(INFO)<<"Talking to SOAP service at "<<c_url<<std::endl;
    struct soap soap;
    HTTP_ClientSOAP s(c_url.c_str(),&soap);
    soap.namespaces=file_soap_namespaces;
    if(s.connect() != 0) {
      odlog(ERROR)<<"Failed to connect to "<<c_url<<std::endl;
      CHECK_PROXY("start_reading_httpg",failure_code);
      DataHandleCommon::stop_reading();
      return false;
    };
    ns__infoResponse rr;
    std::string soap_url = c_url;
    std::string::size_type n = soap_url.find(':');
    // make gSOAP create http headers (should be removed in a future)
    if(n != std::string::npos) soap_url.replace(0,n,"http");
    if(soap_call_ns__info(&soap,soap_url.c_str(),"info",NULL,rr) != SOAP_OK) {
      odlog(INFO)<<"Failed to execute remote soap call 'info' at "<<c_url<<std::endl;
      DataHandleCommon::stop_reading();
      return false;
    };
    if(rr.error_code != 0) {
      odlog(INFO)<<"Failed ("<<rr.error_code<<") to find remote file "<<c_url<<std::endl;
      DataHandleCommon::stop_reading();
      return false;
    };
    if(rr.__size_file <= 0) { // it must be at least one entry
      odlog(INFO)<<"Failed to find remote file "<<c_url<<std::endl;
      DataHandleCommon::stop_reading();
      return false;
    };
    // and it must be only one
    if((rr.file[0].__size_url > 0) && (rr.file[0].url[0])) {
      store_url=rr.file[0].url[0];
      // hack
      for(n=0;n<rr.file[0].__size_url;n++) {
        odlog(INFO)<<"File is available at: "<<rr.file[0].url[n]<<std::endl;
        if(strncasecmp(rr.file[0].url[n],"https://",8) == 0) // prefer SSL
          store_url=rr.file[0].url[n];
      };
    };
    if(rr.file[0].size) {
      /* provide some metadata */
      odlog(INFO)<<"start_reading_httpg: obtained size: "<<*(rr.file[0].size)<<std::endl;
      url->meta_size(*(rr.file[0].size));
    };
    if(rr.file[0].created) {
      time_t t;
      if(stringtotime(t,rr.file[0].created) == 0) url->meta_created(t);
    };
  };
  if(httpg_stat == NULL) httpg_stat=new httpg_info_t;
  httpg_stat->buffer=&buf; buffer=&buf;
  httpg_stat->url=strdup(store_url.c_str());
  httpg_stat->chunk_size=buf.buffer_size();
  httpg_stat->streams=transfer_streams;
  httpg_stat->requested_size=0; // amount of data currently being downloaded
  httpg_stat->threads=0;
  httpg_stat->cancel=false;
  httpg_stat->have_threads=false;
  if(httpg_stat->channels) free(httpg_stat->channels);
  httpg_stat->channels=(httpg_state_t*)malloc(sizeof(httpg_state_t)*httpg_stat->streams);
  if(httpg_stat->channels==NULL) {
    DataHandleCommon::stop_reading();
    return false;
  };
  httpg_stat->handle = this;
  httpg_stat->point = url;
  httpg_stat->failure_code=failure_code;
  // start threads: one per stream
  httpg_stat->lock.reset();
  int started = 0;
  pthread_attr_t attr;
  if(pthread_attr_init(&attr) != 0) {
    DataHandleCommon::stop_reading();
    return false;
  };
  if(pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_DETACHED) != 0) return false;
  for(;started < httpg_stat->streams;started++) { 
    httpg_state_t* tstat = httpg_stat->channels+started;
    if(pthread_create(&(tstat->thr),&attr,&read_thread,httpg_stat) != 0) break;
  };
  odlog(DEBUG)<<"start_reading_httpg: streams started: "<<started<<std::endl;
  pthread_attr_destroy(&attr);
  if(started==0) {
    DataHandleCommon::stop_reading();
    return false;
  };
  return true;
}

bool DataHandleHTTPg::stop_reading(void) {
  if(!DataHandleCommon::stop_reading()) return false;
  httpg_stat->lock.block();
  failure_code=httpg_stat->failure_code;
  if(!buffer->eof_read()) {
    buffer->error_read(true);
    // cancel transfer
    httpg_stat->cancel=true;
    for(int i = 0;i<httpg_stat->streams;i++) {
      if(httpg_stat->channels[i].s) httpg_stat->channels[i].s->disconnect();
    };
  };
  while(httpg_stat->threads > 0) {
    httpg_stat->lock.wait_nonblock();
  };
  httpg_stat->lock.unblock();
  free((char*)(httpg_stat->url));
  delete httpg_stat;
  return true;
}

static void* write_thread(void *arg) {
  httpg_info_t* istat = (httpg_info_t*)arg;
  istat->lock.block();
  if(istat->cancel) { istat->lock.unblock(); return NULL; };
  httpg_state_t* tstat = istat->channels+istat->threads;
  istat->threads++; istat->have_threads=true;
  odlog(DEBUG)<<"write_thread: threads: "<<istat->threads<<std::endl;
  bool encryption = true;
  if(strncasecmp(istat->url,"https://",8) == 0) encryption=false;
  HTTP_Client s(istat->url,encryption); // use relaxed encryption (integrity only)
  tstat->s=&s;
  istat->lock.unblock();
  tstat->stat=istat;
  bool failed = true;

  if(s) for(;;) {
    /// get first buffer
    if(!istat->buffer->for_write(tstat->buffer_h,tstat->buffer_size,tstat->offset,true)) {
      if(!istat->buffer->error()) failed=false; // eof
      break;
    };
    tstat->buffer=(*(istat->buffer))[tstat->buffer_h];
    // connect
    if(s.connect() != 0) break;
    // send chunk
    int get_res = s.PUT("",tstat->offset,tstat->buffer_size,
                        (unsigned char*)(tstat->buffer),
                        istat->have_content_size?istat->content_size:0,false);
    if(get_res != 0) { // failed
      istat->buffer->is_notwritten(tstat->buffer_h);
      istat->buffer->error_write(true);
      break;
    } else {
      istat->buffer->is_written(tstat->buffer_h);
    };
  };
  istat->lock.block();
  istat->threads--;
  if(istat->threads==0) {
    if(failed) odlog(DEBUG)<<"write_thread: last thread: failured"<<std::endl;
    DataHandle* handle = istat->handle;
    DataPoint* point = istat->point;
    if(point && !failed) {
      if(!istat->buffer->checksum_valid()) { failed=true; }
      else if(strncasecmp(point->current_location(),"se://",5) == 0) {
        // pass meta information to an SE
        struct soap soap;
        HTTP_ClientSOAP s(istat->url,&soap);
        soap.namespaces=file_soap_namespaces;
        if(s.connect() != 0) {
          odlog(ERROR)<<"Failed to connect to "<<istat->url<<std::endl;
          failed=true;
        } else {
          ns__fileinfo info;
          info.size=NULL;
          // only cheksum is important here
          info.checksum=NULL;
          char checksum_[100];
          info.checksum=NULL;
          if(istat->buffer->checksum_valid()) { 
            const CheckSum* cs = istat->buffer->checksum_object();
            if(cs) {
              cs->print(checksum_,100);
              // int l_ = snprintf(checksum_,99,"cksum:%16LX",cs->end()); 
              // checksum_[l_]=0;
              info.checksum=checksum_;
            };
          };
          info.acl=NULL;
          info.id=NULL; 
          info.created=NULL;
          std::string created("");
          if(point->meta_created_available()) {
            struct tm* t;
            struct tm tt;
            time_t t_ = point->meta_created();
            t=gmtime_r(&t_,&tt);
            if(t) {
              if(timetostring(*t,created) == 0) {
                info.created=(char*)(created.c_str());
              };
            };
          };
          odlog(DEBUG)<<"write_thread: update: created: "<<info.created<<std::endl;
          odlog(DEBUG)<<"write_thread: update: created: "<<created<<std::endl;
          ns__updateResponse rr;
          int soap_err = SOAP_OK;
          std::string soap_url = istat->url;
          std::string::size_type n = soap_url.find(':');
          // make gSOAP create http headers (should be removed in a future)
          if(n != std::string::npos) soap_url.replace(0,n,"http");
          if((soap_err=soap_call_ns__update(&soap,soap_url.c_str(),"update",
                                    &info,rr)) != SOAP_OK) {
            odlog(INFO)<<"Failed to execute remote soap call 'update' at "<<istat->url<<std::endl;
            failed=true;
          } else if(rr.error_code != 0) {
            odlog(INFO)<<"Failed ("<<rr.error_code<<") to update remote file "<<info.id<<" at "<<istat->url<<std::endl;
            failed=true;
          };
        };
      };
    };
    if(failed) {
      istat->buffer->error_write(true);
      CHECK_PROXY("write_thread",istat->failure_code);
    };
    istat->buffer->eof_write(true);
  };
  tstat->s=NULL;
  istat->lock.signal_nonblock();
  istat->lock.unblock();
  return NULL;  
}

bool DataHandleHTTPg::start_writing(DataBufferPar &buf,DataCallback *space_cb) {
  if(!DataHandleCommon::start_writing(buf)) return false;
  std::string store_url(c_url.c_str());
  // registration at SE in case of SE url
  if(strncasecmp(url->current_location(),"se://",5) == 0) {
    // pass meta information directly to an SE
    std::string service_url = url->current_location();
    service_url.replace(0,2,"httpg");
    std::string service_path = "";
    {
      std::string::size_type n = service_url.find('?');
      if(n != std::string::npos) {
        service_path=service_url.c_str()+n+1;
        service_url.resize(n);
      };
    };
    canonic_url(service_url);
    store_url=service_url;
    odlog(INFO)<<"Talking to SOAP service at "<<service_url<<std::endl;
    struct soap soap;
    HTTP_ClientSOAP s(service_url.c_str(),&soap);
    soap.namespaces=file_soap_namespaces;
    if(s.connect() != 0) {
      odlog(ERROR)<<"Failed to connect to "<<service_url<<std::endl;
      CHECK_PROXY("start_writing_httpg",failure_code);
      DataHandleCommon::stop_writing();
      return false;
    };
    ns__fileinfo info;
    ULONG64 size_ = url->meta_size();
    if(url->meta_size_available()) info.size=&size_;
    char checksum_[100];
    info.checksum=NULL;
    int l_ = snprintf(checksum_,99,"%16LX",url->meta_checksum()); 
    checksum_[l_]=0;
    if(url->meta_checksum_available()) info.checksum=checksum_;
    info.acl=NULL;
    std::string acl("");
    {
      try {
        Certificate ci;
        acl=ci.GetIdentitySN();
      } catch(std::exception) { };
      if(acl.length()) acl="<gacl><entry><person><dn>"+acl+"</dn></person><allow><read/><write/><list/><admin/></allow></entry></gacl>";
      info.acl=(char*)(acl.c_str());
    };
    if(url->lfn() && url->lfn()[0]) {
      info.id=strdup(url->lfn()); 
    } else if(service_path.length()) {
      info.id=strdup(service_path.c_str()); 
    } else {
      odlog(ERROR)<<"Missing file id for SE URL"<<std::endl;
      DataHandleCommon::stop_writing();
      return false;
    };
    info.created=NULL;
    if(url->meta_created_available()) {
      struct tm* t;
      struct tm tt;
      time_t t_ = url->meta_created();
      t=gmtime_r(&t_,&tt);
      if(t) {
        std::string created;
        if(timetostring(*t,created) == 0) {
          info.created=strdup(created.c_str());
        };
      };
    };
    odlog(DEBUG)<<"start_writing_httpg: id(lfn): "<<info.id<<std::endl;
    odlog(DEBUG)<<"start_writing_httpg: acl: "<<info.acl<<std::endl;
    odlog(DEBUG)<<"start_writing_httpg: created: "<<info.created<<std::endl;
    ns__addResponse rr;
    int soap_err = SOAP_OK;
    std::string soap_url = service_url;
    std::string::size_type n = soap_url.find(':');
    // make gSOAP create http headers (should be removed in a future)
    if(n != std::string::npos) soap_url.replace(0,n,"http");
    if((soap_err=soap_call_ns__add(&soap,soap_url.c_str(),"add",&info,0,NULL,
                                                 rr)) != SOAP_OK) {
      odlog(INFO)<<"Failed to execute remote soap call 'add' at "<<service_url<<std::endl;
      if(info.id) free(info.id);
      if(info.created) free(info.created);
      DataHandleCommon::stop_writing();
      return false;
    };
    if(rr.error_code != 0) {
      odlog(INFO)<<"Failed ("<<rr.error_code<<") to create remote file "<<info.id<<" at "<<service_url<<std::endl;
      if(info.id) free(info.id);
      if(info.created) free(info.created);
      DataHandleCommon::stop_writing();
      return false;
    };
    if(info.id) free(info.id);
    if(info.created) free(info.created);
    if((rr.file.url) && (rr.file.__size_url>0) && (rr.file.url[0])) {
      store_url=rr.file.url[0];
      // hack
      for(n=0;n<rr.file.__size_url;n++) {
        odlog(INFO)<<"File is available at: "<<rr.file.url[n]<<std::endl;
        if(strncasecmp(rr.file.url[n],"https://",8) == 0) // prefer SSL
          store_url=rr.file.url[n];
      };
    } else {
      if(store_url[store_url.length()-1] != '/') store_url+="/";
      if(url->lfn()[0] == '/') {
        store_url+=(url->lfn()+1);
      } else {
        store_url+=url->lfn();
      };
    };
    odlog(DEBUG)<<"Created slot at: "<<store_url<<std::endl;
  };
  // starting data transfer
  if(httpg_stat == NULL) httpg_stat=new httpg_info_t;
  httpg_stat->buffer=&buf; buffer=&buf;
  httpg_stat->url=strdup(store_url.c_str());
  httpg_stat->chunk_size=buf.buffer_size(); // not actually needed for writing
  httpg_stat->streams=transfer_streams;
  httpg_stat->requested_size=0;
  httpg_stat->threads=0;
  httpg_stat->cancel=false;
  httpg_stat->have_threads=false;
  httpg_stat->have_content_size=url->meta_size_available();
  httpg_stat->content_size=url->meta_size();
  if(httpg_stat->channels) free(httpg_stat->channels);
  httpg_stat->handle = this;
  httpg_stat->point = url;
  httpg_stat->failure_code=failure_code;
  httpg_stat->channels=(httpg_state_t*)malloc(sizeof(httpg_state_t)*httpg_stat->streams);
  if(httpg_stat->channels==NULL) {
    DataHandleCommon::stop_writing();
    return false;
  };
  // start threads: one per stream
  httpg_stat->lock.reset();
  int started = 0;
  pthread_attr_t attr;
  if(pthread_attr_init(&attr) != 0) {
    DataHandleCommon::stop_writing();
    return false;
  };
  if(pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_DETACHED) != 0) {
    DataHandleCommon::stop_writing();
    return false;
  };
  for(;started < httpg_stat->streams;started++) { 
  odlog(DEBUG)<<"start_writing_httpg: started: "<<started<<std::endl;
    httpg_state_t* tstat = httpg_stat->channels+started;
    if(pthread_create(&(tstat->thr),&attr,&write_thread,httpg_stat) != 0) break;
  };
  pthread_attr_destroy(&attr);
  odlog(DEBUG)<<"start_writing_httpg: started: "<<started<<std::endl;
  if(started==0) {
    DataHandleCommon::stop_writing();
    return false;
  };
  return true;
}

bool DataHandleHTTPg::stop_writing(void) {
  if(!DataHandleCommon::stop_writing()) return false;
  httpg_stat->lock.block();
  failure_code=httpg_stat->failure_code;
  if(!buffer->eof_read()) {
    buffer->error_read(true);
    // cancel transfer
    httpg_stat->cancel=true;
    for(int i = 0;i<httpg_stat->streams;i++) {
      if(httpg_stat->channels[i].s) httpg_stat->channels[i].s->disconnect();
    };
  };
  while(httpg_stat->threads > 0) {
    httpg_stat->lock.wait_nonblock();
  };
  httpg_stat->lock.unblock();
  free((char*)(httpg_stat->url));
  delete httpg_stat;
  return true;
}

bool DataHandleHTTPg::remove(void) {
  if(!DataHandleCommon::remove()) return false;
  odlog(DEBUG)<<"DataHandle::remove_httpg: "<<url->current_location()<<" ("<<c_url<<")"<<std::endl;
  if(strncasecmp(url->current_location(),"se://",5) != 0) {
    odlog(ERROR)<<"Removing for URL "<<url->current_location()<<" is not supported"<<std::endl;
    return false;
  };
  struct soap soap;
  HTTP_ClientSOAP s(c_url.c_str(),&soap);
  soap.namespaces=file_soap_namespaces;
  odlog(DEBUG)<<"DataHandle::remove_httpg: created HTTP_ClientSOAP"<<std::endl;
  if(s.connect() != 0) {
    odlog(ERROR)<<"Failed to connect to "<<c_url<<std::endl;
    return false;
  };
  odlog(DEBUG)<<"DataHandle::remove_httpg: HTTP_ClientSOAP connected"<<std::endl;
  ns__delResponse rr;
  int soap_err = SOAP_OK;
  std::string soap_url = c_url;
  std::string::size_type n = soap_url.find(':');
  // make gSOAP create http headers (should be removed in a future)
  if(n != std::string::npos) soap_url.replace(0,n,"http");
  odlog(DEBUG)<<"DataHandle::remove_httpg: calling soap_call_ns__del"<<std::endl;
  if((soap_err=soap_call_ns__del(&soap,soap_url.c_str(),"del",rr))
                                                             != SOAP_OK) {
    odlog(INFO)<<"Failed to execute remote soap call 'del' at "<<c_url<<std::endl;
    return false;
  };
  if(rr.error_code != 0) {
    odlog(INFO)<<"Failed ("<<rr.error_code<<") to delete remote file "<<c_url<<std::endl;
    return false;
  };
  odlog(DEBUG)<<"DataHandle::remove_httpg: soap_call_ns__del finished"<<std::endl;
  return true;
}

static int check_callback(unsigned long long offset,unsigned long long size,unsigned char** buf,unsigned long long* bufsize,void* arg) {
  return 0;
}

bool DataHandleHTTPg::check(void) {
  if(!DataHandleCommon::check()) return false;
  HTTP_Client s(c_url.c_str());
  if(s.connect() != 0) return false; // failed to connect
  int get_res = s.GET("",0,1,&check_callback,NULL); // recieve chunk
  if(get_res != 0) return false;
  unsigned long long int size = s.response().ContentSize();
  if(size) url->meta_size(size);
  if(s.response().haveLastModified()) {
    url->meta_created(s.response().LastModified());
  };
  return true;
}

bool DataHandleHTTPg::list_files(std::list<DataPoint::FileInfo> &files,bool long_list,bool resolve,bool metadata) {
  if(!DataHandleCommon::list_files(files,long_list,resolve,metadata)) return false;
  odlog(DEBUG)<<"list_files_httpg"<<std::endl;

  if(strncasecmp(url->current_location(),"se://",5) != 0) {
    if(!check()) return false;
    std::string::size_type n = c_url.rfind('/');
    if(n == std::string::npos) n=c_url.length()-1;
    const char* id = c_url.c_str()+n+1;
    std::list<DataPoint::FileInfo>::iterator f =
          files.insert(files.end(),DataPoint::FileInfo(id));
    f->metadata["path"] = id;
    f->type=DataPoint::FileInfo::file_type_file;
    f->metadata["type"] = "file";
    if(url->meta_size_available()) {
      f->size=url->meta_size();
      f->size_available=true;
      f->metadata["size"] = inttostring(url->meta_size());
    };
    if(url->meta_created_available()) {
      f->created=url->meta_created();
      f->created_available=true;
      const time_t creationtime = url->meta_created();
      std::string ctimestr = ctime(&creationtime);
      f->metadata["ctime"] = ctimestr.erase(ctimestr.length()-1);
    };
    // odlog(ERROR)<<"Listing for URL "<<url->current_location()<<" is not supported"<<std::endl;
    return true;
  };
  struct soap soap;
  const char* service_pattern = NULL;
  std::string pattern(""); 
  get_url_option(url->base_url(),"pattern",0,pattern);
  if(strchr(url->current_location(),'?') == NULL) {
    service_pattern=".*";
    if(pattern.length()) service_pattern=pattern.c_str();
  };
  // std::string service_url = c_url;
  HTTP_ClientSOAP s(c_url.c_str(),&soap);
  soap.namespaces=file_soap_namespaces;
  odlog(DEBUG)<<"Connecting to service at "<<c_url<<std::endl;
  if(s.connect() != 0) return false; // failed to connect
  ns__infoResponse rr;
  // std::string soap_url = service_url;
  // std::string::size_type n = soap_url.find(':');
  //// make gSOAP create http headers (should be removed in a future)
  // if(n != std::string::npos) soap_url.replace(0,n,"http");
  if(soap_call_ns__info(&soap,s.SOAP_URL(),"info",(char*)service_pattern,rr) != SOAP_OK) {
    odlog(INFO)<<"Failed to execute remote soap call 'info' at "<<c_url<<std::endl;
    return false;
  };
  if(rr.error_code != 0) {
    odlog(INFO)<<"Failed ("<<rr.error_code<<") to list remote files at "<<c_url<<std::endl;
    return false;
  };
  for(int i = 0;i<rr.__size_file;i++) {
    char* id = rr.file[i].id;
    if(id) {
      std::list<DataPoint::FileInfo>::iterator f =
          files.insert(files.end(),DataPoint::FileInfo(id));
      f->type=DataPoint::FileInfo::file_type_file;
      if(rr.file[i].size) {
        f->size=*(rr.file[i].size); f->size_available=true;
      };
      // char* checksum;
      // char* acl;
      // char* created;
      // enum ns__filestate* state;
      // int __size_url;
      // char** url;
    };
  };
  return true;
}

