#ifndef __DATAPOINT_H__
#define __DATAPOINT_H__

#include "../std.h"
#include <string>
#include <list>
#include <map>
#include "../transfer/url_map.h"
#include "../misc/globus_modules.h"
#include "../misc/condition.h"

class RCManager;

#ifdef HAVE_GLOBUS_RLS_CLIENT_H
extern "C" {
#include <globus_rls_client.h>
}
#endif


/// DataPoint is an abstraction of URL. It can handle URLs of type file://, 
/// ftp://, gsiftp://, http://, https://, httpg:// (HTTP over GSI), 
/// se:// (NG web service over HTTPG) and meta-URLs (URLs of Infexing 
/// Services) rc://, rls://.
/// DataPoint provides means to resolve meta-URL into multiple URLs and
/// to loop through them.
/// DataPoint is both base virtual class and wrapper. It's implementation
/// provides only redirection methods and type definitions.

// DataPoint  -> DataPointDirect  -> DataPointMeta
//               |-> DataPointFile   |-> DataPointRLS
//               |-> DataPointFTP    \-> DataPointRC
//               |-> DataPointHTTP
//               \-> DataPointSRM

class DataPoint {
 private:
  DataPoint* instance;
  typedef DataPoint* (*constructor_t)(const char* url);
  static std::list<constructor_t> protocols;
  static LockSimple protocols_lock;
 protected:
  virtual bool process_meta_url(void);
  DataPoint(void):instance(NULL) { };
 public:
  /// FileInfo stores information about file (meta-information). Although all
  /// members are public it is mot desirable to modify them directly outside
  /// DataPoint class.
  class FileInfo {
   public:
    typedef enum {
      file_type_unknown = 0,
      file_type_file = 1,
      file_type_dir = 2
    } Type;
    std::string name;
    std::list<std::string> urls; /// Physical enpoints/URLs at which file can be accessed.
    long long int size;  /// Size of filein bytes.
    bool size_available;          /// If size is known.
    std::string checksum;  /// Checksum of file.
    bool checksum_available;          /// If checksum is known.
    time_t created;          /// Creation/modification time.
    bool created_available;  /// If time is known.
    time_t valid;            /// Valid till time.
    bool valid_available;    /// If validity is known.
    std::string latency;     /// Access latency
    bool latency_available;  /// If latency is known
    Type type;               /// Fyle type - usually file_type_file - ordinary file.
    std::map<std::string, std::string> metadata; /// Generic metadata arrtribute-value pairs.
    FileInfo(const char *name_ = ""):name(name_),size_available(false),checksum_available(false),created_available(false),valid_available(false),latency_available(false),type(file_type_unknown) { };
    operator bool(void) { return (name.length() != 0); }; /// If object is valid
  };

  /// Constructor requires URL or meta-URL to be provided
  DataPoint(const char* url);
  virtual ~DataPoint(void);

  /// Register new protocol. Any new instance of DataPoint will 
  /// recognize it.
  static bool AddProtocol(constructor_t constructor);
  static DataPoint* CreateInstance(const char* url);
  DataPoint* Instance(void) { if(instance) return instance; return this; };
  const DataPoint* constInstance(void) const { if(instance) return instance; return this; };

  /* 
   *  META actions. Valid only for meta-URLs
   */
  /// Resolve meta-URL into list of ordinary URLs and obtain meta-information
  /// about file. Can be called for object representing ordinary URL or
  /// already resolved object.
  /// \param source true if DataPoint object represents source of information
  virtual bool meta_resolve(bool source);
  /// Resolve meta-URL into list of ordinary URLs and obtain meta-information
  /// about file. Also sort obtained list so that URLs mentioned in UrlMap
  /// object are placed first. This is used during transfer to access
  /// local locations first.
  /// \param maps list of mappings of remote URLs to (potentially) local locations.
  virtual bool meta_resolve(bool source,const UrlMap &maps);
  ///  This function registers physical location of file into Indexing
  ///  Service. It should be called *before* actual transfer to that
  ///  location happens.
  ///  \param replication if true then file is being replicated between 2 locations registered in Indexing Service under same name.
  ///  \param force if true, perform registration of new file even if it already exists. Should be used to fix failures in Indexing Service.
  virtual bool meta_preregister(bool replication,bool force = false);
  /// Used for same purpose as meta_preregister. Should be called after
  /// actual transfer of file successfully finished.
  ///  \param replication if true then file is being replicated between 2 locations registered in Indexing Service under same name.
  /// \param failure not used.
  virtual bool meta_postregister(bool replication,bool failure);
  // Same operation as meta_preregister and meta_postregister together.
  virtual bool meta_register(bool replication);
  ///  Should be called if file transfer failed. It removes changes made
  ///  by meta_preregister.
  virtual bool meta_preunregister(bool replication);
  ///  Remove information about file registered in Indexing Service.
  ///  \param all if true information about file itself is (LFN) is removed. Otherwise only particular physical instance is unregistered.
  virtual bool meta_unregister(bool all);
  /// Obtain information about objects and their properties available
  /// under meta-URL of DataPoint object. It works only for meta-URL.
  /// \param files list of obtained objects.
  /// \param long_list if true find details for each file
  /// \param resolve if true resolve location URLs (for meta-URLs only).
  /// \param metadata if true fill all possible metadata in files
  virtual bool list_files(std::list<DataPoint::FileInfo> &files,bool long_list = false,bool resolve = false,bool metadata=false);
  /// Retrieve properties of object pointed by meta-URL of DataPoint
  /// object. It works only for meta-URL.
  /// \param fi contains retrieved information.
  virtual bool get_info(DataPoint::FileInfo &fi);
  /*
   * Set and get corresponding meta-information related to URL.
   * Those attributes can be supported by non-meta-URLs too.  
   */
  /// Check if meta-information 'size' is available.
  virtual bool meta_size_available(void) const;
  /// Set value of meta-information 'size' if not already set.
  virtual void meta_size(unsigned long long int val);
  /// Set value of meta-information 'size'.
  virtual void meta_size_force(unsigned long long int val);
  /// Get value of meta-information 'size'.
  virtual unsigned long long int meta_size(void) const;
  /// Check if meta-information 'checksum' is available.
  virtual bool meta_checksum_available(void) const;
  /// Set value of meta-information 'checksum' if not already set.
  virtual void meta_checksum(const char* val);
  /// Set value of meta-information 'checksum'.
  virtual void meta_checksum_force(const char* val);
  /// Get value of meta-information 'checksum'.
  virtual const char* meta_checksum(void) const;
  /// Check if meta-information 'creation/modification time' is available.
  virtual bool meta_created_available(void) const;
  /// Set value of meta-information 'creation/modification time' if not already set.
  virtual void meta_created(time_t val);
  /// Set value of meta-information 'creation/modification time'.
  virtual void meta_created_force(time_t val);
  /// Get value of meta-information 'creation/modification time'.
  virtual time_t meta_created(void) const;
  /// Check if meta-information 'validity time' is available.
  virtual bool meta_validtill_available(void) const;
  /// Set value of meta-information 'validity time' if not already set.
  virtual void meta_validtill(time_t val);
  /// Set value of meta-information 'validity time'.
  virtual void meta_validtill_force(time_t val);
  /// Get value of meta-information 'validity time'.
  virtual time_t meta_validtill(void) const;
  /// Check if the given metadata attribute is defined
  virtual bool has_meta_attribute(std::string attr) const;
  /// Get the value of the given metadata attribute
  virtual std::string meta_attribute(std::string name) const;
  /// Check if URL is meta-URL.
  virtual bool meta(void) const;
  /// If endpoint can have any use from meta information.
  virtual bool accepts_meta(void);
   /// If endpoint can provide at least some meta information directly.
  virtual bool provides_meta(void);
  /// Acquire meta-information from another object. Defined values a
  /// not overwritten.
  /// \param p object from which information is taken.
  virtual void meta(const DataPoint &p);
   /// Compare meta-information form another object. Undefined values
  /// are not used for comparison. Default result is 'true'.
  /// \param p object to which compare.
  virtual bool meta_compare(const DataPoint &p) const;
  /// Check if file is registered in Indexing Service. Proper value is
  /// obtainable only after meta-resolve.
  virtual bool meta_stored(void);
  /// Check if file is local (URL is something like file://).
  virtual bool local(void) const;
  /// Map url (change it) according to table provided in maps.
  /// \param maps mapping information.
  virtual bool map(const UrlMap &maps);
  /// Sort list of URLs so that those listed in mapping table are put first.
  /// It can also implement any other algoritm too.
  /// \param maps mapping information.
  virtual bool sort(const UrlMap &maps);
  // virtual DataPoint& operator= (const DataPoint &);
  virtual operator bool (void) const;
  virtual bool operator !(void) const;
  /*
   *  Methods to manage list of locations.
   */
  /// Returns current (resolved) URL.
  virtual const char* current_location(void) const;
  /// Returns meta information used to create curent URL. For RC that is
  ///  location's name. For RLS that is equal to pfn.
  virtual const char* current_meta_location(void) const;
  /// Switch to next location in list of URLs. At last location
  /// switch to first if number of allowed retries does not exceeded.
  /// Returns false if no retries left.
  virtual bool next_location(void);
  /// Returns false if out of retries.
  virtual bool have_location(void) const;
  /// Returns true if number of resolved URLs is not 0.
  virtual bool have_locations(void) const;
  /// Remove current URL from list
  virtual bool remove_location(void);
  /// Remove locations present in another DataPoint object
  virtual bool remove_locations(const DataPoint& p);
  /// Returns number of retries left.
  virtual int tries(void);
  /// Set number of retries.
  virtual void tries(int n);
  /// Returns URL which was passed to constructor
  virtual std::string base_url(void) const;
  /// Returns URL which was passed to constructor with location
  /// names and options removed, port number added.
  virtual std::string canonic_url(void) const;
  /// Returns name which is given to file in Indexing Service (aka LFN).
  virtual const char* lfn(void) const;
  /// Add URL to list.
  /// \param meta meta-name (name of location/service).
  /// \param loc URL.
  virtual bool add_location(const char* meta,const char* loc);
};

/// DataPointDirect implements common purpose private attributes and
/// corresponding methods suitable for any kind of URL.
/// It should never be used directly.
class DataPointDirect: public DataPoint {
 protected:
  /// DataPointDirect::Location represents physical service at which files are
  /// located aka "base URL" inculding it's name (as given in Indexing Service).  /// Currently it is used only internally by classes derived from 
  /// DataPointDirect class and for printing debug information.
  class Location {
   friend class DataPointDirect;
   public:
    std::string meta; // Given name of location
    std::string url;  // location aka pfn aka access point
    bool existing;
    void* arg;   // to be used by different pieces of soft differently
    Location(void):existing(true),arg(NULL) { };
    Location(const char* url_):meta(""),url(url_),existing(true),arg(NULL) { };
    Location(const char* meta_,const char* url_,bool existing_ = true):meta(meta_),url(url_?url_:""),existing(existing_),arg(NULL) { };
    Location(const std::string &url_):meta(""),url(url_),existing(true),arg(NULL) { };
    Location(const std::string &meta_,const std::string &url_):meta(meta_),url(url_),existing(true),arg(NULL) { };
  };

  std::list<Location> locations; /// List of locations at which file can be probably found.
  std::list<Location>::iterator location;

  bool is_valid;
  std::string url; /// Initial URL
  std::string common_url_options; /// URL options to be added to all derived URLs
  // meta attributes
  unsigned long long int meta_size_;
  bool meta_size_valid;
  std::string meta_checksum_;
  bool meta_checksum_valid;
  time_t meta_created_;
  bool meta_created_valid;
  time_t meta_validtill_;
  bool meta_validtill_valid;
  std::map<std::string,std::string> meta_attributes;
  // retries
  int tries_left;
  // globus modules
  GlobusModuleErrors error_mod;
  GlobusModuleGSIGSSAPI gsi_gssapi_mod;
 public:
  DataPointDirect(const char* u);
  virtual ~DataPointDirect(void) { };
  virtual operator bool (void) const { return is_valid; };
  virtual bool operator ! (void) const { return !is_valid; };

  virtual std::string base_url(void) const;
  virtual std::string canonic_url(void) const;

  // META attributes
  virtual bool meta_size_available(void) const { return meta_size_valid; };
  virtual void meta_size(unsigned long long int val) {
    if(!meta_size_valid) { meta_size_=val; meta_size_valid=true; };
  };
  virtual void meta_size_force(unsigned long long int val) {
    meta_size_=val; meta_size_valid=true;
  };
  virtual unsigned long long int meta_size(void) const {
    if(meta_size_valid) return meta_size_; return 0;
  };
  virtual bool meta_checksum_available(void) const { return meta_checksum_valid; };
  virtual void meta_checksum(const char* val) {
    if(!meta_checksum_valid) { meta_checksum_=val; meta_checksum_valid=true; };
  };
  virtual void meta_checksum_force(const char* val) {
    meta_checksum_=val; meta_checksum_valid=true;
  };
  virtual const char* meta_checksum(void) const {
    if(meta_checksum_valid) return meta_checksum_.c_str(); return "";
  };
  virtual bool meta_created_available(void) const { return meta_created_valid; };
  virtual void meta_created(time_t val) {
    if(!meta_created_valid) { meta_created_=val; meta_created_valid=true; };
  };
  virtual void meta_created_force(time_t val) {
    meta_created_=val; meta_created_valid=true;
  };
  virtual time_t meta_created(void) const {
    if(meta_created_valid) return meta_created_; return 0;
  };
  virtual bool meta_validtill_available(void) const { return meta_validtill_valid; };
  virtual void meta_validtill(time_t val) {
    if(!meta_validtill_valid) {meta_validtill_=val; meta_validtill_valid=true;};  };
  virtual void meta_validtill_force(time_t val) {
    meta_validtill_=val; meta_validtill_valid=true;
  };
  virtual time_t meta_validtill(void) const {
    if(meta_validtill_valid) return meta_validtill_; return 0;
  };
  virtual bool has_meta_attribute(const std::string attr) const { 
    std::map<std::string,std::string>::const_iterator iter = meta_attributes.find(attr);
    return iter != meta_attributes.end();
  };
  virtual std::string meta_attribute(const std::string name) const {
    std::map<std::string,std::string>::const_iterator iter = meta_attributes.find(name);
    if (iter == meta_attributes.end()) return "";
    return iter->second;
  };
  virtual void meta(const DataPoint &p) {
    if(p.meta_size_available())      meta_size(p.meta_size());
    if(p.meta_checksum_available())  meta_checksum(p.meta_checksum());
    if(p.meta_created_available())   meta_created(p.meta_created());
    if(p.meta_validtill_available()) meta_validtill(p.meta_validtill());
  };
  virtual bool meta_compare(const DataPoint &p) const {
    if(p.meta_size_available() && meta_size_valid)
      if(meta_size_ != p.meta_size()) return false;
    // TODO: compare checksums properly
    if(p.meta_checksum_available() && meta_checksum_valid)
      if(strcasecmp(meta_checksum_.c_str(),p.meta_checksum())) return false;
    if(p.meta_created_available() && meta_created_valid)
      if(meta_created_ != p.meta_created()) return false;
    if(p.meta_validtill_available() && meta_validtill_valid)
      if(meta_validtill_ != p.meta_validtill()) return false;
    return true;
  };

  // non-meta URL
  virtual bool meta(void) const { return false; };
  virtual bool accepts_meta(void) { return false; };
  virtual bool provides_meta(void) { return false; };
  virtual bool meta_resolve(bool source) { return true; };
  virtual bool meta_resolve(bool source,const UrlMap &maps) { return true; };
  virtual bool meta_preregister(bool replication,bool force = false) { return true; };
  virtual bool meta_postregister(bool replication,bool failure) { return true; };
  virtual bool meta_preunregister(bool replication) { return true; };
  virtual bool meta_unregister(bool all) { return true; };
  virtual bool get_info(DataPoint::FileInfo &fi) { return true; };
  virtual bool meta_stored(void) { return false; };

  // Locations and retries 
  virtual const char* current_location(void) const {
    if(((std::list<Location>::const_iterator)location) == 
       locations.end()) return "";
    return location->url.c_str();
  };
  virtual const char* current_meta_location(void) const {
    if(((std::list<Location>::const_iterator)location) ==
       locations.end()) return "";
    return location->meta.c_str();
  };
  virtual bool next_location(void);
  virtual bool have_location(void) const;
  virtual bool have_locations(void) const;
  virtual bool remove_location(void);
  virtual bool remove_locations(const DataPoint& p);
  virtual bool add_location(const char* meta,const char* loc);
  virtual int tries(void);
  virtual void tries(int n);
  virtual bool local(void) const { return false; };
  virtual bool sort(const UrlMap &maps);
  virtual bool map(const UrlMap &maps);
  virtual bool list_files(std::list<DataPoint::FileInfo> &files,bool long_list = false,bool resolve = false,bool metadata = false) { return false; };

};

/// DataPointMeta complements DataPointDirect with attributes
/// common for meta-URLs
/// It should never be used directly.
class DataPointMeta: public DataPointDirect {
 protected:
  bool is_metaexisting;
  bool is_resolved;
  std::string meta_service_url;
  std::string meta_lfn;
  virtual bool process_meta_url(void);
  bool extract_meta_attributes(std::string& lfn);
  void fix_unregistered(bool all);
 public:
  DataPointMeta(const char* u);
  virtual ~DataPointMeta(void) { };
  // meta URL
  virtual bool meta(void) const { return true; };
  virtual bool accepts_meta(void) { return true; };
  virtual bool provides_meta(void) { return true; };
  virtual bool meta_resolve(bool source) { return false; };
  virtual bool meta_resolve(bool source,const UrlMap &maps);
  virtual bool meta_preregister(bool replication,bool force = false) { return false; };
  virtual bool meta_postregister(bool replication,bool failure) { return false; };
  virtual bool meta_register(bool replication) {
    if(!meta_preregister(replication)) return false;
    if(!meta_postregister(replication,false)) return false;
    return true;
  };
  virtual bool meta_preunregister(bool replication) { return false; };
  virtual bool meta_unregister(bool all) { return false; };
  virtual bool get_info(DataPoint::FileInfo &fi);
  virtual bool meta_stored(void) { return is_metaexisting; };
  virtual const char* lfn(void) const { return meta_lfn.c_str(); };
};

class DataPointFile: public DataPointDirect {
 private:
  bool is_stdchannel;
 public:
  DataPointFile(const char* url);  
  virtual ~DataPointFile(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool provides_meta(void) { return true; };
  virtual bool local(void) const { return true; };
};

class DataPointFTP: public DataPointDirect {
 private:
  bool is_secure;
 public:
  DataPointFTP(const char* url);
  virtual ~DataPointFTP(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool provides_meta(void) { return true; };
};

class DataPointHTTP: public DataPointDirect {
 private:
  bool is_se;
  bool is_http;
  bool is_https;
  bool is_httpg;
 public:
  DataPointHTTP(const char* url);
  virtual ~DataPointHTTP(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool provides_meta(void) { return is_se; };
  virtual bool accepts_meta(void) { return is_se; };
};

class DataPointSRM: public DataPointDirect {
 private:
 public:
  DataPointSRM(const char* url);
  virtual ~DataPointSRM(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool provides_meta(void) { return true; };
};


class DataPointRLS: public DataPointMeta {
 private:
  std::string pfn_path;
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  GlobusModuleCommon common_mod;
  GlobusModuleIO io_mod;
  GlobusModuleRLSClient rls_mod;
  static bool meta_resolve_callback(globus_rls_handle_t* h,const char* url,void* arg);
  static bool list_files_callback(globus_rls_handle_t* h,const char* url,void* arg);
  static bool meta_unregister_callback(globus_rls_handle_t* h,const char* url,void* arg);
#endif
 protected:
  bool guid_enabled;
  virtual bool process_meta_url(void);
 public:
  DataPointRLS(const char* url);
  virtual ~DataPointRLS(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool meta_resolve(bool source);
  virtual bool meta_preregister(bool replication,bool force = false);
  virtual bool meta_postregister(bool replication,bool failure);
  virtual bool meta_preunregister(bool replication);
  virtual bool meta_unregister(bool all);
  virtual bool list_files(std::list<DataPoint::FileInfo> &files,bool long_list = false,bool resolve = false,bool metadata = false);
};

class DataPointRC: public DataPointMeta {
 private:
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  GlobusModuleReplicaCatalog replica_mod;
#endif
  RCManager *rc_mgr;
 protected:
  virtual bool process_meta_url(void);
 public:
  DataPointRC(const char* url);
  virtual ~DataPointRC(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool meta_resolve(bool source);
  virtual bool meta_preregister(bool replication,bool force = false);
  virtual bool meta_postregister(bool replication,bool failure);
  virtual bool meta_preunregister(bool replication);
  virtual bool meta_unregister(bool all);
  virtual bool list_files(std::list<DataPoint::FileInfo> &files,bool long_list = false,bool resolve = false,bool metadata = false);
};

class DataPointLFC: public DataPointMeta {
 private:
  std::string guid;
  bool resolveGUIDToLFN();
 protected:
  bool guid_enabled;
  virtual bool process_meta_url(void);
 public:
  DataPointLFC(const char* url);
  virtual ~DataPointLFC(void);
  static DataPoint* CreateInstance(const char* url);
  virtual bool meta_resolve(bool source);
  virtual bool meta_preregister(bool replication,bool force = false);
  virtual bool meta_postregister(bool replication,bool failure);
  virtual bool meta_preunregister(bool replication);
  virtual bool meta_unregister(bool all);
  virtual bool list_files(std::list<DataPoint::FileInfo> &files,bool long_list = false,bool resolve = false,bool metadata = false);
};

std::ostream& operator<<(std::ostream& o,const DataPoint &point);

#endif

