#include "../../std.h"

#include "srm_utils.h"

#include "srm_request.h"

SRMRequest::SRMRequest(int id,const char* userid,const char* type):state_("pending"),id_(id),type_(type) {
  created_=time(NULL);
  if(userid) userid_=userid;
}

SRMRequest::~SRMRequest(void) {
}

SRMRequestFile* SRMRequest::add(SEFiles::iterator& file) {
  std::list<SRMRequestFile>::iterator f = files_.insert(files_.end(),file);
  if(f == files_.end()) return NULL;
  return &(*f);
}

const char* SRMRequest::type(void) {
  return "none";
}


SRMv1Type__RequestStatus* SRMRequest::get(struct soap* sp,const char* service_url) {
  // Allocate structures
  int n = files_.size();
  SRMv1Type__RequestStatus* r=soap_new_SRMv1Type__RequestStatus(sp,-1);
  if(r == NULL) return NULL;
  r->soap_default(sp);
  if(n<=0) return r;
  r->fileStatuses=soap_new_ArrayOfRequestFileStatus(sp,-1);
  if(r->fileStatuses == NULL) return NULL;
  r->fileStatuses->soap_default(sp);
  r->fileStatuses->__size=0;
  r->fileStatuses->__ptr=(SRMv1Type__RequestFileStatus**)soap_malloc(sp,
                                     sizeof(SRMv1Type__RequestFileStatus*)*n);
  if(r->fileStatuses->__ptr == NULL) return NULL;
  r->fileStatuses->__size=n;
  for(int i=0;i<n;i++) {
    r->fileStatuses->__ptr[i]=soap_new_SRMv1Type__RequestFileStatus(sp,-1);
    if(r->fileStatuses->__ptr[i] == NULL) return NULL;
    r->fileStatuses->__ptr[i]->soap_default(sp);
  };
  // 
  time_t* t = (time_t*)soap_malloc(sp,sizeof(time_t)); if(t) time(t);
  // Fill common information
  r->requestId=id_;
  r->state=soap_strdup(sp,state_.c_str());
  r->type=soap_strdup(sp,type_.c_str());
  r->submitTime=&created_;
  r->startTime=&created_;
  r->finishTime=&created_;
  r->estTimeToStart=0;
  r->errorMessage=NULL;
  r->retryDeltaTime=0;
  // Fill information per file
  n=0;
  bool have_failed = false;
  bool have_active = false;
  bool have_pending = false;
  for(std::list<SRMRequestFile>::iterator f_p = files_.begin();
                                            f_p!=files_.end();++f_p,++n) {
    SRMv1Type__RequestFileStatus* f_status = r->fileStatuses->__ptr[n];
    f_status->fileId=n;
    if(!(f_p->f_.exists())) {
      f_status->state="Failed";
      continue;
    };
    SEFile& f = *(f_p->f_);
    bool running = f_p->running_;
    bool shorturl = f_p->shorturl_;
    // Prepare answer about this file
    f_status->size=f.size();
    convert_checksum(sp,f.checksum(),
              &(f_status->checksumType),&(f_status->checksumValue));
    f_status->isPinned=(f.pinned() > 0);
    f_status->isPermanent=true;
    f_status->isCached=true;
    f_status->estSecondsToStart=0;
    f_status->sourceFilename=NULL;
    f_status->destFilename=NULL;
    f_status->queueOrder=0;
    f_status->owner=soap_strdup(sp,f.creator().c_str());
    f_status->group=NULL;
    f_status->permMode=0700;
    if(service_url) {
      if(shorturl) {
        f_status->SURL=soap_strdup(sp,
                  make_SURL_short(std::string(service_url),f.id()).c_str());
      } else {
        f_status->SURL=soap_strdup(sp,
                  make_SURL(std::string(service_url),f.id()).c_str());
      };
      f_status->TURL=soap_strdup(sp,
                make_TURL(std::string(service_url),f.id()).c_str());
    };
    switch(f.state_file()) {
      case FILE_STATE_ACCEPTED:
      case FILE_STATE_REQUESTED:
        // File generated by "put" request and not processed yet
        f_status->state="Pending"; have_pending=true;
      break;
      case FILE_STATE_COLLECTING:
        // File generated by "put" request and ready to be filled or
        // being filled
        if(running) {
          f_status->state="Running"; have_active=true;
        } else {
          f_status->state="Ready"; have_active=true;
        };
      break;
      case FILE_STATE_DOWNLOADING:
        f_status->state="Running"; have_active=true;
      break;
      case FILE_STATE_COMPLETE:
        // File was filled by "put" procedure but checksum is not 
        // computed yet - should be short
        f_status->state="Running"; have_active=true;
      break;
      case FILE_STATE_VALID:
        // File is either ready for "get" request or is beeing 
        // downloaded
        //if(f.pinned(userid_.c_str())) {
        if(running) {
          if(f.sources().size() == 0) {
            f_status->state="Running"; have_active=true;
          } else {
            f_status->state="Ready"; have_active=true;
          };
        } else {
          f_status->state="Done";
        };
      break;
      case FILE_STATE_DELETING:
      default:
        f_status->state="Failed"; have_failed=true;
      break;
    };
  };
  if(have_failed) { r->state="Failed"; }
  else if(have_active) { r->state="Active"; }
  else if(have_pending) { r->state="Pending"; }
  else { r->state="Done"; };
  return r;
}

SRMRequestFile* SRMRequest::file(int fileid) {
  if(fileid < 0) return NULL;
  if(fileid >= files_.size()) return NULL;
  std::list<SRMRequestFile>::iterator f_p = files_.begin();
  for(;f_p!=files_.end() && fileid;++f_p,--fileid) {
 };
  if(f_p == files_.end()) return NULL;
  //return &(*(f_p->f_));
  if(!(f_p->f_.exists())) return NULL;
  return &(*f_p);
}

bool SRMRequest::active(void) {
  // Gather information about current state of request
  bool have_undone = false;
  for(std::list<SRMRequestFile>::iterator f_p = files_.begin();
                                            f_p!=files_.end();++f_p) {
    if(!(f_p->f_.exists())) continue;
    SEFile& f = *(f_p->f_);
    switch(f.state_file()) {
      case FILE_STATE_ACCEPTED:
      case FILE_STATE_REQUESTED:
      case FILE_STATE_COLLECTING:
      case FILE_STATE_DOWNLOADING:
      case FILE_STATE_COMPLETE:
        have_undone=true;
      break;
      case FILE_STATE_VALID:
        if(f.pinned(userid_.c_str())) have_undone=true;
      break;
      case FILE_STATE_DELETING:
      default:
      break;
    };
  };
  return have_undone;
}

SRMRequests::SRMRequests(void) {
}

SRMRequests::~SRMRequests(void) {
}

void SRMRequests::add(SRMRequest* req) {
  lock_.block();
  reqs_.push_back(req);
  lock_.unblock();
}

/// Get request from list and lock it. This is only proper way
/// for external code to obtain link to request.
SRMRequest* SRMRequests::acquire(int id,const char* userid) {
  lock_.block();
  for(std::list<SRMRequest*>::iterator i = reqs_.begin();i!=reqs_.end();++i) {
    if(!(*i)) continue;
    if(id == (*i)->id_) {
      if((*i)->userid_ == userid) {
        (*i)->acquire();
        lock_.unblock();
        return (*i);
      } else {
        lock_.unblock();
        return NULL;
      };
    };
  };
  lock_.unblock();
  return NULL;
}

// Release lock on request. If link to request is available one can 
// use SRMRequest::release() instead.
//bool SRMRequests::release(int id) {
//  lock_.block();
//  for(std::list<SRMRequest*>::iterator i = reqs_.begin();i!=reqs_.end();++i) {
//    if(!(*i)) continue;
//    if(id == (*i)->id_) {
//      (*i)->release();
//      lock_.unblock();
//      return true;
//    };
//  };
//  lock_.unblock();
//  return false;
//}

/// Destroy request
bool SRMRequests::remove(int id) {
  lock_.block();
  for(std::list<SRMRequest*>::iterator i = reqs_.begin();i!=reqs_.end();++i) {
    if(!(*i)) continue;
    if(id == (*i)->id_) {
      (*i)->acquire(); // Make sure nothing uses this request
      SRMRequest* r = *i;
      reqs_.erase(i);  // Remove from list
      r->release(); // Keep mutex clean
      lock_.unblock(); // Release list
      delete r;     // Destroy request
      return true;
    };
  };
  lock_.unblock();
  return false;
}

/// Remove request if finished
bool SRMRequests::maintain(int id) {
  bool res = false;
  lock_.block();
  for(std::list<SRMRequest*>::iterator i = reqs_.begin();i!=reqs_.end();++i) {
    if(!(*i)) continue;
    if(id == (*i)->id_) {
      if(!((*i)->active())) {
        (*i)->acquire(); // Make sure nothing uses this request
        SRMRequest* r = (*i);
        reqs_.erase(i);  // Remove from list
        r->release(); // Keep mutex clean
        delete r;     // Destroy request
        res=true;
      };
      break;
    };
  };
  lock_.unblock();
  return res;
}

/// Remove finished requests 
bool SRMRequests::maintain(void) {
  bool res = false;
  lock_.block();
  for(std::list<SRMRequest*>::iterator i = reqs_.begin();i!=reqs_.end();) {
    if(!(*i)) continue;
    if(!((*i)->active())) {
      (*i)->acquire(); // Make sure nothing uses this request
      SRMRequest* r = (*i);
      i=reqs_.erase(i);  // Remove from list
      r->release(); // Keep mutex clean
      delete r;     // Destroy request
      res=true;
    } else {
      ++i;
    };
  };
  lock_.unblock();
  return res;
}


void SRMRequests_Thread::func(void) {
  while(true) {
    reqs_.maintain();
    idle(60*60*1000); // 1 hour
  };
}

