#include "../../std.h"

#include <fstream>

#include <arc/stringconv.h>

#include "../../misc/log_time.h"

#include "../client/client.h"
#include "srm1_soapH.h"
#include "srm2_soapH.h"

#include "srm_remote_request.h"
#include "srm_file_metadata.h"
#include "srm_file_status.h"
#include "srm_request.h"

extern struct Namespace srm1_soap_namespaces[];
extern struct Namespace srm2_soap_namespaces[];


// -----------------------------------------

ArrayOfstring* SRMRemoteRequest::MakeSURLs(struct soap* soap,const std::list<SRMFile*>& files) {
  ArrayOfstring* SURLs = soap_new_ArrayOfstring(soap,-1);
  if(!SURLs) return NULL;
  SURLs->soap_default(soap);
  SURLs->__ptr=(char**)soap_malloc(soap,sizeof(char*)*files.size());
  if(SURLs->__ptr == NULL) return NULL;
  SURLs->__size=0;
  for(std::list<SRMFile*>::const_iterator f = files.begin();f!=files.end();++f) {
    std::string file_url = (endpoint->url.BaseURL())+((*f)->name);
    SURLs->__ptr[SURLs->__size]=soap_strdup(soap,file_url.c_str());
    if(SURLs->__ptr[SURLs->__size] == NULL) continue;
    SURLs->__size++;
  };
  return SURLs;
}

ArrayOfstring* SRMRemoteRequest::MakeTURLs(struct soap* soap,const std::list<SRMFile*>& files) {
  ArrayOfstring* TURLs = soap_new_ArrayOfstring(soap,-1);
  if(!TURLs) return NULL;
  TURLs->soap_default(soap);
  TURLs->__ptr=(char**)soap_malloc(soap,sizeof(char*)*files.size());
  if(TURLs->__ptr == NULL) return NULL;
  TURLs->__size=0;
  for(std::list<SRMFile*>::const_iterator f = files.begin();f!=files.end();++f)
{
    std::string file_url = (*f)->is_turl?((*f)->name.c_str()):"";
    TURLs->__ptr[TURLs->__size]=soap_strdup(soap,file_url.c_str());
    if(TURLs->__ptr[TURLs->__size] == NULL) continue;
    TURLs->__size++;
  };
  return TURLs;
}

ArrayOfstring* SRMRemoteRequest::MakeProtocols(struct soap* soap,const std::list<std::string>& protocols) {
  ArrayOfstring* Protocols = soap_new_ArrayOfstring(soap,-1);
  if(!Protocols) return NULL;
  Protocols->soap_default(soap);
  Protocols->__ptr=(char**)soap_malloc(soap,sizeof(char*)*protocols.size());
  if(Protocols->__ptr == NULL) return NULL;
  Protocols->__size=0;
  for(std::list<std::string>::const_iterator p = protocols.begin();
                                               p!=protocols.end();++p) {
    Protocols->__ptr[Protocols->__size]=soap_strdup(soap,p->c_str());
    if(Protocols->__ptr[Protocols->__size] == NULL) continue;
    Protocols->__size++;
  };
  return Protocols;
}


// -----------------------------------------
 

SRMRemoteRequest::SRMRemoteRequest(SRMEndpoint* endpoint_,const char* credentials):endpoint(endpoint_),status(NULL),c(NULL),cred(credentials?credentials:"") {
  if(!endpoint) return;
  if(!(endpoint->url)) return;
  c = new HTTP_ClientSOAP(endpoint->url.ContactURL().c_str(),&soap,endpoint->url.GSSAPI());
  if(!c) return;
  if(!*c) { delete c; c=NULL; return; };
  soap.namespaces=srm1_soap_namespaces;
  if(!cred.empty()) c->credentials(cred.c_str());
  //timeout=300;
}

SRMRemoteRequest::SRMRemoteRequest(const SRMRemoteRequest& r) {
  operator=(r);
}

SRMRemoteRequest::~SRMRemoteRequest(void) {
 if(c) { delete c; c=NULL; };
}

SRMRemoteRequest& SRMRemoteRequest::operator=(const SRMRemoteRequest& r) {
  id=r.id;
  endpoint=r.endpoint;
  status=r.status;
  c=NULL;
  if(!endpoint) return (*this);
  if(!(endpoint->url)) return (*this);
  c = new HTTP_ClientSOAP(endpoint->url.ContactURL().c_str(),&soap,endpoint->url.GSSAPI());
  if(!c) return (*this);
  if(!*c) { delete c; c=NULL; return (*this); };
  soap.namespaces=srm1_soap_namespaces;
  cred=r.cred;
  if(!cred.empty()) c->credentials(cred.c_str());
  return (*this);
}

bool SRMRemoteRequest::V1_getFileMetaData(std::list<SRMFile*>& files) {
std::cerr<<"V1_getFileMetaData: "<<c->SOAP_URL()<<std::endl;
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = MakeSURLs(&soap,files);
  if(SURLs == NULL) return false;
  struct SRMv1Meth__getFileMetaDataResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getFileMetaData(&soap,c->SOAP_URL(),
                      "getFileMetaData",SURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getFileMetaData) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if((r._Result == NULL) ||
     (r._Result->__size <= 0) ||
     (r._Result->__ptr == NULL)) {
    odlog(INFO)<<"SRM server did not return any information (getFileMetaData) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return true; // This situation is not an error - maybe server has no files
  };
  for(int n = 0;n<r._Result->__size;n++) {
    SRMv1Type__FileMetaData* mdata = r._Result->__ptr[n];
    if(mdata == NULL) continue;
std::cerr<<"V1_getFileMetaData: Result["<<n<<"] - mdata passed"<<std::endl;
    if(mdata->SURL == NULL) continue;
std::cerr<<"V1_getFileMetaData: Result["<<n<<"] - SURL passed: "<<mdata->SURL<<std::endl;
    std::list<SRMFile*>::iterator f = files.begin();
    for(;f!=files.end();++f) {
      SRM_URL u(mdata->SURL);
std::cerr<<"V1_getFileMetaData: compare "<<(*f)->name<<" to "<<u.FileName()<<std::endl;
      if(!u) continue;
      if((*f)->name == u.FileName()) break;
    };
    if(f!=files.end()) {
std::cerr<<"V1_getFileMetaData: set metadata"<<std::endl;
(*f)->MetaData(new SRMFileMetaData(mdata));
    };
  };
  c->reset(); c->disconnect();
std::cerr<<"V1_getFileMetaData: exit"<<std::endl;
  return true;
}

bool SRMRemoteRequest::V1_getProtocols(std::list<std::string>& protocols) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  struct SRMv1Meth__getProtocolsResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getProtocols(&soap,c->SOAP_URL(),
                      "getProtocols",r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getProtocols) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if((r._Result == NULL) ||
     (r._Result->__size <= 0) ||
     (r._Result->__ptr == NULL)) {
    odlog(INFO)<<"SRM server did not return any information (getProtocols) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  for(int n = 0;n<r._Result->__size;n++) {
    protocols.push_back(std::string(r._Result->__ptr[n]));
  };
  c->reset(); c->disconnect();
  return true;
}

bool SRMRemoteRequest::V1_advisoryDelete(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = MakeSURLs(&soap,files);
  if(SURLs == NULL) return false;
  struct SRMv1Meth__advisoryDeleteResponse r;
  if((soap_err=soap_call_SRMv1Meth__advisoryDelete(&soap,c->SOAP_URL(),
                      "advisoryDelete",SURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (advisoryDelete) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}

bool SRMRemoteRequest::FindFiles(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = MakeSURLs(&soap,files);
  if(SURLs == NULL) return false;
  struct SRMv1Meth__getFileMetaDataResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getFileMetaData(&soap,c->SOAP_URL(),
                      "getFileMetaData",SURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getFileMetaData) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if((r._Result == NULL) ||
     (r._Result->__size <= 0) ||
     (r._Result->__ptr == NULL)) {
    odlog(INFO)<<"SRM server did not return any information (getFileMetaData) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  std::list<SRMFile*>::iterator f = files.begin();
  for(;f!=files.end();) {
    int n = 0;
    for(;n<r._Result->__size;n++) {
      SRMv1Type__FileMetaData* mdata = r._Result->__ptr[n];
      if(mdata == NULL) continue;
      if(mdata->SURL == NULL) continue;
      SRM_URL u(mdata->SURL);
      if(!u) continue;
      if((*f)->name == u.FileName()) break;
    };
    if(n<r._Result->__size) { ++f; }
    else { f=files.erase(f); };
  };
  c->reset(); c->disconnect();
  return true;
}


bool SRMRemoteRequest::V1_mkPermanent(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = MakeSURLs(&soap,files);
  if(SURLs == NULL) return false;
  struct SRMv1Meth__mkPermanentResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__mkPermanent(&soap,c->SOAP_URL(),
                      "mkPermanent",SURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (mkPermanent) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (mkPermanent) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}


bool SRMRemoteRequest::V1_copy(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* sourceSURLs = MakeSURLs(&soap,files);
  if(sourceSURLs == NULL) return false;
  ArrayOfstring* destSURLs = MakeSURLs(&soap,files);
  if(destSURLs == NULL) return false;
  ArrayOfboolean* undefined_arg = soap_new_ArrayOfboolean(&soap,-1);
  if(undefined_arg == NULL) return false;
  undefined_arg->soap_default(&soap);
  undefined_arg->__ptr=(bool*)soap_malloc(&soap,sizeof(bool)*files.size());
  if(undefined_arg->__ptr == NULL) return false;
  undefined_arg->__size=0;
  for(;undefined_arg->__size < files.size();++(undefined_arg->__size)) {
    undefined_arg->__ptr[undefined_arg->__size]=true;
  };
  // Fix source/destination
  int n = 0;
  for(std::list<SRMFile*>::const_iterator f = files.begin();
                                               f!=files.end();++f) {
    if(((*f) != NULL) &&
       ((*f)->Status() != NULL)) {
      if(!((*f)->Status()->sourceFilename.empty())) {
        sourceSURLs->__ptr[n]=soap_strdup(&soap,(*f)->Status()->sourceFilename.c_str());
      };
      if(!((*f)->Status()->destFilename.empty())) {
        destSURLs->__ptr[n]=soap_strdup(&soap,(*f)->Status()->destFilename.c_str());
      };
    };
    n++;
  };
  struct SRMv1Meth__copyResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__copy(&soap,c->SOAP_URL(),
                      "copy",sourceSURLs,sourceSURLs,
                       undefined_arg,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (copy) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (copy) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}


bool SRMRemoteRequest::V1_get(std::list<SRMFile*>& files,const std::list<std::string>& protocols) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = MakeSURLs(&soap,files);
  if(SURLs == NULL) return false;
  ArrayOfstring* Protocols = MakeProtocols(&soap,protocols);
  if(Protocols == NULL) return false;
  struct SRMv1Meth__getResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__get(&soap,c->SOAP_URL(),
                      "get",SURLs,Protocols,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (get) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (get) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}

bool SRMRemoteRequest::V1_put(std::list<SRMFile*>& files,const std::list<std::string>& protocols) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* Dest_file_names = MakeSURLs(&soap,files);
  if(Dest_file_names == NULL) return false;
  ArrayOfstring* Src_file_names = soap_new_ArrayOfstring(&soap,-1);
  if(!Src_file_names) return false;
  Src_file_names->soap_default(&soap);
  Src_file_names->__ptr=(char**)soap_malloc(&soap,sizeof(char*)*files.size());
  if(Src_file_names->__ptr == NULL) return false;
  Src_file_names->__size=0;
  ArrayOflong* sizes = soap_new_ArrayOflong(&soap,-1);
  if(sizes == NULL) return false;
  sizes->soap_default(&soap);
  sizes->__ptr=(LONG64*)soap_malloc(&soap,sizeof(LONG64)*files.size());
  if(sizes->__ptr == NULL) return false;
  sizes->__size=0;
  ArrayOfboolean* wantPermanent = soap_new_ArrayOfboolean(&soap,-1);
  if(wantPermanent == NULL) return false;
  wantPermanent->soap_default(&soap);
  wantPermanent->__ptr=(bool*)soap_malloc(&soap,sizeof(bool)*files.size());
  if(wantPermanent->__ptr == NULL) return false;
  wantPermanent->__size=0;
  for(std::list<SRMFile*>::const_iterator f = files.begin();
                                               f!=files.end();++f) {
    if(((*f) != NULL) &&
       ((*f)->Status() != NULL) &&
       (!((*f)->Status()->sourceFilename.empty()))) {
      Src_file_names->__ptr[Src_file_names->__size]=
            soap_strdup(&soap,(*f)->Status()->sourceFilename.c_str());
    } else {
      Src_file_names->__ptr[Src_file_names->__size]=
            soap_strdup(&soap,(*f)->name.c_str());
    };
    if(Src_file_names->__ptr[Src_file_names->__size] == NULL) 
                    Src_file_names->__ptr[Src_file_names->__size]="";
    Src_file_names->__size++;
    sizes->__ptr[sizes->__size]=0;
    if(((*f) != NULL) &&
       ((*f)->MetaData() != NULL) &&
       ((*f)->MetaData()->size > 0)) {
      sizes->__ptr[sizes->__size]=(*f)->MetaData()->size;
    };
    sizes->__size++;
    wantPermanent->__ptr[wantPermanent->__size]=false;
    if(((*f) != NULL) &&
       ((*f)->MetaData() != NULL)) {
      wantPermanent->__ptr[wantPermanent->__size]=(*f)->MetaData()->isPermanent;
    };
    wantPermanent->__size++;
  };
  ArrayOfstring* Protocols = MakeProtocols(&soap,protocols);
  if(Protocols == NULL) return false;
  struct SRMv1Meth__putResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__put(&soap,c->SOAP_URL(),
                      "put",Src_file_names,Dest_file_names,
                       sizes,wantPermanent,Protocols,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (put) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (put) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}


bool SRMRemoteRequest::V1_getEstGetTime(std::list<SRMFile*>& files,const std::list<std::string>& protocols) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = MakeSURLs(&soap,files);
  if(SURLs == NULL) return false;
  ArrayOfstring* Protocols = MakeProtocols(&soap,protocols);
  if(Protocols == NULL) return false;
  struct SRMv1Meth__getEstGetTimeResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getEstGetTime(&soap,c->SOAP_URL(),
                      "getEstGetTime",SURLs,Protocols,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getEstGetTime) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (getEstGetTime) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}


bool SRMRemoteRequest::V1_getEstPutTime(std::list<SRMFile*>& files,const std::list<std::string>& protocols) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* Dest_file_names = MakeSURLs(&soap,files);
  if(Dest_file_names == NULL) return false;
  ArrayOfstring* Src_file_names = soap_new_ArrayOfstring(&soap,-1);
  if(!Src_file_names) return false;
  Src_file_names->soap_default(&soap);
  Src_file_names->__ptr=(char**)soap_malloc(&soap,sizeof(char*)*files.size());
  if(Src_file_names->__ptr == NULL) return false;
  Src_file_names->__size=0;
  ArrayOflong* sizes = soap_new_ArrayOflong(&soap,-1);
  if(!sizes) return false;
  sizes->__ptr=(LONG64*)soap_malloc(&soap,sizeof(LONG64)*files.size());
  if(sizes->__ptr == NULL) return false;
  sizes->__size=0;
  ArrayOfboolean* wantPermanent = soap_new_ArrayOfboolean(&soap,-1);
  if(!wantPermanent) return false;
  wantPermanent->__ptr=(bool*)soap_malloc(&soap,sizeof(bool)*files.size());
  if(wantPermanent->__ptr == NULL) return false;
  wantPermanent->__size=0;
  for(std::list<SRMFile*>::const_iterator f = files.begin();
                                               f!=files.end();++f) {
    if(((*f) != NULL) &&
       ((*f)->Status() != NULL) &&
       (!((*f)->Status()->sourceFilename.empty()))) {
      Src_file_names->__ptr[Src_file_names->__size]=
            soap_strdup(&soap,(*f)->Status()->sourceFilename.c_str());
    } else {
      Src_file_names->__ptr[Src_file_names->__size]=
            soap_strdup(&soap,(*f)->name.c_str());
    };
    if(Src_file_names->__ptr[Src_file_names->__size] == NULL)
                    Src_file_names->__ptr[Src_file_names->__size]="";
    Src_file_names->__size++;
    sizes->__ptr[sizes->__size]=0;
    if(((*f) != NULL) &&
       ((*f)->MetaData() != NULL) &&
       ((*f)->MetaData()->size > 0)) {
      sizes->__ptr[sizes->__size]=(*f)->MetaData()->size;
    };
    sizes->__size++;
    wantPermanent->__ptr[wantPermanent->__size]=false;
    if(((*f) != NULL) &&
       ((*f)->MetaData() != NULL)) {
      wantPermanent->__ptr[wantPermanent->__size]=(*f)->MetaData()->isPermanent;    };
    wantPermanent->__size++;
  };
  ArrayOfstring* Protocols = MakeProtocols(&soap,protocols);
  if(Protocols == NULL) return false;
  struct SRMv1Meth__getEstPutTimeResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getEstPutTime(&soap,c->SOAP_URL(),
                      "getEstPutTime",Src_file_names,Dest_file_names,
                       sizes,wantPermanent,Protocols,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getEstPutTime) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (getEstPutTime) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files,ReplaceQuicker)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}


bool SRMRemoteRequest::V1_pin(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* TURLs = MakeTURLs(&soap,files);
  if(TURLs == NULL) return false;
  struct SRMv1Meth__pinResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__pin(&soap,c->SOAP_URL(),
                      "pin",TURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (pin) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (pin) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}
 
bool SRMRemoteRequest::V1_unPin(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* TURLs = MakeTURLs(&soap,files);
  if(TURLs == NULL) return false;
  struct SRMv1Meth__unPinResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__unPin(&soap,c->SOAP_URL(),
                      "unPin",TURLs,stringto<int>(id),r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (unPin) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (unPin) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}

bool SRMRemoteRequest::V1_getRequestStatus(std::list<SRMFile*>& files) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  struct SRMv1Meth__getRequestStatusResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getRequestStatus(&soap,c->SOAP_URL(),
                      "getRequestStatus",stringto<int>(id),r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getRequestStatus) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (getRequestStatus) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,files)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;
}

bool SRMRemoteRequest::V1_setFileStatus(SRMFile& file,const std::string& state) {
  if(!c) return false;
  if(c->connect() != 0) return false;
  int soap_err = SOAP_OK;
  struct SRMv1Meth__setFileStatusResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__setFileStatus(&soap,c->SOAP_URL(),
        "setFileStatus",stringto<int>(id),stringto<int>(file.remote_id),
        (char*)(state.c_str()),r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (setFileStatus) - "<<endpoint->url.ContactURL()<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->reset(); c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM server did not return any information (getRequestStatus) - "<<endpoint->url.ContactURL()<<std::endl;
    c->reset(); c->disconnect();
    return false;
  };
  if(!SetStatus(r._Result,file)) {
    c->reset(); c->disconnect();
    return false;
  };
  c->reset(); c->disconnect();
  return true;



}

// -----------------------------------------------------
typedef bool (*StatusCompare)(SRMFile&,SRMv1Type__RequestFileStatus*);


bool SRMRemoteRequest::SetStatus(SRMv1Type__RequestStatus* s,std::list<SRMFile*>& files,SRMRemoteRequest::StatusOptions options) {
  if(!s) return false;
  id=tostring<int>(s->requestId);
  status=new SRMRequestStatus;
  if(!status) return false;
  if(s->state) status->state=s->state;
  if(s->submitTime) status->submitTime=*(s->submitTime);
  if(s->startTime)  status->startTime=*(s->startTime);
  if(s->finishTime) status->finishTime=*(s->finishTime);
  status->estTimeToStart=s->estTimeToStart;
  status->retryDeltaTime=s->retryDeltaTime;
  if(s->errorMessage) status->errorMessage=s->errorMessage;
  // s->type
  if((s->fileStatuses) &&
     (s->fileStatuses->__ptr) &&
     (s->fileStatuses->__size > 0)) {
    for(int n = 0;n<s->fileStatuses->__size;n++) {
      SRMv1Type__RequestFileStatus* fstat = s->fileStatuses->__ptr[n];
      std::string fname;
      std::string turl = fstat->TURL?fstat->TURL:"";
      if(fstat->SURL) {
        SRM_URL u(fstat->SURL);
        if(u) fname=u.FileName();
      };
      if(fname.empty() && turl.empty()) continue;
std::cerr<<"SetStatus: fname: "<<fname<<std::endl;
std::cerr<<"SetStatus: turl: "<<turl<<std::endl;
      std::list<SRMFile*>::iterator f = files.begin();
      for(;f!=files.end();++f) {
std::cerr<<"SetStatus: file fname: "<<(*f)->name<<" ("<<(*f)->is_turl<<")"<<std::endl;
        if( ((!((*f)->is_turl)) && (!fname.empty()) && ((*f)->name == fname)) ||
            (((*f)->is_turl)    && (!turl.empty())  && ((*f)->name == turl)) ) {
          if((fstat->state != NULL) &&
             ((strcasecmp(fstat->state,"pending") == 0) ||
              (strcasecmp(fstat->state,"ready") == 0) ||
              (strcasecmp(fstat->state,"running") == 0) ||
              (strcasecmp(fstat->state,"done") == 0))) {
            bool replace = false;
            switch (options) {
              case Replace: replace=true; break;
              case NotReplace: 
                if((*f)->Status() == NULL) replace=true;
              break;
              case ReplaceQuicker:
                if((*f)->Status() == NULL) { replace=true; break; };
                if(fstat->estSecondsToStart < (*f)->Status()->estSecondsToStart)
                  replace=true;
              break;
            };
            if(replace) {
              (*f)->Status(new SRMFileStatus(fstat));
              (*f)->MetaData(new SRMFileMetaData(fstat));
              (*f)->request=this; // associate
              (*f)->remote_id=tostring<int>(fstat->fileId);
            };
          };
          break;
        };
      };
    };
  };
  return true;
}

bool SRMRemoteRequest::SetStatus(SRMv1Type__RequestStatus* s,SRMFile& file) {
  if(!s) return false;
  id=tostring<int>(s->requestId);
  status=new SRMRequestStatus;
  if(!status) return false;
  if(s->state) status->state=s->state;
  if(s->submitTime) status->submitTime=*(s->submitTime);
  if(s->startTime)  status->startTime=*(s->startTime);
  if(s->finishTime) status->finishTime=*(s->finishTime);
  status->estTimeToStart=s->estTimeToStart;
  status->retryDeltaTime=s->retryDeltaTime;
  if(s->errorMessage) status->errorMessage=s->errorMessage;
  // s->type
  if((s->fileStatuses) &&
     (s->fileStatuses->__ptr) &&
     (s->fileStatuses->__size > 0)) {
    for(int n = 0;n<s->fileStatuses->__size;n++) {
      SRMv1Type__RequestFileStatus* fstat = s->fileStatuses->__ptr[n];
      std::string fname;
      std::string turl = fstat->TURL?fstat->TURL:"";
      if(fstat->SURL) {
        SRM_URL u(fstat->SURL);
        if(u) fname=u.FileName();
      };
      if(fname.empty() && turl.empty()) continue;
      if( ((!file.is_turl) && (!fname.empty()) && (file.name == fname)) ||
          ((file.is_turl)  && (!turl.empty())  && (file.name == turl)) ) {
        file.Status(new SRMFileStatus(fstat));
        file.MetaData(new SRMFileMetaData(fstat));
        file.request=this; // associate
        file.remote_id=tostring<int>(fstat->fileId);
        return true;
      };
    };
  };
  return false;
}




// --------------------------------------

