// TODO: Run multiple clients in parallel

#include "../../std.h"
#include <globus_common.h>
#include <globus_io.h>
#include <string>
#include <list>
#include <fstream>

#include "../../files/info_types.h"
#include "../../misc/log_time.h"
#include <arc/stringconv.h>
#include <arc/url.h>
#include <arc/globuserrorutils.h>

#include "../client/client.h"
#include "../../misc/url_options.h"
#include "../../config/conf.h"

#include "logger_client.h"
#include "logger_common.h"


time_t* soap_new_time(struct soap* sp,const char* s) {
  mds_time t;
  t=s;
  if(!t.defined()) return NULL;
  time_t* p = sp?(time_t*)soap_malloc(sp,sizeof(time_t)):(new time_t);
  if(p) *p=(time_t)t;
  return p;
}

time_t* soap_new_time(struct soap* sp,const std::string& s) {
  return soap_new_time(sp,s.c_str());
}

template<typename T> T* soap_new_i(struct soap* sp,const std::string& s) {
  T n;
  if(!isdigit(s[0])) return NULL;
  std::stringstream ss(s);
  ss >> n;
  T* p = sp?(T*)soap_malloc(sp,sizeof(T)):(new T);
  if(p) *p=n;
  return p;
}

template<typename T> T* soap_new_i(struct soap* sp,const char* s) {
  T n;
  if(!s) return NULL;
  if(!isdigit(s[0])) return NULL;
  std::stringstream ss(s);
  ss >> n;
  T* p = sp?(T*)soap_malloc(sp,sizeof(T)):(new T);
  if(p) *p=n;
  return p;
}

void split(const std::string& str, const std::string delim, std::vector<std::string>& output) {
  std::string::size_type offset = 0;
  std::string::size_type delimIndex = 0;
    
  delimIndex = str.find(delim, offset);

  while (delimIndex != std::string::npos) {
    output.push_back(str.substr(offset, delimIndex - offset));
    offset += delimIndex - offset + delim.length();
    delimIndex = str.find(delim, offset);
  }

  output.push_back(str.substr(offset));
}

int string_to_int(std::string str) {
  std::stringstream ss(str);
  int n;
  ss >> n;
  return n;
}

void split_i(const std::string& str, const std::string delim, std::vector<int>& output) {
  std::string::size_type offset = 0;
  std::string::size_type delimIndex = 0;
    
  delimIndex = str.find(delim, offset);

  while (delimIndex != std::string::npos) {
    try {
      output.push_back(stringtoi(str.substr(offset, delimIndex - offset)));
    } catch(ARCLibError& e) {
      output.push_back(0);
    }
    offset += delimIndex - offset + delim.length();
    delimIndex = str.find(delim, offset);
  }

  try {
    output.push_back(stringtoi(str.substr(offset)));
  } catch(ARCLibError& e) {
     output.push_back(0);
  }
}


class JobRecord {
 private:
  bool valid;
  bool allocated;
 public:
  nl2__UsageRecord info;
  std::string url;
  operator bool(void) { return valid; };
  bool operator!(void) { return !valid; };
  JobRecord(std::istream& i,struct soap* sp = NULL);
  JobRecord(const JobRecord& j);
  ~JobRecord(void);
};

JobRecord::JobRecord(const JobRecord& j) {
  valid=j.valid;
  info=j.info;
  allocated=false;
}

JobRecord::JobRecord(std::istream& i,struct soap* sp):valid(false),url("") {
  allocated=(sp != NULL);
  if(sp) {
    info.soap_default(sp);
  } else {
      info.globaljobid="";
      info.globaluserid="";
	  info.cluster="";
      info.jobdescription=NULL;
	  info.projectname=NULL;
	  info.jobname=NULL;
	  info.submithost=NULL;
	  info.requestedcputime=NULL;
	  info.requestedwalltime=NULL;
	  info.requestedmemory=NULL;
	  info.requesteddisk=NULL;
	  info.submissiontime=NULL;
	  info.localuserid=NULL;
	  info.queue=NULL;
	  info.lrms=NULL;
	  info.localjobid=NULL;
	  info.lrmssubmissiontime=NULL;
	  info.lrmsendtime=NULL;
	  info.nodename.clear();
	  info.nodecount=NULL;
	  info.processors=NULL;
	  info.exitcode=NULL;
	  info.failurestring=NULL;
	  info.usedcputime=NULL;
	  info.usedwalltime=NULL;
	  info.usedmemory=NULL;
	  info.useddisk=NULL;
	  info.status=NULL;
	  info.endtime=NULL;
	  info.downloadtime=NULL;
	  info.uploadtime=NULL;
	  info.processid.clear();
	  info.charge=NULL;
	  info.network=NULL;
	  info.stageindata=NULL;
	  info.stageoutdata=NULL;
	  info.usedswap=NULL;
	  info.servicelevel=NULL;
	  info.runtimeenvironment.clear();
  };
  for(;;) {
    if(i.fail()) goto error;
    if(i.eof()) break;
    std::string value;
    std::string key = config_read_line(i,value,'=');
    if(key=="loggerurl") { url=value; }
    else if(key=="ngjobid") { info.globaljobid=value; }
    else if(key=="usersn") { info.globaluserid=value; }
    else if(key=="usersn") { info.globaluserid=value; }
	else if(key=="cluster") { info.cluster=value; }
    else if(key=="description") { info.jobdescription=soap_new_std__string(sp,value); }
    else if(key=="projectname") { info.projectname=soap_new_std__string(sp,value); }
    else if(key=="jobname") { info.jobname=soap_new_std__string(sp,value); }
    else if(key=="clienthost") { info.submithost=soap_new_std__string(sp,value); }
    else if(key=="requestedcputime") { info.requestedcputime=soap_new_i<int>(sp,value); }
	else if(key=="requestedwalltime") { info.requestedwalltime=soap_new_i<int>(sp,value); }
    else if(key=="requestedmemory") { info.requestedmemory=soap_new_i<int>(sp,value); }
    else if(key=="requesteddisk") { info.requesteddisk=soap_new_i<int>(sp,value); }
    else if(key=="submissiontime") { info.submissiontime=soap_new_time(sp,value); }
    else if(key=="localuser") { info.localuserid=soap_new_std__string(sp,value); }
    else if(key=="queue") { info.queue=soap_new_std__string(sp,value); }
    else if(key=="lrms") { info.lrms=soap_new_std__string(sp,value); }
    else if(key=="localjobid") { info.localjobid=soap_new_std__string(sp,value); }
    else if(key=="lrmssubmissiontime") { info.lrmssubmissiontime=soap_new_time(sp,value); }
    else if(key=="lrmsendtime") { info.lrmsendtime=soap_new_time(sp,value); }
	else if(key=="nodename") { split(value, ",", info.nodename); }
    else if(key=="nodecount") { info.nodecount=soap_new_i<int>(sp,value); }
    else if(key=="processors") { info.processors=soap_new_i<int>(sp,value); }
    else if(key=="exitcode") { info.exitcode=soap_new_i<int>(sp,value); }
    else if(key=="failurestring") { info.failurestring=soap_new_std__string(sp,value); }
    else if(key=="usedcputime") { info.usedcputime=soap_new_i<int>(sp,value); }
    else if(key=="usedwalltime") { info.usedwalltime=soap_new_i<int>(sp,value); }
    else if(key=="usedmemory") { info.usedmemory=soap_new_i<int>(sp,value); }
    else if(key=="useddisk") { info.useddisk=soap_new_i<int>(sp,value); }
    else if(key=="status") { info.status=soap_new_std__string(sp,value); }
    else if(key=="endtime") { info.endtime=soap_new_time(sp,value); }
    else if(key=="downloadtime") { info.downloadtime=soap_new_i<int>(sp,value); }
    else if(key=="uploadtime") { info.uploadtime=soap_new_i<int>(sp,value); }
    else if(key=="processid") { split_i(value, ",", info.processid); }
    else if(key=="charge") { info.charge=soap_new_i<int>(sp,value); }
    else if(key=="network") { info.network=soap_new_std__string(sp,value); }
    else if(key=="stageindata") { info.stageindata=soap_new_i<int>(sp,value); }
    else if(key=="stageoutdata") { info.stageoutdata=soap_new_i<int>(sp,value); }
    else if(key=="usedswap") { info.usedswap=soap_new_i<int>(sp,value); }
    else if(key=="servicelevel") { info.servicelevel=soap_new_std__string(sp,value); }
    else if(key=="runtimeenvironment") { split(value, ",", info.runtimeenvironment); }
  };
  valid=true;
error:
  return;
}

JobRecord::~JobRecord(void) {
  if(allocated) {
    //if(info.cluster) delete info.cluster;
    if(info.jobdescription) delete info.jobdescription;
    if(info.projectname) delete info.projectname;
    if(info.jobname) delete info.jobname;
    if(info.submithost) delete info.submithost;
    if(info.localuserid) delete info.localuserid;
    if(info.queue) delete info.queue;
    if(info.lrms) delete info.lrms;
    if(info.localjobid) delete info.localjobid;
    //if(info.nodename) delete info.nodename;
    if(info.failurestring) delete info.failurestring;
    if(info.status) delete info.status;
    if(info.requestedcputime) delete info.requestedcputime;
    if(info.requestedmemory) delete info.requestedmemory;
    if(info.requesteddisk) delete info.requesteddisk;
    if(info.nodecount) delete info.nodecount;
    if(info.exitcode) delete info.exitcode;
    if(info.usedcputime) delete info.usedcputime;
    if(info.usedmemory) delete info.usedmemory;
    if(info.usedwalltime) delete info.usedwalltime;
    if(info.useddisk) delete info.useddisk;
    if(info.submissiontime) delete info.submissiontime;
    if(info.lrmssubmissiontime) delete info.lrmssubmissiontime;
    if(info.lrmsendtime) delete info.lrmsendtime;
    if(info.endtime) delete info.endtime;
    if(info.downloadtime) delete info.downloadtime;
    if(info.uploadtime) delete info.uploadtime;
  };
}

#define MAX_CLIENTS 10
HTTP_ClientSOAP* clients[MAX_CLIENTS];

int logger(const char* url,char const * const * dirs,int num,time_t ex_period = 0);
 
int main(int argc,char* argv[]) {
  char* url = NULL;

  opterr=0;
  time_t ex_period = 0;
  int n;
  const char* basename = strrchr(argv[0],'/');
  if(basename == NULL) basename=argv[0];
  while((n=getopt(argc,argv,":hu:d:E:")) != -1) {
    switch(n) {
      case ':': { olog<<"Missing argument\n"; return 1; };
      case '?': { olog<<"Unrecognized option\n"; return 1; };
      case 'h': {
        std::cout<<"logger [-h] [-d level] [-u url] [-E expiration_period_days] control_dir ..."<<std::endl;
         return 0;
      };
      case 'u': {
        url=optarg;
      }; break;
      case 'd': {
        char* p;
        int i = strtol(optarg,&p,10);
        if(((*p) != 0) || (i<0)) {
          olog<<"Improper debug level '"<<optarg<<"'"<<std::endl;
          exit(1);
        };
        LogTime::Level(NotifyLevel(FATAL+i));
      }; break;
      case 'E': {
        char* p;
        int i = strtol(optarg,&p,10);
        if(((*p) != 0) || (i<=0)) {
          olog<<"Improper expiration period '"<<optarg<<"'"<<std::endl;
          exit(1);
        };
        ex_period=i; ex_period*=(60*60*24);
      }; break;
      default: { olog<<"Options processing error\n"; return 1; };
    };
  };
  return logger(url,argv+optind,argc-optind,ex_period);
}

int logger(const char* url,char const * const * dirs,int num,time_t ex_period) {
  int ret = 0;
  if(globus_module_activate(GLOBUS_IO_MODULE) != GLOBUS_SUCCESS) exit(1);

  {
  LoggerClient client;
  for(int n = 0;dirs[n] && n<num;n++) {
    std::string logdir(dirs[n]); logdir+="/logs/";
    std::string logger_url(url?url:"");
    struct dirent file_;
    struct dirent *file;
    odlog(INFO)<<"Processing directory: "<<logdir<<std::endl;
    DIR *dir=opendir(logdir.c_str());
    if(dir == NULL) continue;
    for(;;) {
      readdir_r(dir,&file_,&file);
      if(file == NULL) break;
      std::string logfile = logdir+file->d_name;
      struct stat st;
      if(stat(logfile.c_str(),&st) != 0) continue;
      if(!S_ISREG(st.st_mode)) continue;
      odlog(INFO)<<"Processing file: "<<logfile<<std::endl;
      bool result = true;
      /* read file */
      char buf[4096];
      std::ifstream f(logfile.c_str());
      if(!f.is_open() ) continue; /* can't open file */
      JobRecord j(f);
      f.close();
      if(!j) {
        result=false;
        odlog(ERROR)<<"Removing unreadable job information in "<<logfile<<std::endl;
        unlink(logfile.c_str());
      } else {
        if(j.url.length()) logger_url=j.url;
        if(logger_url.length() == 0) {
          odlog(DEBUG)<<"No service URL provided"<<std::endl;
          result = false;
        } else {
          std::list<nl2__UsageRecord> recs;
          recs.push_back(j.info);
          odlog(DEBUG)<<"Reporting to: "<<logger_url<<std::endl;
	  odlog(DEBUG)<<"Reporting about: "<<j.info.globaljobid<<std::endl;
          result = client.Report(logger_url.c_str(),recs);
        };
        if(result) {
          odlog(INFO)<<"Passed information about job "<<j.info.globaljobid<<std::endl;
          unlink(logfile.c_str());
        } else {
          odlog(ERROR)<<"Failed to pass information about job "<<
                  j.info.globaljobid<<std::endl;
          ret=-1;
          // Check creation time and remove it if really too old
          if((ex_period) && 
             (((unsigned int)(time(NULL)-st.st_mtime)) > ex_period)) {
            odlog(ERROR)<<"Removing outdated information about job "<<
                    j.info.globaljobid<<std::endl;
            unlink(logfile.c_str());
          };
        };
      };
    };
    closedir(dir);
  };
  };
  globus_module_deactivate(GLOBUS_IO_MODULE);
  return ret;
}

