#include "../../std.h"

#include <iostream>
#include <fstream>

#include "../../misc/stringtoint.h"
#include "../../misc/escaped.h"
#include "../../misc/time_utils.h"
#include "../../misc/checksum.h"
#include "../../misc/log_time.h"
#include "file.h"


// ----------------------------------------------------------

const char* file_state_str[FILE_STATE_MAX] = {
  "accepted",
  "collecting",
  "requested",
  "downloading",
  "complete",
  "valid",
  "deleting",
  "failed"
};

const char* reg_state_str[REG_STATE_MAX] = {
  "local",
  "registering",
  "announced",
  "unregistering"
};

bool set_state(const char* key,const char* value,void* arg) {
  SEState* s = (SEState*)arg;
  return s->set(key,value);
}

bool SEState::set(const char* key,const char* value) {
  if(strcasecmp(key,"file") == 0) {
    if(value == NULL) return false;
    int n = 0; for(;value[n];n++) if(isspace(value[n])) break;
    if(n == 0) return false;
    int i;
    for(i=0;i<FILE_STATE_MAX;i++) if(strncasecmp(value,file_state_str[i],n) == 0) break;
    if(i>=FILE_STATE_MAX) return false;
    file_=(file_state_t)i;
    if(value[n]) stringtotime(file_last_changed_,value+n+1);
  } else if(strcasecmp(key,"registration") == 0) {
    if(value == NULL) return false;
    int n = 0; for(;value[n];n++) if(isspace(value[n])) break;
    if(n == 0) return false;
    int i;
    for(i=0;i<REG_STATE_MAX;i++) if(strncasecmp(value,reg_state_str[i],n) == 0) break;
    if(i>=REG_STATE_MAX) return false;
    reg_=(reg_state_t)i;
    if(value[n]) stringtotime(reg_last_changed_,value+n+1);
  } else if(strcasecmp(key,"pin") == 0) {
    pin_.add(value);
  } else if(strcasecmp(key,"desc") == 0) {
    file_description_=value;
  } else if(strcasecmp(key,"tries") == 0) {
    stringtoint(value,tries_);
  };
  return true;
}

std::ostream& operator<<(std::ostream& o,const SEState &s) {
  o<<"file="<<file_state_str[s.file_]<<" "
   <<timetostring(s.file_last_changed_)<<std::endl;
  o<<"registration="<<reg_state_str[s.reg_]<<" "
    <<timetostring(s.reg_last_changed_)<<std::endl;
  o<<s.pin_;
  if(s.file_description_.length()) o<<"desc="<<s.file_description_<<std::endl;
  o<<"tries="<<s.tries_<<std::endl;
  return o;
}

bool SEState::maintain(void) {
  return pin_.maintain();
}

// ----------------------------------------------------------

typedef bool(*use_pair_t)(const char* key,const char* value,void* arg);

bool read_pairs(const char* fname,use_pair_t func,void* arg) {
  std::ifstream f(fname);
  if(!f.is_open() ) return false;
  char buf[1024];
  for(;!f.eof();) {
    istream_readline(f,buf,sizeof(buf));
    char* key = buf;
    for(;(*key) && isspace(*key);key++);
    if((*key) == '#') continue;
    char* value = strchr(key,'='); 
    if(value) { (*value)=0; value++; };
    if(!func(buf,value,arg)) return false;
  };
  return true;
}

// ----------------------------------------------------------

static int write_range(const char* fname,SEFileRange* ranges) {
  if(ranges == NULL) {
    unlink(fname); return 0;
  };
  std::ofstream o(fname,std::ios::trunc);
  if(!o) return -1;
  for(int i=0;i<MAX_SEFILE_RANGES;i++) {
    if(ranges[i].start == (uint64_t)(-1)) continue;
    o<<ranges[i].start<<" "<<ranges[i].end<<std::endl;
    if(!o) {

    };
  };
  return 0;
}

static int read_range(const char* fname,SEFileRange* ranges) {
  struct stat st;
  if(stat(fname,&st) != 0) {
    if(errno == ENOENT) {
      ranges=NULL;
      return 1;
    };
  };
  int i;
  for(i=0;i<MAX_SEFILE_RANGES;i++) ranges[i].start=(uint64_t)(-1);
  uint64_t start; 
  uint64_t end;
  std::ifstream in(fname);
  if(!in) return -1;
  i=0;
  for(;i<MAX_SEFILE_RANGES;i++) {
    if(in.eof()) break;
    in>>start; if(!in) return -1;
    in>>end; if(!in) return -1;
    in.ignore(256,'\n');
    ranges[i].start=start; ranges[i].end=end;
  };
  return 0;
}

static int join_range(uint64_t start, uint64_t end,SEFileRange* ranges,int max_ranges=MAX_SEFILE_RANGES) {
  int i;
  int empty_slot = max_ranges;
  for(i=0;i<max_ranges;i++) {
    if(ranges[i].start == (uint64_t)(-1)) { empty_slot=i; continue; };
    if((ranges[i].start >= start) && (ranges[i].start <= end)) break;
    if((ranges[i].end >= start) && (ranges[i].end <= end)) break;
  };
  if(i<max_ranges) {
    // uint64_t min=ranges[i].start;
    // uint64_t max=ranges[i].end;
    if(start < ranges[i].start) ranges[i].start=start;
    if(end > ranges[i].end) ranges[i].end=end;
  } else {
    if(empty_slot == max_ranges) return -1;
    i=empty_slot;
    ranges[i].start=start;
    ranges[i].end=end;
  };
  return i;
}

static int compress_range(SEFileRange* ranges) {
  int i;
  int n = 0;
  for(i=0;i<(MAX_SEFILE_RANGES-1);i++) {
    if(ranges[i].start == (uint64_t)(-1)) continue;
    int r = join_range(ranges[i].start,ranges[i].end,ranges+i+1,MAX_SEFILE_RANGES-i-1);
    if(r != -1) { // range replaced
      n++;
      ranges[i].start=(uint64_t)(-1);
      ranges[i].end=0;
    };
  };
  if(ranges[i].start != (uint64_t)(-1)) n++;
  return n;
}

// ----------------------------------------------------------

int SEFile::open(bool for_read) {
  int i;
  lock_.block();
  if(for_read) {
    if(ranges != NULL) { lock_.unblock(); return -1; }; // incomplete
    if(file_handle==-1) file_handle=::open(path.c_str(),O_RDWR);
    if(file_handle==-1) { lock_.unblock(); return -1; }; // failed to open
    read_count++;
    lock_.unblock();
    return 0;
  };
  if((write_count >= (MAX_SEFILE_RANGES-2)) && (ranges != NULL)) { 
    lock_.unblock(); return -1; // too scattered
  };
  if(file_handle==-1) file_handle=::open(path.c_str(),O_RDWR);
  if(file_handle==-1) { lock_.unblock(); return -1; }; // failed to open
  write_count++;
  lock_.unblock();
  return 0;
}

bool SEFile::state_file(file_state_t f) {
  if((f<0) || (f>=FILE_STATE_MAX)) return false;
  if(f != state_.file()) {
    std::string sname = path+".state";
    std::ofstream o(sname.c_str(),std::ios::trunc);
    if(!o) return false;
    state_.file(f); o<<state_;
    if(!o) return false;
  };
  return true;
}

bool SEFile::state_reg(reg_state_t r) {
  if((r<0) || (r>=REG_STATE_MAX)) return false;
  if(r != state_.reg()) {
    std::string sname = path+".state";
    std::ofstream o(sname.c_str(),std::ios::trunc);
    if(!o) return false;
    state_.reg(r); o<<state_;
    if(!o) return false;
  };
  return true;
}

void SEFile::close(bool for_read) {
  lock_.block();
  if(for_read) { read_count--; }
  else { write_count--; };
  if((read_count==0) && (write_count==0)) {
    ::close(file_handle); file_handle=-1;
  };
  // join ranges
  if(ranges && (!for_read)) {
    int n = compress_range(ranges);
    if(n == 1) {
      // check if full file
      if((ranges[MAX_SEFILE_RANGES-1].start == 0) &&
         (ranges[MAX_SEFILE_RANGES-1].end >= size())) {
        odlog(DEBUG)<<"SEFile::close: file is full (length = "<<ranges[MAX_SEFILE_RANGES-1].end<<")"<<std::endl;
        free(ranges); ranges=NULL;
        space.release();
      };
    };
    // store updated range information
    std::string rname = path+".range";
    if(write_range(rname.c_str(),ranges) == -1) {

    };
  };
  lock_.unblock();
  return;
}

void SEFile::destroy(void) {
  valid=false;
  unlink((path+".cred").c_str());
  unlink((path+".range").c_str());
  unlink((path+".attr").c_str());
  unlink((path+".state").c_str());
  unlink((path+".acl").c_str());
  unlink(path.c_str());
}

void SEFile::destroy_content(void) {
  unlink(path.c_str());
}

static char char_tab[16] = {
   '0','1','2','3',
   '4','5','6','7',
   '8','9','a','b',
   'c','d','e','f'
};

static void file_name(uint32_t n,char* name) {
  for(int i=(sizeof(uint32_t)*2-1);i>=0;i--) {
    name[i] = char_tab[n & 0x0F]; n>>=4;
  };
  name[sizeof(int)*2]=0;
}

/* creating new file */
SEFile::SEFile(const char* dirpath,const SEAttributes& attr,DiskSpace& sp):path(dirpath),SEAttributes(attr),space(0,sp) {
  valid=false;
  ranges=NULL; file_handle=-1; read_count=0; write_count=0;
//  id=id_; // TODO check for same name ?
  odlog(VERBOSE)<<"SEFile::SEFile(new): path: "<<path<<std::endl;
  char fname[sizeof(uint32_t)*2+2]; fname[0]='/';
  unsigned int i=0;
  srandom(time(NULL));
  unsigned int r = random();
  for(;i<INT_MAX;i++) {
    file_name(i+r,fname+1);
    /* Warning: O_EXCL is broken on NFS */
    std::string fpath=path+fname; 
    int nh=::open(fpath.c_str(),O_CREAT | O_EXCL | O_RDWR,S_IRUSR | S_IWUSR);
    if(nh == -1) {
      if(errno == ENOSPC) { /* out of space */
        i=INT_MAX; break;
      };
      continue;
    };
    ::close(nh);
    break;
  };
  if(i == INT_MAX) {
    destroy();
    return;
  };
  path+=fname; 
  name=path.c_str()+strlen(dirpath)+1;
  // create file
  int h = ::open(path.c_str(),O_WRONLY | O_CREAT,S_IRUSR | S_IWUSR);
  if(h == -1) {
    destroy();
    return;
  };
  ::close(h);
  if(size_available()) {
    if(!space.request(size())) { // not enough space
      destroy();
      return;
    };
  };
  // initialize ranges
  if((!size_available()) || (size() != 0)) {
    ranges=(SEFileRange*)malloc(sizeof(SEFileRange)*MAX_SEFILE_RANGES);
    for(int i = 0;i<MAX_SEFILE_RANGES;i++) ranges[i].start=(uint64_t)(-1);
  };
  std::string rname = path+".range";
  if(write_range(rname.c_str(),ranges) == -1) {
    odlog(ERROR)<<"SEFile::SEFile(new): failed to write ranges"<<std::endl;
    destroy();
    return;
  };
  rname = path+".attr";
  if(SEAttributes::write(rname.c_str()) != 0) {
    odlog(ERROR)<<"SEFile::SEFile(new): failed to write attributes"<<std::endl;
    destroy();
    return;
  };
  if(!state_file(FILE_STATE_ACCEPTED)) {
    odlog(ERROR)<<"SEFile::SEFile(new): failed to set file state"<<std::endl;
    destroy();
    return;
  };
  valid=true;
  last_changed_=time(NULL);
}


/* acquire existing file */
SEFile::SEFile(const char* path_,DiskSpace& sp):path(path_),space(0,sp) {
  odlog(VERBOSE)<<"SEFile::SEFile: path: "<<path<<std::endl;
  valid=false;
  std::string tpath = path+".attr";
  if(SEAttributes::read(tpath.c_str()) != 0) {
    return;
  };
  std::string::size_type n = path.rfind('/');
  if(n == std::string::npos) { n=0; } else { n++; };
  name=path.c_str()+n;
  // initialize ranges
  tpath = path+".range";
  ranges=(SEFileRange*)malloc(sizeof(SEFileRange)*MAX_SEFILE_RANGES);
  int err;
  if((err=read_range(tpath.c_str(),ranges)) != 0) {
    if(err == 1) { free(ranges); ranges=NULL; }
    else {
      return;
    };
  };
  if((size() == 0) && ranges) {
    free(ranges); ranges=NULL; write_range(tpath.c_str(),ranges);
  };
  // Add rest size to space request
  if(size_available() && ranges) {
    uint64_t avail = 0;
    for(int i = 0;i<MAX_SEFILE_RANGES;i++) {
      if(ranges[i].start == (uint64_t)(-1)) continue;
      if(ranges[i].start <= ranges[i].end) {
        avail+=(ranges[i].end-ranges[i].start+1);
      };
    };
    if(avail<size()) space.request(size()-avail);
  };
  tpath = path+".state";
  if(!read_pairs(tpath.c_str(),&set_state,&state_)) {
    return;
  };
  if((state_.file() == FILE_STATE_COLLECTING) && size_available() && (size() == 0)) {
    state_file(FILE_STATE_COMPLETE); // empty file is easy to handle
  } else if(state_.file() == FILE_STATE_DOWNLOADING) {
    odlog(ERROR)<<"Warning: intermidiate file state DOWNLOADING found. Setting to REQUESTED."<<std::endl;
    state_file(FILE_STATE_REQUESTED);
  };
  if(state_.reg() == REG_STATE_REGISTERING) {
    state_reg(REG_STATE_LOCAL);
    odlog(ERROR)<<"Warning: intermidiate registration state REGISTERING found. Setting to LOCAL."<<std::endl;
  } else if(state_.reg() == REG_STATE_UNREGISTERING) {
    state_reg(REG_STATE_ANNOUNCED);
    odlog(ERROR)<<"Warning: intermidiate registration state UNREGISTERING found. Setting to ANNOUNCED."<<std::endl;
  };
  file_handle=-1; read_count=0; write_count=0;
  valid=true;
  last_changed_=time(NULL);
  odlog(DEBUG)<<"File at "<<path_<<std::endl;
  odlog(INFO)<<"ID: "<<id()<<std::endl;
  odlog(VERBOSE)<<"size: "<<size()<<std::endl;
  odlog(VERBOSE)<<"checksum: "<<checksum()<<std::endl;
  odlog(VERBOSE)<<"creator: "<<creator()<<std::endl;
  odlog(VERBOSE)<<"created: "<<created()<<std::endl;
}

SEFile::~SEFile(void) {
  odlog(ERROR)<<"SEFile::~SEFile"<<std::endl;
  lock_.block();
  if(ranges) compress_range(ranges);
  
  // update ranges information 

  if(ranges) free(ranges); ranges=NULL;
  if(file_handle != -1) ::close(file_handle); file_handle=-1;
  lock_.unblock();
}


uint64_t SEFile::write(void* buf,uint64_t offset,uint64_t size) {
  odlog(VERBOSE)<<"SEFile::write - size: "<<size<<std::endl;
  odlog(VERBOSE)<<"SEFile::write - offset: "<<offset<<std::endl;
  if(size==0) return 0;
  if(ranges == NULL) return size; // full file is already available
  // store data
  uint64_t o = offset;
  uint64_t s = size;
  char* b = (char*)buf;
  for(;;) {
    ssize_t l = pwrite(file_handle,b,s,o);
    if(l == -1) return 0;
    s-=l;
    if(s == 0) break;
    o+=l; b+=l;
  };
  last_changed_=time(NULL);
  lock_.block();
  if(ranges) {
    if(join_range(offset,offset+size,ranges) == -1) { // file is too fragmented
      lock_.unblock();
      return 0;
    };
  };
  lock_.unblock();
  space.release(size);
  return size;
}

uint64_t SEFile::read(void* buf,uint64_t offset,uint64_t size) {
  odlog(VERBOSE)<<"SEFile::read - size: "<<size<<std::endl;
  odlog(VERBOSE)<<"SEFile::read - offset: "<<offset<<std::endl;
  uint64_t o = offset;
  uint64_t s = size;
  char* b = (char*)buf;
  for(;;) {
    ssize_t l = pread(file_handle,b,s,o);
    if(l == -1) {
      perror("pread");
      return 0;
    };
    if(l == 0) break;
    s-=l;
    if(s == 0) break;
    o+=l; b+=l;
  };
  return (size-s);
}

int SEFile::write_attr(void) {
  std::string rname = path+".attr";
  rname = path+".attr";
  odlog(VERBOSE)<<"SEFile::write_attr: to file: "<<rname<<std::endl;
  return SEAttributes::write(rname.c_str());
}

int SEFile::free_ranges(int n,SEFileRange r[]) {
  if(ranges == NULL) return 0; // complete file
  if(n<=0) return 0;
  int i = 0;
  int i_r = 0;
  uint64_t start = 0;
  uint64_t end = 0;
  for(;(i<n) && (i_r<MAX_SEFILE_RANGES);) {
    if(ranges[i_r].start == (uint64_t)(-1)) { i_r++; continue; };
    if(ranges[i_r].start > start) {
      r[i].start=start; r[i].end=ranges[i_r].start-1;
      start=ranges[i_r].end+1;
      i++; i_r++;
      continue;
    };
    start=ranges[i_r].end+1;
    i_r++;
  };
  if(i>=n) return i;
  if(!size_available()) {
    r[i].start=start;  r[i].end=(uint64_t)(-1); // max possible value
    i++;
  } else if(start<size()) {
    r[i].start=start; r[i].end=size()-1; 
    i++;
  };
  return i;
}

// 0 - passed, 1 - try later, -1 - error
int SEFile::verify(void) {
  odlog(VERBOSE)<<"SEFile::verify: start"<<std::endl;
  if(ranges) return 1;   // content is not complete yet
  if(!SEAttributes::complete()) return 1; // metadata is not complete yet
  // complete means checksum is available.
  // checksums's format is "type: value".

  CheckSumAny::type t_ex = CheckSumAny::Type(checksum().c_str());
  CheckSumAny::type t_req = CheckSumAny::md5;
  if(t_ex == CheckSumAny::unknown) {
    olog<<"Unknown checksum type - file can't be verified: "<<id()<<std::endl;
    return -1;
  };
  if(t_ex == CheckSumAny::undefined) {
    // Client said it wants checksum to be computed by service
    // Bad behavior, but needed for SRM.
  } else {
    t_req=t_ex;
  };
  // Compute checksum of file
  if(open(true) != 0) {
    olog<<"Failed to open content for reading - verification failed: "<<id()<<std::endl;
    return -1;
  };
  unsigned char ckbuf[1024*1024];
  uint64_t o = 0;
  CheckSumAny ck(t_ex);
  ck.start();
  for(;;) {
    uint64_t l = read(ckbuf,o,sizeof(ckbuf));
    if(l==0) break;
    if(l==-1) break;
    ck.add(ckbuf,l);
    o+=l;
  };
  close(true);
  ck.end();
  ck.print((char*)ckbuf,sizeof(ckbuf));
  if(t_ex != CheckSumAny::undefined) {
    odlog(VERBOSE)<<"SEFile::verify: computed checksum: "<<(char*)ckbuf<<std::endl;
    // compare checksums
    CheckSumAny ck_(t_ex);
    ck_.scan(checksum().c_str());
    ck_.print((char*)ckbuf,sizeof(ckbuf));
    odlog(VERBOSE)<<"SEFile::verify: provided checksum: "<<(char*)ckbuf<<std::endl;
    if(ck != ck_) {
      odlog(ERROR)<<"SEFile::verify: checksums differ"<<std::endl;
      return -1;
    };
    odlog(VERBOSE)<<"SEFile::verify: checksums same"<<std::endl;
  } else {
    checksum(std::string((char*)ckbuf));
    if(write_attr() != 0) {
      odlog(ERROR)<<"Failed to write attributes."<<std::endl;
      return -1;
    };
  };
  return 0;
}

// Compute checksum of file
int SEFile::checksum_compute(const char* type) {
  CheckSumAny ck(type);
  if(!ck.active()) {
    olog<<"Failed to create checksum of type "<<type<<std::endl;
    return -1;
  };
  if(open(true) != 0) {
    olog<<"Failed to open content for reading - verification failed: "<<id()<<std::endl;
    return -1;
  };
  unsigned char buf[1024*1024];
  uint64_t o = 0;
  ck.start();
  for(;;) {
    uint64_t l = read(buf,o,sizeof(buf));
    if(l==0) break;
    ck.add(buf,l);
    o+=l;
  };
  close(true);
  ck.end();
  ck.print((char*)buf,sizeof(buf));
  odlog(VERBOSE)<<"SEFile:cheksum: computed checksum: "<<(char*)buf<<std::endl;
  checksum(std::string((char*)buf));
  return 0;
}

bool SEFile::pin(const char* id,int valid_for) {
  if(state_.pin(id,valid_for)) {
    std::string sname = path+".state";
    std::ofstream o(sname.c_str(),std::ios::trunc);
    if(!o) return false;
    o<<state_;
    if(!o) return false;
  };
  return true;
}

bool SEFile::unpin(const char* id) {
  if(state_.unpin(id)) {
    std::string sname = path+".state";
    std::ofstream o(sname.c_str(),std::ios::trunc);
    if(!o) return false;
    o<<state_;
    if(!o) return false;
  };
  return true;
};

//bool SEFile::pin(void) {
//  if(state_.pin()) {
//    std::string sname = path+".state";
//    std::ofstream o(sname.c_str(),std::ios::trunc);
//    if(!o) return false;
//    o<<state_;
//    if(!o) return false;
//  };
//  return true;
//}

//bool SEFile::unpin(void) {
//  if(state_.unpin()) {
//    std::string sname = path+".state";
//    std::ofstream o(sname.c_str(),std::ios::trunc);
//    if(!o) return false;
//    o<<state_;
//    if(!o) return false;
//  };
//  return true;
//};

void SEFile::Maintain(void) {
  if(state_.maintain()) {
    std::string sname = path+".state";
    std::ofstream o(sname.c_str(),std::ios::trunc);
    if(!o) return;
    o<<state_;
  };
}

// -------------------------------------------------------------

#define skip_space(__p) for(;*__p;__p++) if(!isspace(*__p)) break;
#define skip_till_space(__p) for(;*__p;__p++) if(isspace(*__p)) break;

SEAttributes::SEAttributes(const char* fname) {
  valid=false;
  size_b=false;
  checksum_b=false;
  created_b=false;
  read(fname);
}

int SEAttributes::read(const char* fname) {
  odlog(VERBOSE)<<"SEAttributes::read"<<std::endl;
  std::ifstream f(fname);
  if(!f) return -1;
  char buf[1024];
  bool have_id = false;
  for(;!f.eof();) {
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    char* p = buf;
    skip_space(p);
    char* command = p;
    if(*command == '#') continue;
    odlog(VERBOSE)<<"SEAttributes::read: line "<<p<<std::endl;
    skip_till_space(p);
    int command_len = p-command;
    if(!command_len) continue;
    if((command_len==4) && (strncmp(command,"size",command_len)==0)) {
      odlog(VERBOSE)<<"SEAttributes::read: command size: "<<p<<std::endl;
      std::string value(p); size_b=false;
      if(!stringtoint(value,size_i)) {
        odlog(ERROR)<<"Size value is bad: "<<value<<std::endl; return -1;
      } else { size_b=true; };
    } else if((command_len==2) && (strncmp(command,"id",command_len)==0)) {
      odlog(VERBOSE)<<"SEAttributes::read: id: "<<p<<std::endl;
      if(input_escaped_string(p,id_i) == 0) {
        odlog(ERROR)<<"Can't read file's ID"<<std::endl; return -1;
      } else { have_id=true; };
    } else if((command_len==8) && (strncmp(command,"checksum",command_len)==0)){
      odlog(VERBOSE)<<"SEAttributes::read: checksum: "<<p<<std::endl;
      for(;*p;++p) if(!isspace(*p)) break;
      checksum_b=true; checksum_i=p;
    } else if((command_len==7) && (strncmp(command,"creator",command_len)==0)){
      odlog(VERBOSE)<<"SEAttributes::read: creator: "<<p<<std::endl;
      if(input_escaped_string(p,creator_i) == 0) {
        odlog(ERROR)<<"Can't read file's creator"<<std::endl; return -1;
      };
    } else if((command_len==7) && (strncmp(command,"created",command_len)==0)){
      odlog(VERBOSE)<<"SEAttributes::read: created: "<<p<<std::endl;
      std::string created_str;
      if(input_escaped_string(p,created_str) == 0) {	 
        odlog(ERROR)<<"Can't read creation time"<<std::endl; return -1;
      } else {
        odlog(VERBOSE)<<"SEAttributes::read: created: time string: "<<created_str<<std::endl;
        if(stringtotime(created_i,created_str)) {
          odlog(ERROR)<<"Can't interpret creation time"<<std::endl; return -1;
        };
        created_b=true;
      };
    } else if((command_len==6) && (strncmp(command,"source",command_len)==0)){
      odlog(VERBOSE)<<"SEAttributes::read: source: "<<p<<std::endl;
      std::string source_str;
      input_escaped_string(p,source_str); 
      sources_.push_back(source_str);
    };
  };
  if(!have_id)   { odlog(ERROR)<<"Missing file's ID"<<std::endl;   return -1; };
  // if(!size_b) { odlog(ERROR)<<"Missing file's size"<<std::endl; return -1; };
  valid=true;
  return 0;
}

void SEAttributes::created(const time_t* c) {
  struct tm* t;
  struct tm tt;
  created_b=false;
  t=gmtime_r(c,&tt);
  if(!t) return;
  memcpy(&created_i,t,sizeof(struct tm));
  created_b=true;
}

void SEAttributes::created(const char* c) { 
  created_b=false;
  if(!c) return;
  std::string s(c);
  if(stringtotime(created_i,s)) {
    odlog(ERROR)<<"Can't interpret creation time: "<<s<<std::endl; return;
  };
  created_b=true;
}

bool SEAttributes::created_compare(const char* c) {
  if(!c) return false;
  std::string s(c);
  struct tm t;
  if(stringtotime(t,s)) return false;
  return (memcmp(&t,&created_i,sizeof(struct tm)) == 0); 
}

int SEAttributes::write(const char* fname) {
  odlog(VERBOSE)<<"SEAttributes::write: "<<fname<<std::endl;
  // if(!valid) return -1; write whatever we have
  odlog(VERBOSE)<<"SEAttributes::write: valid"<<std::endl;
  std::ofstream o(fname,std::ios::trunc);
  if(!o) return -1;
  odlog(VERBOSE)<<"SEAttributes::write: opened"<<std::endl;
  std::string tmp = id_i; make_escaped_string(tmp);
  o<<"id "<<tmp<<std::endl;
  if(size_b) o<<"size "<<size_i<<std::endl;
  tmp = creator_i; make_escaped_string(tmp);
  o<<"creator "<<tmp<<std::endl;
  if(checksum_b) o<<"checksum "<<checksum_i<<std::endl;
  if(created_b) {
    o<<"created ";
    int w = o.width(4); int c = o.fill('0');
    o<<(created_i.tm_year+1900);
    o.width(2); o<<created_i.tm_mon;
    o.width(2); o<<created_i.tm_mday;
    o.width(2); o<<created_i.tm_hour;
    o.width(2); o<<created_i.tm_min;
    o.width(2); o<<created_i.tm_sec;
    o.width(w); o.fill(c);
    o<<std::endl;
  };
  for(std::list<std::string>::iterator i = sources_.begin();i!=sources_.end();i++) {
    std::string tmp = *i; make_escaped_string(tmp);
    o<<"source "<<tmp<<std::endl;
  };
  odlog(VERBOSE)<<"SEAttributes::write: written"<<std::endl;
  if(!o) return -1;
  odlog(VERBOSE)<<"SEAttributes::write: exiting"<<std::endl;
  return 0;
}

bool SEAttributes::enough(void) {
  return (valid && created_b && size_b && id_i.length() && creator_i.length());
}

bool SEAttributes::complete(void) {
  odlog(VERBOSE)<<"SEAttributes::complete: valid: "<<valid<<std::endl<<
            "SEAttributes::complete: created: "<<created_b<<std::endl<<
            "SEAttributes::complete: size: "<<size_b<<std::endl<<
            "SEAttributes::complete: id: "<<id_i<<std::endl<<
            "SEAttributes::complete: creator: "<<creator_i<<std::endl<<
            "SEAttributes::complete: checksum: "<<checksum_b<<std::endl;

  return (valid && created_b && size_b && id_i.length() && creator_i.length() && checksum_b);
}

