#ifndef __SE_FILE_H__
#define __SE_FILE_H__

#include <string>
#include <list>
#include <iostream>

#include "disk_space.h"
#include "se_ns.h"
#include "../../misc/condition.h"
#include "../../auth/gacl_auth.h"
#include "../../misc/safelist.h"

// Access control
#define FILE_ACC_NONE        GACL_PERM_NONE
#define FILE_ACC_LIST        GACL_PERM_LIST
#define FILE_ACC_READ        GACL_PERM_READ
#define FILE_ACC_WRITE       GACL_PERM_WRITE
#define FILE_ACC_CREATE      GACL_PERM_WRITE
#define FILE_ACC_DELETE      GACL_PERM_WRITE
#define FILE_ACC_READ_META   (GACL_PERM_ADMIN | GACL_PERM_LIST)
#define FILE_ACC_WRITE_META  GACL_PERM_ADMIN
#define FILE_ACC_ALL         0xFFFF 

typedef enum {
  FILE_STATE_ACCEPTED     =  0,    // File accepted at SE (initial state)
  FILE_STATE_COLLECTING   =  1,    // File is being uploaded by client
  FILE_STATE_REQUESTED    =  2,    // File should be downloaded by service
  FILE_STATE_DOWNLOADING  =  3,    // File is being downloaded
  FILE_STATE_COMPLETE     =  4,    // File is physically complete
  FILE_STATE_VALID        =  5,    // File's content is valid
  FILE_STATE_DELETING     =  6,    // File is being deleted
  FILE_STATE_FAILED       =  7     // Processing of file failed
} file_state_t;

#define FILE_STATE_MAX            8
#define FILE_STATE_UNCHANGED      FILE_STATE_MAX

/* ***********************************************************************
ACCEPTED -> COLLECTING   - immediately, during processing of request.
ACCEPTED -> REQUESTED    - immediately, during processing of request.
COLLECTING -> COMPLETE   - after whole content is stored by client.
REQUESTED -> DOWNLOADING - file is picked from queue, service started 
  downloading.
DOWNLOADING -> REQUESTED - file is put back to queue because it can't 
  be downloaded now.
DOWLOADING -> COMPLETE   - after whole content is downloaded by service.
COMPLETE -> VALID        - after checksum is computed and optionally
  verified.
* -> DELETING            - upon request from client, file will disappear
  soon.
COLLECTING -> FAILED     - if client is uploading content too slowly.
REQUESTED -> FAILED      - if all allowed attempts to download file failed.
COMPLETE -> FAILED       - if checksum verification failed.
VALID -> FAILED          - if file content was corrupted somehow.
 File is kelp in FAILED for predefined time and then removed.
 While in failed content is already deleted.
   ********************************************************************** */

typedef enum {
  REG_STATE_LOCAL         =  0,
  REG_STATE_REGISTERING   =  1,  // File claimed - registration in progress
  REG_STATE_ANNOUNCED     =  2,
  REG_STATE_UNREGISTERING =  3   // File being deleted - unregistration in progress
} reg_state_t;

#define REG_STATE_MAX             4
#define REG_STATE_UNCHANGED       REG_STATE_MAX

/* ***********************************************************************
  LOCAL -> REGISTERING  - after file changed stae from ACCEPTED to COLLECTING or DOWNLOADING.
  REGISTERING -> LOCAL - if registration attempt failed. Regitration will be retried. Number of retries is unlimited.
  REGISTRING -> ANNOUNCED - after file was registered.
  ANNOUNCED -> UNREGISTERING - if file being deleted or somehow else going to be removed. File is not removed before it is unregistered? 
   ********************************************************************** */

extern const char* file_state_str[];
extern const char* reg_state_str[];

#define FILE_STATE_ACCEPTED_STR      file_state_str[FILE_STATE_ACCEPTED]
#define FILE_STATE_COLLECTING_STR    file_state_str[FILE_STATE_COLLECTING
#define FILE_STATE_COMPLETE_STR      file_state_str[FILE_STATE_COMPLETE]
#define FILE_STATE_VALID_STR         file_state_str[FILE_STATE_COMPLETE]
#define FILE_STATE_DELETING_STR      file_state_str[FILE_STATE_DELETING]

#define REG_STATE_LOCAL_STR          reg_state_str[REG_STATE_LOCAL]
#define REG_STATE_REGISTERING_STR    reg_state_str[REG_STATE_REGISTERING]
#define REG_STATE_ANNOUNCED_STR      reg_state_str[REG_STATE_ANNOUNCED]
#define REG_STATE_UNREGISTERING_STR  reg_state_str[REG_STATE_UNREGISTERING]

// Undefined number of tries. Will be changed to configured value.
#define DEFAULT_TRIES (-1)

typedef struct {
  uint64_t start;
  uint64_t end;
} SEFileRange;
#define MAX_SEFILE_RANGES 100
#define MAX_RANGE_VALUE ((uint64_t)(-1))
#define MIN_RANGE_VALUE (0)

class SEAuthorization {

};

// File metadata
class SEAttributes {
 private:
  bool valid;
  uint64_t size_i;
  bool size_b;
  std::string id_i;
  std::string creator_i;
  std::string checksum_i;
  bool checksum_b;
  struct tm created_i;
  bool created_b;
  std::list<std::string> sources_;
 public:
  operator bool(void) { return valid; };
  bool operator!(void) { return !valid; };
  bool enough(void);
  bool complete(void);
  SEAttributes(void):valid(false),size_b(false),checksum_b(false),created_b(false) { };
  SEAttributes(const char* id,uint64_t size,const AuthUser user):id_i(id),size_b(true),size_i(size),valid(true) {
    checksum_b=false;
    created_b=false;
    creator_i=user.DN();
  };
  SEAttributes(const char* id,const AuthUser user):id_i(id),size_b(false),valid(false) {
    checksum_b=false;
    created_b=false;
    creator_i=user.DN();
  };
  SEAttributes(const SEAttributes& attr):valid(attr.size_b) {
    id_i=attr.id_i;
    size_i=attr.size_i; size_b=attr.size_b;
    if((checksum_b=attr.checksum_b)) checksum_i=attr.checksum_i;
    if((created_b=attr.created_b)) memcpy(&created_i,&(attr.created_i),sizeof(created_i));
    creator_i=attr.creator_i;
    sources_=attr.sources_;
  };
  SEAttributes(const char* fname);
  void operator=(const SEAttributes& attr) {
    valid=attr.valid;
    id_i=attr.id_i;
    size_i=attr.size_i; size_b=attr.size_b;
    checksum_b=attr.checksum_b; checksum_i=attr.checksum_i;
    created_b=attr.created_b; memcpy(&created_i,&(attr.created_i),sizeof(created_i));
    creator_i=attr.creator_i;
    sources_=attr.sources_;
  };
  int read(const char* fname);
  int write(const char* fname);
  uint64_t size(void) const {
    if(!size_b) return MAX_RANGE_VALUE;
    return size_i;
  };
  void size(uint64_t s) { size_i=s; size_b=true; valid=true; };
  bool size_available(void) { return size_b; };
  const char* id(void) const { return id_i.c_str(); };
  void id(const char* i) { id_i=i; };
  const std::string& creator(void) const { return creator_i; };
  const std::string& checksum(void) const { return checksum_i; };
  void checksum(const std::string& m) { checksum_i=m; checksum_b=true; };
  bool checksum_available(void) const { return checksum_b; };
  // TODO: rewrite 
  bool checksum_compare(const std::string& m) { return (checksum_i==m); };
  const struct tm* created(void) const { return &created_i; };
  void created(const struct tm* c) { memcpy(&created_i,c,sizeof(created_i)); created_b=true; };
  void created(const time_t* c);
  void created(const char* c);
  void created(const std::string& c) { created(c.c_str()); };
  bool created_available(void) const { return created_b; };
  bool created_compare(const char* c);
  // void md5_string(std::string &s);
  const std::list<std::string> & sources(void) const { return sources_; };
  void source(const std::string s) { sources_.push_back(s); };
  void source(const char* s) { sources_.push_back(std::string(s)); };
};

class SEReqAttr {
 private:
  std::string who_;
  time_t till_;
  bool parse(const char* s);
 public:
  SEReqAttr(const char* id,int valid):who_(id) {
    till_=time(NULL)+valid;
  };
  SEReqAttr(const char* s) { parse(s); };
  SEReqAttr(std::istream& i);
  operator bool(void) const { return (who_.length() > 0); };
  bool operator!(void) const { return (who_.length() <= 0); };
  const char* id(void) const { return who_.c_str(); };
  int left(void) const { return (till_-time(NULL)); };
  time_t till(void) const { return till_; };
  bool extend(int valid);
};

std::ostream& operator<<(std::ostream& o,const SEReqAttr &a);

// Simple blind pin
class SEPin {
 private:
  bool pin_;
 public:
  SEPin(void):pin_(false) { };
  bool add(const char* s);
  bool add(void) { if(pin_) return false; pin_=true; return true; };
  bool remove(void) { if(!pin_) return false; pin_=false; return true; };
  int pinned(void) const { return pin_; };
};

std::ostream& operator<<(std::ostream& o,const SEPin &a);

// Multiple pin's with id
#define DEFAULT_PIN_TIME (8*3600)
class SEPins {
 friend std::ostream& operator<<(std::ostream& o,const SEPins &a);
 private:
  std::list<SEReqAttr> pins_;
  bool add(const SEReqAttr& a);
 public:
  SEPins(void);
  bool add(const char* s);
  bool add(std::istream& i);
  bool add(const char* id,int valid);
  bool remove(const char* id);
  bool maintain(void);
  int pinned(void) const;
  int pinned(const char* id) const;
};

std::ostream& operator<<(std::ostream& o,const SEPins &a);

class SEState {
 friend std::ostream& operator<<(std::ostream& o,const SEState &s);
 private:
  file_state_t file_;
  reg_state_t reg_;
  time_t file_last_changed_;
  time_t reg_last_changed_;
  SEPins pin_;
  //SEPin pin_;
  std::string file_description_;
  int tries_;
 public:
  SEState(void):file_(FILE_STATE_ACCEPTED),reg_(REG_STATE_LOCAL),file_description_("") {
    file_last_changed_=time(NULL);
    reg_last_changed_=time(NULL);
    tries_=DEFAULT_TRIES;
  };
  SEState(file_state_t f,reg_state_t r):file_(f),reg_(r),file_description_("") {
    file_last_changed_=time(NULL);
    reg_last_changed_=time(NULL);
  };
  ~SEState() { };
  file_state_t file(void) const { return file_; };
  reg_state_t reg(void) const { return reg_; };
  file_state_t file(file_state_t f) {
    if((f>=FILE_STATE_UNCHANGED) || (f<0)) return file_;
    if(f != file_) file_last_changed_=time(NULL);
    return file_=f;
  };
  void file_description(const char* s) { file_description_=s; };
  void file_description(const std::string& s) { file_description_=s; };
  reg_state_t reg(reg_state_t r) {
    if((r>=REG_STATE_UNCHANGED) || (r<0)) return reg_;
    if(r != reg_) reg_last_changed_=time(NULL);
    return reg_=r;
  };
  int pinned(void) { return pin_.pinned(); };
  int pinned(const char* id) { return pin_.pinned(id); };
  bool pin(const char* id,int valid) { return pin_.add(id,valid); };
  bool unpin(const char* id) { return pin_.remove(id); };
  //bool pin(void) { return pin_.add(); };
  //bool unpin(void) { return pin_.remove(); };
  bool set(const char* key,const char* value);
  time_t file_changed(void) { return file_last_changed_; };
  time_t reg_changed(void) { return reg_last_changed_; };
  bool tries(void) { return (tries_>0); };
  void tries(int n) { tries_=n; };
  bool tries_dec(void) { if(tries_>0) tries_--; return (tries_>0); };
  bool maintain(void);
};

std::ostream& operator<<(std::ostream& o,const SEState &s);

class SEFileHandle;

// File itself
class SEFile: public SEAuthorization, public SEAttributes {
 private:
  bool valid;

  std::string path;     // Path to directory to store file
  const char* name;     // Name on file system

  SEFileRange* ranges;  // Ranges available (file is complete if NULL)
  LockSimple lock;      // use to atomically perform modifications (used 
                        // only from outside of that class)
  LockSimple lock_;     // Lock for internal operations
  int read_count;       // how many times it is open for reading
  int write_count;      // how many times it is open for writing
  int usage_count;      // not used yet
  int file_handle;      // Handle for file access
  time_t last_changed_;
  DiskSpaceLink space;  // Disk space request handler
  SEState state_;       // State of file and registration (is handled 
                        // outside of this class yet)

 public:
  operator bool(void) { return valid; };
  bool operator!(void) { return !valid; };
  /// Create new file
  SEFile(const char* dirpath,const SEAttributes&,DiskSpace& sp);
  /// Acquire stored file
  SEFile(const char *fname,DiskSpace& sp);
  ~SEFile(void);
  bool check_id(const char* id_) const { return (strcmp(id(),id_)==0); };
  bool check_name(const char* name_) const { return (strcmp(name,name_)==0); };
  int open(bool for_read);
  void close(bool for_read);
  uint64_t write(void* buf,uint64_t offset,uint64_t size);
  uint64_t read(void* buf,uint64_t offset,uint64_t size);
  int check_acl(const AuthUser &user);
  int write_acl(const AuthUser &user,const char* acl);
  int read_acl(const AuthUser &user,std::string& acl);
  int write_credentials(const char* cred);
  int read_credentials(std::string& cred);
  std::string credentials_file(void);
  int write_attr(void);
  bool state_file(file_state_t f);
  bool state_reg(reg_state_t r);
  file_state_t state_file(void) { return state_.file(); };
  reg_state_t state_reg(void) { return state_.reg(); };
  void state_file_description(const char* s) { state_.file_description(s); };
  void state_file_description(const std::string& s) { state_.file_description(s); };
  bool state_tries(void) { return state_.tries(); };
  void state_tries(int n) { state_.tries(n); };
  bool state_tries_dec(void) { return state_.tries_dec(); };
  void acquire(void) { lock.block(); };     // lock for atomic operation
  void release(void) { lock.unblock(); };   // release after atomic operation
  void destroy(void);
  void destroy_content(void);
  time_t state_file_changed(void) { return state_.file_changed(); };
  time_t state_reg_changed(void) { return state_.reg_changed(); };
  time_t data_changed(void) { return last_changed_; };
  bool pin(const char* id,int valid);
  bool unpin(const char* id);
  //bool pin(void);
  //bool unpin(void);
  int pinned(void) { return state_.pinned(); };
  int pinned(const char* id) { return state_.pinned(id); };
  int free_ranges(int num_ranges,SEFileRange ranges[]);
  bool full(void) { return (ranges==NULL); };
  /// Verify file's content. Returns:
  ///   0 - test passed
  ///   1 - some (meta)data is not yet available, try later
  ///  -1 - verification failed
  int verify(void);
  int checksum_compute(const char* type);
  void Maintain(void);
};

#endif // __SE_FILE_H__
