#include "../../std.h"

#include <iostream>
#include <string>
#include <list>

#define LDAP_DEPRECATED 1
#include <ldap.h>

#include "se_ldap.h"

#include "../../misc/inttostring.h"

#if defined(HAVE_SASL_H) || defined(HAVE_SASL_SASL_H)
#ifdef HAVE_SASL_H
#include <sasl.h>
#endif
#ifdef HAVE_SASL_SASL_H
#include <sasl/sasl.h>
#endif

#define SASLMECH "GSI-GSSAPI"

static int sasl_interact(LDAP* ld,unsigned flags,void* defaults,void* in) {
  sasl_interact_t* interact = (sasl_interact_t*)in;
  while(interact->id != SASL_CB_LIST_END) {
    const char* dflt = interact->defresult;
    interact->result = strdup( (dflt && *dflt) ? dflt : "" );
    interact->len = interact->result ? strlen((char*)(interact->result)) : 0;
    interact++;
  };
  return LDAP_SUCCESS;
}
#endif

#define TIMEOUT 60

LDAPConnector::LDAPConnector(const char* host_,int port_):connection(NULL),host(host_),port(port_) {
  connect();
}

int LDAPConnector::connect(void) {
  const int version = LDAP_VERSION3;
  int timeout = TIMEOUT;
  timeval tout; tout.tv_sec = timeout; tout.tv_usec = 0;
#if defined(HAVE_SASL_H) || defined(HAVE_SASL_SASL_H)
  unsigned sasl_flags = LDAP_SASL_QUIET;
  const char* sasl_mech = SASLMECH;
#endif
  //struct berval passwd = { 0, NULL };
  void* defaults;
  ldap_initialize(&connection, ("ldap://" + host + ':' + inttostring(port)).c_str());
  if(!connection) {
    std::cerr<<"Could not open LDAP connection to "<<host<<":"<<port<<std::endl;
    goto errorexit;
  };
  if(ldap_set_option(connection,LDAP_OPT_NETWORK_TIMEOUT,&tout) != LDAP_OPT_SUCCESS) {
    std::cerr<<"Could not set LDAP network timeout"<<std::endl;
    goto errorexit;
  };
  if(ldap_set_option(connection,LDAP_OPT_TIMELIMIT,&timeout) != LDAP_OPT_SUCCESS) {
    std::cerr<<"Could not set LDAP timelimit"<<std::endl;
    goto errorexit;
  };
  if(ldap_set_option(connection,LDAP_OPT_PROTOCOL_VERSION,&version) != LDAP_OPT_SUCCESS) {
    std::cerr << "Could not set LDAP protocol version" << std::endl;
    goto errorexit;
  };
  int ldresult;
#if defined(HAVE_SASL_H) || defined(HAVE_SASL_SASL_H)
  if((ldresult=ldap_sasl_interactive_bind_s(connection,NULL,
         sasl_mech,NULL,NULL,sasl_flags,&sasl_interact,NULL)) != LDAP_SUCCESS) {
#else
  if((ldresult=ldap_simple_bind_s(connection,NULL,NULL)) != LDAP_SUCCESS) {
#endif
    std::cerr<<"Connection failed to "<<host<<":"<<port<<std::endl;
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    goto errorexit;
  };
  return 0;
 errorexit:
  if (connection) {
    ldap_unbind_ext(connection, NULL, NULL);
    connection=NULL;
  };
  return -1;
}

LDAPConnector::~LDAPConnector(void) {
  if (connection) {
    ldap_unbind_ext(connection, NULL, NULL);
    connection=NULL;
  };
}



// Check if 'filter' is suitable for selecting object at 'base'
// 0 - yes, 1 - no, -1 - error
int LDAPConnector::CheckEntry(const char* base,const char* filter) {
  if (!connection) {
    std::cerr<<"no LDAP connection to "<<host<<":"<<port<<std::endl;
    return -1;
  };
  timeval tout;
  tout.tv_sec = TIMEOUT;
  tout.tv_usec = 0;
  int messageid;
  const char* attrs[] = {"dn", NULL};
  int ldresult = ldap_search_ext(connection,base,LDAP_SCOPE_BASE,filter,
                       (char**)attrs,0,NULL,NULL,&tout,0,&messageid);
  if (ldresult != LDAP_SUCCESS) {
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };

  bool done = false;
  bool found = false;
  LDAPMessage* res = NULL;
  while(!done && (ldresult=ldap_result(connection,messageid,LDAP_RES_ANY,
                                       &tout,&res)) > 0) {
    for(LDAPMessage* msg = ldap_first_message (connection, res); msg;
        msg = ldap_next_message (connection, msg)) {
      switch (ldap_msgtype (msg)) {
        case LDAP_RES_SEARCH_ENTRY:
          found=true;
        break;
        case LDAP_RES_SEARCH_RESULT:
          done = true;
        break;
      };
    };
    ldap_msgfree(res);
  };
  if (ldresult == 0) {
    std::cerr<<"LDAP query to "<<host<<":"<<port<<" timed out"<<std::endl;
    return -1;
  };
  if (ldresult == -1) {
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };
  if(found) return 0;
  return 1;
}

static void attr2attr(std::list<LDAPConnector::Attribute> &attrs,LDAPMod** atsp,LDAPMod* ats) {
  LDAPMod* a = ats;
  int n = 0;
  for(std::list<LDAPConnector::Attribute>::iterator i = attrs.begin();i!=attrs.end();){
    atsp[n]=a;
    a->mod_type=NULL;
    a->mod_values=(char**)malloc(sizeof(char*)*(attrs.size()+1));
    if(a->mod_values) {
      for(int nn = 0;nn<=attrs.size();nn++) a->mod_values[nn]=NULL;
    };
    ++i; a++; n++;
  };
  atsp[n]=NULL;
  int max_n = -1;
  for(std::list<LDAPConnector::Attribute>::iterator i = attrs.begin();i!=attrs.end();){
    a=NULL;
    for(n=0;atsp[n];n++) {
      if(atsp[n]->mod_type == NULL) break;
      if(i->attr == atsp[n]->mod_type) { a=atsp[n]; break; };
    };
    if(!a) { max_n++; a=atsp[max_n]; };
    a->mod_op=LDAP_MOD_ADD;
    a->mod_type=(char*)(i->attr.c_str());
    if(a->mod_values) {
      int nn;
      for(nn=0;a->mod_values[nn];nn++) { };
      a->mod_values[nn]=(char*)(i->value.c_str());
    };
    ++i;
  };
  atsp[++max_n]=NULL;
}

int LDAPConnector::SetAttributes(const char* base,std::list<Attribute> &attrs) {
  if(attrs.size() <= 0) return -1;
  int atsn = attrs.size();
  LDAPMod* ats = (LDAPMod*)malloc(sizeof(LDAPMod)*attrs.size());
  if(ats == NULL) return -1;
  LDAPMod** atsp = (LDAPMod**)malloc(sizeof(LDAPMod*)*(attrs.size()+1));
  if(atsp == NULL) { free(ats); return -1; };
  attr2attr(attrs,atsp,ats);
  int ldresult = ldap_modify_ext_s(connection,base,atsp,NULL,NULL);
  for(int n=0;n<atsn;n++) if(ats[n].mod_values) free(ats[n].mod_values);
  free(ats); free(atsp);
  if (ldresult != LDAP_SUCCESS) {
    if(ldresult == LDAP_ALREADY_EXISTS) {
      return 1;
    };
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };
  return 0;
}

int LDAPConnector::CreateEntry(const char* base,std::list<Attribute> &attrs) {
  int atsn = attrs.size();
  LDAPMod* ats = NULL;
  if(atsn) ats = (LDAPMod*)malloc(sizeof(LDAPMod)*attrs.size());
  if(ats == NULL) return -1;
  LDAPMod** atsp = (LDAPMod**)malloc(sizeof(LDAPMod*)*(attrs.size()+1));
  if(atsp == NULL) { free(ats); return -1; };
  attr2attr(attrs,atsp,ats);
  int ldresult = ldap_add_ext_s(connection,base,atsp,NULL,NULL);
  for(int n=0;n<atsn;n++) if(ats[n].mod_values) free(ats[n].mod_values);
  free(ats); free(atsp);
  if (ldresult != LDAP_SUCCESS) {
    if(ldresult == LDAP_ALREADY_EXISTS) {
      return 1;
    };
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };
  return 0;
}

int LDAPConnector::GetAttributes(const char* base,std::list<Attribute> &attrs) {
  bool use_attrs = (attrs.size() > 0);
  if (!connection) {
    std::cerr<<"no LDAP connection to "<<host<<":"<<port<<std::endl;
    return -1;
  };
  timeval tout;
  tout.tv_sec = TIMEOUT;
  tout.tv_usec = 0;
  int messageid;
  const char** ats = NULL;
  if(use_attrs) {
    ats=(const char**)malloc((attrs.size()+1)*sizeof(char*));
    if(ats) {
      std::list<Attribute>::iterator i = attrs.begin();
      int ii=0;
      for(;i != attrs.end();++i,ii++) ats[ii]=i->attr.c_str();
      ats[ii]=NULL;
    };
    attrs.clear();
  };
  const char* filter = NULL;
  int ldresult = ldap_search_ext(connection,base,LDAP_SCOPE_BASE,filter,
                       (char**)ats,0,NULL,NULL,&tout,0,&messageid);
  if (ldresult != LDAP_SUCCESS) {
    free(ats);
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };
  free(ats);

  bool done = false;
  bool found = false;
  LDAPMessage* res = NULL;
  while(!done && (ldresult=ldap_result(connection,messageid,LDAP_RES_ANY,
                                       &tout,&res)) > 0) {
    for(LDAPMessage* msg = ldap_first_message (connection, res); msg;
        msg = ldap_next_message (connection, msg)) {
      BerElement * ber = NULL;
      switch (ldap_msgtype (msg)) {
        case LDAP_RES_SEARCH_ENTRY:
          for(char* attr = ldap_first_attribute(connection,msg,&ber); attr;
              attr=ldap_next_attribute(connection,msg,ber)) {
            BerValue ** bval;
            if (bval = ldap_get_values_len(connection,msg,attr)) {
              for (int i = 0;bval[i];i++) {
                if(bval[i]->bv_val) {
                  attrs.push_back(Attribute(attr,bval[i]->bv_val));
                } else {
                  attrs.push_back(Attribute(attr));
                };
              };
              ber_bvecfree (bval);
            };
          };
          if(ber) ber_free(ber, 0);
        break;
        case LDAP_RES_SEARCH_RESULT:
          done = true;
        break;
      }
    }
    ldap_msgfree (res);
  };
  if (ldresult == 0) {
    std::cerr<<"LDAP query to "<<host<<":"<<port<<" timed out"<<std::endl;
    return -1;
  };
  if (ldresult == -1) {
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };
  return 0;
  //  check if entry present return 1;
}

int LDAPConnector::Query(const char* base,
           const char* filter,int scope,const char* attrs[],
           query_callback_t callback,void* ref) {
  if (!connection) {
    std::cerr<<"no LDAP connection to "<<host<<":"<<port<<std::endl;
    return -1;
  };
  timeval tout;
  tout.tv_sec = TIMEOUT;
  tout.tv_usec = 0;
  int messageid;
  int ldresult = ldap_search_ext(connection,base,scope,filter,
                       (char**)attrs,0,NULL,NULL,&tout,0,&messageid);
  if (ldresult != LDAP_SUCCESS) {
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    return -1;
  };

  bool done = false;
  LDAPMessage* res = NULL;

  while(!done && (ldresult=ldap_result(connection,messageid,LDAP_RES_ANY,
                                       &tout,&res)) > 0) {
    for(LDAPMessage* msg = ldap_first_message (connection, res); msg;
        msg = ldap_next_message (connection, msg)) {
      BerElement * ber = NULL;
      switch (ldap_msgtype (msg)) {
        case LDAP_RES_SEARCH_ENTRY: {
          char* dn = ldap_get_dn(connection,msg);
          for(char * attr = ldap_first_attribute (connection, msg, &ber); attr;
             attr = ldap_next_attribute (connection, msg, ber)) {
            BerValue ** bval;
            if (bval = ldap_get_values_len (connection, msg, attr)) {
              for (int i = 0; bval[i]; i++)
                callback (dn, attr, bval[i]->bv_val, ref);
              ber_bvecfree (bval);
            };
          };
          if(ber) ber_free (ber, 0);
          if(dn) ldap_memfree(dn);
        }; break;
        case LDAP_RES_SEARCH_RESULT:
          done = true;
        break;
      };
    };
    ldap_msgfree (res);
  }

  int retval = 0;

  if (ldresult == 0) {
    std::cerr << "LDAP query to " << host << " timed out" << std::endl;
    retval = 1;
  }

  if (ldresult == -1) {
    std::cerr<<ldap_err2string(ldresult)<<std::endl;
    retval = 1;
  }

  return 0;
}

