#include "../../std.h"

#include <iostream>

#include "../../datamove/datapoint.h"
#include "../../datamove/datahandle.h"
#include "../../datamove/databufferpar.h"
#include "../../misc/log_time.h"

#include "se_replicator.h"


SEReplicator::SEReplicator(void) {
  timeout=-1;
}

SEReplicator::~SEReplicator(void) {
  lock.block(); // after mutex destroyed waiting threads will be informed
}

// File state is not checked. It must be already DOWNLOADING.
int SEReplicator::Obtain(SEFile& file) {
  const std::list<std::string> &sources = file.sources();
  lock.block();
  if(sources.size() == 0) { lock.unblock(); return -1; };
  bool downloaded = false;
  for(std::list<std::string>::const_iterator u = sources.begin();u!=sources.end();++u) {
    if(downloaded) break;
    // TODO - download from multiple sources simultaneously
    // Little hack to deal with wrong records created by fixed bug
    std::string u_ = *u;
    if(strncmp(u_.c_str(),"lrc://",6) == 0) u_.replace(0,3,"rls");
    DataPoint source(u_.c_str());
    if(!source) {
      odlog(ERROR)<<"Failed to acquire source: "<<source<<std::endl;
      continue;
    };
    source.tries(1);
    if(!source.meta_resolve(true,UrlMap())) {
      odlog(ERROR)<<"Failed to resolve source: "<<source<<std::endl;
      continue;
    };
    // Do some tricks with meta data (size and checksum so far).
    {
      bool modified = false;
      if(source.meta_size_available()) {
        unsigned long long int size = source.meta_size();
        if(!file.size_available()) file.size(size);
        modified=true;
      };
      if(source.meta_checksum_available()) {
        std::string check = source.meta_checksum();
        if(!file.checksum_available()) file.checksum(check);
        modified=true;
      };
      if(modified) file.write_attr();
    };
    if(!source.have_locations()) {
      odlog(ERROR)<<"No locations for source found: "<<source<<std::endl;
      continue;
    };
    for(;;) {  // locations
      if(downloaded) break;
      if(!source.have_location()) {
        odlog(INFO)<<"Out of tries: "<<source<<std::endl;
        break;
      };
      DataHandle source_h(&source);
//      source_h.secure(force_secure);
//      source_h.passive(force_passive);
//      source_h.additional_checks(do_checks);
      int bufnum = 1;
      long int bufsize = (1024*1024);
      {
      DataHandle::analyze_t hint;
      source_h.analyze(hint);
      if(hint.bufnum>bufnum) bufnum=hint.bufnum;
      if(hint.bufsize>bufsize) bufsize=hint.bufsize;
      };
      bool seekable = source_h.out_of_order();
      if(bufnum>10) bufnum=10;
      if(bufsize>(10*1024*1024)) bufsize=10*1024*1024;
      bufnum=bufnum*2;
      odlog(DEBUG)<<"Creating buffer: "<<bufsize<<" x "<<bufnum<<std::endl;
//      CheckSum crc;
      DataBufferPar buffer(NULL,bufsize,bufnum);
      if(!buffer) {
        odlog(INFO)<<"Buffer creation failed !"<<std::endl;
        source.next_location();
        continue; // there is still hope to create buffer with different sizes 
      };
      if(timeout > 0) {
        buffer.speed.set_min_speed(1024*1024/timeout+1,600);
        buffer.speed.set_min_average_speed(1024*1024/timeout+1);
        if(file.size_available()) {
          buffer.speed.set_max_inactivity_time(timeout * file.size()
                                               /1024/1024 + 1);
        };
      };
      buffer.speed.verbose(false);
      for(;;) { // ranges
        // choose range
        SEFileRange range;
        int n=file.free_ranges(1,&range);
        if(n==0) {
          odlog(ERROR)<<"No ranges to download. File must be complete."<<std::endl;
          downloaded=true;
          break;
        };
        // TODO: decrease range if too big
        odlog(ERROR)<<"Range to download: "<<range.start<<" - "<<range.end<<std::endl;
        if(file.open(false) != 0) { 
          odlog(ERROR)<<"Failed to open file."<<std::endl;
          lock.unblock();
          return -1;
        };
        source_h.range(range.start,range.end);
        // download
        if(!source_h.start_reading(buffer)) {
          file.close(false);
          odlog(ERROR)<<"Failed to start reading from source"<<std::endl;
          source.next_location();
          break;
        };
        unsigned long long int max_size = 0;
        for(;;) { // download
          /* 1. claim buffer */
          int h;
          unsigned int l;
          unsigned long long int p;
          if(!buffer.for_write(h,l,p,true)) {
            /* failed to get buffer - must be error or request to exit */
            if(!buffer.eof_read()) buffer.error_write(true);
            buffer.eof_write(true);
            break;
          };
          if(buffer.error()) {
            buffer.is_written(h);
            buffer.eof_write(true);
            break;
          };
          /* 2. write */
          if(file.write(buffer[h],p,l) != l) {
            odlog(ERROR)<<"Failed to write to file."<<std::endl;
            buffer.is_written(h);
            buffer.error_write(true);
            buffer.eof_write(true);
            source_h.stop_reading();
            file.close(false);
            lock.unblock();
            return -1;
          };
          buffer.is_written(h);
          if((p+l) > max_size) max_size=p+l;
        }; // download
        source_h.stop_reading();
        if(buffer.error()) {
          file.close(false);
          odlog(ERROR)<<"Error while downloading"<<std::endl;
          source.next_location();
          break;
        };
        if(!file.size_available()) {
          // That is not good and should not happen. But if it happened
          // try to fix that.
          if(source.meta_size_available()) {
            // 1. Try to use information obtained during resolving.
            file.size(source.meta_size()); file.write_attr();
            odlog(INFO)<<"Using size obtained from remote server: "<<file.size()<<" - "<<source.meta_size()<<std::endl;
          } else if(max_size <= range.end) { 
            // 2. Range was downloaded without error. If it is not
            // complete, that means file finished.
            file.size(max_size); file.write_attr();
            odlog(INFO)<<"Using size obtained from incomplete range: "<<file.size()<<" - "<<max_size<<std::endl;
          };
        };
        file.close(false);
      }; // ranges
    }; // location
  }; // sources
  lock.unblock();
  if(!downloaded) return -1;
  return 0;
}

// ---------------------------------------------

// Do replication
void SEReplicator_Thread::func(void) {
  while(true) {
    files.block();
    // go through all storages
    for(SEFilesList::iterator fs = files.begin();fs!=files.end();++fs) {
      files.unblock();
      if(*fs) (*fs)->Replicate();
      files.block();
    };
    files.unblock();
    idle(10*60*1000); // 10 minutes
  };
}

