#include "../../std.h"
#include <arpa/inet.h>
#include <sys/socket.h>
#include <openssl/err.h>

#include <globus_common.h>
#include <globus_gsi_credential.h>
#include <globus_gsi_system_config.h>

#include "../../misc/log_time.h"
#include "../../misc/inttostring.h"
#include <arc/globuserrorutils.h>
#include "../../config/environment.h"

#include "../httpsd.h"

#ifndef BUFLEN
#define BUFLEN 1024
#endif

static int curr_connections = 0;
static int pre_connections = 2;
static int max_connections = 100;
static int curr_pid = 1;

extern unsigned int conections_processed;

static char* service_not_found_str = "Service not found\r\n";

#ifndef HAVE_STRERROR_R
static char * strerror_r (int errnum, char * buf, size_t buflen) {
  char * estring = strerror (errnum);
  strncpy (buf, estring, buflen);
  return buf;
}
#endif


void HTTP_Connector::loop_in_thread(void) {
  globus_thread_t handle;
  int r;
  // TODO: replace with ordinary pthread
  r=globus_thread_create(GLOBUS_NULL,GLOBUS_NULL,&loop_thread,(void*)this);
  if(r != GLOBUS_SUCCESS) {
    odilog(ERROR,pid)<<"Failed to create thread"<<std::endl;
    delete this;
    conections_processed++;
  };
}

void* HTTP_Connector::loop_thread(void* arg) {
  globus_thread_yield();
  HTTP_Connector* it = (HTTP_Connector*)arg;
  it->loop_start();
  // Assuming user's identity was already set here
  it->user.add_vo(it->vos); // match user against configured VOs
  it->loop();
  it->loop_end();
  delete it;
  conections_processed++;
  return NULL;
}

void HTTP_Connector::loop_start(void) {
}

void HTTP_Connector::loop_end(void) {
}

void HTTP_Connector::loop(void) {
  char buf[1024];
  int keep_alive = 1;
  for(;;) {
    /* read first line of request header */
    odilog(DEBUG,pid)<<"Waiting for request"<<std::endl;
    size_t l = readline(buf,sizeof(buf)-1);
    if(l == 0) break;
    odilog(DEBUG,pid)<<"Request: "<<buf<<std::endl;
    // parse request
    char* http_URI;
    char* http_version;
    char* http_method = buf;
    char* t = buf;
    t=strchr(t,' ');
    if(t == NULL) {
      odilog(ERROR,pid)<<"Broken HTTP request (no URI found) - "<<buf<<std::endl; break;
    };
    (*t)=0; t++; for(;(*t)==' ';t++);
    http_URI=t;
    t=strchr(t,' ');
    if(t == NULL) {
      odilog(ERROR,pid)<<"Broken HTTP request (no version found) - "<<http_URI<<std::endl; break;
    };
    (*t)=0; t++; for(;(*t)==' ';t++);
    if(strncmp("HTTP/",t,5) != 0) {
      odilog(ERROR,pid)<<"Unknown protocol: "<<t<<std::endl; break;
    };
    http_version=t+5;
    if(keep_alive) {
      if(strcmp(http_version,"1.0") == 0) {
        keep_alive=1;
      } else {
        keep_alive=2;
      };
    };
    HTTP_Error err = HTTP_OK;
    // adjust URI
    std::string http_url = http_URI;
    if(strncasecmp("https://",http_url.c_str(),8) &&
       strncasecmp("httpg://",http_url.c_str(),8) &&
       strncasecmp("http://",http_url.c_str(),7)) {
      // Path in request, convert to absolute
      if(http_url[0] != '/') http_url="/"+http_url;
      // Convert to full url
      http_url=base_url+http_url;
    };
    // if(http_url[http_url.length()-1] == '/') http_url.resize(http_url.length()-1);
    odilog(DEBUG,pid)<<"Request on "<<http_url<<std::endl;
    // make sure request is for proper url
    if(strncasecmp(http_url.c_str(),base_url.c_str(),base_url.length()) != 0) {
      odilog(ERROR,pid)<<"Improper URL in request: "<<http_url<<" != "<<base_url<<std::endl;
      err=HTTP_NOT_FOUND;
    } else {
      // obtain service to use 
      HTTP_Service* service = services.get(*this,http_url.c_str());
      if(service == NULL) {
        odilog(ERROR,pid)<<"Corresponding service not found: "<<http_url<<std::endl;
        err=HTTP_NOT_FOUND;
      } else {
        if(strcasecmp(http_method,"GET") == 0) {
          odilog(DEBUG,pid)<<"Calling GET"<<std::endl;
          err=service->get(http_url.c_str(),keep_alive);
        } else if(strcasecmp(http_method,"PUT") == 0) {
          odilog(DEBUG,pid)<<"Calling PUT"<<std::endl;
          err=service->put(http_url.c_str(),keep_alive);
        } else if(strcasecmp(http_method,"POST") == 0) {
          odilog(DEBUG,pid)<<"Calling POST"<<std::endl;
          err=service->post(http_url.c_str(),keep_alive);
        } else {
          odilog(DEBUG,pid)<<"Unsupported method: "<<http_method<<std::endl;
          err=HTTP_NOT_IMPLEMENTED;
        };
      };
    };
    if((err==HTTP_NOT_IMPLEMENTED) || (err==HTTP_NOT_FOUND)) {
      odilog(INFO,pid)<<"Method is not implemented in requested service"<<std::endl;
      if(skip_request(keep_alive) != 0) break;
    };
    if((err != HTTP_OK) && (err != HTTP_FAILURE)) { /* send error response */
      odilog(INFO,pid)<<"Sending error response: "<<err<<std::endl;
      if(send_error_response(keep_alive,err,NULL,NULL) != 0) {
        keep_alive=0; break;
      };
    };
    if(keep_alive == 0) {
      odilog(DEBUG,pid)<<"Can't keep connection alive"<<std::endl;
    } else {
      odilog(DEBUG,pid)<<"Keeping connection alive"<<std::endl;
    };
    if(keep_alive == 0) break;
  };
  odilog(INFO,pid)<<"Closing connection"<<std::endl;
  // close connection
  services.close(*this);
}

HTTP_Connector::HTTP_Connector(const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths_,std::list<AuthVO>& vos_):services(serv),auths(auths_),vos(vos_) {
  initialized=false;
  if(url) base_url=url;
  return;
}

HTTP_Connector::~HTTP_Connector(void) {
}

size_t HTTP_Connector::readline(char* buf, size_t l) {
  char c = 0;
  unsigned int ll = 0;
  for(;;) {
    size_t ll_ = read(&c,1);
    if(ll_ == 0) break; // unexpected eof
    if(ll<l) buf[ll]=c; ll++;
    if(c == 0x0a) break; // eol
  };
  // eol found
  if(ll > l) ll=l;
  l=ll;
  for(;l>0;) {
    l--;
    if((buf[l] != 0x0a) && (buf[l] != 0x0d)) break;
    buf[l]=0;
  };
  return ll;
}

size_t HTTP_Connector::read(char* buf, size_t l) {
  return 0;
}

int HTTP_Connector::write(const char* buf, size_t l) {
  return -1;
}

int HTTP_Connector::skip_request(int &keep_alive) {
  bool length_passed=false;
  unsigned long long int length;
  char buf[1024];
  // skip header
  for (;;) {
    if(this->readline(buf,1023) == 0) return HTTP_ERROR; /* connection broken */
    if(buf[0] == 0) break; /* empty line - end of header */
    char* token = buf; for(;*token;token++) if(isspace(*token)) break;
    int l = token-buf;
    for(;*token;token++) if(!isspace(*token)) break;
    if(strncasecmp("Connection:",buf,l) == 0) {
      if(strcasecmp("close",token) == 0) { keep_alive=0; return HTTP_OK; }
      else if(strcasecmp("keep-alive",token) == 0) { keep_alive=2; };
    } else if(strncasecmp("Content-Length:",buf,l) == 0) {
      char *e;
      length=strtoull(token,&e,10);
      if((*e) == 0) length_passed=true;
      if(length > 1024) { keep_alive=0; return HTTP_OK; };        
    };
  };
  // skip body
  if(!length_passed) { keep_alive=0; return HTTP_OK; };
  for (;;) {
    size_t l = 1024;
    if(l>length) l=length;
    l=this->read(buf,0);
    if(l==0) { keep_alive=0; return HTTP_OK; };
    length-=l; 
    if(length == 0) break;
  };
  return HTTP_OK;
}

int HTTP_Connector::send_error_response(int keep_alive,int code,char* type,char* content) {
  unsigned long long int size = 0;
  if(content) {
    size=strlen(content);
    if(send_response_header(keep_alive,code,type,size) != 0) return -1;
    return this->write(content,size);
  } else {
    std::string fname = nordugrid_loc+"/share/error"+inttostring(code)+".html";
    stat_file(fname.c_str(),size);
    if(send_response_header(keep_alive,code,"text/html",size) != 0) return -1;
    if(size) return send_file(fname.c_str());
  };
  return 0;
}

int HTTP_Connector::send_response_header(int keep_alive,int code,char* type,int size) {
  std::string header = "HTTP/1.1 "+inttostring(code);
  if(code == 200) { header+=" OK\r\n"; } else { header+=" NOREASON\r\n"; };
  if(keep_alive && (size>=0)) {
    header+="Connection: Keep-Alive\r\n";
  } else {
    header+="Connection: close\r\n";
  };
  if(size>=0) header+="Content-Length: "+inttostring(size)+"\r\n";
  if(type) { header+="Content-Type: "; header+=type; header+="\r\n"; };
  header+="\r\n";
  odilog(ERROR,pid)<<"Sending response:\n"<<header<<std::endl;
  return this->write(header.c_str(),header.length());
}

int stat_file(const char* fname,unsigned long long int &size) {
  struct stat st;
  if(stat(fname,&st) != 0) return -1;
  if(!S_ISREG(st.st_mode)) return -1;
  size=st.st_size;
  return 0;
}

int HTTP_Connector::send_file(const char* fname) {
  int h = ::open(fname,O_RDONLY);
  if(h == -1) return 1;
  char buf[1024];
  for(;;) {
    int l = ::read(h,buf,sizeof(buf));
    if(l == 0) break;
    if(l == -1) { ::close(h); return 1; };
    if(this->write(buf,l)) { ::close(h); return 1; };
  };
  ::close(h);
  return 0;
}

int HTTP_Connector::skip_header(int &keep_alive) {
/* TODO - control keep_alive */
  char buf[1024], *s;
  for (;;) {
    if(this->readline(buf,1023) == 0) return 1; /* connection broken */
    if(buf[0] == 0) break; /* empty line */
  };
  return 0;
}

// -------------------------------------------------------
// ------------------------- Connectors ------------------
// -------------------------------------------------------

// --------------------------- No wrapping ---------------

HTTP_Plain_Connector::HTTP_Plain_Connector(int socket,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths_,std::list<AuthVO>& vos_):HTTP_Connector(url,serv,auths_,vos_),s(socket) {
  if(s != -1) initialized=true;
  return;
}

HTTP_Plain_Connector::~HTTP_Plain_Connector(void) {
  if(s != -1) close(s);
}

int HTTP_Plain_Connector::write(const char* buf, size_t l) {
  size_t n = 0;
  for(;n<l;) {
    ssize_t ll = 0;
    globus_result_t res;
    if((ll=::write(s,buf+n,l-n)) == -1) {
      char* err;
      char errbuf[BUFLEN];
#ifndef _AIX
      err=strerror_r(errno,errbuf,sizeof(errbuf));
#else
      errbuf[0]=0; err=errbuf;
      strerror_r(errno,errbuf,sizeof(errbuf));
#endif
      odilog(ERROR,pid)<<"Failed to write to network: "<<(err?err:"")<<std::endl;
      return -1;
    };
    n+=ll;
  };
  return 0;
}

size_t HTTP_Plain_Connector::read(char* buf, size_t l) {
  ssize_t ll = ::read(s,buf,l);
  if(ll == -1) {
    char* err;
    char errbuf[BUFLEN];
#ifndef _AIX
    err=strerror_r(errno,errbuf,sizeof(errbuf));
#else
    errbuf[0]=0; err=errbuf;
    strerror_r(errno,errbuf,sizeof(errbuf));
#endif
    odilog(ERROR,pid)<<"Failed to read from network: "<<(err?err:"")<<std::endl;
    return 0;
  };
  return ll;
}

void HTTP_Plain_Connector::loop_start(void) {
  odilog(ERROR,pid)<<"Session started"<<std::endl;
}

// --------------------------- GSI ifrom Globus -----------------------

#if GLOBUS_IO_VERSION<5
extern "C" {
extern void globus_i_io_handle_destroy(globus_io_handle_t * handle);
};
#else
void globus_i_io_handle_destroy(globus_io_handle_t * handle) {
};
#endif

HTTP_Globus_Connector::HTTP_Globus_Connector(globus_io_handle_t *h_,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths_,std::list<AuthVO>& vos_):HTTP_Connector(url,serv,auths_,vos_),h(h_),r_start(0),r_end(0) {
  initialized=true;
  return;
}

HTTP_Globus_Connector::~HTTP_Globus_Connector(void) {
  // remove own entry from global list
  if(h) {
    globus_result_t res;
    res=globus_io_close(h);
    if(res != GLOBUS_SUCCESS) odilog(ERROR,pid)<<"Failed to close connection: "<<GlobusResult(res)<<std::endl;
    globus_i_io_handle_destroy(h);
    free(h);
  };
}

void HTTP_Globus_Connector::loop_start(void) {
  odilog(ERROR,pid)<<"Session started"<<std::endl;
  {
    char abuf[1024]; abuf[sizeof(abuf)-1]=0;
    int host[4] = {0,0,0,0};
    unsigned short port = 0;
    if(globus_io_tcp_get_local_address(h,host,&port) == GLOBUS_SUCCESS) {
      struct hostent he;
      struct hostent* he_p;
      snprintf(abuf,sizeof(abuf)-1,"%u.%u.%u.%u",
                              host[0],host[1],host[2],host[3]);
      struct in_addr a;
      if(inet_aton(abuf,&a) != 0) {
        int h_errnop;
        char buf[1024];
        he_p=globus_libc_gethostbyaddr_r((char*)&a,strlen(abuf),AF_INET,
                                             &he,buf,sizeof(buf),&h_errnop);
        if(he_p) {
          if(strcmp(he_p->h_name,"localhost") == 0) {
            abuf[sizeof(abuf)-1]=0;
            if(globus_libc_gethostname(abuf,sizeof(abuf)-1) != 0) {
              strcpy(abuf,"localhost");
            };
          };
        };
      };
    };
    odilog(INFO,pid)<<"My name is: "<<abuf<<" ("<<host[0]<<"."<<host[1]<<"."<<host[2]<<"."<<host[3]<<":"<<port<<")"<<std::endl;
    abuf[0]=0;
    if(globus_io_tcp_get_remote_address(h,host,&port) == GLOBUS_SUCCESS) {
      snprintf(abuf,sizeof(abuf)-1,"%u.%u.%u.%u:%u",
                     host[0],host[1],host[2],host[3],(unsigned int)port);
    };
    odilog(INFO,pid)<<"Connection from: "<<abuf<<std::endl;
  };
}

void HTTP_Globus_Connector::identity(globus_io_handle_t* handle,const char* subject) {
  // modify entry in global list to reflect identity of client
#if GLOBUS_IO_VERSION<5
  gss_cred_id_t cred = handle->delegated_credential;
  gss_ctx_id_t  ctx  = handle->context;
#else
  gss_cred_id_t cred;
  gss_ctx_id_t ctx;
  if(globus_io_tcp_get_delegated_credential(handle,&cred) != GLOBUS_SUCCESS) {
    cred=GSS_C_NO_CREDENTIAL;
  };
  if(globus_io_tcp_get_security_context(handle,&ctx)  != GLOBUS_SUCCESS) {
    ctx=GSS_C_NO_CONTEXT;
  };
#endif
  user.set(subject,ctx,cred);
  odilog(ERROR,pid)<<"Subject is "<<user.DN()<<std::endl;
  if(user.is_proxy() && user.proxy() && user.proxy()[0]) {
    odilog(ERROR,pid)<<"Delegated credentials in "<<user.proxy()<<std::endl;
  } else {
    odilog(ERROR,pid)<<"No delegated credentials"<<std::endl;
  };
}

// Hide EOF error, this is not an error. 
#define print_globus_result_not_eof(o,prefix,res) {                 \
  globus_object_t* err = globus_error_get(res);                     \
  if(err == NULL) { o<<prefix<<"<undefined>"<<std::endl; }               \
  else {                                                            \
    char* tmp = globus_object_printable_to_string(err);             \
    if(tmp == NULL) {                                               \
      o<<prefix<<"<undefined>"<<std::endl; globus_object_free(err);      \
    } else {                                                        \
      if(strstr(tmp,"end-of-file") == NULL) o<<prefix<<tmp<<std::endl;   \
      free(tmp);                                                    \
    };                                                              \
    globus_object_free(err);                                        \
  };                                                                \
}

int HTTP_Globus_Connector::write(const char* buf, size_t l) {
  size_t n = 0;
  for(;n<l;) {
    globus_size_t ll = 0;
    globus_result_t res;
    if((res=globus_io_write(h,(globus_byte_t*)buf+n,l-n,&ll)) != GLOBUS_SUCCESS) {
      odilog(ERROR,pid)<<"Failed to write to network: "<<GlobusResult(res)<<std::endl;
      return -1;
    };
    n+=ll;
  };
  return 0;
}

size_t HTTP_Globus_Connector::read(char* buf, size_t l) {
  globus_result_t res;
  globus_size_t ll = 0;
  if(r_start<r_end) {
    ll=r_end-r_start;
    if(ll<l) { memcpy(buf,r_buf+r_start,ll); r_end=0; r_start=0; }
    else { memcpy(buf,r_buf+r_start,l); r_start+=l; return l; };
    l-=ll; buf+=ll;
  };
  globus_size_t ll_;
  if(l > sizeof(r_buf)) {
    if((res=globus_io_read(h,(globus_byte_t*)buf,l,ll?0:1,&ll_))
                                                          !=GLOBUS_SUCCESS) {
      if(ll) return ll;
      print_globus_result_not_eof(odilog(ERROR,pid),"Failed to read from network: ",res);
      return 0;
    };
    return (ll+ll_);
  };
  if((res=globus_io_read(h,(globus_byte_t*)r_buf,sizeof(r_buf),ll?0:1,&ll_))
                                                          !=GLOBUS_SUCCESS) {
    if(ll) return ll;
    print_globus_result_not_eof(odilog(ERROR,pid),"Failed to read from network: ",res);
    return 0;
  };
  r_start=0; r_end=ll_; if(ll_>l) ll_=l;
  memcpy(buf,r_buf,ll_); r_start+=ll_;
  return (ll+ll_);
}

size_t HTTP_Globus_Connector::readline(char* buf, size_t l) {
  char c = 0;
  unsigned int ll = 0;
  for(;;) {
    if(ll > 65536) return 0; // must be veeeeery strange line
    if(r_start>=r_end) {
      globus_size_t ll_;
      globus_result_t res;  
      if((res=globus_io_read(h,(globus_byte_t*)r_buf,sizeof(r_buf),1,&ll_))
                                                          !=GLOBUS_SUCCESS) {
        print_globus_result_not_eof(odilog(ERROR,pid),"Failed to read from network: ",res);
        return 0; // unexpected eof
      };
      if(ll_ == 0) return 0; // unexpected eof
      r_start=0; r_end=ll_;
    };
    char c = r_buf[r_start];
    if(ll<l) buf[ll]=c;
    ++r_start; ++ll;
    if(c == 0x0a) break; // eol
  };
  // eol found
  if(ll > l) ll=l;
  if(ll == 0) return 0;
  l=ll;
  for(;l>0;) {
    l--;
    if((buf[l] != 0x0a) && (buf[l] != 0x0d)) break;
    buf[l]=0;
  };
//buf[ll]=0;
  return ll;
}

// --------------------------- SSL -----------------------

static int verify_callback(int ok, X509_STORE_CTX *ctx) {
  if(ok) return ok;
  /*
  char buf[256]; buf[0]=0;
  X509 *cert=X509_STORE_CTX_get_current_cert(ctx);
  STACK_OF(X509) *chain=X509_STORE_CTX_get_chain(ctx);
  int err=X509_STORE_CTX_get_error(ctx);
  X509_NAME_oneline(X509_get_subject_name(cert),buf,sizeof(buf));
  olog<<"SSL verify error for "<<buf<<" : "<<err<<" "<<
                      X509_verify_cert_error_string(err)<<std::endl;
  */
  /*
  if((err != X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) &&
     (err != X509_V_ERR_CERT_UNTRUSTED) &&
     (err != X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) &&
     (err != X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION)) {
    return ok;
  };
  */
  // Allow any certificate here - later Globus routines will be used 
  return 1;
}

HTTP_SSL_Connector::HTTP_SSL_Connector(SSL_CTX *ctx,int socket,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos):HTTP_Connector(url,serv,auths,vos),con(NULL),s(socket) {
  if(s == -1) return;
  // New connection
  con = SSL_new(ctx);
  SSL_clear(con);
  BIO* sbio=BIO_new_socket(s,BIO_NOCLOSE);
  SSL_set_bio(con,sbio,sbio);
  SSL_set_accept_state(con);
  initialized=true;
}

HTTP_SSL_Connector::~HTTP_SSL_Connector(void) {
  if(con != NULL) {
    SSL_set_shutdown(con,SSL_SENT_SHUTDOWN|SSL_RECEIVED_SHUTDOWN);
    close(s);
    SSL_free(con);
  };
}

void HTTP_SSL_Connector::loop_start(void) {
  if(con == NULL) return;
  if(!SSL_is_init_finished(con)) {
    int i;
    i=SSL_accept(con);
    if(i != 1) {
      /* if (BIO_sock_should_retry(i)) */
      close(s); SSL_free(con); con=NULL;
      return; // Errors are not handled yet
    };
  };
  odilog(ERROR,pid)<<"Using Globus methods to verify credentials"<<std::endl;
  // Use Globus routines to verify certificates in a Globus way
  globus_result_t res;
  globus_gsi_cred_handle_t handle;
  globus_gsi_callback_data_t data;
  char* cert_dir;
  res=GLOBUS_GSI_SYSCONFIG_GET_CERT_DIR(&cert_dir);
  if(res != GLOBUS_SUCCESS) {
    odilog(ERROR,pid)<<"GLOBUS_GSI_SYSCONFIG_GET_CERT_DIR failed: "<<GlobusResult(res)<<std::endl;
    close(s); SSL_free(con); con=NULL;
    return;
  };
  res=globus_gsi_cred_handle_init(&handle,GLOBUS_NULL);
  if(res != GLOBUS_SUCCESS) {
    odilog(ERROR,pid)<<"globus_gsi_cred_handle_init failed: "<<GlobusResult(res)<<std::endl;
    globus_free(cert_dir);
    close(s); SSL_free(con); con=NULL;
    return;
  };
  STACK_OF(X509) *new_chain = sk_X509_new_null();
  if(!new_chain) {
    globus_gsi_cred_handle_destroy(handle);
    globus_free(cert_dir);
    close(s); SSL_free(con); con=NULL;
    return;
  };
  int i = 0;
  bool user_set = false;
  X509 *peer_cert = SSL_get_peer_certificate(con);
  X509 *cert = peer_cert;
  STACK_OF(X509) *chain = SSL_get_peer_cert_chain(con);
  if(chain == NULL) return;
  // Go through chain till user's certificate
  int chain_index;
  for(chain_index = 0;;++chain_index) {
    if(cert) {
      X509 *tmp_cert = X509_dup(cert);
      if(tmp_cert) sk_X509_insert(new_chain,tmp_cert,i++);
      X509_NAME *name = X509_get_subject_name(cert);
      char buf[256]; buf[0]=0;
      if(name) {
        X509_NAME_oneline(name,buf,sizeof(buf));
        odilog(INFO,pid)<<"Name: "<<buf<<std::endl;
      };
    };
    if(chain_index >= sk_X509_num(chain)) break;
    cert = sk_X509_value(chain,chain_index);
  };
  X509_free(peer_cert);
  user.set(NULL,new_chain);
  res=globus_gsi_cred_set_cert_chain(handle,new_chain);
  if(res != GLOBUS_SUCCESS) {
    odilog(ERROR,pid)<<"globus_gsi_cred_set_cert_chain failed: "<<GlobusResult(res)<<std::endl;
    globus_gsi_cred_handle_destroy(handle);
    globus_free(cert_dir);
    sk_X509_pop_free(new_chain,X509_free);
    close(s); SSL_free(con); con=NULL;
    return;
  };
  globus_gsi_callback_data_init(&data);
  globus_gsi_callback_set_cert_dir(data,cert_dir);
  res=globus_gsi_cred_verify_cert_chain(handle,data);
  if(res != GLOBUS_SUCCESS) {
    odilog(ERROR,pid)<<"globus_gsi_cred_verify_cert_chain failed: "<<GlobusResult(res)<<std::endl;
    globus_gsi_callback_data_destroy(data);
    globus_gsi_cred_handle_destroy(handle);
    globus_free(cert_dir);
    sk_X509_pop_free(new_chain,X509_free);
    close(s); SSL_free(con); con=NULL;
    return;
  };
  globus_gsi_callback_data_destroy(data);
  globus_gsi_cred_handle_destroy(handle);
  globus_free(cert_dir);
  sk_X509_pop_free(new_chain,X509_free);
}

size_t HTTP_SSL_Connector::read(char* buf, size_t l) {
  if(con == NULL) return 0;
  for(;;) {
    // SSL_pending(con);
    int i;
    i=SSL_read(con,(char *)buf,l);
    switch(SSL_get_error(con,i)) {
      case SSL_ERROR_NONE: return i;
      case SSL_ERROR_WANT_WRITE:
      case SSL_ERROR_WANT_READ:
      case SSL_ERROR_WANT_X509_LOOKUP: {
        // wait for activity on socket
        fd_set readfds;
        FD_ZERO(&readfds);
        FD_SET(s,&readfds);
        i=select(s+1,&readfds,NULL,NULL,NULL);
        if(i<=0) return 0; // Error not handled yet
      }; break;
      case SSL_ERROR_SYSCALL:
      case SSL_ERROR_SSL: return 0; // Error not handled yet
      case SSL_ERROR_ZERO_RETURN: return 0; // Error not handled yet
    };
  };
  return 0;
}

int HTTP_SSL_Connector::write(const char* buf, size_t l) {
  if(con == NULL) return -1;
  int ll = 0;
  for(;ll<l;) {
    int i = SSL_write(con,buf+ll,l-ll);
    switch(SSL_get_error(con,i)) {
      case SSL_ERROR_NONE: {
        ll+=i;
      }; break;
      // TODO: timeout
      case SSL_ERROR_WANT_WRITE:
      case SSL_ERROR_WANT_READ:
      case SSL_ERROR_WANT_X509_LOOKUP: break;
      case SSL_ERROR_SYSCALL:
      case SSL_ERROR_SSL: return -1;
      case SSL_ERROR_ZERO_RETURN: return -1;
    };
  };
  return 0;
}

// --------------------------- GSI from CERN ---------------

HTTP_GSSAPI_Connector::HTTP_GSSAPI_Connector(int socket,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths_,std::list<AuthVO>& vos_):HTTP_Plain_Connector(socket,url,serv,auths_,vos_) {
  if(initialized) {
    initialized=false;
    OM_uint32 major_status = 0;
    OM_uint32 minor_status = 0;
    server_credentials=GSS_C_NO_CREDENTIAL;
    context=GSS_C_NO_CONTEXT;
    data_tok.length=0; data_tok.value=NULL;
    data_tok_offset=0;


    major_status = gss_acquire_cred(&minor_status,
                                    GSS_C_NO_NAME,
                                    0,
                                    GSS_C_NULL_OID_SET,
                                    GSS_C_ACCEPT,
                                    &server_credentials,
                                    NULL,
                                    NULL);
    if (major_status != GSS_S_COMPLETE) {
      odlog(ERROR)<<"Failed to acquire server credentials"<<std::endl;
      return;
    };
    initialized=true;
  };
  return;
}

void HTTP_GSSAPI_Connector::loop_start(void) {
  OM_uint32 major_status = 0;
  OM_uint32 minor_status = 0;
  OM_uint32 acc_sec_min_stat;
  OM_uint32 ret_flags =  0;
  OM_uint32 time_req;
  gss_buffer_desc recv_tok;
  gss_buffer_desc send_tok;
  gss_buffer_desc name;
  gss_channel_bindings_t input_chan_bindings = GSS_C_NO_CHANNEL_BINDINGS;
  gss_name_t client = GSS_C_NO_NAME;
  gss_OID doid = GSS_C_NO_OID;
  gss_cred_id_t delegated_cred_handle = GSS_C_NO_CREDENTIAL;
  int l;

  do {
    l=read_SSL_token(&(recv_tok.value));
    if(l == -1) {
      initialized=false;
      return;
    };
    recv_tok.length=l;
    major_status = gss_accept_sec_context(&acc_sec_min_stat,
                                          &context,
                                          server_credentials,
                                          &recv_tok,
                                          input_chan_bindings,
                                          &client,
                                          &doid,
                                          &send_tok,
                                          &ret_flags,
                                          &time_req,
                                          &delegated_cred_handle);
    free(recv_tok.value);
    if((major_status!=GSS_S_COMPLETE) &&
       (major_status!=GSS_S_CONTINUE_NEEDED)) {
      odlog(ERROR)<<"Failed to authenticate"<<std::endl;
      initialized=false;
      return;
    };
    if(send_tok.length != 0) {
      if(HTTP_Plain_Connector::write((char*)(send_tok.value),send_tok.length) == -1) {
        gss_release_buffer(&minor_status,&send_tok);
        if(context != GSS_C_NO_CONTEXT)
          gss_delete_sec_context(&minor_status,&context,GSS_C_NO_BUFFER);
        context=GSS_C_NO_CONTEXT;
        initialized=false;
        return;
      };
    };
  } while(major_status!=GSS_S_COMPLETE);
  major_status=gss_display_name(&minor_status,client,&name,(gss_OID *)NULL);
  if(major_status != GSS_S_COMPLETE) name.value=NULL;
  user.set((char*)(name.value),context,delegated_cred_handle);
  gss_release_name(&minor_status,&client);
  gss_release_buffer(&minor_status,&name);
  odilog(ERROR,pid)<<"Subject is "<<user.DN()<<std::endl;
  if(user.is_proxy() && user.proxy() && user.proxy()[0]) {
    odilog(ERROR,pid)<<"Delegated credentials in "<<user.proxy()<<std::endl;
  } else {
    odilog(ERROR,pid)<<"No delegated credentials"<<std::endl;
  };
  odilog(ERROR,pid)<<"Session started"<<std::endl;
}

int HTTP_GSSAPI_Connector::read_SSL_token(void** val) {
  unsigned char header[5]; // SSL V3 header
  (*val)=NULL;
  size_t l = HTTP_Plain_Connector::read((char*)header,sizeof(header));
  if(l <= 0) return -1;
  if(header[0] == (unsigned char)0x80) {
    /* SSL V2 - 2 byte header */
    l=((unsigned int)(header[1]))-3;
  } else if((header[0] >= 20) &&
            (header[0] <= 26) &&
            (header[1] == 3) &&
            (header[2] == 0 || header[2] == 1)) {
         /* SSL V3 */
    l=(((unsigned int)(header[3])) << 8) | ((unsigned int)(header[4]));
  } else {
    odlog(ERROR)<<"Urecognized token received"<<std::endl;
    return -1;
  };
  unsigned char* data = (unsigned char*)malloc(l+5);
  if(data == NULL) return -1;
  memcpy(data,header,5);
  if(l) {
    size_t ll = HTTP_Plain_Connector::read((char*)(data+5),l);
    if(ll <= 0) {
      free(data);
      return -1;
    };
  };
  (*val)=data;
  return (l+5);
}

HTTP_GSSAPI_Connector::~HTTP_GSSAPI_Connector(void) {
  OM_uint32 minor_status;
  if(context != GSS_C_NO_CONTEXT)
    gss_delete_sec_context(&minor_status,&context,GSS_C_NO_BUFFER);
  context=GSS_C_NO_CONTEXT;
  if(data_tok.length != 0)
    gss_release_buffer(&minor_status,&data_tok);
  data_tok.length=0; data_tok.value=NULL; data_tok_offset=0;
}

int HTTP_GSSAPI_Connector::write(const char* buf, size_t l) {
  OM_uint32 major_status;
  OM_uint32 minor_status;
  gss_buffer_desc send_tok;
  gss_buffer_desc data_tok;
  int conf_state;
  data_tok.length=l;
  data_tok.value=(void*)buf;
  major_status = gss_wrap(&minor_status,
                          context,
                          0,
                          GSS_C_QOP_DEFAULT,
                          &data_tok,
                          &conf_state,
                          &send_tok);
  if(major_status != GSS_S_COMPLETE) {
    odlog(ERROR)<<"Failed wraping GSI token"<<std::endl;
    return 0;
  };
  int r = HTTP_Plain_Connector::write((char*)(send_tok.value),send_tok.length);
  gss_release_buffer(&minor_status,&send_tok);
  return r;
}

size_t HTTP_GSSAPI_Connector::read(char* buf, size_t l) {
  OM_uint32 major_status;
  OM_uint32 minor_status;
  size_t result = 0;
  if(data_tok.length != 0) {
    result=data_tok.length-data_tok_offset;
    if(result > l) result=l;
    memcpy(buf,(char*)(data_tok.value)+data_tok_offset,result);
    data_tok_offset+=result; 
    if(data_tok_offset < data_tok.length) return result;
    gss_release_buffer(&minor_status,&data_tok);
    data_tok.length=0; data_tok.value=NULL; data_tok_offset=0;
    //l-=result; buf+=result;
    return result;
  };
  gss_buffer_desc recv_tok;
  int ll = read_SSL_token(&(recv_tok.value));
  if(ll == -1) return 0;
  recv_tok.length=ll;
  major_status = gss_unwrap(&minor_status,
                            context,
                            &recv_tok,
                            &data_tok,
                            NULL,
                            NULL);
  free(recv_tok.value);
  if(major_status != GSS_S_COMPLETE) {
    odlog(ERROR)<<"Failed unwraping GSI token"<<std::endl;
    return 0;
  };
  ll=data_tok.length; if(ll > l) ll=l;
  memcpy(buf,data_tok.value,ll); result+=ll;
  if(ll < data_tok.length) {
    data_tok_offset=ll;
  } else {
    gss_release_buffer(&minor_status,&data_tok);
    data_tok.length=0; data_tok.value=NULL; data_tok_offset=0;
  };
  return result;
}


// -------------------------------------------------------
// ------------------------- Listeners -------------------
// -------------------------------------------------------

// --------------------------- GSI from Globus -----------------------

void HTTP_Globus_Listener::listen_cb(void *callback_arg,globus_io_handle_t *handle,globus_result_t res) {
  if(res != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"listen failed: "<<GlobusResult(res)<<std::endl;
    return;
  };
  HTTP_Globus_Listener& it = *((HTTP_Globus_Listener*)callback_arg);
  it.connect(NULL);
  res=globus_io_tcp_register_listen(handle,&listen_cb,&it);
  if(res != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Failed to listen: "<<GlobusResult(res)<<std::endl;
  };
}

void HTTP_Globus_Listener::accept_cb(void *callback_arg,globus_io_handle_t *handle,globus_result_t result) {
  HTTP_Globus_Listener& it = *((HTTP_Globus_Listener*)callback_arg);
  // Extract passed credentials
  std::string subject("");
  pthread_mutex_lock(&it.cons_lock);
  std::list<cred_info_t>::iterator co = it.cons.begin();
  for(;co!=it.cons.end();++co) {
    if(co->h == handle) {
      subject=co->subject; it.cons.erase(co); break;
    };
  };
  pthread_mutex_unlock(&it.cons_lock);
  if(result != GLOBUS_SUCCESS) {
    char abuf[1024]; abuf[0]=0;
    int host[4] = {0,0,0,0};
    unsigned short port = 0;
    if(globus_io_tcp_get_remote_address(handle,host,&port) == GLOBUS_SUCCESS) {
      snprintf(abuf,sizeof(abuf)-1,"%u.%u.%u.%u:%u",
                     host[0],host[1],host[2],host[3],(unsigned int)port);
    };
    odlog(ERROR)<<"Connection from: "<<abuf<<std::endl;
    odlog(ERROR)<<"Failed to accept connection: "<<GlobusResult(result)<<std::endl;
    free(handle); // ????!!!!
    conections_processed++;
    return;
  };
  // Create connector
  HTTP_Globus_Connector* c = NULL;
  c = new HTTP_Globus_Connector(handle,it.base_url.c_str(),it.services,it.auths,it.vos);
  if(c) {
    odilog(INFO,curr_pid)<<"Connection accepted"<<std::endl;
    c->pid=(curr_pid++);
    if(!(*c)) {
      odlog(ERROR)<<"Failed to create connector"<<std::endl;
      delete c;
      conections_processed++;
    } else {
      c->identity(handle,subject.c_str());
      c->loop_in_thread();
    };
  } else {
    odlog(ERROR)<<"Failed to create connector"<<std::endl;
    free(handle);
  };
  globus_thread_blocking_will_block();
}

#if GLOBUS_IO_VERSION<4
globus_bool_t HTTP_Globus_Listener::auth_cb(void* arg,globus_io_handle_t* h,globus_result_t result,char* identity,gss_ctx_id_t* context_handle) {
#else
globus_bool_t HTTP_Globus_Listener::auth_cb(void* arg,globus_io_handle_t* h,globus_result_t result,char* identity,gss_ctx_id_t context_handle) {
#endif
  if(result != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"auth failed: "<<GlobusResult(result)<<std::endl;
    return GLOBUS_FALSE;
  };
  HTTP_Globus_Listener& it = *((HTTP_Globus_Listener*)arg);
  pthread_mutex_lock(&it.cons_lock);
  it.cons.push_back(cred_info_t(h,identity));
  pthread_mutex_unlock(&it.cons_lock);
  return GLOBUS_TRUE;
}

HTTP_Globus_Listener::HTTP_Globus_Listener(int port,globus_io_attr_t* attr,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos):HTTP_Listener(url,serv,auths,vos),port_num(port) {
  globus_result_t res;
  pthread_mutex_init(&cons_lock,NULL);

  // Prepare for authorization 
  globus_io_secure_authorization_data_t auth;
  globus_io_secure_authorization_data_initialize(&auth);
  globus_io_secure_authorization_data_set_callback(&auth,&auth_cb,this);
  globus_io_attr_set_secure_authorization_mode(attr,GLOBUS_IO_SECURE_AUTHORIZATION_MODE_CALLBACK,&auth);

  // Create listener
  res=globus_io_tcp_create_listener(&port_num,-1,attr,&s);
  if(res != GLOBUS_SUCCESS) {
    olog<<"Failed to listen on port "<<port_num<<" : "<<GlobusResult(res)<<std::endl;
    return;
  };

  // Start listening
  res=globus_io_tcp_register_listen(&s,&listen_cb,this);
  if(res != GLOBUS_SUCCESS) {
    olog<<"Failed to listen on port "<<port_num<<" : "<<GlobusResult(res)<<std::endl;
    return;
  };
  initialized=true;
}

void HTTP_Globus_Listener::connect(void* arg) {
  // accept connection
  globus_result_t res;
  globus_io_handle_t* h = 
               (globus_io_handle_t*)malloc(sizeof(globus_io_handle_t));
  memset(h,0,sizeof(globus_io_handle_t));
  res=globus_io_tcp_register_accept(&s,GLOBUS_NULL,h,&accept_cb,this);
  if(res != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Failed to accept connection: "<<GlobusResult(res)<<std::endl;
    free(h);
    conections_processed++;
    return;
  };
}

HTTP_Globus_Listener::~HTTP_Globus_Listener() {
  pthread_mutex_destroy(&cons_lock);
  // TODO: close and destroy listening handle
}

// --------------------------- SSL -----------------------

HTTP_SSL_Listener::HTTP_SSL_Listener(int port,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos):HTTP_Listener(url,serv,auths,vos),s(-1),ctx(NULL),port_num(port) {
  // Setup SSL
  SSL_METHOD *meth = SSLv23_server_method();
  int verify = SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT|SSL_VERIFY_CLIENT_ONCE;
  // int verify_depth = -1;
  char* cert_file = NULL;
  char* key_file = NULL;
  char* CApath = NULL;
  char* CAfile = NULL;
  char *cipher = NULL;
  int off = SSL_OP_NO_SSLv2 | SSL_OP_NO_TLSv1;
  int session_id = 1;
  globus_result_t res;

  res=GLOBUS_GSI_SYSCONFIG_GET_CERT_DIR(&CApath);
  if(res != GLOBUS_SUCCESS) {
    olog<<"GLOBUS_GSI_SYSCONFIG_GET_CERT_DIR failed: "<<GlobusResult(res)<<std::endl;
    return;
  };
  res=GLOBUS_GSI_SYSCONFIG_GET_HOST_CERT_FILENAME(&cert_file,&key_file);
  if(res != GLOBUS_SUCCESS) {
    olog<<"GLOBUS_GSI_SYSCONFIG_GET_HOST_CERT_FILENAME failed: "<<GlobusResult(res)<<std::endl;
    return;
  };

  SSL_load_error_strings();
  OpenSSL_add_ssl_algorithms();
  ctx=SSL_CTX_new(meth);
  if (ctx == NULL) {
    ERR_print_errors_fp(stderr); // ERR_print_errors(bio_err);
    return;
  };

  SSL_CTX_set_options(ctx,off);
  SSL_CTX_sess_set_cache_size(ctx,128);

  // Maintain CA
  if ((!SSL_CTX_load_verify_locations(ctx,CAfile,CApath)) ||
      (!SSL_CTX_set_default_verify_paths(ctx))) {
    ERR_print_errors_fp(stderr); // ERR_print_errors(bio_err);
    SSL_CTX_free(ctx);
    globus_free(cert_file);
    globus_free(key_file);
    globus_free(CApath);
    return;
  };
  // Load certificate/key
  if(SSL_CTX_use_certificate_file(ctx,cert_file,SSL_FILETYPE_PEM) <= 0) {
    olog<<"Unable to get certificate from "<<cert_file<<std::endl;
    SSL_CTX_free(ctx);
    globus_free(cert_file);
    globus_free(key_file);
    globus_free(CApath);
    return;
  };
  if (SSL_CTX_use_PrivateKey_file(ctx,key_file,SSL_FILETYPE_PEM) <= 0) {
    olog<<"Unable to get private key from "<<cert_file<<std::endl;
    SSL_CTX_free(ctx);
    globus_free(cert_file);
    globus_free(key_file);
    globus_free(CApath);
    return;
  };
  if (!SSL_CTX_check_private_key(ctx)) {
    olog<<"Private key does not match the certificate public key"<<std::endl;
    SSL_CTX_free(ctx);
    globus_free(cert_file);
    globus_free(key_file);
    globus_free(CApath);
    return;
  };
  globus_free(cert_file);
  globus_free(key_file);
  globus_free(CApath);

  SSL_CTX_set_verify(ctx,verify,verify_callback);
  SSL_CTX_set_verify_depth(ctx,100);
  SSL_CTX_set_session_id_context(ctx,(unsigned char*)&session_id,sizeof(session_id));

  // Create socket and listen
  struct sockaddr_in server;
  memset((char *)&server,0,sizeof(server));
  server.sin_family=AF_INET;
  server.sin_port=htons(port_num);
  server.sin_addr.s_addr=INADDR_ANY;
  s=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
  if(s==-1) { SSL_CTX_free(ctx); return; };
  if(bind(s,(struct sockaddr *)&server,sizeof(server)) == -1) {
    perror("bind"); close(s); SSL_CTX_free(ctx); return;
  };
  if(listen(s,-1) == -1) {
    perror("listen"); close(s); SSL_CTX_free(ctx); return;
  };
  // Run new thread to accept new connections
  globus_thread_t handle;
  int r;
  // TODO: replace with ordinary pthread
  r=globus_thread_create(GLOBUS_NULL,GLOBUS_NULL,&accept_thread,(void*)this);
  if(r != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Failed to create thread"<<std::endl;
    close(s); SSL_CTX_free(ctx); return;
  };
  initialized=true;
}

HTTP_SSL_Listener::~HTTP_SSL_Listener() {
  if(initialized) {
    if(s != -1) close(s);
    if(ctx != NULL) SSL_CTX_free(ctx);
  };
}

void* HTTP_SSL_Listener::accept_thread(void* arg) {
  HTTP_SSL_Listener& it = *((HTTP_SSL_Listener*)arg);
  static struct sockaddr_in from;
  for(;;) {
    memset((char *)&from,0,sizeof(from));
    socklen_t len = sizeof(from);
    int ret = accept(it.s,(struct sockaddr *)&from,&len);
    if(ret == -1) {
      if(errno != EINTR) {
        perror("accept");
        sleep(10);
      };
      continue;
    };
    it.connect(&ret);
  };
  return NULL;
}

void HTTP_SSL_Listener::connect(void* arg) {
  int h = *((int*)arg);
  HTTP_SSL_Connector* c = 
            new HTTP_SSL_Connector(ctx,h,base_url.c_str(),services,auths,vos);
  if(c) {
    struct sockaddr_in from;
    socklen_t from_l = sizeof(from);
    if(getpeername(h,(struct sockaddr*)&from,&from_l) == 0) {
      odilog(INFO,curr_pid)<<"Connection accepted from "<<
       (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[0])<<"."<<
       (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[1])<<"."<<
       (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[2])<<"."<<
       (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[3])<<":"<<
       ntohs(from.sin_port)<<std::endl;
    } else {
      odilog(INFO,curr_pid)<<"Connection accepted"<<std::endl;
    };
    c->pid=(curr_pid++);
    if(!(*c)) {
      odlog(ERROR)<<"Failed to create connector"<<std::endl;
      conections_processed++;
    } else {
      // c->identity(handle,subject.c_str());
      c->loop_in_thread();
    };
  } else {
    odlog(ERROR)<<"Failed to create connector"<<std::endl;
  };
}

// --------------------------- No wrapping -----------------------

HTTP_Plain_Listener::HTTP_Plain_Listener(int port,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos):HTTP_Listener(url,serv,auths,vos),s(-1),port_num(port) {
  // Create socket and listen
  struct sockaddr_in server;
  memset((char *)&server,0,sizeof(server));
  server.sin_family=AF_INET;
  server.sin_port=htons(port_num);
  server.sin_addr.s_addr=INADDR_ANY;
  s=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
  if(s==-1) { return; };
  if(bind(s,(struct sockaddr *)&server,sizeof(server)) == -1) {
    perror("bind"); close(s); return;
  };
  if(listen(s,-1) == -1) {
    perror("listen"); close(s); return;
  };
  // Run new thread to accept new connections
  globus_thread_t handle;
  int r;
  // TODO: replace with ordinary pthread
  r=globus_thread_create(GLOBUS_NULL,GLOBUS_NULL,&accept_thread,(void*)this);
  if(r != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Failed to create thread"<<std::endl;
    close(s); return;
  };
  initialized=true;
}

HTTP_Plain_Listener::~HTTP_Plain_Listener() {

}

bool HTTP_Plain_Listener::new_connection(HTTP_Plain_Connector& c,int h) {
  struct sockaddr_in from;
  socklen_t from_l = sizeof(from);
  if(getpeername(h,(struct sockaddr*)&from,&from_l) == 0) {
    odilog(INFO,curr_pid)<<"Connection accepted from "<<
     (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[0])<<"."<<
     (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[1])<<"."<<
     (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[2])<<"."<<
     (unsigned int)(((unsigned char*)(&(from.sin_addr.s_addr)))[3])<<":"<<
     ntohs(from.sin_port)<<std::endl;
  } else {
    odilog(INFO,curr_pid)<<"Connection accepted"<<std::endl;
  };
  c.pid=(curr_pid++);
  if(!c) {
    odlog(ERROR)<<"Failed to create connector"<<std::endl;
    conections_processed++;
    return false;
  };
  c.loop_in_thread();
  return true;
}

void HTTP_Plain_Listener::connect(void* arg) {
  int h = *((int*)arg);
  HTTP_Plain_Connector* c =
            new HTTP_Plain_Connector(h,base_url.c_str(),services,auths,vos);
  if(c) new_connection(*c,h);
}

void* HTTP_Plain_Listener::accept_thread(void* arg) {
  HTTP_Plain_Listener& it = *((HTTP_Plain_Listener*)arg);
  static struct sockaddr_in from;
  for(;;) {
    memset((char *)&from,0,sizeof(from));
    socklen_t len = sizeof(from);
    int ret = accept(it.s,(struct sockaddr *)&from,&len);
    if(ret == -1) {
      if(errno != EINTR) {
        perror("accept");
        sleep(10);
      };
      continue;
    };
    it.connect(&ret);
  };
  return NULL;
}

// --------------------------- GSI from CERN -----------------------

HTTP_GSSAPI_Listener::HTTP_GSSAPI_Listener(int port,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos):HTTP_Plain_Listener(port,url,serv,auths,vos) {

}

HTTP_GSSAPI_Listener::~HTTP_GSSAPI_Listener() {

}

void HTTP_GSSAPI_Listener::connect(void* arg) {
  int h = *((int*)arg);
  HTTP_GSSAPI_Connector* c =
            new HTTP_GSSAPI_Connector(h,base_url.c_str(),services,auths,vos);
  if(c) new_connection(*c,h);
}







