#include "std.h"

#include <string>
#include <list>
#include <fstream>

#include "datamove/datapoint.h"
#include "datamove/datahandle.h"
#include "misc/log_time.h"
#include "https/SRM/srm_client.h"
#include "ngdata.h"

#include <arc/url.h>
#include <arc/error.h>

/**
 * add a request to stage some files
 */
void stage_add(const std::list<std::string>& urls,
               bool dryrun,
               int recursion,
               int timeout
               );

/**
 * query a request
 */
void stage_query(std::string request_id,
                 std::string endpoint,
                 int timeout);

/**
 * cancel a request
 */
void stage_cancel(std::string request_id,
                  std::string endpoint,
                  int timeout);
 
/**
 * list all request tokens
 */
void stage_list(std::string endpoint,
                int timeout);

/**
 * list directories, recursively if required
 * TODO move this to a common area to be used by ngls -r
 */
void list_dirs(std::list<std::string>& urls,
               int recursion);

/**
 * Implementing arcstage from arccli.h
 */
void arcstage (const std::list<std::string>& urls,
               const std::string& request_id,
               const std::string& endpoint,
               bool query,
               bool cancel,
               bool list_ids,
               bool dryrun,
               int recursion,
               int timeout
               ) {

  LogTime::Active(false);
  LogTime::Level(GetNotifyLevel());

  if(query) {stage_query(request_id, endpoint, timeout);} // do query
  else if(cancel) {stage_cancel(request_id, endpoint, timeout);} // do cancel
  else if(list_ids) {stage_list(endpoint, timeout);} // do list ids
  else {stage_add(urls, dryrun, recursion, timeout);}; // add stage request

};

void stage_add(const std::list<std::string>& urls,
               bool dryrun,
               int recursion,
               int timeout
               ) {

  // take urls and resolve any wildcards so we have a single list of urls
  std::list<std::string> url_list;

  for (std::list<std::string>::const_iterator i = urls.begin(); i != urls.end(); i++) {
    std::string url = *i;
    // remove trailing slashes
    if (url.find_last_of("/") == url.length()-1) url = url.substr(0, url.length()-1);

    // TODO: support meta urls
    // if (url.compare(0, 6, "srm://") != 0) {
    if(strncasecmp(url.c_str(),"srm://",6) != 0) {
      throw ARCCLIDataError("Only SRM URLs are supported at the moment");
      return;
    };
    
    // check for wildcards
    std::string::size_type wildcard_pos = url.find("*");
    if (wildcard_pos == std::string::npos) {
      url_list.push_back(url);
      continue;
    };
      
    // wildcards only allowed at the end
    if (wildcard_pos != url.length()-1) {
      throw ARCCLIDataError("Wildcards are only supported as the final character");
      return;
    };
      
    // list directory and take entries matching wildcard
    std::string::size_type slash_pos = url.find_last_of("/");
    // check url is properly formed
    if (slash_pos == std::string::npos || slash_pos < 7) {
      throw ARCCLIDataError("Badly formed URL: "+url);
      return;
    };
    std::string dir = url.substr(0, slash_pos);

    // init objects to handle listing
    DataPoint datapoint(dir.c_str());
    if (!datapoint) {
      throw ARCCLIDataError("Unsupported URL: "+dir);
      return;
    };
    DataHandle datahandle(&datapoint);
    std::list<DataPoint::FileInfo> files;

    odlog(DEBUG)<<"Listing "<<dir<<std::endl;
    if (!datahandle.list_files(files, true, false, false)) {
      throw ARCCLIDataError("Failed to list files in dir "+dir);
      return;
    };
    
    // match to path part of original wildcard url
    std::string basename = url.substr(url.find_last_of("/")+1);
    basename.erase(basename.length()-1);
    for(std::list<DataPoint::FileInfo>::iterator j = files.begin();
        j!=files.end();++j) {
      
      // TODO: if and when dcache supports file locality in dir
      // listing, use it here
      
      if(j->name.find(basename) == 0) {
        std::string full_url = url.substr(0, url.rfind("/", wildcard_pos)) + "/" + j->name;
        odlog(DEBUG)<<"Adding matching URL "<<full_url<<std::endl;
        url_list.push_back(full_url);
        };
    };
  }; // for

  // recursively list dirs so we are only left with files
  // TODO: ls -R srm://...
  list_dirs(url_list, recursion);
  
  // remove any duplicate elements from the list
  url_list.sort();
  url_list.unique();

  if (url_list.empty()) {
    throw ARCCLIDataError("No files were found matching the given URL(s)\nRemember to use the -r option when giving directories");
    return;
  };

  // if dryrun, exit here
  if(dryrun) {
    odlog(WARNING)<<"Files to be staged:"<<std::endl;
    for (std::list<std::string>::iterator i = url_list.begin(); i != url_list.end(); i++) {
      odlog(WARNING)<<"   "<<(*i)<<std::endl;
    };
    odlog(WARNING)<<" Total: "<<url_list.size()<<std::endl;
    return;
  };

  odlog(INFO)<<"Files to be staged:"<<std::endl;
  for (std::list<std::string>::iterator i = url_list.begin(); i != url_list.end(); i++) {
    odlog(INFO)<<"   "<<(*i)<<std::endl;
  };
  odlog(INFO)<<" Total: "<<url_list.size()<<std::endl;

  // do the srmBringOnline call
  SRMClient * client = SRMClient::getInstance(url_list.front(), timeout);
  if(!client) return;
  SRMClientRequest * req = new SRMClientRequest(url_list);
  if(!req) return;

  if(client->requestBringOnline(*req) != SRM_OK) {
    throw ARCCLIDataError("Error adding the bring online request");
    return;
  };

  if (req->request_token() == "") {
    throw ARCCLIDataError("Error: request token was lost. Please try again");
  }
  else {
    // make sure this is printed
    odlog(FATAL)<<" Request id: "<<req->request_token()<<std::endl;
  };
};


void stage_query(std::string request_id,
                 std::string endpoint,
                 int timeout) {

  // do the statusOfBringOnlineRequest call
  SRMClient * client = SRMClient::getInstance(endpoint, timeout);
  if(!client) return;
  //  SRMClientRequest * req = new SRMClientRequest(std::atoi(request_id.c_str()));
  SRMClientRequest * req = new SRMClientRequest("", request_id);
  if(!req) return;

  SRMReturnCode result = client->requestBringOnlineStatus(*req);
  if(result == SRM_ERROR_SOAP) {
    throw ARCCLIDataError("Error querying status of request");
  }
  else if(result != SRM_OK) {
    throw ARCCLIDataError("Error with request id "+request_id);
  };

  // look at the status
  int success = 0;
  int staging = 0;
  int error = 0;
  SRMRequestStatus status = req->status();
    
  // list the status of each file
  std::map<std::string, SRMFileLocality> statuses = req->surl_statuses();
  for (std::map<std::string, SRMFileLocality>::iterator i = statuses.begin();
       i != statuses.end();
       ++i) {
      
    SRMFileLocality file_locality = (*i).second;
    if (file_locality == SRM_ONLINE) {
      odlog(INFO)<<"ONLINE  "<<(*i).first<<std::endl;
      success++;
    }
    else if (file_locality == SRM_NEARLINE) {
      odlog(INFO)<<"STAGING "<<(*i).first<<std::endl;
      staging++;
    }
    else if (file_locality == SRM_STAGE_ERROR) {
      odlog(INFO)<<"ERROR   "<<(*i).first<<std::endl;
      odlog(INFO)<<"  Reason: "<<req->surl_failures()[(*i).first]<<std::endl;
      error++;
    }
    else {
      odlog(INFO)<<"UNKNOWN "<<(*i).first<<std::endl;
    };
  };

  // on some conditions we don't get individual file statuses
  if (!statuses.empty())
    odlog(INFO)<<" "<<success<<" files online, "<<staging<<" files staging, "<<error<<" errors "<<std::endl;
  
  if (status == SRM_REQUEST_FINISHED_SUCCESS) 
    {odlog(WARNING)<<" Status: Success"<<std::endl;}
  else if (status == SRM_REQUEST_FINISHED_PARTIAL_SUCCESS)
    {odlog(WARNING)<<" Status: Partial Success"<<std::endl;}
  else if (status == SRM_REQUEST_FINISHED_ERROR)
    {odlog(WARNING)<<" Status: Failed"<<std::endl;}
  else if (status == SRM_REQUEST_CANCELLED)
    {odlog(WARNING)<<" Status: Cancelled"<<std::endl;}
  else 
    {odlog(WARNING)<<" Status: Pending"<<std::endl;};
};


void stage_cancel(std::string request_id,
                  std::string endpoint,
                  int timeout) {

  // call srmAbortRequest
  SRMClient * client = SRMClient::getInstance(endpoint, timeout);
  if(!client) return;
  SRMClientRequest * req = new SRMClientRequest("", request_id);
  if(!req) return;

  if(!client->abort(*req)) {
    throw ARCCLIDataError("Error aborting request");
    return;
  };


};

void stage_list(std::string endpoint,
                int timeout) {

  // call srmGetRequestTokens
  SRMClient * client = SRMClient::getInstance(endpoint, timeout);
  if(!client) return;

  std::list<std::string> tokens;

  // get the request description from the user name
  std::string desc = "";
  char * user = getlogin();
  if (user != "") {
    desc = std::string(user);
    odlog(DEBUG)<<"userRequestDescription is "<<desc<<std::endl;
  };


  if(client->getRequestTokens(tokens, desc) != SRM_OK) {
    throw ARCCLIDataError("Error listing requests");
    return;
  };

  for (std::list<std::string>::iterator i = tokens.begin(); i != tokens.end(); i++) {
    odlog(WARNING)<<*i<<std::endl;
  };

};

void list_dirs(std::list<std::string>& urls, int recursion) {

  std::list<std::string> entries;

  // make a copy of urls so we can delete entries from it
  std::list<std::string> urls_copy(urls);

  for (std::list<std::string>::iterator i = urls_copy.begin();
       i != urls_copy.end(); i++) {
    std::string url = *i;
    // do list
    // init objects to handle listing
    DataPoint datapoint(url.c_str());
    if (!datapoint) {
      throw ARCCLIDataError("Unsupported URL: "+url);
      urls.remove(url);
      continue;
    };
    DataHandle datahandle(&datapoint);
    std::list<DataPoint::FileInfo> files;

    odlog(INFO)<<"Listing "<<url<<std::endl;
    if (!datahandle.list_files(files, true, false, false)) {
      throw ARCCLIDataError("Failed to list "+url);
      urls.remove(url);
      continue;
    };
    
    // empty directory
    if (files.empty()) {
      urls.remove(url);
      continue;
    }
    
    // look at the first entry to see if we are dealing with a file or directory
    // if there are slashes in the path we have one file
    // if not we have listed a directory
    if (files.front().name.find("/") != std::string::npos) {
      std::string full_url = url.substr(0, url.find("/", 7)) + files.front().name;
      odlog(DEBUG)<<full_url<<" is a file"<<std::endl;
      entries.push_back(full_url);
      continue;
    };

    // here means we have a dir
    urls.remove(url);
    if (url.find_last_of("/") != url.length()-1) { url+="/"; };
    
    // if recursion examine the listing
    if (recursion > 0) {
      std::list<std::string> dirs;
      for(std::list<DataPoint::FileInfo>::iterator j = files.begin();
          j!=files.end();++j) {
        
        std::string full_url = url + j->name;
        if (j->type == DataPoint::FileInfo::file_type_file) {
          odlog(DEBUG)<<full_url<<" is a file"<<std::endl;
          entries.push_back(full_url);
        }
        else { // directory so add to list to call list_dirs(recursion-1) with
          odlog(DEBUG)<<full_url<<" is a dir"<<std::endl;
          dirs.push_back(full_url);
        };
      };
      if (recursion > 1) {
        list_dirs(dirs, recursion-1);
        for (std::list<std::string>::iterator i = dirs.begin(); i != dirs.end(); i++) {
          entries.push_back(*i);
        };
      };
    };
  };
  // add all entries to urls
  for (std::list<std::string>::iterator i = entries.begin(); i != entries.end(); i++) {
    urls.push_back(*i);
  };

};
