#include "../std.h"
#include <globus_common.h>
#include <globus_object.h>
#include "../misc/checksum.h"
#include "../misc/inttostring.h"
#include "../misc/stringtoint.h"
#include "../misc/url_options.h"
#include <arc/globuserrorutils.h>
#include "../misc/log_time.h"
#include "replica.h"

#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
pthread_mutex_t RCManager::sasl_mutex = PTHREAD_MUTEX_INITIALIZER;
#endif

RCLocation::RCLocation(const std::string &url_,const std::string &name_) {
  RCLocation(url_.c_str(),name_.c_str());
}

RCLocation::RCLocation(const char* url_,const char* name_):url(""),name("") {
  if(url_) url=url_;
  if(name_) name=name_;
  if(name.length() == 0) name=get_url_host(url.c_str());
};

bool RCManager::GetFile(RCFile &file) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  globus_result_t res;
  if(!inited) return false;
  char* attr_names[] = {
    "filechecksum",
    "size",
    "modifytime",
    GLOBUS_NULL
  };
  globus_replica_catalog_attribute_set_t lf_attr;
  globus_replica_catalog_attribute_set_init(&lf_attr);
  if((res=globus_replica_catalog_logicalfile_list_attributes(
    &handle,(char*)(file.name.c_str()),attr_names,&lf_attr)) != GLOBUS_SUCCESS) {
//    odlog(DEBUG)<<"globus_replica_catalog_logicalfile_list_attributes failed"<<std::endl;
//    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  globus_replica_catalog_attribute_set_first(&lf_attr);
  std::string checksum;
  std::string size;
  std::string timestamp;
  for(;globus_replica_catalog_attribute_set_more(&lf_attr);) {
    char* a_name;
    char** a_values;
    globus_replica_catalog_attribute_set_get_name(&lf_attr,&a_name);
    globus_replica_catalog_attribute_set_get_values(&lf_attr,&a_values);
    if(!strcasecmp("filechecksum",a_name)) { checksum=a_values[0]; }
    else if(!strcasecmp("size",a_name)) { size=a_values[0]; }
    else if(!strcasecmp("modifytime",a_name)) { timestamp=a_values[0]; }
    globus_replica_catalog_attribute_set_free_values(a_values);
    globus_libc_free(a_name);
    globus_replica_catalog_attribute_set_next(&lf_attr);
  };
  globus_replica_catalog_attribute_set_destroy(&lf_attr);
  file.checksum=checksum; file.checksum_valid=true;
  file.size=size; file.size_valid=true;
  file.timestamp=timestamp; file.timestamp_valid=true;
  /*
  if(file.checksum.length() != 0) {
    if(!stringtoint(file.checksum,file.checksum_i)) {
      file.checksum=""; file.checksum_valid=false;
    };
  };
  */
  if(size.length() != 0) {
    if(!stringtoint(file.size,file.size_i)) {
      file.size=""; file.size_valid=false;
    };
  };
  if(timestamp.length() != 0) {
    if(!stringtoint(file.timestamp,file.timestamp_i)) {
      file.timestamp="";
      file.timestamp_valid=false;
    };
  };
  return true;
#else
  return false;
#endif
}

/* this removes file, but does not remove location */
bool RCManager::RemoveFile(const RCFile &file) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(!inited) return false;
  char* fn[2];
  fn[0]=(char*)(file.name.c_str());
  fn[1]=NULL;
  globus_result_t res;
  if((res=globus_replica_catalog_logicalfile_delete(&handle,
        (char*)(file.name.c_str()))) != GLOBUS_SUCCESS) {
    /* maybe it is already been delete or did never exist */
//    return false;
    odlog(DEBUG)<<"globus_replica_catalog_logicalfile_delete failed: not critical"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
  };
  if((res=globus_replica_catalog_collection_delete_filenames(&handle,fn))
         != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"globus_replica_catalog_colletion_delete_filenames failed"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  return true;
#else
  return false;
#endif
}

bool RCManager::RemoveFileLocation(const RCFile &file,const std::string &host) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(!inited) return false;
  char* fn[2];
  fn[0]=(char*)(file.name.c_str());
  fn[1]=NULL;
  globus_result_t res;
  if((res=globus_replica_catalog_location_delete_filenames(&handle,
                        (char*)host.c_str(),fn)) != GLOBUS_SUCCESS) {
    globus_object_t* err = globus_error_get(res);
    char* tmp=globus_object_printable_to_string(err);
    if(strstr(tmp,"No such attribute") == NULL) { // already removed
      odlog(DEBUG)<<"globus_replica_catalog_location_delete_filenames failed"<<std::endl;
      odlog(DEBUG)<<"Globus error: "<<tmp<<std::endl;
      free(tmp); globus_object_free(err);
      return false;
    };
    // it is ok if pfn already missing - that is that we want
    free(tmp); globus_object_free(err);
  };
  return true;
#else
  return false;
#endif
}

/* this only adds possibly missing attributes */
bool RCManager::UpdateFile(const RCFile &file) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(file.timestamp_valid) globus_replica_catalog_logicalfile_add_attribute(
        &handle,
        (char*)(file.name.c_str()),
        "modifytime",(char*)(file.timestamp.c_str()));
  if(file.checksum_valid) globus_replica_catalog_logicalfile_add_attribute(
        &handle,
        (char*)(file.name.c_str()),
        "filechecksum",(char*)(file.checksum.c_str()));
  return true;
#else
  return false;
#endif
}

bool RCManager::AddFile(const RCFile &file,bool unique) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(!inited) return false;
  char* fn[2];
  fn[0]=(char*)(file.name.c_str());
  fn[1]=NULL;
  globus_result_t res;
  globus_bool_t add = (unique)?GLOBUS_FALSE:GLOBUS_TRUE;
  if((res=globus_replica_catalog_collection_add_filenames(&handle,fn,add)) 
         != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"globus_replica_catalog_collection_add_filenames failed"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  /* !!!!!!! size is necessary !!!!!!!! */
  if((res=globus_replica_catalog_logicalfile_create(&handle,
        (char*)(file.name.c_str()),file.size_i,NULL)) != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"globus_replica_catalog_logicalfile_create failed"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    globus_replica_catalog_collection_delete_filenames(&handle,fn);
    return false;
  };
  if(file.timestamp_valid) globus_replica_catalog_logicalfile_add_attribute(
        &handle,
        (char*)(file.name.c_str()),
        "modifytime",(char*)(file.timestamp.c_str()));
  if(file.checksum_valid) globus_replica_catalog_logicalfile_add_attribute(
        &handle,
        (char*)(file.name.c_str()),
        "filechecksum",(char*)(file.checksum.c_str()));
  globus_replica_catalog_logicalfile_add_attribute(&handle,
        (char*)(file.name.c_str()),
        "filetype","file");
  return true;
#else
  return false;
#endif
}

bool RCManager::AddFileLocation(const RCFile &file,
                const std::string &host,const std::string &root_dir,bool unique) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(!inited) return false;
  if(root_dir != "*") { /* means - do not try to create location */
    bool use_url = false;
    if(root_dir.find("://") != std::string::npos) use_url=true;
    odlog(DEBUG) << "trying to create location" << std::endl;
    if(use_url) {
      if(globus_replica_catalog_location_create(&handle,
          (char*)(host.c_str()),(char*)(root_dir.c_str()),
          GLOBUS_NULL,GLOBUS_NULL) != GLOBUS_SUCCESS) {
//      return false; Not an error if location already exists
      };
    }
    else {
      if(globus_replica_catalog_location_create(&handle,
          (char*)(host.c_str()),(char*)(host.c_str()),
          GLOBUS_NULL,GLOBUS_NULL) != GLOBUS_SUCCESS) {
//      return false; Not an error if location already exists
      }
      else {
        if(globus_replica_catalog_location_add_attribute(&handle,
                (char*)(host.c_str()),"path",(char*)(root_dir.c_str()))
                                                       !=GLOBUS_SUCCESS){
//        return false;
        };
      };
    };
  };
  char* fn[2];
  fn[0]=(char*)(file.name.c_str());
  fn[1]=NULL;
  globus_result_t res;
  globus_bool_t add = (unique)?GLOBUS_FALSE:GLOBUS_TRUE;
  if((res=globus_replica_catalog_location_add_filenames(&handle,
        (char*)(host.c_str()),fn,add)) != GLOBUS_SUCCESS) {
    globus_object_t* err = globus_error_get(res);
    char* tmp=globus_object_printable_to_string(err);
    if(strstr(tmp,"Type or value exists") == NULL) {
      odlog(DEBUG)<<"globus_replica_catalog_location_add_filenames failed"<<std::endl;
      odlog(DEBUG)<<"Globus error: "<<tmp<<std::endl;
      free(tmp); globus_object_free(err);
      return false; 
    };
    // it is ok if pfn already exists - that is that we want
    free(tmp); globus_object_free(err);
  };
  return true;
#else
  return false;
#endif
}

/* host is list of locations of format host|host|host... */
bool RCManager::GetLocations(const std::string &host,std::list<RCLocation> &locs,const char* file_name,bool reverse) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(!inited) return false;
  std::string url="";
  bool res=true;
  char* attr_list[3];
  attr_list[0]="uc";
  attr_list[1]="path";
  attr_list[2]=0;
  int count;
  globus_replica_catalog_entry_set_t loc_list;
  globus_replica_catalog_entry_set_init(&loc_list);
  globus_result_t gres;
  char* fnames[2] = { NULL, NULL };
  if(file_name == NULL) {
    gres=globus_replica_catalog_collection_list_locations(
                                         &handle,GLOBUS_NULL,&loc_list);
  }
  else {
    fnames[0]=(char*)file_name;
    gres=globus_replica_catalog_collection_find_locations(
          &handle,fnames,true,GLOBUS_NULL,&loc_list);
  };
  if(gres != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"globus_replica_catalog_collection_list/find_locations failed"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(gres)<<std::endl;
    res=false;
  }
  else {
      globus_replica_catalog_entry_set_first(&loc_list);
      for(;globus_replica_catalog_entry_set_more(&loc_list);) {
        char* name = NULL;
        char* uc = NULL;
        char* path = NULL;
        globus_replica_catalog_entry_set_get_name(&loc_list,&name);
        globus_replica_catalog_attribute_set_t loc_attr;
        globus_replica_catalog_attribute_set_init(&loc_attr);
        globus_replica_catalog_entry_set_get_attributes(&loc_list,&loc_attr);
        globus_replica_catalog_attribute_set_first(&loc_attr);
        for(;globus_replica_catalog_attribute_set_more(&loc_attr);) {
          char* a_name;
          char** a_values;
          globus_replica_catalog_attribute_set_get_name(&loc_attr,&a_name);
          globus_replica_catalog_attribute_set_get_values(&loc_attr,&a_values);
          if(!strcasecmp("uc",a_name)) uc=strdup(a_values[0]);
          if(!strcasecmp("path",a_name)) path=strdup(a_values[0]);
          globus_replica_catalog_attribute_set_free_values(a_values);
          globus_libc_free(a_name);
          globus_replica_catalog_attribute_set_next(&loc_attr);
        };
        if(name) odlog(DEBUG)<<"location: name: "<<name<<std::endl;
        if(uc)   odlog(DEBUG)<<"location: uc:   "<<uc<<std::endl;
        if(path) odlog(DEBUG)<<"location: path: "<<path<<std::endl;
        /* check if current location is in list and add to list in url */
        bool present = true;
        std::string options;
        if(host.length() != 0) {
          const char *pos_s = host.c_str();
          char* pos_c;
          if((pos_c=strstr(pos_s,name)) != NULL) {
            std::string::size_type pos = (pos_c-pos_s);
            if(pos > 0) if(host[pos-1] != '|') present=false;
            if(present) {
              pos+=strlen(name);
              if(pos<host.length()) {
                /* ';' to allow options in location name */
                if((host[pos]!='|') && (host[pos]!=';')) { present=false; }
                else if(host[pos]==';') {
                  pos++;
                  std::string::size_type pos_=host.find('|',pos);
                  if(pos_ == std::string::npos) pos_=host.length();
                  options=host.substr(pos,pos_-pos);
                };
              };
            };
          }
          else { present=false; };
        };
        if(present) {
          /* create url */
          if(!uc) uc=name;  /* missing uc - take name instead */
          /* check if uc is actually uc or just hostname (gdmp way) */
          url.erase();
          if(strstr(uc,"://") == NULL) { url="gsiftp://"; };
          url+=uc;
          /* add path if exists */
          if(path) {
            if(path[0] != '/') { url+="/"; };
            url+=path;
          };
          if(url[url.length()-1] != '/') url+="/";
          if(options.length() != 0) {
            odlog(DEBUG)<<"Adding options: "<<options<<std::endl;
            add_url_options(url,options.c_str(),-1);
          };
          if(uc==name) uc=NULL;
          locs.push_back(RCLocation(url.c_str(),name));
        };
        if(uc) globus_libc_free(uc);
        if(path) globus_libc_free(path);
        globus_libc_free(name);
        globus_replica_catalog_attribute_set_destroy(&loc_attr);
        globus_replica_catalog_entry_set_next(&loc_list);
      };
  };
  globus_replica_catalog_entry_set_destroy(&loc_list);
//  if(url.length() == 0) return false;
  return res;
#else
  return false;
#endif
}

bool RCManager::ListFiles(std::list<RCFile> &files) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  globus_result_t res;
  if(!inited) return false;
  char** filenames;

  if((res=globus_replica_catalog_collection_list_filenames(
                              &handle,&filenames)) != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"globus_replica_catalog_collection_list_filenames failed"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  if(filenames==NULL) return true;
  for(;*filenames;filenames++) {
    files.push_back(RCFile(*filenames));
  };
  return true;
#else
  return false;
#endif
}

RCManager::RCManager(const std::string &url,const std::string &manager,const std::string &pwd,bool secure) {
   inited=false;
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
   globus_replica_catalog_collection_handleattr_init(&attrs);
   if(secure) {
     globus_replica_catalog_collection_handleattr_set_authentication_mode(&attrs,GLOBUS_REPLICA_CATALOG_AUTHENTICATION_MODE_SECURE);
   }
   else {
     char* mn = (char*)(manager.c_str());
     char* pw = (char*)(pwd.c_str());
     if(manager.length() == 0) mn=NULL;
     if(manager.length() == 0) pw=NULL;
     globus_replica_catalog_collection_handleattr_set_authentication_mode(&attrs,GLOBUS_REPLICA_CATALOG_AUTHENTICATION_MODE_CLEARTEXT,mn,pw);
   };
   pthread_mutex_lock(&sasl_mutex);
   globus_result_t res;
   if((res=globus_replica_catalog_collection_open(&handle,&attrs,(char*)(url.c_str()))) ==
      GLOBUS_SUCCESS) {
     inited=true;
   }
   else {
     odlog(DEBUG)<<"globus_replica_catalog_collection_open failed"<<std::endl;
     odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
   };
   pthread_mutex_unlock(&sasl_mutex);
#endif
}

RCManager::~RCManager(void) {
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  if(inited) globus_replica_catalog_collection_close(&handle);
  globus_replica_catalog_collection_handleattr_destroy(&attrs); 
#endif
}

RCFile::RCFile(void) {
  name="";
  path="";
  size=""; checksum=""; timestamp=""; 
  size_i=0; /* checksum_i=0; */ timestamp_i=0;
  size_valid=false; checksum_valid=false; timestamp_valid=false;
}

/*

This constructor is disabled till situation with checksums becomes more clear

RCFile::RCFile(const std::string &name_,const std::string &path_) {
  name.resize(0); * marks undefined *
  odlog(DEBUG) <<"RCFile constructor: "<<name_<<std::endl;
  odlog(DEBUG) <<"RCFile constructor: "<<path_<<std::endl;
  * compute file information *
  struct stat st;
  if(lstat(path_.c_str(),&st) != 0) return;
  odlog(DEBUG) <<"RCFile constructor: lstat"<<std::endl;
  if(!S_ISREG(st.st_mode)) return;
  odlog(DEBUG) <<"RCFile constructor: ISREG"<<std::endl;
  size=inttostring(st.st_size); 
  timestamp=inttostring(st.st_ctime); 
  size_i=st.st_size; 
  timestamp_i=st.st_ctime;
  size_valid=true;
  timestamp_valid=true;
  char buf[1024];
  int h=open(path_.c_str(),O_RDONLY);
  if(h==-1) return;
  odlog(DEBUG) <<"RCFile constructor: open"<<std::endl;
  unsigned long ll;
  CheckSum crc;
  long l;
  crc.start();
  for(;;) {
    if((l=read(h,buf,1024)) == 0) break;
    if(l==-1) { close(h); return; };
    crc.add(buf,l); ll+=l;
  };
  odlog(VERBOSE) <<"RCFile constructor: cksum"<<std::endl;
  close(h);
  crc.end();
  {
    char buf[100];
    crc.print(buf,sizeof(buf));
    checksum=buf;
  };
  // checksum=inttostring(crc.crc());
  // checksum_i=crc.crc();
  checksum_valid=true;
  name=name_;
olog <<"RCFile constructor: setting name to: "<<name<<std::endl;
  path=path_;
}
*/

RCFile::RCFile(const std::string &name_) {
  name=name_; path.resize(0);
  size_valid=false; checksum_valid=false; timestamp_valid=false;
}

/*
RCFile::RCFile(const std::string &name_,
               ullint size_,ullint checksum_,ullint timestamp_) {
  name=name_; path.resize(0);
  size=inttostring(size_); size_valid=true; size_i=size_;
  checksum=inttostring(checksum_); checksum_valid=true; * checksum_i=checksum_; *
  timestamp=inttostring(timestamp_); timestamp_valid=true; timestamp_i=timestamp_;
}
*/

RCFile::RCFile(const std::string &name_,
               ullint size_,const char* checksum_,ullint timestamp_) {
  name=name_; path.resize(0);
  size=inttostring(size_); size_valid=true; size_i=size_;
  checksum=checksum_; checksum_valid=true; /* checksum_i=checksum_; */
  timestamp=inttostring(timestamp_); timestamp_valid=true; timestamp_i=timestamp_;
}

/*
RCFile::RCFile(const std::string &name_,
               ullint size_,bool size_valid_,
               ullint checksum_,bool checksum_valid_,
               ullint timestamp_,bool timestamp_valid_){
  name=name_;
  * no path means no such physical file *
  path.resize(0);
  size_valid=false; checksum_valid=false; timestamp_valid=false;
  if(size_valid_) {
    size=inttostring(size_); size_valid=true; size_i=size_; 
  };
  if(checksum_valid_) {
    checksum=inttostring(checksum_); checksum_valid=true; * checksum_i=checksum_; *
  };
  if(timestamp_valid_) {
    timestamp=inttostring(timestamp_); timestamp_valid=true; timestamp_i=timestamp_; 
  };
}
*/

RCFile::RCFile(const std::string &name_,
               ullint size_,bool size_valid_,
               const char* checksum_,bool checksum_valid_,
               ullint timestamp_,bool timestamp_valid_){
  name=name_;
  /* no path means no such physical file */
  path.resize(0);
  size_valid=false; checksum_valid=false; timestamp_valid=false;
  if(size_valid_) {
    size=inttostring(size_); size_valid=true; size_i=size_;
  };
  if(checksum_valid_) {
    checksum=checksum_; checksum_valid=true; /* checksum_i=checksum_; */
  };
  if(timestamp_valid_) {
    timestamp=inttostring(timestamp_); timestamp_valid=true; timestamp_i=timestamp_;
  };
}

RCFile::~RCFile(void) {
}

RCFile& RCFile::operator= (const RCFile &file) {
  name=file.name;
  path=file.path;
  size=file.size; 
  checksum=file.checksum;
  timestamp=file.timestamp; 
  size_i=file.size_i; 
  // checksum_i=file.checksum_i;
  timestamp_i=file.timestamp_i; 
  size_valid=file.size_valid; 
  checksum_valid=file.checksum_valid;
  timestamp_valid=file.timestamp_valid; 
  return *this;
}

