/*
  upload files from user interface to CE .
  result: 0 - ok, 1 - unrecoverable error, 2 - potentially recoverable.
*/
#include "std.h"

#include <string>
#include <list>
#include <vector>

#include <globus_common.h>
#include <globus_ftp_client.h>
#include <globus_object.h>
#include "files/info_types.h"
#include "files/info_files.h"
#include "misc/canonical_dir.h"
#include "misc/globus_modules.h"
#include "misc/log_time.h"
#include "transfer/ftpsubmit.h"
#include "datamove/datamovepar.h"

#include "ui_uploader.h"

/* maximum number simultaneous uploads */
#define MAX_UPLOADS 2

//static list<FileData> job_files;
//static JobId job_id;
//static int upload_done=0;

static bool make_lurl(std::string &path) {
  if(path[0] == '/') {
    path=std::string("file://")+path;
  }
  else {
    char buf[BUFSIZ];
    if(getcwd(buf,sizeof(buf)) == NULL) {
      odlog(ERROR) << "Current directory path is too long." << std::endl;
      return false;
    };
    path=std::string("file://")+buf+"/"+path;
  };
  return true;
};

#ifndef __CREATE_EXEC__
int ui_uploader(const char* resource,const char* rsl,char** job_id,const char* session_url,rsl_action act,const std::vector<std::string> &filenames,int debug_level,int timeout) {
  char* jobid = NULL;
  if(job_id) {
    if(*job_id) jobid=strdup(*job_id);
    (*job_id)=NULL;
  };
  LogTime::Level(NotifyLevel(FATAL+debug_level));
#else
int main(int argc,char** argv) {
  const char* resource = NULL;
  const char* rsl = NULL;
  char* jobid = NULL;
  const char* session_url = NULL;
  rsl_action act = RSL_ACTION_REQUEST;
  LogTime::level=2;
  int timeout = -1;
#endif
  int res=0;
  int i;
  int retries=0;
  LogTime::Active(false);

#ifdef __CREATE_EXEC__
  for(;;) {
    opterr=0;
    int optc=getopt(argc,argv,"+hcCpi:r:R:");
    if(optc == -1) break;
    switch(optc) {
      case 'h': {
        cerr<<"ui-uploader [-r resource] [-R rsl] [-c|C|p] [-i job_id] session_dir_url|session_root_url [list]"<<std::endl;
        exit(1);
      }; break;
      case 'c': {
        if(act != RSL_ACTION_REQUEST) {
          cerr<<"Only one non-default action can be specified"<<std::endl;
          exit(1);
        };
        act=RSL_ACTION_CANCEL;
      }; break;
      case 'C': {
        if(act != RSL_ACTION_REQUEST) {
          cerr<<"Only one non-default action can be specified"<<std::endl;
          exit(1);
        };
        act=RSL_ACTION_CLEAN;
      }; break;
      case 'p': {
        if(act != RSL_ACTION_REQUEST) {
          cerr<<"Only one non-default action can be specified"<<std::endl;
          exit(1);
        };
        act=RSL_ACTION_RENEW;
      }; break;
      case 'i': {
        if(jobid != NULL) {
          cerr<<"Only one id can be given"<<std::endl;
          exit(1);
        };
        jobid=strdup(optarg);
      }; break;
      case 'r': {
        if(resource != NULL) {
          cerr<<"Only one resource can be given"<<std::endl;
          exit(1);
        };
        resource=optarg;
      }; break;
      case 'R': {
        if(rsl != NULL) {
          cerr<<"Only one rsl can be specified"<<std::endl;
          exit(1);
        };
        rsl=optarg;
      }; break;
      case '?': {
        cerr<<"Unsupported option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      case ':': {
        cerr<<"Missing parameter for option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      default: {
        cerr<<"Undefined processing error"<<std::endl;
        exit(1);
      };
    };
  };
#endif
  char* rsl_act = NULL;
  char* rsl_act_templ="&(executable=/bin/echo)(savestate=yes)(action=%s)(jobid=%s)";
  if((act == RSL_ACTION_CANCEL) || (act == RSL_ACTION_CLEAN) || (act == RSL_ACTION_RENEW) || (act == RSL_ACTION_RESTART)) {
    if(!resource) { odlog(ERROR)<<"Missing resource contact string"<<std::endl; return 1; };
    if(!jobid) { odlog(ERROR)<<"Missing job identifier"<<std::endl; return 1; };
    rsl_act=(char*)malloc(strlen(rsl_act_templ)+strlen(jobid)+7+1);
    if(!rsl_act) { odlog(ERROR)<<"Memory allocation error"<<std::endl; return 1; };
    if(act == RSL_ACTION_CLEAN) {
      sprintf(rsl_act,rsl_act_templ,"clean",jobid);
    } else if(act == RSL_ACTION_RESTART) {
      sprintf(rsl_act,rsl_act_templ,"restart",jobid);
    } else if(act == RSL_ACTION_RENEW) {
      sprintf(rsl_act,rsl_act_templ,"renew",jobid);
    } else {
      sprintf(rsl_act,rsl_act_templ,"cancel",jobid);
    };
    rsl=rsl_act; 
  }
  else if(act == RSL_ACTION_REQUEST) {
#ifdef __CREATE_EXEC__
    session_url = argv[optind];
#endif
    if(!session_url) {
      odlog(ERROR)<< "Missing session directory url" << std::endl; return 1;
    };
  }
  else {
    odlog(ERROR)<<"This action is not supported."<<std::endl;
    return 1;
  };
  GlobusModuleCommon mod_common;
  if(!mod_common.active()) {
    odlog(ERROR)<<"COMMON module activation failed\n"; return 1;
  };
  if(resource) {
    if(rsl == NULL) {
      odlog(ERROR)<<"Missing RSL"<<std::endl;
      return 1;
    };
    /* submission through gsiftp */
    GlobusModuleFTPControl mod_ftp_control;
    if(!mod_ftp_control.active()) {
      odlog(ERROR)<<"FTP_CONTROL module activation failed\n"; 
      return 1;
    };
    globus_url_t url_;
    if(globus_url_parse(resource,&url_) != GLOBUS_URL_SUCCESS) {
      odlog(ERROR)<<"Bad resource url"<<std::endl;
      return 1;
    };
    char* host = url_.host;
    if(!host) {
      odlog(ERROR)<<"Missing host in resource url"<<std::endl;
      return 1;
    };
    unsigned short port = url_.port;
    if(port == 0) {
      odlog(ERROR)<<"Missing or wrong port in resource url"<<std::endl;
      return 1;
    };
    char* path = url_.url_path;
    if(!path) {
      odlog(ERROR)<<"Missing path in resource url"<<std::endl;
      return 1;
    };

    if(act == RSL_ACTION_CANCEL) {
      if(!ftpsubmit(host,port,path,ftpsubmit_cancel_req,&jobid,timeout)) {
        odlog(ERROR)<<"Cancel request failed"<<std::endl;
        return 1;
      };
    }
    else if(act == RSL_ACTION_CLEAN) {
      if(!ftpsubmit(host,port,path,ftpsubmit_clean_req,&jobid)) {
        odlog(ERROR)<<"Clean request failed"<<std::endl;
        return 1;
      };
    }
    else if(act == RSL_ACTION_RENEW) {
      if(!ftpsubmit(host,port,path,ftpsubmit_renew_req,&jobid)) {
        odlog(ERROR)<<"Proxy renew request failed"<<std::endl;
        return 1;
      };
    }
    else if(act == RSL_ACTION_RESTART) {
      if(!ftpsubmit(host,port,path,rsl,&jobid)) {
        odlog(ERROR)<<"Job restart request failed"<<std::endl;
        return 1;
      };
    }
    else {
      if(!ftpsubmit(host,port,path,rsl,&jobid)) {
        odlog(ERROR)<<"RSL submission failed"<<std::endl;
        return 1;
      };
      odlog(INFO)<<"Assigned job id: "<<jobid<<std::endl;
#ifndef __CREATE_EXEC__
      if(job_id) if((*job_id) == NULL) (*job_id)=jobid;
#endif
    };
  };
  if(act != RSL_ACTION_REQUEST) {
    if(rsl_act) free(rsl_act);
    return 0;
  };
  /* 2 arguments required: url of session dir and file containing
     list of files to upload (pairs: remote file name + local file name;
     if local file name not given it is the same as remote; if
     list is not given - read from stdin */
#ifdef __CREATE_EXEC__
  char* file_list = argv[optind+1];
#endif
  std::string base_url=session_url;
  if(base_url.length() == 0) {
    odlog(ERROR)<< "Session directory url is empty" << std::endl; return 1;
  };
  if(base_url[base_url.length()-1] == '/') {
    base_url.erase(base_url.length()-1,1);
  };
  if(jobid != NULL) base_url=base_url+"/"+jobid;
  DataMovePar mover;
  if(timeout > 0) {
    mover.set_default_min_speed(0,timeout);
    mover.set_default_max_inactivity_time(timeout);
  };
#ifdef __CREATE_EXEC__
  if(file_list) {
    if(!job_Xput_read_file(std::string(file_list),job_files)) {
      odlog(ERROR)<< "Can't read list of files from " << file_list << std::endl;
      res=1; goto exit;
    };
  }
  else {
    if(!job_Xput_read_file(job_files)) {
      odlog(ERROR)<< "Can't parse list of files" << std::endl;
      res=1; goto exit;
    };
  };
  /* push names to mover */
  for(FileData::iterator i=job_files.begin();i!=job_files.end();++i) {
    std::string source;
    if(i->lfn.length() != 0) { source=i->lfn; }
    /* pfn always starts with '/' */
    else { source=i->pfn.substr(1); };
    std::string destination = base_url+i->pfn;
    if(!make_lurl(source)) {
      odlog(INFO)<<"Failed converting local name to local url"<<std::endl;
      res=1; goto exit;
    };
    if(!mover.Add(source.c_str(),destination.c_str())) {
      odlog(ERROR)<<"Can't add data pair to list: "<<std::endl; res=1; goto exit;
    };
  };
#else
  /* push names to mover */
  for(std::vector<std::string>::const_iterator iv=filenames.begin();iv!=filenames.end();++iv) {
    std::string destination = *iv;
    std::string source; 
    canonical_dir(destination);
    ++iv; if(iv==filenames.end()) { source=""; }
    else { source = *iv; };
    if(source.length() == 0) { source=destination.substr(1); };
    destination = base_url+destination;
    if(!make_lurl(source)) {
      odlog(INFO)<< "Failed converting local name to local url" << std::endl;
      res=1; goto exit;
    };
    if(!mover.Add(source.c_str(),destination.c_str())) {
      odlog(ERROR)<<"Can't add data pair to list: "<<std::endl; res=1; goto exit;
    };
    if(iv==filenames.end()) break;
  };
#endif
  mover.secure(false);
  mover.passive(true);  /* make it work even from behind firewall */
  if(!mover.Transfer(FileCache(),UrlMap(),MAX_UPLOADS)) {
    odlog(ERROR)<<"FAILED to transfer files"<<std::endl; res=2; goto exit;
  };
  // Check if all files have been properly uploaded
  for(;;) {
    std::string source;
    std::string destination;
    DataMovePar::result dres;
    if(!mover.Get(source,destination,dres)) { // no more data
      break;
    };
    if(dres == DataMovePar::success) {  // remove this file from list
      odlog(INFO)<<"Uploaded "<<source<<" to "<<destination<<std::endl;
    }
    else {
      odlog(INFO)<<"Failed "<<source<<" to "<<destination<<std::endl;
      res=1;
    };
  };
  /* still can be uploads running - must cancel here */
  odlog(INFO) << "Leaving uploader" << std::endl;
exit:
#ifndef __CREATE_EXEC__
  if((!job_id) || (!*job_id)) { if(jobid) free(jobid); };
#else
  if(jobid) free(jobid);
#endif
  return res;
}
