/*
  Upload files specified in job.ID.output.
  result: 0 - ok, 1 - unrecoverable error, 2 - potentially recoverable,
  3 - certificate error, 4 - should retry.
*/
#include "std.h"
#include <openssl/ssl.h>
#include "misc/proxy.h"
#include "files/info_types.h"
#include "files/info_files.h"
#include "jobs/job.h"
#include "jobs/users.h"
#include "misc/delete.h"
#include "misc/log_time.h"
#include "misc/stringtoint.h"
#include "datamove/datamovepar.h"
#include "config/config_map.h"
#include "config/config_cache.h"
#include "config/environment.h"

/* check for user uploaded files every 60 seconds */
#define CHECK_PERIOD 60
/* maximum number of retries (for every source/destination) */
#define MAX_RETRIES 5
/* maximum number simultaneous uploads */
#define MAX_UPLOADS 5
/* maximum time for user files to upload (per file) */
#define MAX_USER_TIME 600

bool is_checksum(std::string str) {
  /* check if have : */
  std::string::size_type n;
  const char *s = str.c_str();
  if((n=str.find('.')) == std::string::npos) return false;
  if(str.find('.',n+1) != std::string::npos) return false;
  for(int i=0;i<n;i++) { if(!isdigit(s[i])) return false; };
  for(int i=n+1;s[i];i++) { if(!isdigit(s[i])) return false; };
  return true;
}

int clean_files(std::list<FileData> &job_files,char* session_dir) {
  std::string session(session_dir);
  if(delete_all_files(session,job_files,true) != 2) return 0;
  return 1;
}

int expand_files(std::list<FileData> &job_files,char* session_dir) {
  for(FileData::iterator i = job_files.begin();i!=job_files.end();) {
    std::string url = i->lfn;
    // Only ftp and gsiftp can be expanded to directories so far
    if(strncasecmp(url.c_str(),"ftp://",6) && 
       strncasecmp(url.c_str(),"gsiftp://",9)) { ++i; continue; };
    // user must ask explicitly
    if(url[url.length()-1] != '/') { ++i; continue; };
    std::string path(session_dir); path+="/"; path+=i->pfn;
    int l = strlen(session_dir) + 1;
    DIR *dir = opendir(path.c_str());
    if(dir) {
      struct dirent file_;
      struct dirent *file;
      for(;;) {
        readdir_r(dir,&file_,&file);
        if(file == NULL) break;
        if(!strcmp(file->d_name,".")) continue;
        if(!strcmp(file->d_name,"..")) continue;
        std::string path_ = path; path_+="/"; path+=file->d_name;
        struct stat st;
        if(lstat(path_.c_str(),&st) != 0) continue; // do not follow symlinks
        if(S_ISREG(st.st_mode)) {
          std::string lfn = url+file->d_name;
          job_files.push_back(FileData(path_.c_str()+l,lfn.c_str()));
        } else if(S_ISDIR(st.st_mode)) {
          std::string lfn = url+file->d_name+"/"; // cause recursive search
          job_files.push_back(FileData(path_.c_str()+l,lfn.c_str()));
        };
      };
      i=job_files.erase(i);
    } else {
      ++i;
    }; 
  };
}

int main(int argc,char** argv) {
  int res=0;
  int n_threads = 1;
  int n_files = MAX_UPLOADS;
  LogTime::Active(false);
  LogTime::Level(DEBUG);
  /* used to find caches used by this user */
  std::string file_owner_username = "";
  uid_t file_owner = 0;
  gid_t file_group = 0;
  std::vector<std::string> caches;
  std::vector<std::string> remote_caches;
  bool use_conf_cache = false;
  unsigned long long int min_speed = 0;
  time_t min_speed_time = 300;
  unsigned long long int min_average_speed = 0;
  time_t max_inactivity_time = 300;
  bool secure = true;
  bool passive = false;
  bool userfiles_only = false;
  std::string failure_reason("");
  // process optional arguments
  for(;;) {
    opterr=0;
    int optc=getopt(argc,argv,"+hclpZfn:t:u:U:s:S:a:i:d:");
    if(optc == -1) break;
    switch(optc) {
      case 'h': {
        olog<<"Usage: uploader [-hclpZf] [-n files] [-t threads] [-U uid]"<<std::endl;
        olog<<"          [-u username] [-s min_speed] [-S min_speed_time]"<<std::endl;
        olog<<"          [-a min_average_speed] [-i min_activity_time]"<<std::endl;
        olog<<"          [-d debug_level] job_id control_directory"<<std::endl;
        olog<<"          session_directory [cache options]"<<std::endl; 
        exit(1);
      }; break;
      case 'c': {
        secure=false;
      }; break;
      case 'f': {
        use_conf_cache=true;
      }; break;
      case 'l': {
        userfiles_only=true;
      }; break;
      case 'p': {
        passive=true;
      }; break;
      case 'Z': {
        central_configuration=true;
      }; break;
      case 'd': {
        LogTime::Level(NotifyLevel(atoi(optarg)));
      }; break;
      case 't': {
        n_threads=atoi(optarg);
        if(n_threads < 1) {
          olog<<"Wrong number of threads"<<std::endl; exit(1);
        };
      }; break;
      case 'n': {
        n_files=atoi(optarg);
        if(n_files < 1) {
          olog<<"Wrong number of files"<<std::endl; exit(1);
        };
      }; break;
      case 'U': {
        unsigned int tuid;
        if(!stringtoint(std::string(optarg),tuid)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        struct passwd pw_;
        struct passwd *pw;
        char buf[BUFSIZ];
        getpwuid_r(tuid,&pw_,buf,BUFSIZ,&pw);
        if(pw == NULL) {
          olog<<"Wrong user name"<<std::endl; exit(1);
        };
        if(pw->pw_name) file_owner_username=pw->pw_name;
        file_owner=pw->pw_uid;
        file_group=pw->pw_gid;
        if((getuid() != 0) && (getuid() != file_owner)) {
          olog<<"Specified user can't be handled"<<std::endl; exit(1);
        };
      }; break;
      case 'u': {
        struct passwd pw_;
        struct passwd *pw;
        char buf[BUFSIZ];
        getpwnam_r(optarg,&pw_,buf,BUFSIZ,&pw);
        if(pw == NULL) {
          olog<<"Wrong user name"<<std::endl; exit(1);
        };
        if(pw->pw_name) file_owner_username=pw->pw_name;
        file_owner=pw->pw_uid;
        file_group=pw->pw_gid;
        if((getuid() != 0) && (getuid() != file_owner)) {
          olog<<"Specified user can't be handled"<<std::endl; exit(1);
        };
      }; break;
      case 's': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_speed=tmpi;
      }; break;
      case 'S': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_speed_time=tmpi;
      }; break;
      case 'a': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_average_speed=tmpi;
      }; break;
      case 'i': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        max_inactivity_time=tmpi;
      }; break;
      case '?': {
        olog<<"Unsupported option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      case ':': {
        olog<<"Missing parameter for option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      default: {
        olog<<"Undefined processing error"<<std::endl;
        exit(1);
      };
    };
  };
  // process required arguments
  char* id = argv[optind+0];
  if(!id) { olog << "Missing job id" << std::endl; return 1; };
  char* control_dir = argv[optind+1];
  if(!control_dir) { olog << "Missing control directory" << std::endl; return 1; };
  char* session_dir = argv[optind+2];
  if(!session_dir) { olog << "Missing session directory" << std::endl; return 1; };
  
  // Initialize SSL - hack till official Globus initializes it properly
  SSL_library_init();
  // prepare Job and User descriptions (needed for substitutions in cache dirs)
  JobDescription desc(id,session_dir);
  uid_t uid;
  gid_t gid;
  if(file_owner != 0) { uid=file_owner; }
  else { uid= getuid(); };
  if(file_group != 0) { gid=file_group; }
  else { gid= getgid(); };
  desc.set_uid(uid,gid);
  JobUser user(uid);
  user.SetControlDir(control_dir);
  user.SetSessionRoot(session_dir);
  
  // if u or U option not set, use our username
  if (file_owner_username == "") {
    struct passwd pw_;
    struct passwd *pw;
    char buf[BUFSIZ];
    getpwuid_r(getuid(),&pw_,buf,BUFSIZ,&pw);
    if(pw == NULL) {
      olog<<"Wrong user name"<<std::endl; exit(1);
    }
    if(pw->pw_name) file_owner_username=pw->pw_name;
  }
  
  if(use_conf_cache) {
    // use cache dir(s) from conf file
    try {
      CacheConfig * cache_config = new CacheConfig(file_owner_username);
      std::list<std::string> conf_caches = cache_config->getCacheDirs();
      // add each cache to our list
      for (std::list<std::string>::iterator i = conf_caches.begin(); i != conf_caches.end(); i++) {
        user.substitute(*i);
        caches.push_back(*i);
      }
      std::list<std::string> remote_conf_caches = cache_config->getRemoteCacheDirs();
      // add each cache to our list
      for (std::list<std::string>::iterator i = remote_conf_caches.begin(); i != remote_conf_caches.end(); i++) {
        user.substitute(*i);
        remote_caches.push_back(*i);
      }
    }
    catch (CacheConfigException e) {
      olog<<"Error with cache configuration: "<<e.what()<<std::endl;
      olog<<"Cannot clean up any cache files"<<std::endl;
    }
  }
  else {
    if(argv[optind+3]) {
      std::string cache_path = argv[optind+3];
      if(argv[optind+4]) cache_path += " "+std::string(argv[optind+4]);
      caches.push_back(cache_path);
    };
  }
  LogTime::Active(true);
  if(min_speed != 0) { olog<<"Minimal speed: "<<min_speed<<" B/s during "<<min_speed_time<<" s"<<std::endl; };
  if(min_average_speed != 0) { olog<<"Minimal average speed: "<<min_average_speed<<" B/s"<<std::endl; };
  if(max_inactivity_time != 0) { olog<<"Maximal inactivity time: "<<max_inactivity_time<<" s"<<std::endl; }; 

  prepare_proxy();

  if(n_threads > 10) {
    olog<<"Won't use more than 10 threads"<<std::endl;
    n_threads=10;
  };
  UrlMapConfig url_map;
  olog<<"Uploader started"<<std::endl;

  FileCache * cache;
  if(!caches.empty()) {
    try {
      cache = new FileCache(caches,remote_caches,std::string(id),uid,gid);
    } catch (FileCacheException e) {
      olog << "Error creating cache: " << e.what() << std::endl;
      exit(1);
    }
  }
  else {
    // if no cache defined, use null cache
    cache = new FileCache();
  }

  // get the list of input files
  DataMovePar mover;
  mover.secure(secure);
  mover.passive(passive);
  if(min_speed != 0)
    mover.set_default_min_speed(min_speed,min_speed_time);
  if(min_average_speed != 0)
    mover.set_default_min_average_speed(min_average_speed);
  if(max_inactivity_time != 0)
    mover.set_default_max_inactivity_time(max_inactivity_time);
  std::list<FileData> job_files;
  bool all_data = false;
  bool transfered = true;
  bool credentials_expired = false;
  FileData::iterator next_di;
  
  if(!job_output_read_file(desc.get_id(),user,job_files)) {
    olog << "WARNING: Can't read list of output files - whole output will be removed" << std::endl;
  };
  // add any output files dynamically added by the user during the job
  for(next_di = job_files.begin(); next_di != job_files.end() ; ++next_di) {
    if(next_di->pfn.find("@") == 1) { // GM puts a slash on the front of the local file
      std::string outputfilelist = session_dir + std::string("/") + next_di->pfn.substr(2);
      olog << "Reading output files from user generated list " << outputfilelist << std::endl;
      if (!job_Xput_read_file(outputfilelist, job_files)) {
        olog << "Error reading user generated output file list in " << outputfilelist << std::endl; res=1; goto exit;
      }
    }
  }
  // remove dynamic output file lists from the files to upload
  next_di = job_files.begin();
  while (next_di != job_files.end()) {
    if(next_di->pfn.find("@") == 1) next_di = job_files.erase(next_di);
    else next_di++;
  }

  // remove bad files
  if(clean_files(job_files,session_dir) != 0) {
    failure_reason+="Internal error in uploader\n";
    olog << "Can't remove junk files from session dir " << session_dir << std::endl; res=1; goto exit;
  };
  expand_files(job_files,session_dir);
  // initialize structures to handle download
  /* TODO: add threads=# to all urls if n_threads!=1 */
  desc.GetLocalDescription(user);

  // Push data transfer pair into mover
  for(next_di=job_files.begin();next_di!=job_files.end();++next_di) {
    if(next_di->lfn.find(":") != std::string::npos) { /* is it lfn ? */
      /* have place and file to download */
      /* define place to store */
      std::string stdlog;
      std::string source;
      JobLocalDescription* local = desc.get_local();
      if(local) stdlog=local->stdlog;
      if(stdlog.length() > 0) stdlog="/"+stdlog+"/";
      if((stdlog.length() > 0) &&
         (strncmp(stdlog.c_str(),next_di->pfn.c_str(),stdlog.length()) == 0)) {
        stdlog=next_di->pfn.c_str()+stdlog.length();
        source=std::string("file://")+control_dir+"/job."+id+"."+stdlog;
      } else {
        source=std::string("file://")+session_dir+next_di->pfn;
      };
      std::string destination = next_di->lfn;
      if(strncasecmp(destination.c_str(),"file://",7) == 0) {
        failure_reason+=std::string("User requested to store output locally ")+destination.c_str()+"\n";
        olog<<"FATAL ERROR: local destination for uploader"<<destination<<std::endl; res=1; goto exit;
      };
      if(!mover.Add(source.c_str(),destination.c_str())) {
        failure_reason+="Internal error in uploader\n";
        olog<<"Can't add data pair to list: "<<std::endl; res=1; goto exit;
      };
    };
  };
  if(!userfiles_only) {
    // Start transfer 
    if(!mover.Transfer(*cache,url_map,n_files)) {
      failure_reason+="Internal error in uploader\n";
      odlog(ERROR)<<"FAILED to start transfering files"<<std::endl; res=2; goto exit;
    };
    // Check if all files have been properly uploaded
    for(next_di=job_files.begin();next_di!=job_files.end();) {
      if(next_di->lfn.find(":") != std::string::npos) { /* is it lfn ? */
        // Files should be of same order as when pushed. Error if not.
        if(all_data) {
          failure_reason+="Internal error in uploader\n";
          odlog(ERROR)<<"Data structure was destroyed."<<std::endl; res=2; goto exit;
        };
        std::string src = std::string("file://") + session_dir + next_di->pfn;
        std::string source;
        std::string destination;
        DataMovePar::result dres;
        if(!mover.Get(source,destination,dres)) { // no more data
          all_data=true; ++next_di; continue;
        };
        if(src != source) {
          failure_reason+="Internal error in uploader\n";
          odlog(ERROR)<<"Data structure was destroyed."<<std::endl; res=2; goto exit;
        };
        if(dres == DataMovePar::success) {  // remove this file from list
          next_di=job_files.erase(next_di);
          odlog(ERROR)<<"Uploaded "<<destination<<std::endl;
        }
        else {
          failure_reason+="Output file: "+next_di->lfn+" - "+
                             DataMove::get_result_string(dres)+"\n";
          if(dres == DataMove::credentials_expired_error) credentials_expired=true;
          transfered=false; ++next_di;
          odlog(ERROR)<<"Failed to upload "<<destination<<std::endl;
        };
      }
      else { ++next_di; };
    };
    if(!transfered) {
      odlog(INFO)<<"Some uploads failed"<<std::endl; res=2;
      if(credentials_expired) res=3;
    };
    /* all files left should be kept */
    for(next_di=job_files.begin();next_di!=job_files.end();) {
      next_di->lfn=""; ++next_di;
    };
    if(!job_output_write_file(desc,user,job_files)) {
      olog << "WARNING: Failed writing changed output file." << std::endl;
    };
  };
exit:
  // release input files used for this job
  cache->release();
  olog << "Leaving uploader ("<<res<<")"<<std::endl;
  // clean uploaded files here 
  clean_files(job_files,session_dir);
  remove_proxy();
  if(res != 0) {
    job_failed_mark_add(desc,user,failure_reason);
  };
  return res;
}
