#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <vector>
#include <string>
#include <fstream>

#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include "Preferences.h"
#include "DateTime.h"
#include "MdsQuery.h"
#include "Xrsl.h"
#include "Target.h"
#include "Environment.h"
#include "JobSubmission.h"

#define __UI_LIBRARY__
#include "ui_uploader.h"

int JobSubmission (std::vector <Cluster> & clusterlist, const std::string & oldcluster,
		   const Xrsl & xrsl, const std::string & joblist,
		   const bool dryrun, const bool dumpxrsl, const int timeout,
		   const int debug, std::string* jobid_return) {

  static bool init = false;
  if (!init) {
    srand (time (NULL));
    init = true;
  }

  std::vector <Target> targetlist;

  for (std::vector <Cluster>::iterator cli = clusterlist.begin();
       cli != clusterlist.end(); cli++) {

    // skip the old cluster if we are resubmitting
    if (oldcluster == cli->GetName()) continue;

    for (std::vector <Queue>::iterator qli = cli->queues.begin();
	 qli != cli->queues.end(); qli++) {

      Target target (&*cli, &*qli);

      if (debug) {
	std::cout << "Cluster: " << target.cluster->GetLongName() << std::endl;
	std::cout << "Queue: " << target.queue->GetName() << std::endl;
      }

      if (target.queue->GetStatus() != "active") {
	if (debug) std::cout << "Queue rejected due to status: "
			     << target.queue->GetStatus() << std::endl;
	continue;
      }

      if (target.queue->GetUserFreeCpus() == UNDEFINED) {
	if (debug) std::cout << "Queue rejected because user not authorized"
			     << std::endl;
	continue;
      }

      if (!target.cluster->HaveKeys ()) {
	if (debug) std::cout << "Queue rejected because of missing public keys"
			     << std::endl;
	continue;
      }

      if (target.queue->GetMaxQueuable() != UNDEFINED &&
	  target.queue->GetQueued() >= target.queue->GetMaxQueuable()) {
	if (debug) std::cout << "Queue rejected because the maximum number "
			     << "of queued jobs is reached" << std::endl;
	continue;
      }

      if (target.GetTotalCpus() < 1) {
	if (debug) std::cout << "Queue rejected because it does not have any CPUs"
			     << std::endl;
	continue;
      }

      std::string failattr;
      if (xrsl.Test (target, failattr)) return 1;
      if (!target) {
	if (debug) std::cout << "Queue rejected because it does not match "
			     << "the XRSL specification (" << failattr << ")"
			     << std::endl;
	continue;
      }

      long int cputime;
      if (target.GetCpuTime (&cputime)) return 1;
      if ((target.queue->GetMaxCpuTime() != UNDEFINED &&
	   target.queue->GetMaxCpuTime() < cputime) ||
	  (target.queue->GetMinCpuTime() != UNDEFINED &&
	   target.queue->GetMinCpuTime() > cputime)) {
	if (debug) std::cout << "Queue rejected because its cputime limits "
			     << "does not match the XRSL description" << std::endl;
	continue;
      }

      long int walltime;
      if (target.GetWallTime (&walltime)) return 1;
      if ((target.queue->GetMaxCpuTime() != UNDEFINED &&
	   target.queue->GetMaxCpuTime() < walltime) ||
	  (target.queue->GetMinCpuTime() != UNDEFINED &&
	   target.queue->GetMinCpuTime() > walltime)) {
	if (debug) std::cout << "Queue rejected because its walltime limits "
			     << "does not match the XRSL description" << std::endl;
	continue;
      }

      if (target.CalculateNeededFileSizes (timeout)) return 1;

      if (target.GetRemoteSize() == UNDEFINED) {
	if (debug) std::cout << "Queue rejected because no valid location was "
			     << "found for the inputfiles" << std::endl;
	continue;
      }

      if (debug > 1) std::cout << "Total size of remote files: "
			       << target.GetRemoteSize() << " bytes" << std::endl;
      if (debug > 1) std::cout << "Total size of local files: "
			       << target.GetLocalSize() << " bytes" << std::endl;

      if (target.cluster->GetCacheFree() == UNDEFINED) {
	if (debug > 1) std::cout << "Total size of files: "
				 << target.GetNeededTotalSize()
				 << " bytes" << std::endl;

	if (target.GetNeededTotalSize() > target.queue->GetUserDiskSpace()) {
	  if (debug) std::cout << "Queue rejected due to insufficient disk space"
			       << std::endl;
	  continue;
	}
      }
      else {
	if (debug > 1) std::cout << "Total size of files to be cached: "
				 << target.GetNeededCacheSize()
				 << " bytes" << std::endl;
	if (debug > 1) std::cout << "Total size of files in session directory: "
				 << target.GetNeededSessDirSize()
				 << " bytes" << std::endl;

	if (target.GetNeededCacheSize() > target.cluster->GetCacheFree()) {
	  if (debug) std::cout << "Queue rejected due to insufficient cache space"
			       << std::endl;
	  continue;
	}
	if (target.GetNeededSessDirSize() > target.queue->GetUserDiskSpace()) {
	  if (debug) std::cout << "Queue rejected due to insufficient disk space"
			       << std::endl;
	  continue;
	}
      }

      targetlist.push_back (target);
      if (debug)
	std::cout << "Queue accepted as possible submission target with "
		  << target.queue->GetUserFreeCpus (walltime) << " free CPUs"
		  << std::endl;
    }
  }

  bool submitted = false;
  while (!submitted) {

    int freecpus = 0;
    long long int minremotesize = UNDEFINED;
    long long int minlocalsize = UNDEFINED;

    for (std::vector <Target>::iterator vti = targetlist.begin();
	 vti != targetlist.end(); vti++) {

      if (vti->queue->GetMaxRunning() != UNDEFINED &&
	  vti->queue->GetRunning() >= vti->queue->GetMaxRunning()) continue;

      long int time;
      if (vti->GetWallTime (&time)) return 1;
      int userfreecpus = vti->queue->GetUserFreeCpus (time);

      int count;
      if (vti->xrsl.GetCount (&count)) return 1;

      if (userfreecpus < count) continue;

      if (minremotesize == UNDEFINED ||
	  vti->GetRemoteSize() < minremotesize) {
	minremotesize = vti->GetRemoteSize();
	minlocalsize = vti->GetLocalSize();
	freecpus = userfreecpus;
      }
      else if (vti->GetRemoteSize() == minremotesize) {
	if (vti->GetLocalSize() < minlocalsize) {
	  minlocalsize = vti->GetLocalSize();
	  freecpus = userfreecpus;
	}
	else
	  freecpus += userfreecpus;
      }
    }

    std::vector <Target>::iterator target = targetlist.end();

    if (freecpus > 0) {
      float rn = rand() / (RAND_MAX + 1.0) * freecpus;
      int counter = 0;
      bool found = false;

      for (std::vector <Target>::iterator vti = targetlist.begin();
	   !found && (vti != targetlist.end()); vti++) {

	if (vti->queue->GetMaxRunning() != UNDEFINED &&
	    vti->queue->GetRunning() >= vti->queue->GetMaxRunning()) continue;

	long int time;
	if (vti->GetWallTime (&time)) return 1;
	int userfreecpus = vti->queue->GetUserFreeCpus (time);

	int count;
	if (vti->xrsl.GetCount (&count)) return 1;

	if (userfreecpus < count) continue;

	if (vti->GetRemoteSize() == minremotesize &&
	    vti->GetLocalSize() == minlocalsize) {
	  counter += userfreecpus;
	  if (rn < counter) {
	    target = vti;
	    found = true;
	  }
	}
      }
    }

    else {
      if (debug) {
	std::cout << "No cluster with enough free CPUs found" << std::endl;
	std::cout << "Submitting using fallback resource search algorithm" << std::endl;
      }

      for (std::vector <Target>::iterator vti = targetlist.begin();
	   vti != targetlist.end(); vti++) {
	if (target == targetlist.end() ||
	    (float) vti->GetQueued() / vti->GetTotalCpus() <
	    (float) target->GetQueued() / target->GetTotalCpus())
	  target = vti;
      }
    }

    if (target == targetlist.end()) {
      std::cerr << "No suitable cluster matching the specified criteria found"
		<< std::endl;
      return 1;
    }

    if (debug) {
      std::cout << target->cluster->GetLongName() << " selected" << std::endl;
      std::cout << "queue " << target->queue->GetName() << " selected" << std::endl;
    }

    long int cputime;
    if (target->GetCpuTime (&cputime)) return 1;
    long int walltime;
    if (target->GetWallTime (&walltime)) return 1;

    target->xrsl.AddSimpleRelation ("queue", target->queue->GetName());
    target->xrsl.AddSimpleRelation ("action", "request");
    target->xrsl.AddSimpleRelation ("savestate", "yes");
    target->xrsl.AddSimpleRelation ("clientxrsl", xrsl.str());

    if (dryrun) target->xrsl.AddSimpleRelation ("dryrun", "yes");

    if ((target->xrsl.FixJoin()) ||
	(target->xrsl.FixInOut ("executable", "inputfiles")) ||
	(target->xrsl.FixInOut ("stdin", "inputfiles")) ||
	(target->xrsl.FixInOut ("stdout", "outputfiles")) ||
	(target->xrsl.FixInOut ("stderr", "outputfiles")) ||
	(target->xrsl.FixInOut ("gmlog", "outputfiles")) ||
	(target->xrsl.FixExecs()) ||
	(target->xrsl.FixTime ("starttime")) ||
	(target->xrsl.FixPeriod ("lifetime")) ||
	(target->xrsl.RemoveRelation ("cputime")) ||
	(target->xrsl.RemoveRelation ("walltime")) ||
	(target->xrsl.RemoveRelation ("gridtime")) ||
	(target->xrsl.RemoveRelation ("benchmarks")))
      return 1;

    // For historical (i.e. globus gram) reasons pre-0.5.26 servers expect a
    // fake executable attribute and the real executable as the first argument
    if (target->cluster->GetServerVersion() &&
	target->cluster->GetServerVersion() < EnvVersion (0, 5, 26))
      if (target->xrsl.FixExec())
	return 1;

    if (cputime != UNDEFINED) {
#ifdef HAVE_SSTREAM
      std::stringstream ss;
#else
      std::strstream ss;
#endif
      if (target->cluster->GetServerVersion() &&
	  target->cluster->GetServerVersion() < EnvVersion (0, 5, 27))
	// use minutes instead of seconds for pre-0.5.27 servers
	ss << cputime / 60;
      else
	ss << cputime;
#ifndef HAVE_SSTREAM
      ss << ends;
#endif
      target->xrsl.AddSimpleRelation ("cputime", ss.str());
#ifndef HAVE_SSTREAM
      ss.freeze (false);
#endif
    }

    if (walltime != UNDEFINED) {
      if (!target->cluster->GetServerVersion() ||
	  target->cluster->GetServerVersion() >= EnvVersion (0, 5, 27)) {
#ifdef HAVE_SSTREAM
	std::stringstream ss;
#else
	std::strstream ss;
#endif
	ss << walltime;
#ifndef HAVE_SSTREAM
	ss << ends;
#endif
	target->xrsl.AddSimpleRelation ("walltime", ss.str());
#ifndef HAVE_SSTREAM
	ss.freeze (false);
#endif
      }
    }

    char hostname[1024];
    globus_libc_gethostname (hostname, 1024);
    target->xrsl.AddSimpleRelation ("hostname", hostname);
    target->xrsl.AddSimpleRelation ("clientsoftware",
				    std::string ("nordugrid-") + VERSION);

    std::string jobname;
    if (target->xrsl.GetJobName (jobname)) return 1;

    std::vector <std::string> filenames;
    if (target->xrsl.PrepareUpload (filenames)) return 1;

    if ((target->xrsl.Collect (*target, "runtimeenvironment")) ||
	(target->xrsl.Collect (*target, "middleware")) ||
	(target->xrsl.Collect (*target, "opsys"))) return 1;

    if (debug > 1) {
      std::cout << "Parsed output XRSL:" << std::endl;
      target->xrsl.Print();
    }

    // dump xrsl

    if (dumpxrsl) {
      std::cout << target->xrsl.str() << std::endl;
      submitted = true;
      continue;
    }

    // submit the jobs

    char * shortjobid = NULL;

    int err = ui_uploader (target->cluster->GetContact().c_str(),
			   target->xrsl.str().c_str(), &shortjobid,
			   target->cluster->GetContact().c_str(),
			   RSL_ACTION_REQUEST, filenames, debug, timeout);

    if (err || !shortjobid) {
      std::cout << "Job submission failed to queue " << target->queue->GetName()
		<< " at " << target->cluster->GetLongName() << std::endl;

      if (shortjobid) {
	std::cout << "Deleting partially submitted job" << std::endl;
	std::vector <std::string> filenames;
	ui_uploader (target->cluster->GetContact().c_str(), NULL,
			       &shortjobid, NULL, RSL_ACTION_CLEAN,
			       filenames, debug, timeout);
	free (shortjobid);
      }

      std::cout << "Trying remaining queues" << std::endl;
      targetlist.erase (target);
      continue;
    }

    if (target->Accept()) return 1;

    std::string jobid = target->cluster->GetContact() + '/' + shortjobid;

    free (shortjobid);

    std::string oldfilename = GetEnv ("HOME");
    oldfilename.append ("/.ngjobs");
    std::ifstream oldngjobs (oldfilename.c_str());
    std::string newfilename = oldfilename + ".tmp";
    std::ofstream newngjobs (newfilename.c_str());
    std::string jobidname;
    bool written = false;
    while (getline (oldngjobs, jobidname)) {
      int pos = jobidname.find('#');
      std::string jobname_ = jobidname.substr (pos + 1);
      if (!written && jobname_ > jobname) {
	newngjobs << jobid << '#' << jobname << std::endl;
	written = true;
      }
      newngjobs << jobidname << std::endl;
    }
    if (!written) newngjobs << jobid << '#' << jobname << std::endl;
    oldngjobs.close();
    newngjobs.close();
    remove (oldfilename.c_str());
    rename (newfilename.c_str(), oldfilename.c_str());

    std::string histfilename = GetEnv ("HOME");
    histfilename.append ("/.nghistory");
    std::ofstream nghist (histfilename.c_str(), std::ios::app);
    nghist << TimeStamp() << "  " << jobid << std::endl;
    nghist.close();

    if (!joblist.empty()) {
      std::ofstream jobs (joblist.c_str(), std::ios::app);
      jobs << jobid << std::endl;
      jobs.close();
    }

    std::cout << "Job submitted with jobid " << jobid << std::endl;
	if (jobid_return) {
		*jobid_return = jobid.c_str();
	}

    submitted = true;
  }

  return 0;
}
