#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <string>
#include <fstream>
#include <vector>
#include <map>

#include <globus_common.h>
#include <globus_rsl.h>
#include <globus_gsi_credential.h>
#include <globus_gsi_system_config.h>
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
#include <globus_replica_catalog.h>
#endif
#include <globus_ftp_client.h>
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
extern "C" {
#include <globus_rls_client.h>
}
#endif

#include "Preferences.h"

std::string GetEnv (const std::string & env) {

  const char * temp = getenv (env.c_str());
  return temp ? temp : "";
}

std::string GetDef (const std::string & def) {

  static bool doinit = true;
  static std::map <std::string, std::string> defaults;

  if (doinit) {
    std::string conffilename = GetEnv ("HOME");
    conffilename.append ("/.ngrc");
    std::ifstream conffile (conffilename.c_str());
    std::string line;
    while (getline (conffile, line)) {
      if (line.empty()) continue;
      if (line[0] == '#') continue;
      std::string::size_type pos = line.find ('=');
      if (pos == std::string::npos)
	std::cerr << "Warning: Syntax error in " << conffilename << ": " << line
		  << std::endl;
      else {
	int quot = 0;
	if (line[pos + 1] == '"' &&
	    line[line.length() - 1] == '"') quot = 1;
	if (line[pos + 1] == '\'' &&
	    line[line.length() - 1] == '\'') quot = 1;
	defaults [line.substr (0, pos)] =
	  line.substr (pos + 1 + quot, line.length() - pos - 1 - 2 * quot);
      }
    }
    conffile.close();
    doinit = false;
  }

  if (defaults.find (def) != defaults.end())
    return defaults [def];
  else
    return "";
}

std::vector <std::string> ResolveAliases (const std::vector <std::string> & clusters) {

  static bool doinit = true;
  static std::map <std::string, std::vector <std::string> > alias;

  if (doinit) {
    std::string conffilename = GetEnv ("HOME");
    conffilename.append ("/.ngalias");
    std::ifstream conffile (conffilename.c_str());
    std::string line;
    while (getline (conffile, line)) {
      if (line.empty()) continue;
      if (line[0] == '#') continue;
      std::string::size_type pos = line.find ('=');
      if (pos == std::string::npos)
	std::cerr << "Warning: Syntax error in " << conffilename << ": " << line
		  << std::endl;
      else {
	std::string aliaskey = line.substr (0, pos);
	std::string aliasval = line.substr (pos + 1);
	if ((aliasval[0] == '"' && aliasval[aliasval.length() - 1] == '"') ||
	    (aliasval[0] == '\'' && aliasval[aliasval.length() - 1] == '\''))
	  aliasval = aliasval.substr (1, aliasval.length() - 2);
	std::vector <std::string> aliasvec;
	std::string::size_type pos1 = 0;
	std::string::size_type pos2 = 0;
	while (pos2 != std::string::npos) {
	  pos1 = aliasval.find_first_not_of (" \t", pos2);
	  if (pos1 == std::string::npos) break;
	  pos2 = aliasval.find_first_of (" \t", pos1);
	  std::string val;
	  if (pos2 == std::string::npos)
	    val = aliasval.substr (pos1);
	  else
	    val = aliasval.substr (pos1, pos2 - pos1);
	  if (alias.find (val) != alias.end())
	    aliasvec.insert (aliasvec.end(),
			     alias[val].begin(), alias[val].end());
	  else
	    aliasvec.push_back (val);
	}
	if (!aliasvec.empty()) alias [aliaskey] = aliasvec;
      }
    }
    conffile.close();
    doinit = false;
  }

  std::vector <std::string> resolved;

  for (std::vector <std::string>::const_iterator vsi = clusters.begin();
       vsi != clusters.end(); vsi++) {
    if (alias.find (*vsi) != alias.end())
      resolved.insert (resolved.end(), alias[*vsi].begin(), alias[*vsi].end());
    else
      resolved.push_back (*vsi);
  }

  return resolved;
}

int iGetEnv (const std::string & env) {

  std::string temp = GetEnv (env);
  if (temp.empty())
    return UNDEFINED;
  else
    return atoi (temp.c_str());
}

int iGetDef (const std::string & def) {

  std::string temp = GetDef (def);
  if (temp.empty())
    return UNDEFINED;
  else
    return atoi (temp.c_str());
}

std::vector <std::string> GetJobIDs (const std::vector <std::string> & jobs,
				     const std::vector <std::string> & joblists,
				     const std::vector <std::string> & clusterselect,
				     const std::vector <std::string> & clusterreject,
				     const std::vector <std::string> & status,
				     bool all) {

  bool allbut = false;
  if (jobs.empty() && joblists.empty() && clusterselect.empty() &&
      (!clusterreject.empty() || !status.empty())) allbut = true;

  std::vector <std::string> jobids;

  std::string filename = GetEnv ("HOME");
  filename.append ("/.ngjobs");
  std::ifstream ngjobs (filename.c_str());

  if (all || allbut || !clusterselect.empty()) {
    std::string jobidname;
    while (getline (ngjobs, jobidname)) {
      int pos = jobidname.find('#');
      std::string jobid = jobidname.substr (0, pos);
      std::string jobname = jobidname.substr (pos + 1);
      if (!all && !clusterselect.empty()) {
	int pos0 = 0;
	if (jobid.substr (0, 9) == "gsiftp://") pos0 = 9;
	std::string::size_type pos = jobid.find_first_of (":/", pos0);
	std::string clustername;
	if (pos == std::string::npos)
	  clustername = jobid.substr (pos0);
	else
	  clustername = jobid.substr (pos0, pos - pos0);
	bool found = false;
	for (std::vector <std::string>::const_iterator vsci = clusterselect.begin();
	     !found && vsci != clusterselect.end(); vsci++)
	  if (*vsci == clustername) found = true;
	if (!found) continue;
      }
      bool old = false;
      for (std::vector <std::string>::iterator vsi = jobids.begin();
	   !old && vsi != jobids.end(); vsi++)
	if (*vsi == jobid) old = true;
      if (!old) jobids.push_back (jobid);
    }
  }

  for (std::vector <std::string>::const_iterator vsci = jobs.begin();
       vsci != jobs.end(); vsci++) {
    bool found = false;
    ngjobs.clear();
    ngjobs.seekg (std::ios::beg);
    std::string jobidname;
    while (getline (ngjobs, jobidname)) {
      int pos = jobidname.find('#');
      std::string jobid = jobidname.substr (0, pos);
      std::string jobname = jobidname.substr (pos + 1);
      if (jobid == *vsci || jobname == *vsci) {
	bool old = false;
	for (std::vector <std::string>::iterator vsi = jobids.begin();
	     !old && vsi != jobids.end(); vsi++)
	  if (*vsi == jobid) old = true;
	if (!old) jobids.push_back (jobid);
	found = true;
      }
    }
    if (!found) {
      bool old = false;
      for (std::vector <std::string>::iterator vsi = jobids.begin();
	   !old && vsi != jobids.end(); vsi++)
	if (*vsi == *vsci) old = true;
      if (!old) jobids.push_back (*vsci);
    }
  }

  ngjobs.close();

  if (!joblists.empty()) {
    for (std::vector <std::string>::const_iterator vsci = joblists.begin();
	   vsci != joblists.end(); vsci++) {
      std::ifstream joblist (vsci->c_str());
      if (!joblist) std::cerr << "Warning: can't open file " << *vsci << std::endl;
      std::string jobid;
      while (getline (joblist, jobid)) {
	if (jobid.empty()) continue;
	bool old = false;
	for (std::vector <std::string>::iterator vsi = jobids.begin();
	     !old && vsi != jobids.end(); vsi++)
	  if (*vsi == jobid) old = true;
	if (!old) jobids.push_back (jobid);
      }
      joblist.close();
    }
  }

  if (!clusterreject.empty()) {
    std::vector <std::string> filteredjobids;
    for (std::vector <std::string>::iterator vsi = jobids.begin();
	 vsi != jobids.end(); vsi++) {
      int pos0 = 0;
      if (vsi->substr (0, 9) == "gsiftp://") pos0 = 9;
      std::string::size_type pos = vsi->find_first_of (":/", pos0);
      std::string clustername;
      if (pos == std::string::npos)
	clustername = vsi->substr (pos0);
      else
	clustername = vsi->substr (pos0, pos - pos0);
      bool found = false;
      for (std::vector <std::string>::const_iterator vsci = clusterreject.begin();
	   !found && vsci != clusterreject.end(); vsci++)
	if (clustername == *vsci) found = true;
      if (!found) filteredjobids.push_back (*vsi);
    }
    jobids = filteredjobids;
  }

  return jobids;
}

void RemoveJobID (const std::string & jobid) {

  std::string oldfilename = GetEnv ("HOME");
  oldfilename.append ("/.ngjobs");
  std::ifstream oldngjobs (oldfilename.c_str());
  std::string newfilename = oldfilename + ".tmp";
  std::ofstream newngjobs (newfilename.c_str());
  std::string jobidname;
  while (getline (oldngjobs, jobidname)) {
    int pos = jobidname.find('#');
    std::string jobid_ = jobidname.substr (0, pos);
    if (jobid_ != jobid) newngjobs << jobidname << std::endl;
  }
  oldngjobs.close();
  newngjobs.close();
  remove (oldfilename.c_str());
  rename (newfilename.c_str(), oldfilename.c_str());
}

void ActivateGlobus() {

  globus_module_activate (GLOBUS_COMMON_MODULE);
  globus_module_activate (GLOBUS_RSL_MODULE);
  globus_module_activate (GLOBUS_GSI_CREDENTIAL_MODULE);
  globus_module_activate (GLOBUS_GSI_SYSCONFIG_MODULE);
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  globus_module_activate (GLOBUS_REPLICA_CATALOG_MODULE);
#endif
  globus_module_activate (GLOBUS_FTP_CLIENT_MODULE);
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  globus_module_activate (GLOBUS_RLS_CLIENT_MODULE);
#endif
}

void DeactivateGlobus() {

#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  globus_module_deactivate (GLOBUS_RLS_CLIENT_MODULE);
#endif
  globus_module_deactivate (GLOBUS_FTP_CLIENT_MODULE);
#ifdef HAVE_GLOBUS_REPLICA_CATALOG_H
  globus_module_deactivate (GLOBUS_REPLICA_CATALOG_MODULE);
#endif
  globus_module_deactivate (GLOBUS_GSI_SYSCONFIG_MODULE);
  globus_module_deactivate (GLOBUS_GSI_CREDENTIAL_MODULE);
  globus_module_deactivate (GLOBUS_RSL_MODULE);
  globus_module_deactivate (GLOBUS_COMMON_MODULE);
}
